﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.CrmePerson
{
    public class CrmePersonRetrieveMultiplePostGetConsultsRunner : PluginRunner
    {
        public CrmePersonRetrieveMultiplePostGetConsultsRunner(IServiceProvider serviceProvider) : base(serviceProvider) { }

        public override string McsSettingsDebugField => "crme_person";

        private int VimtTimeout { get; set; }

        public override void Execute()
        {
            try
            {
                var qe = (QueryExpression)PluginExecutionContext.InputParameters["Query"];
                PersonSearchPluginUtilities.TryGetConditionOptionSet(qe, "cvt_IntegrationType", out var integrationTypeValue);
                if (integrationTypeValue != (int)crme_personcvt_IntegrationType.GetConsultsForPatient) //Change to Option Set
                    return;

                var patientSiteVal = string.Empty;
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_patientsiteid", out patientSiteVal);

                string patientContactId = string.Empty;
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "crme_contactid", out patientContactId);

                string providerSiteId = string.Empty;
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_providersiteid", out providerSiteId);

                string sft = string.Empty;
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_issft", out sft);

                string homeMobile = string.Empty;
                PersonSearchPluginUtilities.TryGetFilterValue(qe, "cvt_ishomemobile", out homeMobile);

                string providerStationNumber = GetStationNumberFromSiteId(providerSiteId, Logger);
                string patientStationNumber = GetStationNumberFromSiteId(patientSiteVal, Logger);

                var isSft = Convert.ToBoolean(sft);
                var isHomeMobile = Convert.ToBoolean(homeMobile);

                var proUserDuz = string.Empty;
                var vimtUrl=string.Empty;
                using (var context = new Xrm(OrganizationService))
                {
                    VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, this.GetType().Name);
                    var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                    if (vimtUrlSetting != null) vimtUrl = vimtUrlSetting.mcs_value;
                    else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");
                }
                Logger.WriteToFile("Populating Request");

                List<Guid> providerIds = new List<Guid>();
                List<Guid> patientIds = new List<Guid>();

                providerIds.Add(PluginExecutionContext.UserId);
                patientIds.Add(new Guid(patientContactId));
                var request = new GetConsultsForPatientRequest
                {
                    Debug = true,
                    ProviderIds = providerIds,
                    PatientIds = patientIds,
                    PatLoginSiteCode = patientSiteVal,
                    ProLoginSiteCode = providerSiteId,
                    SAMLToken = VistaPluginHelpers.GetUserSaml(OrganizationService, PluginExecutionContext.UserId, Logger),
                    LogRequest = true,
                    UserId = PluginExecutionContext.UserId,
                    OrganizationName = PluginExecutionContext.OrganizationName,
                    IsHomeMobile = isHomeMobile,
                    IsStoreForward = isSft,
                    PatUserDuz = VistaPluginHelpers.GetDuzFromStationCode(PluginExecutionContext.UserId, Logger, OrganizationService, patientStationNumber),
                    ProUserDuz = VistaPluginHelpers.GetDuzFromStationCode(PluginExecutionContext.UserId, Logger, OrganizationService, providerStationNumber)
                };
                Logger.WriteDebugMessage("About to send/receive message.");
                var Response = VRMRest.Utility.SendReceive<GetConsultsForPatientResponse>(new Uri(vimtUrl), MessageRegistry.GetConsultsForPatientRequest, request, null, VimtTimeout, out int lag);
                Logger.WriteDebugMessage("Finished send/receive message.");
                Response.VimtLagMs = lag;
                Logger.WriteDebugMessage("Sent Request To VIMT");
                var mappedResponse = MapResponseToCrm(Response, patientSiteVal, providerSiteId, isSft);
                Logger.WriteDebugMessage("Response received: " + Response.VimtResponse.ToString());

                ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.AddRange(mappedResponse);
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException(ex.Message);
            }
        }

        
        public List<DataModel.crme_person> MapResponseToCrm(GetConsultsForPatientResponse response, string patSiteId, string proSiteId, bool isSft)
        {
            Logger.WriteDebugMessage($"MapResponseToCrm Started. patSiteId: {patSiteId}, proSiteId:{proSiteId}, isSft: {isSft}");
            if (response == null)
                return null;
            var persons = new List<DataModel.crme_person>();
            persons.Add(new crme_person
            {
                cvt_ConsultIEN = response.PatUserDuz,
                cvt_ConsultText = response.ProUserDuz,
                crme_url = "neither",
                Id = new Guid()
            });
            var patStationCode = string.Empty;
            var proStationCode = string.Empty;
            using (var srv = new Xrm(OrganizationService))
            {
                var patSite = !string.IsNullOrEmpty(patSiteId) ? srv.mcs_siteSet.FirstOrDefault(s => s.Id == new Guid(patSiteId)) : null;
                var proSite = !string.IsNullOrEmpty(proSiteId) ? srv.mcs_siteSet.FirstOrDefault(s => s.Id == new Guid(proSiteId)) : null;
                if (patSite != null && patSite.mcs_FacilityId != null)
                {
                    var facility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == patSite.mcs_FacilityId.Id);
                    patStationCode = facility?.mcs_StationNumber ?? "";
                }
                else
                    Logger.WriteDebugMessage("Unable to get site for patient side.");
                if (proSite != null && proSite.mcs_FacilityId != null)
                {
                    var facility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == proSite.mcs_FacilityId.Id);
                    proStationCode = facility?.mcs_StationNumber ?? "";
                }
                else
                    Logger.WriteDebugMessage("Unable to get site for provider side.");
            }
            var sameFacilityCode = patStationCode == proStationCode;

            if (response.PatGetConsultsForPatientRespTaggedConsultArraysInfo != null && response.PatGetConsultsForPatientRespTaggedConsultArraysInfo.Count > 0)
            {
                var label = "patient";
                if (sameFacilityCode && !isSft)
                    label = "provider";

                var patPersons = MapResponse(response.PatGetConsultsForPatientRespTaggedConsultArraysInfo, label);
                persons.AddRange(patPersons);
            }
            if (response.ProGetConsultsForPatientRespTaggedConsultArraysInfo != null &&
                response.ProGetConsultsForPatientRespTaggedConsultArraysInfo.Count > 0)
            {
                var proPersons = MapResponse(response.ProGetConsultsForPatientRespTaggedConsultArraysInfo, "provider");
                persons.AddRange(proPersons);
            }
            Logger.WriteDebugMessage("MapResponseToCrm Ended");
            return persons;
        }
        private static string Base64Decode(string base64EncodedData)
        {
            var base64EncodedBytes = Convert.FromBase64String(base64EncodedData);
            return Encoding.UTF8.GetString(base64EncodedBytes);
        }

        public List<DataModel.crme_person> MapResponse(GetConsultsForPatientRespTaggedConsultArrays responseConsultsArray, string side)
        {
            Logger.WriteDebugMessage($"MapResponse Started. Side {side}");

            var persons = new List<DataModel.crme_person>();
            if (responseConsultsArray.GetConsultsForPatientRespArraysInfo != null)
            {
                if (responseConsultsArray.GetConsultsForPatientRespFault24Info != null)
                {
                    Logger.WriteDebugMessage($"responseConsultsArray.GetConsultsForPatientRespFault24Info.Message - {responseConsultsArray.GetConsultsForPatientRespFault24Info?.Message}");
                    persons.Add(CreateFailedCrmePersonObject(responseConsultsArray.GetConsultsForPatientRespFault24Info.Message));
                }
                else
                {
                    foreach (var arr in responseConsultsArray.GetConsultsForPatientRespArraysInfo)
                    {
                        if (arr != null && arr.GetConsultsForPatientRespConsultsInfo != null)
                        {
                            foreach (var consult in arr.GetConsultsForPatientRespConsultsInfo)
                            {
                                Logger.WriteDebugMessage("About to get next person");
                                var person = new DataModel.crme_person
                                {
                                    cvt_ConsultIEN = consult.Id,
                                    cvt_ConsultStatus = consult.Status,
                                    cvt_ConsultText = Base64Decode(consult.Text),
                                    cvt_ConsultTimestamp = consult.Timestamp,
                                    cvt_ConsultTitle = consult.Title,
                                    Id = new Guid(),
                                    crme_url = side //arbitrarily using this variable to temporarily send the pat/pro string across to the js

                                };
                                Logger.WriteDebugMessage("Successfully got person: " + person.Id);
                                persons.Add(person);
                            }
                        }
                        else if (arr.GetConsultsForPatientRespFault23Info != null)
                        {
                            Logger.WriteDebugMessage($"arr.GetConsultsForPatientRespFault24Info.Message - {arr.GetConsultsForPatientRespFault23Info?.Message}");
                            persons.Add(CreateFailedCrmePersonObject(arr.GetConsultsForPatientRespFault23Info.Message));
                        }
                    }
                }
            }

            Logger.WriteDebugMessage("MapResponse Ended");
            return persons;
        }
        
        private crme_person CreateFailedCrmePersonObject(string message)
        {
            return new crme_person
            {
                Id = new Guid(),
                crme_url = "fail", //flag read by javascript in html page to look for a failure
                cvt_ConsultTitle = message
            };
        }

        public string GetStationNumberFromSiteId(string siteId, MCSUtilities2011.MCSLogger Logger)
        {
            Logger.WriteDebugMessage("Starting GetStationNumberFromSiteId");
            var output = "";
            Guid siteGUID = new Guid();
            if (siteId != String.Empty)
                siteGUID = Guid.Parse(siteId);
            else
                return output;

            using (var context = new Xrm(OrganizationService))
            {
                var siteRecord = context.mcs_siteSet.FirstOrDefault(s => s.Id == siteGUID);
                if (siteRecord != null)
                {
                    Logger.WriteDebugMessage("Found TMP Site");
                    mcs_site obj = (mcs_site)siteRecord;
                    if (obj.mcs_FacilityId != null)
                    {
                        var facilityRecord = context.mcs_facilitySet.FirstOrDefault(f => f.Id == obj.mcs_FacilityId.Id);
                        if (facilityRecord != null)
                        {
                            Logger.WriteDebugMessage("Found Facility");
                            output = facilityRecord.mcs_StationNumber;
                        }
                    }
                }
            }
            return output;
        }
    }
}