﻿using MCSShared;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Appointment
{
    public class AppointmentUpdateVistaHealthSharePostStageRunner : PluginRunner
    {
        public AppointmentUpdateVistaHealthSharePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public override string McsSettingsDebugField => "mcs_appointmentplugin";

        public string VimtUrl { get; set; }

        public int VimtTimeout { get; set; }

        public bool Success;

        public override void Execute()
        {
            var appointment = OrganizationService.Retrieve(DataModel.Appointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.Appointment>();

            var sa = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, appointment.cvt_serviceactivityid.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();
            bool runIntegration;

            using (var srv = new Xrm(OrganizationService)) runIntegration = VistaPluginHelpers.RunVistaIntegration(sa, srv, Logger);

            if (runIntegration) SendVistaIntegration(appointment, sa);
            else
            {
                Success = true;
                Logger.WriteDebugMessage("Vista switch turned off");
            }
        }

        public void SendVistaIntegration(DataModel.Appointment appointment, DataModel.ServiceAppointment serviceAppointment)
        {
            using (var context = new Xrm(OrganizationService))
            {
                VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, GetType().Name);

                var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
                else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");
            }

            Logger.WriteDebugMessage("Beginning VistA Book/Cancel");

            var changedPatientIds = VistaPluginHelpers.GetChangedPatients(appointment, OrganizationService, Logger, out var isBookRequest);

            // Use the version of the record that got passed in (so that the appointment is only considered canceled if the user actually hit the "Cancel Dialog" - aka the Integration Booking Status was changed to a "canceled state")
            var isWholeAppointmentCanceled = VistaPluginHelpers.FullAppointmentCanceled(PrimaryEntity.ToEntity<DataModel.Appointment>());

            if (isBookRequest) SendReceiveVistaBook(appointment, changedPatientIds);
            else if (isWholeAppointmentCanceled)
            {
                if (changedPatientIds != null && changedPatientIds.Any())
                {
                    using (var context = new Xrm(OrganizationService))
                    {
                        var vistaIntegrationResult = context.cvt_vistaintegrationresultSet.FirstOrDefault(x =>
                            x.cvt_Appointment.Id == appointment.Id &&
                            x.cvt_Veteran.Id == changedPatientIds.First());

                        if (vistaIntegrationResult == null) throw new InvalidPluginExecutionException("Cannot find the VistA Integration Result for Group Cancel");

                        SendReceiveVistaCancel(appointment, changedPatientIds, vistaIntegrationResult);
                    }
                }
            }
            else
            {
                Logger.WriteDebugMessage("Patient removed from Block Resource as result of Individual Cancellation, no further action needed.  Ending plugin now.");
                //throw new InvalidPluginExecutionException("Invalid Plugin Registration for Vista Integration");
                Success = true;
            }
        }

        private void SendReceiveVistaBook(DataModel.Appointment appointment, List<Guid> addedPatients)
        {
            var failures = 0;
            var vistaStatus = -1;

            if (addedPatients != null && addedPatients.Any() && appointment.cvt_serviceactivityid != null)
            {
                var request = new HealthShareMakeCancelOutboundRequestMessage
                {
                    Patients = addedPatients,
                    ServiceAppointmentId = appointment.cvt_serviceactivityid.Id,
                    AppointmentId = appointment.Id,
                    OrganizationName = PluginExecutionContext.OrganizationName,
                    UserId = PluginExecutionContext.UserId,
                    LogRequest = true,
                    VisitStatus = VIMT.Shared.VistaStatus.SCHEDULED.ToString()
                };

                Logger.WriteDebugMessage("Set up HealthShareMakeCancelOutboundRequestMessage request object.");

                var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
                HealthShareMakeCancelOutboundResponseMessage response = null;

                try
                {
                    Logger.WriteDebugMessage($"Sending HealthShare Make Group Appointment Request Message to VIMT: {vimtRequest}.");
                    response = Utility.SendReceive<HealthShareMakeCancelOutboundResponseMessage>(new Uri(VimtUrl), MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, request, null, VimtTimeout, out int lag);
                    response.VimtLagMs = lag;
                    Logger.WriteDebugMessage($"Finished Sending HealthShare Make Appointment Request Message to VIMT for Appointment with Id: {PrimaryEntity.Id}");

                    foreach (var patientIntegrationResultInformation in response.PatientIntegrationResultInformation)
                    {
                        var status = ProcessVistaMakeApptResponse(response, typeof(HealthShareMakeCancelOutboundRequestMessage), typeof(HealthShareMakeCancelOutboundResponseMessage),
                            patientIntegrationResultInformation);

                        if (status == (int) serviceappointment_statuscode.ReservedScheduled) continue;

                        failures++;
                        vistaStatus = status;
                    }
                }
                catch (Exception ex)
                {
                    var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                    IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Make Vista Appointment", errorMessage, vimtRequest, 
                        typeof(HealthShareMakeCancelOutboundRequestMessage).FullName, typeof(HealthShareMakeCancelOutboundResponseMessage).FullName, 
                        MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, PrimaryEntity.Id, OrganizationService, response?.VimtRequest, response?.VimtResponse,
                        response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);
                    Logger.WriteToFile(errorMessage);
                    failures++;
                }
            }
            else Logger.WriteToFile("Either Added Patient is null/empty or NO Service Activity is associated");

            if (failures > 0)
            {
                Success = false;
                IntegrationPluginHelpers.UpdateAppointment(OrganizationService, PrimaryEntity.Id, (Appointmentcvt_IntegrationBookingStatus)vistaStatus);
            }
            else Success = true;
        }

        private int ProcessVistaMakeApptResponse(HealthShareMakeCancelOutboundResponseMessage response, Type requestType, Type responseType, PatientIntegrationResultInformation patientIntegrationResultInformation)
        {
            if (response == null) return (int)Appointmentcvt_IntegrationBookingStatus.VistaFailure;

            var errorMessage = patientIntegrationResultInformation.ExceptionOccured ? patientIntegrationResultInformation.ExceptionMessage : string.Empty;

            IntegrationPluginHelpers.CreateAppointmentIntegrationResult("Group Book to Vista", patientIntegrationResultInformation.ExceptionOccured, errorMessage,
                patientIntegrationResultInformation.VimtRequest, response.SerializedInstance, patientIntegrationResultInformation.VimtResponse, requestType.FullName, 
                responseType.FullName, MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, PrimaryEntity.Id, OrganizationService, response.VimtLagMs,
                patientIntegrationResultInformation.EcProcessingMs, response.VimtProcessingMs, false, patientIntegrationResultInformation);

            var status = Appointmentcvt_IntegrationBookingStatus.ReservedScheduled;

            if (response.ExceptionOccured || patientIntegrationResultInformation.ExceptionOccured) status = Appointmentcvt_IntegrationBookingStatus.VistaFailure;
            
            return (int)status;
        }

        private void SendReceiveVistaCancel(DataModel.Appointment appointment, List<Guid> removedPatients, cvt_vistaintegrationresult vistaIntegrationResult)
        {
            var failures = 0;
            
            try
            {
                var appointmentRequestMessage = new HealthShareMakeCancelOutboundRequestMessage
                {
                    AppointmentId = appointment.Id,
                    ServiceAppointmentId = appointment.cvt_serviceactivityid.Id,
                    Patients = removedPatients,
                    LogRequest = true,
                    OrganizationName = PluginExecutionContext.OrganizationName,
                    UserId = PluginExecutionContext.UserId,
                    VisitStatus = VIMT.Shared.VistaStatus.CANCELED.ToString(),
                    VistaIntegrationResultId = vistaIntegrationResult.Id
                };

                Logger.WriteDebugMessage("Sending to CancelGroup Request to VIMT");
                var response = Utility.SendReceive<HealthShareMakeCancelOutboundResponseMessage>(new Uri(VimtUrl), MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, appointmentRequestMessage, null, VimtTimeout, out int lag);
                Logger.WriteDebugMessage("CancelGroup Response Received from VIMT");

                if (response != null)
                {
                    response.VimtLagMs = lag;
                    foreach (var patientIntegrationResultInformation in response.PatientIntegrationResultInformation)
                    {
                        var failed = ProcessVistaCancelResponse(response, typeof(HealthShareMakeCancelOutboundRequestMessage), typeof(HealthShareMakeCancelOutboundResponseMessage), appointment, true, patientIntegrationResultInformation);
                        if (failed) failures++;
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.WriteToFile($"Cancel Vista Appointment Failed on appt: {appointment.Subject}. Error Message: {CvtHelper.BuildExceptionMessage(ex)}");
                failures++;
            }

            if (failures > 0)
            {
                Success = false;
                IntegrationPluginHelpers.UpdateAppointment(OrganizationService, PrimaryEntity.Id, Appointmentcvt_IntegrationBookingStatus.CancelFailure);
            }
            else Success = true;
        }

        private bool ProcessVistaCancelResponse(HealthShareMakeCancelOutboundResponseMessage response, Type requestType, Type responseType, DataModel.Appointment appointment, bool wholeAppointmentCanceled, PatientIntegrationResultInformation patientIntegrationResultInformation)
        {
            if (response == null) return true;

            var errorMessage = patientIntegrationResultInformation.ExceptionOccured ? patientIntegrationResultInformation.ExceptionMessage : string.Empty;

            IntegrationPluginHelpers.CreateAppointmentIntegrationResult("Group Cancel to Vista", patientIntegrationResultInformation.ExceptionOccured, errorMessage,
                patientIntegrationResultInformation.VimtRequest, response.SerializedInstance, patientIntegrationResultInformation.VimtResponse, requestType.FullName, 
                responseType.FullName, MessageRegistry.HealthShareMakeCancelOutboundRequestMessage, PrimaryEntity.Id, OrganizationService, response.VimtLagMs,
                patientIntegrationResultInformation.EcProcessingMs, response.VimtProcessingMs, false, patientIntegrationResultInformation);

            var status = wholeAppointmentCanceled ? appointment.cvt_IntegrationBookingStatus.Value : (int)Appointmentcvt_IntegrationBookingStatus.PatientCanceled;

            if (response.ExceptionOccured || patientIntegrationResultInformation.ExceptionOccured) status = (int)Appointmentcvt_IntegrationBookingStatus.CancelFailure;
            
            if (!wholeAppointmentCanceled)
            {
                Logger.WriteDebugMessage($"Individual Cancellation, not updating entire appointment status to {(Appointmentcvt_IntegrationBookingStatus)status}");
                return false;
            }

            if (appointment.cvt_IntegrationBookingStatus.Value != status) IntegrationPluginHelpers.UpdateAppointment(OrganizationService, appointment.Id, (Appointmentcvt_IntegrationBookingStatus)status);
            else Logger.WriteDebugMessage("Appointment Booking Status has not changed, no need to update appointment on cancel");

            return status == (int)Appointmentcvt_IntegrationBookingStatus.CancelFailure;
        }
    }
}