﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Appointment
{
    public class AppointmentUpdateVVSPostStageRunner : PluginRunner
    {

        public AppointmentUpdateVVSPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public AppointmentUpdateVVSPostStageRunner(IServiceProvider serviceProvider, Dictionary<Guid, bool> veteransDelta) : base(serviceProvider)
        {
            VeteransDelta = veteransDelta;
        }

        public override string McsSettingsDebugField => "mcs_appointmentplugin";

        public string VimtUrl { get; private set; }

        private int VimtTimeout { get; set; }

        public bool Success { get; internal set; }

        public Dictionary<Guid, bool> VeteransDelta;

        public override void Execute()
        {
            bool callVvs;
            var appt = OrganizationService.Retrieve(DataModel.Appointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.Appointment>();

            using (var srv = new Xrm(OrganizationService))
            {
                var vimtUrlSetting = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
                else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  VVS Canceled.");

                var sa = srv.ServiceAppointmentSet.FirstOrDefault(s => s.Id == appt.cvt_serviceactivityid.Id);
                if (sa == null) throw new InvalidPluginExecutionException("No Service Activity found for this appointment");
                callVvs = VistaPluginHelpers.RunVvs(sa, srv, Logger);
            }
            if (callVvs)
            {
                SendVistaMessage(appt);
            }
            else
            {
                Success = true;
                Logger.WriteDebugMessage("VVS Integration Bypassed, Updating Service Activity Status");
            }
        }

        #region Vista Booking and Cancelation
        private void SendVistaMessage(DataModel.Appointment appointment)
        {
            // Default the success flag to false so that it is false until it successfully reaches a completed Vista Booking
            Success = false;
            List<Guid> changedPatients;
            using (var context = new Xrm(OrganizationService)) VimtTimeout = IntegrationPluginHelpers.GetVimtTimeout(context, Logger, this.GetType().Name);
            Logger.WriteDebugMessage("Beginning VVS Book/Cancel");
            bool isBookRequest;

            if (VeteransDelta != null)
            {
                Logger.WriteDebugMessage("Using Patient passed in from Proxy Add Plugin");
                
                // If there are any veterans that were added, this is considered a book, otherwise view it as a cancel
                isBookRequest = VeteransDelta.Any(kvp => kvp.Value);
                changedPatients = VeteransDelta.Where(kvp => kvp.Value == isBookRequest).Select(kvp => kvp.Key).ToList();
            }
            else
            {
                Logger.WriteDebugMessage("Using Vista Integration Results to determine Patient Delta");
                changedPatients = VistaPluginHelpers.GetChangedPatients(appointment, OrganizationService, Logger, out isBookRequest);
            }

            if (isBookRequest)
            {
                bool isUpdate;

                using (var srv = new Xrm(OrganizationService))
                {
                    isUpdate = srv.mcs_integrationresultSet.Where(
                        i => i.mcs_appointmentid.Id == appointment.Id
                        && i.mcs_status.Value != (int)mcs_integrationresultmcs_status.Error).ToList()
                        .Any(i => i.mcs_VimtMessageRegistryName.ToLower().Contains("accenture"));
                }

                if (!isUpdate)
                {
                    var createRequest = new VideoVisitCreateRequestMessage
                    {
                        AppointmentId = appointment.Id,
                        LogRequest = true,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        AddedPatients = changedPatients
                    };

                    Logger.WriteDebugMessage("Sending Create VVS");
                    var response = Utility.SendReceive<VideoVisitCreateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitCreateRequestMessage, createRequest, null, VimtTimeout, out int lag);
                    response.VimtLagMs = lag;
                    ProcessVistaCreateResponse(response, typeof(VideoVisitCreateRequestMessage), typeof(VideoVisitCreateResponseMessage), appointment);
                }
                else
                {
                    var updateRequest = new VideoVisitUpdateRequestMessage
                    {
                        AppointmentId = appointment.Id,
                        LogRequest = true,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        Contacts = changedPatients
                    };

                    Logger.WriteDebugMessage("Sending Update VVS");
                    var response = Utility.SendReceive<VideoVisitUpdateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitUpdateRequestMessage, updateRequest, null, VimtTimeout, out int lag);
                    response.VimtLagMs = lag;
                    ProcessVistaUpdateResponse(response, typeof(VideoVisitUpdateRequestMessage), typeof(VideoVisitUpdateResponseMessage), appointment);
                }
            }
            else
            {
                var isWholeApptCanceled = VistaPluginHelpers.FullAppointmentCanceled(PrimaryEntity.ToEntity<DataModel.Appointment>());
                if (isWholeApptCanceled)
                {
                    var cancelRequest = new VideoVisitDeleteRequestMessage
                    {
                        AppointmentId = appointment.Id,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        LogRequest = true,
                        CanceledPatients = changedPatients,
                        WholeAppointmentCanceled = true
                    };
                    Logger.WriteDebugMessage("Sending Cancel VVS");
                    var response = Utility.SendReceive<VideoVisitDeleteResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitDeleteRequestMessage, cancelRequest, null, VimtTimeout, out int lag);
                    response.VimtLagMs = lag;
                    ProcessVistaCancelResponse(response, typeof(VideoVisitDeleteRequestMessage), typeof(VideoVisitDeleteResponseMessage), appointment, true);
                }
                else
                {
                    Logger.WriteDebugMessage("Individual patient was canceled through Cancel Dialog and has already been sent to VVS in previous plugin instance.  No action needed here, ending thread.");
                }
            }
        }

        private void ProcessVistaBookResponse(bool isCreate, DataModel.Appointment appointment, string errorMessage, bool exceptionOccured, string vimtRequest, string serializedInstance, string vimtResponse, int vimtLag, int ecProcessingTime, int vimtProcessingTime, WriteResults writeResults)
        {

            var name = isCreate ? "Group VVS" : "Group Update to VVS";
            var reqType = isCreate ? typeof(VideoVisitCreateRequestMessage).FullName : typeof(VideoVisitUpdateRequestMessage).FullName;
            var respType = isCreate ? typeof(VideoVisitCreateResponseMessage).FullName : typeof(VideoVisitUpdateResponseMessage).FullName;
            var regName = isCreate ? MessageRegistry.VideoVisitCreateRequestMessage : MessageRegistry.VideoVisitUpdateRequestMessage;
            IntegrationPluginHelpers.CreateAppointmentIntegrationResult(name, exceptionOccured, errorMessage, vimtRequest, serializedInstance, vimtResponse, reqType, respType, regName, appointment.Id, OrganizationService, vimtLag, ecProcessingTime, vimtProcessingTime);

            if (exceptionOccured)
            {
                Logger.WriteToFile("Exception Occurred in Group Booking: " + errorMessage);
                IntegrationPluginHelpers.UpdateAppointment(OrganizationService, PrimaryEntity.Id, Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure);
            }

            Success = !exceptionOccured;
        }

        private void ProcessVistaCreateResponse(VideoVisitCreateResponseMessage response, Type requestType, Type responseType, DataModel.Appointment appointment)
        {
            if (response == null) return;
            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            ProcessVistaBookResponse(true, appointment, errorMessage, response.ExceptionOccured, response.VimtRequest, response.SerializedInstance, response.VimtResponse, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs, response.WriteResults);
        }

        private void ProcessVistaUpdateResponse(VideoVisitUpdateResponseMessage response, Type requestType, Type responseType, VA.TMP.DataModel.Appointment appointment)
        {
            if (response == null) return;
            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            ProcessVistaBookResponse(false, appointment, errorMessage, response.ExceptionOccured, response.VimtRequest, response.SerializedInstance, response.VimtResponse, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs, response.WriteResults);
        }

        private void ProcessVistaCancelResponse(VideoVisitDeleteResponseMessage response, Type requestType, Type responseType, VA.TMP.DataModel.Appointment appointment, bool wholeAppointmentCanceled)
        {
            if (response == null) return;

            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;

            var integrationResultId = IntegrationPluginHelpers.CreateAppointmentIntegrationResult("Group Cancel to VVS", response.ExceptionOccured, errorMessage, response.VimtRequest,
                response.SerializedInstance, response.VimtResponse, requestType.FullName, responseType.FullName,
                MessageRegistry.VideoVisitDeleteRequestMessage, PrimaryEntity.Id, OrganizationService, response.VimtLagMs, response.EcProcessingMs, response.VimtProcessingMs, false);

            var status = wholeAppointmentCanceled ? appointment.cvt_IntegrationBookingStatus.Value : (int)Appointmentcvt_IntegrationBookingStatus.PatientCanceled;

            if (response.ExceptionOccured) status = (int)Appointmentcvt_IntegrationBookingStatus.CancelFailure;

            if (!wholeAppointmentCanceled)
            {
                Logger.WriteDebugMessage("Individual Cancellation, not updating entire appointment status to " + ((Appointmentcvt_IntegrationBookingStatus)status));
                return;
            }
            if (appointment.cvt_IntegrationBookingStatus.Value != status)
                IntegrationPluginHelpers.UpdateAppointment(OrganizationService, appointment.Id, (Appointmentcvt_IntegrationBookingStatus)status);
            else
                Logger.WriteDebugMessage("Appointment Booking Status has not changed, no need to update appointment on cancel");

            Success = status != (int)serviceappointment_statuscode.CancelFailure;
        }

        #endregion
    }
}