﻿using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.Appointment
{
    public class AppointmentIntegrationOrchestratorUpdatePostStageRunner : PluginRunner
    {
        public AppointmentIntegrationOrchestratorUpdatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public override string McsSettingsDebugField => "mcs_appointmentplugin";

        public Dictionary<Guid, bool> VeteransDelta { get; set; }

        public override void Execute()
        {
            Logger.WriteDebugMessage("AppointmentIntegrationOrchestratorUpdatePostStageRunner -Execute() Initiated.");
            var apptImage = PrimaryEntity.ToEntity<DataModel.Appointment>();
            if (apptImage == null) throw new InvalidPluginExecutionException("Invalid Plugin Registration");

            var entityCollection = PrimaryEntity.GetAttributeValue<EntityCollection>("optionalattendees");
            
            var patientsChanged = entityCollection?.Entities != null && entityCollection.Entities.Count() > 0;
            var statusChange = apptImage.cvt_IntegrationBookingStatus != null;
            var retryInitiated = apptImage.cvt_RetryIntegration;

            Logger.WriteDebugMessage($"+=+=+=+=+= Patients Changed: {patientsChanged} +=+=+=+=+=");
            Logger.WriteDebugMessage($"+=+=+=+=+= Status Changed: {statusChange} +=+=+=+=+=");
            Logger.WriteDebugMessage($"+=+=+=+=+= Retry Initiated: {retryInitiated.HasValue && retryInitiated.Value} +=+=+=+=+=");

            if (retryInitiated.HasValue && retryInitiated.Value) ExecuteRetry();
            else if (patientsChanged) ExecuteBook();
            else if (statusChange)
            {
                switch (apptImage.cvt_IntegrationBookingStatus.Value)
                {
                    case (int)Appointmentcvt_IntegrationBookingStatus.PartialVistaFailure:
                    case (int)Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure:
                    case (int)Appointmentcvt_IntegrationBookingStatus.CancelFailure:
                    case (int)Appointmentcvt_IntegrationBookingStatus.VistaFailure:
                        Logger.WriteDebugMessage("Vista Integration Failures don't trigger cancellations, ending plugin now");
                        break;
                    case (int)Appointmentcvt_IntegrationBookingStatus.TechnologyFailure:
                    case (int)Appointmentcvt_IntegrationBookingStatus.SchedulingError:
                    case (int)Appointmentcvt_IntegrationBookingStatus.PatientNoShow:
                    case (int)Appointmentcvt_IntegrationBookingStatus.PatientCanceled:
                    case (int)Appointmentcvt_IntegrationBookingStatus.ClinicCancelled:
                        ExecuteCancel();
                        break;
                    default:
                        Logger.WriteDebugMessage("Vista Integration not triggered from status " + ((Appointmentcvt_IntegrationBookingStatus)apptImage.cvt_IntegrationBookingStatus.Value).ToString());
                        break;
                }
            }
        }

        public void ExecuteBook()
        {
            if (RunProxyAdd() && RunVista() && RunVvs()) Logger.WriteDebugMessage("Successfully Completed all integration pipeline");
        }

        public void ExecuteCancel()
        {
            if (RunVista() && RunVvs()) Logger.WriteDebugMessage("Successfully Completed all integration pipeline");
        }

        private void ExecuteRetry()
        {
            List<mcs_integrationresult> failedIrs;
            using (var srv = new Xrm(OrganizationService)) failedIrs = srv.mcs_integrationresultSet.Where(ir =>
                ir.mcs_appointmentid.Id == PrimaryEntity.Id && 
                ir.mcs_status.Value == (int)mcs_integrationresultmcs_status.Error).ToList();

            var retryIr = IntegrationPluginHelpers.FindEarliestFailure(failedIrs, Logger);

            if (retryIr == null) Logger.WriteDebugMessage("Skipping Retry as Integration Result with error status could not be found.");
            else RunRetry(retryIr);

            // Reset the Retry integration field post run retry execution
            Logger.WriteDebugMessage("BEGIN Update Retry Flag to False");
            var apt = new DataModel.Appointment { Id = PrimaryEntity.Id, cvt_RetryIntegration = false };
            OrganizationService.Update(apt);
            Logger.WriteDebugMessage("END Update Retry Flag to False");
        }

        private void RunRetry(mcs_integrationresult integrationResult)
        {
            var typeOfIntegration = integrationResult.mcs_VimtMessageRegistryName;
            var successfulRetry = false;
            var finishedIntegrationPipeline = false;
            switch (typeOfIntegration)
            {
                case MessageRegistry.ProxyAddRequestMessage:
                    successfulRetry = RunProxyAdd();
                    if (successfulRetry && RunVista() && RunVvs()) finishedIntegrationPipeline = true;
                    break;
                case MessageRegistry.VideoVisitCreateRequestMessage:
                    successfulRetry = RunVvs();
                    finishedIntegrationPipeline = successfulRetry;
                    break;
                case MessageRegistry.HealthShareMakeCancelOutboundRequestMessage:
                    successfulRetry = RunVista();
                    if (successfulRetry) finishedIntegrationPipeline = RunVvs();
                    break;
                case MessageRegistry.VideoVisitDeleteRequestMessage:
                    successfulRetry = RunVvs();
                    finishedIntegrationPipeline = successfulRetry;
                    break;
                default:
                    Logger.WriteDebugMessage("Unknown Message Type, skipping retry");
                    break;
            }

            if (successfulRetry)
            {
                Logger.WriteDebugMessage("BEGIN Update Failed IR Status");
                ChangeIrStatusToRetrySuccess(integrationResult);
                Logger.WriteDebugMessage("BEGIN Update Failed IR Status");
            }
            else Logger.WriteDebugMessage("IR Status is not updated because successfulRetry is false");

            Logger.WriteToFile(string.Format("Remaining Integrations All {0}.", finishedIntegrationPipeline ? "Succeeded" : "Did Not Succeed"));
        }

        private void ChangeIrStatusToRetrySuccess(mcs_integrationresult ir)
        {
            Logger.WriteDebugMessage($"Updating IR {ir.mcs_VimtMessageRegistryName} to retry success");

            var updateIr = new mcs_integrationresult
            {
                Id = ir.Id,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.ErrorRetrySuccess)
            };

            OrganizationService.Update(updateIr);
        }

        private bool RunProxyAdd()
        {
            var proxyAddRunner = new AppointmentUpdatePostStageRunner(ServiceProvider);
            proxyAddRunner.RunPlugin(ServiceProvider);
            VeteransDelta = proxyAddRunner.VeteransDelta;

            if (!proxyAddRunner.Success) Logger.WriteToFile("Proxy Add did not succeed, Ending Integration Series");

            return proxyAddRunner.Success;
        }

        private bool RunVista()
        {
            var vistaRunner = new AppointmentUpdateVistaHealthSharePostStageRunner(ServiceProvider);
            vistaRunner.RunPlugin(ServiceProvider);

            if (!vistaRunner.Success) Logger.WriteToFile("Vista did not succeed, ending integrations");

            return vistaRunner.Success;
        }

        private bool RunVvs()
        {
            var vvsRunner = new AppointmentUpdateVVSPostStageRunner(ServiceProvider, VeteransDelta);
            vvsRunner.RunPlugin(ServiceProvider);

            if (!vvsRunner.Success) Logger.WriteToFile("Vvs did not succeed");

            return vvsRunner.Success;
        }
    }
}