﻿using Microsoft.Xrm.Sdk;
using System;
using System.Activities;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.CustomActivities.Messages;
using VRMRest;

namespace VA.TMP.Integration.CustomActivities
{
    public class ViaLoginActionRunner : CustomActionRunner
    {
        public ViaLoginActionRunner(CodeActivityContext context) : base(context) { }

        public override string McsSettingsDebugField => "cvt_serviceactivityplugin";

        private string VimtUrl { get; set; }

        public int VimtTimeout { get; set; }

        public string StationNumber { get; set; }

        public string SamlToken { get; set; }

        private string AccessCode { get; set; }

        private string VerifyCode { get; set; }

        public bool SuccessfulLogin { get; private set; }

        public string ErrorMessage { get; private set; }

        public string UserDuz { get; private set; }

        public override void Execute(AttributeCollection attributes)
        {
            GetIntegrationSettings();
            GetPassedInVariables(attributes);
            var response = CallVimt();
            ProcessResponse(response);
        }

        private void GetPassedInVariables(AttributeCollection inputs)
        {
            StationNumber = inputs["StationNumber"].ToString();
            SamlToken = inputs["SamlToken"].ToString();
            AccessCode = inputs["AccessCode"]?.ToString() ?? "";
            VerifyCode = inputs["VerifyCode"]?.ToString() ?? "";
        }

        private void GetIntegrationSettings()
        {
            using (var srv = new Xrm(OrganizationService))
            {
                var vimtTimeoutObj = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VimtTimeout_ViaLoginRunner");
                if (vimtTimeoutObj != null)
                    Convert.ToInt32(vimtTimeoutObj.mcs_value);
                vimtTimeoutObj = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VimtTimeout");
                if (vimtTimeoutObj != null)
                    VimtTimeout = Convert.ToInt32(vimtTimeoutObj.mcs_value);
                else
                    VimtTimeout = 60;
                var vimtUrlObj = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                if (vimtUrlObj == null) throw new InvalidPluginExecutionException("No VIMT URL Integration Setting was found");
                VimtUrl = vimtUrlObj.mcs_value;
            }
        }

        private ViaLoginResponseMessage CallVimt()
        {
            var request = new ViaLoginRequestMessage
            {
                SamlToken = SamlToken,
                StationNumber = StationNumber,
                AccessCode = AccessCode,
                VerifyCode = VerifyCode,
                UserId = WorkflowExecutionContext.UserId,
                OrganizationName = WorkflowExecutionContext.OrganizationName
            };
            try
            {
                Logger.WriteDebugMessage("Sending Login request to VIMT");
                var response = Utility.SendReceive<ViaLoginResponseMessage>(new Uri(VimtUrl), CustomActionMessageRegistry.ViaLoginRequestMessage, request, null, VimtTimeout, out int lag);
                response.VimtLagMs = lag - response.VimtProcessingMs;
                Logger.WriteDebugMessage("Finished sending Login request to VIMT");
                return response;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Failed to Call VIMT");
                return new ViaLoginResponseMessage
                {
                    ExceptionOccured = true,
                    ExceptionMessage = ex.Message
                };
            }
        }

        private void ProcessResponse(ViaLoginResponseMessage response)
        {
            if (response == null) throw new Exception("No Response was returned from VIMT");
            var requestString = string.Format("VIMT Request: {0}\n|EC Request: {1}\n|VIMT Response {2}", response.VimtRequest, response.SerializedInstance, response.VimtResponse);
            Logger.WriteDebugMessage(requestString.Length > 20000 ? requestString.Substring(0, 20000) : requestString);
            SuccessfulLogin = !response.ExceptionOccured;
            ErrorMessage = response.ExceptionMessage;
            UserDuz = response.UserDuz;
            Logger.WriteDebugMessage("User Duz: " + UserDuz + "|Error Message: " + ErrorMessage + "|Success: " + SuccessfulLogin);
        }
    }
}
