﻿using MCSUtilities2011;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.VIMT.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Shared
{
    public static class VistaPluginHelpers
    {
        private static List<Guid> GetPatients(VA.TMP.DataModel.ServiceAppointment serviceAppointment)
        {
            return serviceAppointment.Customers.Select(c => c.PartyId.Id).ToList();
        }

        private static List<Guid> GetPatients(DataModel.Appointment appointment)
        {
            return appointment.OptionalAttendees.Select(c => c.PartyId.Id).ToList();
        }

        private static List<Guid> GetPatients(Entity record)
        {
            if (record.LogicalName == DataModel.Appointment.EntityLogicalName)
                return GetPatients(record.ToEntity<DataModel.Appointment>());
            if (record.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
                return GetPatients(record.ToEntity<DataModel.ServiceAppointment>());
            return new List<Guid>();
        }

        private static List<Guid> GetPreviouslyBookedPatients(IOrganizationService OrganizationService, EntityReference bookingRecord)
        {
            List<Guid> previouslyBookedPatients = new List<Guid>();
            List<cvt_vistaintegrationresult> pastBookings;
            using (var srv = new Xrm(OrganizationService))
            {
                if (bookingRecord.LogicalName == DataModel.Appointment.EntityLogicalName)
                    pastBookings = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_Appointment.Id == bookingRecord.Id && vir.cvt_VistAStatus != VIMT.Shared.VistaStatus.CANCELED.ToString() && vir.cvt_VistAStatus != VIMT.Shared.VistaStatus.FAILED_TO_SCHEDULE.ToString()).ToList();
                else if (bookingRecord.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
                    pastBookings = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_ServiceActivity.Id == bookingRecord.Id && vir.cvt_VistAStatus != VIMT.Shared.VistaStatus.CANCELED.ToString() && vir.cvt_VistAStatus != VIMT.Shared.VistaStatus.FAILED_TO_SCHEDULE.ToString()).ToList();
                else
                    throw new InvalidPluginExecutionException("Invalid Entity Type: Unable to retrieve patient changes.");
                foreach (var book in pastBookings)
                {
                    var contactId = book.cvt_Veteran?.Id ?? IntegrationPluginHelpers.GetPatIdFromIcn(book.cvt_PersonId, OrganizationService);
                    //Only want distinct patients since WriteResults (aka vista integration results) will return 2 copies for a clinic based (1 for pat side and 1 for pro side with the same patient)
                    if (!previouslyBookedPatients.Contains(contactId))
                        previouslyBookedPatients.Add(contactId);
                }
            }
            return previouslyBookedPatients;
        }

        ///// <summary>
        ///// Overload used when getting the answer as to whether this is a book or cancel is irrelevant
        ///// </summary>
        ///// <param name="currentRecord"></param>
        ///// <param name="OrganizationService"></param>
        ///// <param name="Logger"></param>
        ///// <returns></returns>
        //internal static List<Guid> GetChangedPatients(Entity currentRecord, IOrganizationService OrganizationService, MCSLogger Logger)
        //{
        //    bool throwAwayValue;
        //    return GetChangedPatients(currentRecord, OrganizationService, Logger, out throwAwayValue);
        //}

        internal static List<Guid> GetChangedPatients(Entity currentRecord, IOrganizationService organizationService, MCSLogger logger, out bool isAdd)
        {
            isAdd = true;
            List<Guid> currentPatients;

            switch (currentRecord.LogicalName)
            {
                case DataModel.Appointment.EntityLogicalName:
                    currentPatients = currentRecord.ToEntity<DataModel.Appointment>().OptionalAttendees?.Select(ap => ap.PartyId.Id).ToList();
                    break;
                case DataModel.ServiceAppointment.EntityLogicalName:
                    currentPatients = currentRecord.ToEntity<DataModel.ServiceAppointment>().Customers?.Select(ap => ap.PartyId.Id).ToList();
                    break;
                default:
                    throw new InvalidPluginExecutionException("Invalid Entity: Cannot retrieve Patient changes");
            }

            var previousPatients = GetPreviouslyBookedPatients(organizationService, currentRecord.ToEntityReference());
            var newPatients = currentPatients.Except(previousPatients).ToList();

            logger.WriteDebugMessage($"Found {newPatients.Count} new patients");

            if (newPatients.Count > 0) return newPatients;

            isAdd = false;
            var removedPatients = previousPatients; //.Except(currentPatients).ToList();
            logger.WriteDebugMessage($"Found {removedPatients.Count} patients that are in the process of being canceled");
            return removedPatients;
        }

        internal static string GetUserDuz(DataModel.ServiceAppointment serviceAppointment, IOrganizationService organizationService, Guid userId, MCSLogger logger, out string proUserDuz)
        {
            logger.WriteDebugMessage("Starting GetUserDuz");
            string patUserDuz = string.Empty;
            proUserDuz = string.Empty;
            //TSA is deprecated!!
            //if (serviceAppointment != null && serviceAppointment.mcs_relatedtsa != null)
            if(serviceAppointment!=null)
            {
                using (var context = new Xrm(organizationService))
                {
                    mcs_site patSite = new mcs_site();
                    mcs_site proSite = new mcs_site();
                    mcs_facility proFacility = new mcs_facility();
                    mcs_facility patFacility = new mcs_facility();
                    //var proFacility = (from s in context.mcs_servicesSet
                    //                   join f in context.mcs_facilitySet on s.cvt_ProviderFacility.Id equals f.mcs_facilityId.Value
                    //                   where s.mcs_servicesId.Value == serviceAppointment.mcs_relatedtsa.Id
                    //                   select f).FirstOrDefault();

                    //TSA is deprecated, so have to get to facility by taking a different route:
                    if (serviceAppointment.mcs_relatedprovidersite != null)
                    {
                        proSite = context.mcs_siteSet.FirstOrDefault(pro => pro.Id == serviceAppointment.mcs_relatedprovidersite.Id);
                    }

                    if ((proSite.Id != null) && (proSite.Id != Guid.Empty))
                    {
                        proFacility = context.mcs_facilitySet.FirstOrDefault(p => p.Id == proSite.mcs_FacilityId.Id);
                    }

                    if (proFacility != null)
                    {
                        logger.WriteDebugMessage("Provider Facility record was found.  Attempting GetDuzFromStationCode with Station Code: " + proFacility?.mcs_StationNumber);
                        proUserDuz = GetDuzFromStationCode(userId, logger, organizationService, proFacility?.mcs_StationNumber);
                        logger.WriteDebugMessage("ProFacility Station Code: " + proFacility?.mcs_StationNumber + ". ProUserDuz = " + proUserDuz);
                    }

                    //TSA is deprecated, so have to get to facility by taking a different route:
                    //var patFacility = (from s in context.mcs_servicesSet
                    //                   join f in context.mcs_facilitySet on s.cvt_PatientFacility.Id equals f.mcs_facilityId.Value
                    //                   where s.mcs_servicesId.Value == serviceAppointment.mcs_relatedtsa.Id
                    //                   select f).FirstOrDefault();

                    if (serviceAppointment.mcs_relatedsite != null)
                    {
                        patSite = context.mcs_siteSet.FirstOrDefault(pat => pat.Id == serviceAppointment.mcs_relatedsite.Id);
                    }
                    if ((patSite.Id != null) && (patSite.Id != Guid.Empty))
                    {
                        patFacility = context.mcs_facilitySet.FirstOrDefault(f => f.Id == patSite.mcs_FacilityId.Id);
                    }

                    if (patFacility != null)
                    {
                        logger.WriteDebugMessage("Patient Facility record was found.  Attempting GetDuzFromStationCode with Station Code: " + patFacility?.mcs_StationNumber);
                        patUserDuz = GetDuzFromStationCode(userId, logger, organizationService, patFacility?.mcs_StationNumber);
                        logger.WriteDebugMessage("PatFacility Station Code: " + patFacility?.mcs_StationNumber + ". PatUserDuz = " + patUserDuz);
                    }
                }
            }

            return patUserDuz;
        }

        internal static string GetDuzFromStationCode(Guid userId, MCSLogger logger, IOrganizationService organizationService, string stationNumber)
        {
            string userDuz = string.Empty;
            if (!string.IsNullOrWhiteSpace(stationNumber))
            {
                using (var context = new Xrm(organizationService))
                {
                    var duz = context.cvt_userduzSet.FirstOrDefault(u => u.cvt_StationNumber == stationNumber && u.cvt_User.Id == userId && u.statecode.Value == (int)cvt_userduzState.Active);
                    if (duz != null && !string.IsNullOrWhiteSpace(duz.cvt_name))
                    {
                        userDuz = duz.cvt_name;
                    }
                    else
                    {
                        logger.WriteToFile($"The user duz for the facility with the station number {stationNumber} is either not available or empty for the user with Id: {userId}");
                    }
                }
            }
            else
            {
                logger.WriteToFile("The facility is either not associated to the Service Appointment or the facility do not have a station number");
            }

            return userDuz;
        }

        internal static string GetUserSaml(IOrganizationService organizationService, Guid userId, MCSLogger logger)
        {
            var userSaml = string.Empty;
            var user = (SystemUser)organizationService.Retrieve(SystemUser.EntityLogicalName, userId, new ColumnSet(true));
            if (user == null)
            {
                logger.WriteToFile($"The user with id:{userId} is not available");
            }
            else
            {
                userSaml = string.IsNullOrEmpty(user.cvt_SAMLToken) ? string.Empty : user.cvt_SAMLToken;
            }

            return userSaml;
        }

        internal static bool FullAppointmentCanceled(DataModel.Appointment appointment)
        {
            var isCanceled = StatusIsCanceled(appointment.cvt_IntegrationBookingStatus?.Value ?? -1);

            if (isCanceled) return true;
            if (appointment.StateCode == null) return false;
            if (appointment.StateCode.Value == AppointmentState.Canceled) return true;

            return false;
        }

        internal static bool FullAppointmentCanceled(DataModel.ServiceAppointment sa)
        {
            var isCanceled = StatusIsCanceled(sa.StatusCode?.Value ?? -1);
            if (isCanceled)
                return true;
            if (sa.StateCode == null)
                return false;
            if (sa.StateCode.Value == ServiceAppointmentState.Canceled)
                return true;

            return false;
        }

        private static bool StatusIsCanceled(int statusCode)
        {
            switch (statusCode)
            {
                case (int)Appointmentcvt_IntegrationBookingStatus.PatientCanceled:
                case (int)Appointmentcvt_IntegrationBookingStatus.PatientNoShow:
                case (int)Appointmentcvt_IntegrationBookingStatus.ClinicCancelled:
                case (int)Appointmentcvt_IntegrationBookingStatus.TechnologyFailure:
                case (int)Appointmentcvt_IntegrationBookingStatus.SchedulingError:
                    return true;
                default:
                    return false;
            }
        }

        /// <summary>
        /// Create an instance of the Video Vist Service request and send to VIMT.
        /// </summary>
        /// <returns>VideoVisitCreateResponseMessage.</returns>
        internal static VideoVisitCreateResponseMessage CreateAndSendVideoVisitService(Entity entity, Guid userId, string organizationName, string vimtUrl, IOrganizationService organizationService, MCSLogger logger, int vimtTimeout = 0)
        {
            var request = new VideoVisitCreateRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                AppointmentId = entity.Id,
                AddedPatients = GetPatients(entity)
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
            VideoVisitCreateResponseMessage response = null;
            try
            {
                logger.WriteDebugMessage("Sending Create VVS to VIMT");
                response = Utility.SendReceive<VideoVisitCreateResponseMessage>(new Uri(vimtUrl), MessageRegistry.VideoVisitCreateRequestMessage, request, null, vimtTimeout, out int lag);
                response.VimtLagMs = lag;
                logger.WriteDebugMessage("VVS Create Successfully sent to VIMT");
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                if (entity.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
                    IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Create VVS", errorMessage, vimtRequest, typeof(VideoVisitCreateRequestMessage).FullName,
                        typeof(VideoVisitCreateResponseMessage).FullName, MessageRegistry.VideoVisitCreateRequestMessage, entity.Id, organizationService, response?.VimtRequest, 
                        response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);
                else
                    IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Create VVS", errorMessage, vimtRequest, 
                        typeof(VideoVisitCreateRequestMessage).FullName, typeof(VideoVisitCreateResponseMessage).FullName, MessageRegistry.VideoVisitCreateRequestMessage, 
                        entity.Id, organizationService, response?.VimtRequest, response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);

                logger.WriteToFile(errorMessage);

                return null;
            }
        }

        internal static VideoVisitUpdateResponseMessage UpdateAndSendVideoVisitService(Entity entity, Guid userId, string organizationName, string vimtUrl, IOrganizationService organizationService, MCSLogger logger, int vimtTimeout)
        {
            var request = new VideoVisitUpdateRequestMessage
            {
                LogRequest = true,
                AppointmentId = entity.Id,
                OrganizationName = organizationName,
                UserId = userId,
                Contacts = GetPatients(entity)
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
            VideoVisitUpdateResponseMessage response = null;
            try
            {
                logger.WriteDebugMessage("Sending Update Retry to VIMT");
                response = Utility.SendReceive<VideoVisitUpdateResponseMessage>(new Uri(vimtUrl), MessageRegistry.VideoVisitUpdateRequestMessage, request, null, vimtTimeout, out int lag);
                response.VimtLagMs = lag;
                logger.WriteDebugMessage("Retry Successfully sent to VIMT");
                return response;
            }
            catch(Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                if (entity.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
                    IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Update VVS", errorMessage, vimtRequest, typeof(VideoVisitUpdateRequestMessage).FullName, 
                        typeof(VideoVisitUpdateResponseMessage).FullName, MessageRegistry.VideoVisitUpdateRequestMessage, entity.Id, organizationService, response?.VimtRequest, 
                        response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);
                else
                    IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Update VVS", errorMessage, vimtRequest, typeof(VideoVisitUpdateRequestMessage).FullName, 
                        typeof(VideoVisitUpdateResponseMessage).FullName, MessageRegistry.VideoVisitUpdateRequestMessage, entity.Id, organizationService, response?.VimtRequest, 
                        response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs);

                logger.WriteToFile(errorMessage);
                return null;
            }
        }

        /// <summary>
        /// Create an instance of the Video Vist Service request and send to VIMT.
        /// </summary>
        /// <returns>VideoVisitDeleteResponseMessage.</returns>
        internal static VideoVisitDeleteResponseMessage CancelAndSendVideoVisitServiceSa(DataModel.ServiceAppointment serviceAppointment, Guid userId, string organizationName, string vimtUrl, IOrganizationService organizationService, MCSLogger logger, int vimtTimeout)
        {
            var cancels = IntegrationPluginHelpers.GetPatientsFromActivityPartyList(serviceAppointment.Customers.ToList());

            var request = new VideoVisitDeleteRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                AppointmentId = serviceAppointment.Id,
                CanceledPatients = cancels,
                WholeAppointmentCanceled = true
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
            VideoVisitDeleteResponseMessage response = null;
            try
            {
                logger.WriteDebugMessage("Sending Cancel to VIMT");
                response = Utility.SendReceive<VideoVisitDeleteResponseMessage>(new Uri(vimtUrl), MessageRegistry.VideoVisitDeleteRequestMessage, request, null, vimtTimeout, out int lag);
                response.VimtLagMs = lag;
                logger.WriteDebugMessage("Cancel successfully sent to VIMT");
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Cancel VVS", errorMessage, vimtRequest, typeof(VideoVisitDeleteRequestMessage).FullName,
                    typeof(VideoVisitDeleteResponseMessage).FullName, MessageRegistry.VideoVisitDeleteRequestMessage, serviceAppointment.Id, organizationService, 
                    response?.VimtRequest, response?.VimtResponse, response?.VimtLagMs, response?.EcProcessingMs, response?.VimtProcessingMs, true);
                logger.WriteToFile(errorMessage);

                return null;
            }
        }
        
        //Check for switches
        //MAIN VA Video Connect
        //MAIN SFT
        //Record Specialty
        //Record Specialty SubType
        //Record Provider Facility
        //MAIN Interfacility
        //Record Patient Facility
        public static bool RunVistaIntegrationForThisAppointment(mcs_setting settings, DataModel.ServiceAppointment sa, Xrm context, MCSLogger logger, string pathToVista)
        {
            mcs_site patSite = new mcs_site();
            mcs_site proSite = new mcs_site();
            mcs_facility proFacility = new mcs_facility();
            mcs_facility patFacility = new mcs_facility();
            logger.setMethod = "RunVistaIntegrationForThisAppointment";
            logger.WriteDebugMessage("starting RunVistaIntegrationForThisAppointment");

            var isHm = sa.cvt_Type ?? false;

            if (isHm)
            {
                var hmSwitchIsOn = settings.cvt_UseVVSHomeMobile.HasValue && settings.cvt_UseVVSHomeMobile.Value;
                LogSwitchOutput(logger, hmSwitchIsOn, "Main VA Video Connect Encounter", pathToVista);

                if (!hmSwitchIsOn) return false;
            }

            var isSft = sa.cvt_TelehealthModality ?? false;

            if (isSft)
            { 
                var sftSwitchIsOn = settings.cvt_UseVVSSingleEncounterNonHomeMobile.HasValue && settings.cvt_UseVVSSingleEncounterNonHomeMobile.Value;
                LogSwitchOutput(logger, sftSwitchIsOn, "Main Non VA Video Connect Single Encounter", pathToVista);

                if (!sftSwitchIsOn) return false;
            }

            var specialtyRecord = context.mcs_servicetypeSet.FirstOrDefault(s => s.Id == sa.mcs_servicetype.Id);
            var specialtySwitch = specialtyRecord != null && (specialtyRecord.cvt_UseVVS ?? false);
            
            // Checking Specialty
            LogSwitchOutput(logger, specialtySwitch, "Specialty", pathToVista);

            if (!specialtySwitch) return false;
            
            var subSpecialtyRecord = sa.mcs_servicesubtype != null ? context.mcs_servicesubtypeSet.FirstOrDefault(s => s.Id == sa.mcs_servicesubtype.Id) : null;

            // Checking Specialty Sub-type
            if (subSpecialtyRecord == null) logger.WriteDebugMessage("Sub-Specialty not specified on Service Activity.");
            else
            {
                var subSpecialtySwitch = subSpecialtyRecord.cvt_UseVVS ?? false;
                LogSwitchOutput(logger, subSpecialtySwitch, "Specialty Sub-Type", pathToVista);

                if (!subSpecialtySwitch) return false;
            }

            // Provider Facility
            //TSA (mcs_services) is deprecated.  Have to get to Facility via a different path

            //var tsa = context.mcs_servicesSet.FirstOrDefault(s => s.Id == sa.mcs_relatedtsa.Id);
            //var proFacility = context.mcs_facilitySet.FirstOrDefault(f => f.Id == tsa.cvt_ProviderFacility.Id);
            if (sa.mcs_relatedprovidersite != null)
            {
                proSite = context.mcs_siteSet.FirstOrDefault(pro => pro.Id == sa.mcs_relatedprovidersite.Id);
            }

            if ((proSite.Id != null) && (proSite.Id != Guid.Empty))
            {
                proFacility = context.mcs_facilitySet.FirstOrDefault(p => p.Id == proSite.mcs_FacilityId.Id);
            }

            var proFacilitySwitchIsOn = proFacility != null && (proFacility.cvt_UseVistaIntrafacility ?? false);

            if (proFacility != null) LogSwitchOutput(logger, proFacilitySwitchIsOn, "(" + proFacility.mcs_name + ") Provider Facility", pathToVista);

            if (!proFacilitySwitchIsOn) return false;

            //get patFacility to compare
            if (sa.mcs_relatedsite != null)
            {
                patSite = context.mcs_siteSet.FirstOrDefault(pat => pat.Id == sa.mcs_relatedsite.Id);
            }
            if ((patSite.Id != null) && (patSite.Id != Guid.Empty))
            {
                patFacility = context.mcs_facilitySet.FirstOrDefault(f => f.Id == patSite.mcs_FacilityId.Id);
            }

            //var isTsaInterfacility = tsa?.cvt_ServiceScope != null && tsa.cvt_ServiceScope.Value == (int)mcs_servicescvt_ServiceScope.InterFacility;
            bool isTsaInterfacility = false;

            if ((proFacility != null) && (proFacility.Id != Guid.Empty))
            {
                if ((patFacility != null) && (patFacility.Id != Guid.Empty))
                {
                    if (patFacility.Id != proFacility.Id)
                    {
                        isTsaInterfacility = true;
                    }
                }
            }

            if (isTsaInterfacility)
            {
                // Need to check for the actual Interfacility switch
                var ifcSwitchIsOn = settings.cvt_UseVVSInterfacility ?? false;
                LogSwitchOutput(logger, ifcSwitchIsOn, "Main Interfacility", pathToVista);

                if (!ifcSwitchIsOn) return false;

                // Check Patient Facility
                //already have patFacility
                //var patFacility = context.mcs_facilitySet.FirstOrDefault(f => f.Id == tsa.cvt_PatientFacility.Id);
                var patFacilitySwitchIsOn = patFacility != null && (patFacility.cvt_UseVistaIntrafacility ?? false);

                if (patFacility != null) LogSwitchOutput(logger, patFacilitySwitchIsOn, "(" + patFacility.mcs_name + ") Patient Facility", pathToVista);

                if (!patFacilitySwitchIsOn) return false;
            }

            logger.WriteDebugMessage("Returning true for RunVistaIntegrationForThisAppointment.");

            return true;
        }

        public static bool RunVistaIntegration(DataModel.ServiceAppointment sa, Xrm context, MCSLogger Logger)
        {
            var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
            if (settings == null) throw new InvalidPluginExecutionException("Active Settings Cannot be Null");
            if (settings.cvt_UseVistaIntegration.HasValue && settings.cvt_UseVistaIntegration.Value)
            {
                return RunVistaIntegrationForThisAppointment(settings, sa, context, Logger, "VIA/HealthShare");
            }

            Logger.WriteDebugMessage("Vista Integration (through VIA/HealthShare) is turned off system wide, skipping VVS integration");
            return false;
        }

        public static bool RunVvs(DataModel.ServiceAppointment sa, Xrm context, MCSLogger Logger)
        {
            var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
            if (settings == null) throw new InvalidPluginExecutionException("Active Settings Cannot be Null");
            if (settings.cvt_UseVVS != null && settings.cvt_UseVVS.Value)
            {
                return RunVistaIntegrationForThisAppointment(settings, sa, context, Logger, "VVS");
            }

            Logger.WriteDebugMessage("VVS is turned off system wide, skipping VVS integration");
            return false;
        }

        public static void LogSwitchOutput(MCSLogger logger, bool switchIsOn, string appointmentType, string pathToVista)
        {
            logger.WriteDebugMessage(string.Format("{0} Vista Booking switch is {1}", appointmentType, switchIsOn ? "turned on." : $"turned off, skipping {pathToVista} Integration."));
        }
    }
}