using System;
using System.Diagnostics;
using System.Linq;
using VA.TMP.Integration.VIMT.Mvi.StateObject;
using VRM.Integration.Mvi.PersonSearch.Messages;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.Shared
{
    /// <summary>
    /// Static class helpers for the MVI pipeline steps.
    /// </summary>
    public static class MviUtilities
    {
        /// <summary>
        /// Call the Proxy Add Enterprise Component.
        /// </summary>
        /// <param name="state">State object.</param>
        /// <param name="isPatient">Whether a Patient or Provider.</param>
        public static void SendIdentifierToProxyAddEc(ref ProxyAddStateObject state, bool isPatient)
        {
            AddPersonResponse response = null;
            try
            {
                switch (state.FakeResponseType)
                {
                    // Fake Success
                    case "0": 
                        response = isPatient ? CreateFakeSuccessResponse(state.PatientSite) : CreateFakeSuccessResponse(state.ProviderSite);
                        break;

                    // DB Down
                    case "1": 
                        response = CreateFakeDbDownResponse();
                        break;

                    // Failure to Proxy Add
                    case "2": 
                        response = CreateFakeFailureResponse();
                        break;

                    // Server Down
                    case "3": 
                        response = CreateFakeServerDownResponse();
                        break;

                    // Default
                    default :
                        var timer = new Stopwatch();
                        timer.Start();
                        response = state.ProxyAddToVistaRequest.SendReceive<AddPersonResponse>(MessageProcessType.Local);
                        timer.Stop();
                        state.EcProcessingTimeMs += (int)timer.ElapsedMilliseconds;
                        break;
                }
                    
                if (response == null) Logger.Instance.Error(string.Format("Proxy Add to Vista Response for {0} is null", isPatient ? "Patient" : "Provider"));
                else Logger.Instance.Debug(string.Format("Proxy Add to Vista Response for {0} is {1}", isPatient ? "Patient" : "Provider", PipelineUtilities.SerializeInstance(response)));
            }
            catch (Exception ex)
            {
                Logger.Instance.Error(string.Format("Unable to serialize the output of the Proxy Add to Vista response. Error was {0}", ex));
            }

            if (response != null && response.Acknowledgement != null && response.Acknowledgement.TypeCode != null)
            {
                switch (response.Acknowledgement.TypeCode)
                {
                    case "AA":
                        if (response.Acknowledgement.AcknowledgementDetails == null || response.Acknowledgement.AcknowledgementDetails.Length == 0)
                        {
                            state.ExceptionOccured = true;
                            state.ExceptionMessage = "No Acknowledgment Details Listed in Proxy Add to Vista Success Response";
                            Logger.Instance.Error("No Acknowledgment Details Listed in Proxy Add to Vista Success Response");
                        }
                        else
                        {
                            var ack = response.Acknowledgement.AcknowledgementDetails.FirstOrDefault();
                            if (ack != null && ack.Code != null)
                            {
                                if (!string.IsNullOrEmpty(ack.Code.Code))
                                {
                                    var parts = ack.Code.Code.Split('^');
                                    if (parts.Length < 3)
                                    {
                                        state.ExceptionOccured = true;
                                        state.ExceptionMessage = "Unexpected Acknowledgment Code in Success, expecting at least 3 parts separated by ^";
                                        Logger.Instance.Error("Unexpected Acknowledgment Code in Success, expecting at least 3 parts separated by ^");
                                    }
                                    else
                                    {
                                        var identiferId = parts[0];

                                        Logger.Instance.Debug(string.Format("{0} - {1}", identiferId, ack.Text));

                                        if (isPatient)
                                        {
                                            state.PatientSideIdentifierToAdd.mcs_identifier = identiferId;
                                            state.OrganizationServiceProxy.Create(state.PatientSideIdentifierToAdd);
                                            Logger.Instance.Debug(string.Format("VistA Station {0} successfully added to TMP", identiferId));
                                        }
                                        else
                                        {
                                            state.ProviderSideIdentifierToAdd.mcs_identifier = identiferId;
                                            state.OrganizationServiceProxy.Create(state.ProviderSideIdentifierToAdd);
                                            Logger.Instance.Debug(string.Format("VistA Station {0} successfully added to TMP", identiferId));
                                        }
                                    }
                                }
                                else
                                {
                                    state.ExceptionOccured = true;
                                    state.ExceptionMessage = "No Acknowledgment Code Listed in Proxy Add to Vista Success Response";
                                    Logger.Instance.Error("No Acknowledgment Code Listed in Proxy Add to Vista Success Response");
                                }
                            }
                            else
                            {
                                state.ExceptionOccured = true;
                                state.ExceptionMessage = "Acknowledgment or Acknowledgment Code Success is null";
                                Logger.Instance.Error("Acknowledgment or Acknowledgment Code Success is null");
                            }
                        }
                        break;
                    case "AR":
                    case "AE":
                        if (response.Acknowledgement.AcknowledgementDetails == null || response.Acknowledgement.AcknowledgementDetails.Length == 0)
                        {
                            state.ExceptionOccured = true;
                            state.ExceptionMessage = "No Acknowledgment Details Listed in Proxy Add to Vista Fail Response";
                            Logger.Instance.Error("No Acknowledgment Details Listed in Proxy Add to Vista Fail Response");
                        }
                        else
                        {
                            var ack2 = response.Acknowledgement.AcknowledgementDetails.FirstOrDefault();
                            if (ack2 != null && ack2.Code != null)
                            {
                                if (!string.IsNullOrEmpty(ack2.Code.Code))
                                {
                                    var exceptionMessage = string.Format("Proxy Add to Vista failed with the following: {0} - {1} - {2}", ack2.Code.Code, ack2.Code.DisplayName, ack2.Text);
                                    state.ExceptionOccured = true;
                                    state.ExceptionMessage = exceptionMessage;
                                    Logger.Instance.Error(exceptionMessage);

                                }
                                else
                                {
                                    state.ExceptionOccured = true;
                                    state.ExceptionMessage = "No Acknowledgment Code Listed in Proxy Add to Vista Fail Response";
                                    Logger.Instance.Error("No Acknowledgment Code Listed in Proxy Add to Vista Fail Response");
                                }
                            }
                            else
                            {
                                state.ExceptionOccured = true;
                                state.ExceptionMessage = "Acknowledgment or Acknowledgment Code Fail is null";
                                Logger.Instance.Error("Acknowledgment or Acknowledgment Code Fail is null");
                            }
                        }
                        break;
                }
            }
            else
            {
                if (response == null)
                {
                    state.ExceptionOccured = true;
                    state.ExceptionMessage = "No Response from Proxy Add to Vista Enterprise Component";
                    Logger.Instance.Error("Response was null");
                }
                else if (response.Acknowledgement == null)
                {
                    state.ExceptionOccured = true;
                    state.ExceptionMessage = string.Format("Unexpected Response from Proxy Add to Vista Enterprise Component.  The following message was returned: {0}.  RawMVIExceptionMessage: {1}.  Possibly MVI Server Down.", response.Message, response.RawMviExceptionMessage);
                    Logger.Instance.Error(state.ExceptionMessage);
                }
                else if (response.Acknowledgement.TypeCode == null)
                {
                    state.ExceptionOccured = true;
                    state.ExceptionMessage = "Unexpected Response from Proxy Add to Vista Enterprise Component, no Acknowledgment Type Code returned from Enterprise Component";
                    Logger.Instance.Error("TypeCode was null");
                }
                else
                {
                    state.ExceptionOccured = true;
                    state.ExceptionMessage = "Unexpected Response from Proxy Add to Vista Enterprise Component";
                    Logger.Instance.Error(state.ExceptionMessage);
                }
            }
        }

        /// <summary>
        /// Creates a Fake Proxy Add Success response.
        /// </summary>
        /// <param name="stationNumber">VistA station number.</param>
        /// <returns>AddPersonResponse</returns>
        private static AddPersonResponse CreateFakeSuccessResponse(string stationNumber)
        {
            return new AddPersonResponse
            {
                MessageId = Guid.NewGuid().ToString(),
                ExceptionOccured = false,
                Message = "Correlation Proxy Added to VistA",
                RawMviExceptionMessage = string.Empty,
                Acknowledgement = new Acknowledgement
                {
                    TypeCode = "AA",
                    TargetMessage = "1.2.840.114350.1.13.0.1.7.1.1",
                    AcknowledgementDetails = new[]
                    {
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                CodeSystemName = "MVI",
                                DisplayName = "IEN",
                                Code = string.Format("7172149^PI^{0}^", stationNumber)
                            },
                            Text = "Correlation Proxy Added to VistA"
                        }
                    }
                }
            };
        }

        /// <summary>
        /// Creates a Fake Proxy Add Failure response.
        /// </summary>
        /// <returns>AddPersonResponse.</returns>
        private static AddPersonResponse CreateFakeFailureResponse()
        {
            return new AddPersonResponse
            {
                MessageId = Guid.NewGuid().ToString(),
                ExceptionOccured = false,
                Message = "Communication Failure with requested VistA Station",
                RawMviExceptionMessage = string.Empty,
                Acknowledgement = new Acknowledgement
                {
                    TypeCode = "AE",
                    TargetMessage = "1.2.840.114350.1.13.0.1.7.1.1",
                    AcknowledgementDetails = new[]
                    {
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                DisplayName = "Internal System Error",
                                Code = "INTERR"
                            },
                            Text = "Communication Failure with requested VistA Station"
                        },
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                CodeSystemName = string.Empty,
                                DisplayName = string.Empty,
                                Code = string.Empty
                            },
                            Text = "Communication Failure with requested VistA Station"
                        }
                    }
                }
            };
        }

        /// <summary>
        /// Creates a Fake Proxy Add MVI DB is down response.
        /// </summary>
        /// <returns>AddPersonResponse.</returns>
        private static AddPersonResponse CreateFakeDbDownResponse()
        {
            return new AddPersonResponse
            {
                MessageId = Guid.NewGuid().ToString(),
                ExceptionOccured = false,
                Message = "MVI[S]:INVALID REQUEST",
                RawMviExceptionMessage = string.Empty,
                Acknowledgement = new Acknowledgement
                {
                    TypeCode = "AE",
                    TargetMessage = "1.2.840.114350.1.13.0.1.7.1.1",
                    AcknowledgementDetails = new[]
                    {
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                DisplayName = "Internal System Error",
                                Code = "INTERR"
                            },
                            Text = "MVI[S]:INVALID REQUEST"
                        },
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                CodeSystemName = string.Empty,
                                DisplayName = string.Empty,
                                Code = string.Empty
                            },
                            Text = "MVI[S]:INVALID REQUEST"
                        }
                    }
                }
            };
        }

        /// <summary>
        /// Creates a Fake Proxy Add MVI Server is down response.
        /// </summary>
        /// <returns>AddPersonResponse.</returns>
        private static AddPersonResponse CreateFakeServerDownResponse()
        {
            return new AddPersonResponse
            {
                MessageId = Guid.NewGuid().ToString(),
                ExceptionOccured = true,
                Message = "An unexpected error occured during the Add Person Operation",
                RawMviExceptionMessage = "Internal Error (from server)",
            };
        }
    }
}