using MCSShared;
using MCSUtilities2011;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Metadata;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.VIMT.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Shared
{
    /// <summary>
    /// Static class of reusable methods to facilitate integration plugin development.
    /// </summary>
    public static class IntegrationPluginHelpers
    {
        public const string VimtServerDown = "Error - VIMT is likely unavailable. Details of the error are as follows: {0}";

        /// <summary>
        ///  Update Service Appointment's Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="serviceAppointmentId">Service Appointment Id.</param>
        /// <param name="statusCode">Service Appointment Status.</param>
        public static void UpdateServiceAppointmentStatus(IOrganizationService organizationService, Guid serviceAppointmentId, serviceappointment_statuscode statusCode)
        {
            var sa = organizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointmentId, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();
            ChangeEntityStatus(organizationService, sa, (int)statusCode, true);
        }

        /// <summary>
        ///  Update Service Appointment's Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="vodId">Service Appointment Id.</param>
        /// <param name="statusCode">Service Appointment Status.</param>
        public static void UpdateVodStatus(IOrganizationService organizationService, Guid vodId, cvt_vod_statuscode statusCode)
        {
            var vod = organizationService.Retrieve(cvt_vod.EntityLogicalName, vodId, new ColumnSet(true)).ToEntity<cvt_vod>();
            ChangeEntityStatus(organizationService, vod, (int)statusCode, true);
        }

        public static void UpdateAppointment(IOrganizationService organizationService, Guid appointmentId, Appointmentcvt_IntegrationBookingStatus status)
        {
            var appt = organizationService.Retrieve(DataModel.Appointment.EntityLogicalName, appointmentId, new ColumnSet(true)).ToEntity<DataModel.Appointment>();
            if (appt.StatusCode.Value == (int)status)
                return;
            var appointmentToUpdate = new DataModel.Appointment
            {
                Id = appointmentId,
                cvt_IntegrationBookingStatus = new OptionSetValue((int)status)
            };
            organizationService.Update(appointmentToUpdate);
        }
        
        /// <summary>
        /// Update Integration Result to ErrorRetrySuccess Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="integrationResultId">Integration Result Id.</param>
        public static void UpdateIntegrationResultToErrorRetrySuccessStatus(IOrganizationService organizationService, Guid integrationResultId)
        {
            var updateIntegrationResult = new mcs_integrationresult
            {
                Id = integrationResultId,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.ErrorRetrySuccess),
                mcs_retry = false
            };
            organizationService.Update(updateIntegrationResult);
        }

        /// <summary>
        /// Serializes an instance of a class to a string.
        /// </summary>
        /// <typeparam name="T">The Type of class.</typeparam>
        /// <param name="classInstance">The instance of the class.</param>
        /// <returns>Serialized instance of class.</returns>
        public static string SerializeInstance<T>(T classInstance)
        {
            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(T));
                serializer.Serialize(stream, classInstance);

                return Encoding.ASCII.GetString(stream.ToArray());
            }
        }

        /// <summary>
        /// Invoke SendReceive method using .NET reflection to send the request to VIMT.
        /// </summary>
        /// <param name="requestType">Request Type.</param>
        /// <param name="responseType">Response Type.</param>
        /// <param name="serializedRequest">Serialized Request.</param>
        /// <param name="registryMessage">Message Registry Name.</param>
        /// <param name="vimtBaseUri">VIMT base URI.</param>
        /// <returns></returns>
        public static object InvokeVimtDynamically(Type requestType, Type responseType, string serializedRequest, string registryMessage, string vimtBaseUri, int vimtTimeout = 0)
        {
            var serializer = new XmlSerializer(requestType);

            using (TextReader reader = new StringReader(serializedRequest))
            {
                var result = serializer.Deserialize(reader);
                var method = typeof(Utility).GetMethod("SendReceive", new Type[] { typeof(Uri), typeof(string), typeof(object), typeof(LogSettings), typeof(int) });
                var generic = method.MakeGenericMethod(responseType);
                var response = generic.Invoke(null, new[] { new Uri(vimtBaseUri), registryMessage, result, null, vimtTimeout });

                var responseInstance = response;
                if (responseInstance == null) throw new Exception("The response value was not of the proper type");

                return responseInstance;
            }
        }

        /// <summary>
        /// Create Integration Result on successful completion of VIMT call.
        /// </summary>
        /// <param name="integrationResultName">>Name of the Integration Result.</param>
        /// <param name="exceptionOccured">Whether an exception occured.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="serviceAppointId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        public static Guid CreateIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid serviceAppointId,
            IOrganizationService organizationService, 
            int vimtLagTime, 
            int EcProcessingTime, 
            int vimtProcessingTime, PatientIntegrationResultInformation patientIntegrationResultInformation = null)
        {
            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, exceptionOccured, vimtRequest, integrationRequest, vimtResponse, vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, EcProcessingTime, vimtProcessingTime, null, patientIntegrationResultInformation);
            integrationResult.mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointId);

            var id = organizationService.Create(integrationResult);

            if (exceptionOccured)
            {
                UpdateServiceAppointmentStatus(organizationService, serviceAppointId, serviceappointment_statuscode.InterfaceVIMTFailure);
            }

            return id;
        }

        /// <summary>
        /// Create Integration Result on successful completion of VIMT call.
        /// </summary>
        /// <param name="integrationResultName">>Name of the Integration Result.</param>
        /// <param name="exceptionOccured">Whether an exception occured.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="vodId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        public static Guid CreateVodIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid vodId,
            IOrganizationService organizationService,
            int vimtLagTime,
            int eCProcessingTime,
            int vimtProcessingTime)
        {
            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, exceptionOccured, vimtRequest, integrationRequest, vimtResponse, vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, eCProcessingTime, vimtProcessingTime);

            integrationResult.cvt_vod = new EntityReference(cvt_vod.EntityLogicalName, vodId);

            var id = organizationService.Create(integrationResult);

            if (exceptionOccured) UpdateVodStatus(organizationService, vodId, cvt_vod_statuscode.Failure);

            return id;
        }


        /// <summary>
        /// Creates Integration Result.
        /// </summary>
        /// <param name="integrationResultName">Integration Result Name.</param>
        /// <param name="exceptionOccured">Whether an exception occurred.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Request Message Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Response Message Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="appointmentId">Appointment Id</param>
        /// <param name="organizationService">Organization Service Proxy.</param>
        /// <param name="updateAppointment">If true, updates appointment status (success or failure), if false, only updates status on failure.</param>
        /// <param name="patientIntegrationResultInformation"></param>
        public static Guid CreateAppointmentIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid appointmentId,
            IOrganizationService organizationService,
            int vimtLagTime,
            int eCProcessingTime,
            int vimtProcessingTime,
            bool updateAppointment = true,
            PatientIntegrationResultInformation patientIntegrationResultInformation = null)
        {
            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, exceptionOccured, vimtRequest, integrationRequest, vimtResponse, 
                vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, eCProcessingTime, vimtProcessingTime, null, patientIntegrationResultInformation);

            integrationResult.mcs_appointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, appointmentId);
            
            var id = organizationService.Create(integrationResult);

            if (updateAppointment)
            {
                UpdateAppointment(organizationService, appointmentId, exceptionOccured
                    ? Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure
                    : Appointmentcvt_IntegrationBookingStatus.ReservedScheduled);
            }
            else
            {
                if (exceptionOccured)
                    UpdateAppointment(organizationService, appointmentId, Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure);
            }
            return id;
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Name of the Integration Result.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="serviceAppointId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        /// <param name="isCancelRequest">Whether call is for a Cancel Request.</param>
        public static void CreateIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string integrationRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid serviceAppointId,
            IOrganizationService organizationService,
            string vimtRequest,
            string vimtResponse,
            int? vimtLagTime,
            int? eCProcessingTime,
            int? vimtProcessingTime,
            bool isCancelRequest = false)
        {
            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, true, vimtRequest, integrationRequest, vimtResponse, vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, eCProcessingTime, vimtProcessingTime);

            integrationResult.mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointId);
            organizationService.Create(integrationResult);

            if (isCancelRequest) return;

            UpdateServiceAppointmentStatus(organizationService, serviceAppointId, serviceappointment_statuscode.InterfaceVIMTFailure);
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Name of the Integration Result.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="vodId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        /// <param name="isCancelRequest">Whether call is for a Cancel Request.</param>
        public static void CreateVodIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string integrationRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid vodId,
            IOrganizationService organizationService,
            string vimtRequest,
            string vimtResponse,
            int? vimtLagTime,
            int? eCProcessingTime,
            int? vimtProcessingTime)
        {
            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, true, vimtRequest, integrationRequest, vimtResponse, vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, eCProcessingTime, vimtProcessingTime);
            integrationResult.cvt_vod = new EntityReference(DataModel.cvt_vod.EntityLogicalName, vodId);
            organizationService.Create(integrationResult);
            
            UpdateVodStatus(organizationService, vodId, cvt_vod_statuscode.Failure);
        }

        public static void CreateIntegrationResultOnPersonSearch(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            IOrganizationService organizationService,
            int vimtLagTime,
            int eCProcessingTime,
            int vimtProcessingTime, MCSLogger logger)
        {
            logger.WriteDebugMessage($"PrepareGenericIntegrationResultRecord Initiated\nParameter values: \nintegrationResultName - {integrationResultName}, exceptionOccured - {exceptionOccured}, errorMessage - {errorMessage}, vimtRequest - {vimtRequest}, integrationRequest - {integrationRequest}, vimtResponse - {vimtResponse}, vimtRequestMessageType - {vimtRequestMessageType}, vimtResponseMessageType - {vimtResponseMessageType},vimtMessageRegistryName - {vimtMessageRegistryName},organizationService - {organizationService}, vimtLagTime - {vimtLagTime},eCProcessingTime - { eCProcessingTime},vimtProcessingTime - { vimtProcessingTime}");

            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, exceptionOccured, vimtRequest, integrationRequest, vimtResponse, vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, eCProcessingTime, vimtProcessingTime);
            logger.WriteDebugMessage("PrepareGenericIntegrationResultRecord Call Complete");

            try
            {
                organizationService.Create(integrationResult);
            }
            catch (Exception ex)
            {
                logger.WriteToFile($"CreateIntegrationResultOnPersonSearch {ex.Message}, integrationResultName - {integrationResultName}, exceptionOccured - {exceptionOccured}, errorMessage - {errorMessage}, vimtRequest - {vimtRequest}, integrationRequest - {integrationRequest}, vimtResponse - {vimtResponse}, vimtRequestMessageType - {vimtRequestMessageType}, vimtResponseMessageType - {vimtResponseMessageType},vimtMessageRegistryName - {vimtMessageRegistryName},organizationService - {organizationService}, vimtLagTime - {vimtLagTime},eCProcessingTime - { eCProcessingTime},vimtProcessingTime - { vimtProcessingTime}");
                throw;
            }
        }

        public static void CreateIntegrationResultOnGetPersonIdentifiers(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            IOrganizationService organizationService,
            int vimtLagTime,
            int eCProcessingTime,
            int vimtProcessingTime, MCSLogger logger)
        {
            logger.WriteDebugMessage($"PrepareGenericIntegrationResultRecord Initiated\nParameter values: \nintegrationResultName - {integrationResultName}, exceptionOccured - {exceptionOccured}, errorMessage - {errorMessage}, vimtRequest - {vimtRequest}, integrationRequest - {integrationRequest}, vimtResponse - {vimtResponse}, vimtRequestMessageType - {vimtRequestMessageType}, vimtResponseMessageType - {vimtResponseMessageType},vimtMessageRegistryName - {vimtMessageRegistryName},organizationService - {organizationService}, vimtLagTime - {vimtLagTime},eCProcessingTime - { eCProcessingTime},vimtProcessingTime - { vimtProcessingTime}");

            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, exceptionOccured, vimtRequest, integrationRequest, vimtResponse, vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, eCProcessingTime, vimtProcessingTime);
            logger.WriteDebugMessage("PrepareGenericIntegrationResultRecord Call Complete");

            try
            {
                organizationService.Create(integrationResult);
            }
            catch (Exception ex)
            {
                logger.WriteToFile($"CreateIntegrationResultOnGetPersonIdentifiers {ex.Message}, integrationResultName - {integrationResultName}, exceptionOccured - {exceptionOccured}, errorMessage - {errorMessage}, vimtRequest - {vimtRequest}, integrationRequest - {integrationRequest}, vimtResponse - {vimtResponse}, vimtRequestMessageType - {vimtRequestMessageType}, vimtResponseMessageType - {vimtResponseMessageType},vimtMessageRegistryName - {vimtMessageRegistryName},organizationService - {organizationService}, vimtLagTime - {vimtLagTime},eCProcessingTime - { eCProcessingTime},vimtProcessingTime - { vimtProcessingTime}");
                throw;
            }
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Integration Result Name.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Request Message Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Response Message Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="appointmentId">Appointment Id.</param>
        /// <param name="organizationService">Organization Service Proxy.</param>
        public static void CreateAppointmentIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string integrationRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid appointmentId,
            IOrganizationService organizationService, 
            string vimtRequest,
            string vimtResponse,
            int? vimtLagTime,
            int? eCProcessingTime,
            int? vimtProcessingTime)
        {
            var integrationResult = PrepareGenericIntegrationResultRecord(integrationResultName, errorMessage, true, vimtRequest, integrationRequest, vimtResponse, vimtRequestMessageType, vimtResponseMessageType, vimtMessageRegistryName, organizationService, vimtLagTime, eCProcessingTime, vimtProcessingTime);
            integrationResult.mcs_appointmentid = new EntityReference(DataModel.Appointment.EntityLogicalName, appointmentId);
            organizationService.Create(integrationResult);

            UpdateAppointment(organizationService, appointmentId, Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure);
        }

        private static mcs_integrationresult PrepareGenericIntegrationResultRecord(string integrationResultName,
            string errorMessage,
            bool exceptionOccured,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            IOrganizationService organizationService,
            int? vimtLagTime,
            int? eCProcessingTime,
            int? vimtProcessingTime,
            MCSLogger logger = null, PatientIntegrationResultInformation patientIntegrationResultInformation = null)
        {
            var lag = vimtLagTime == null ? decimal.Zero : ConvertSecondsToMilliseconds(vimtLagTime.Value);
            var vimtProcessing = vimtProcessingTime == null ? decimal.Zero : ConvertSecondsToMilliseconds(vimtProcessingTime.Value);
            var ecProcessing = eCProcessingTime == null ? decimal.Zero : ConvertSecondsToMilliseconds(eCProcessingTime.Value);
            logger?.WriteDebugMessage("Timers calculated. Creating the result object");

            //Added ternaries in case the timers' precision on the 2 different machines aren't exact and it leads to a negative result.  Set to 0 in this case, otherwise use true value
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = BoundString(errorMessage, 25000),
                mcs_vimtrequest = BoundString(vimtRequest, 5000),
                mcs_integrationrequest = BoundString(integrationRequest, 5000),
                mcs_vimtresponse = BoundString(vimtResponse, 50000),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                cvt_ECProcessing = ecProcessing,
                cvt_VIMTProcessing = ecProcessing > vimtProcessing ? 0 : vimtProcessing - ecProcessing,
                cvt_VIMTLag = vimtProcessing > lag ? 0 : lag - vimtProcessing,
                cvt_controlid = patientIntegrationResultInformation?.ControlId
            };

            if (exceptionOccured)
                integrationResult.mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.Error);
            else
                integrationResult.mcs_status = (patientIntegrationResultInformation == null) ? new OptionSetValue((int)mcs_integrationresultmcs_status.Complete) : new OptionSetValue((int)mcs_integrationresultmcs_status.WaitingForResponse);

            return integrationResult;
        }
        
        public static int WriteVistaResults(WriteResults results, Guid parentRecordId, Guid apptId, Guid saId, string message, IOrganizationService organizationService, MCSLogger logger, EntityReference patient, int status = 0)
        {
            var failures = 0;
            foreach (var result in results.WriteResult)
            {
                CreateOrUpdateWriteResult(result, apptId, saId, parentRecordId, organizationService, logger, patient);

                if (result.VistaStatus == VIMT.Shared.VistaStatus.FAILED_TO_SCHEDULE || result.VistaStatus == VIMT.Shared.VistaStatus.FAILED_TO_CANCEL)
                {
                    failures++;
                    logger.WriteDebugMessage(string.Format("Failure in Cancel or Booking for {0} at {1}", result.Name.ToString(), result.FacilityName));
                }
            }
            if (message.ToLower() == "book")
                return ReturnBookStatusFromWriteResults(failures, results.WriteResult.Length);
            if (message.ToLower() == "cancel")
                return ReturnCancelStatusFromWriteResults(failures, results.WriteResult.Length, status);
            throw new InvalidPluginExecutionException("unknown operation for WriteVistaResults");
        }

        private static void CreateOrUpdateWriteResult(WriteResult result, Guid apptId, Guid saId, Guid IntegrationResultId, IOrganizationService organizationService, MCSLogger logger, EntityReference patient)
        {
            logger.WriteDebugMessage("Creating/Updating Write Result");
            cvt_vistaintegrationresult crmResult = null;
            var reason = !string.IsNullOrEmpty(result.Reason) && result.Reason.Length > 4000 ? BoundString(result.Reason, 3999) + "*" : result.Reason;

            using (var srv = new Xrm(organizationService))
            {
                //Figure out how to determine which Vista Booking 
                if (saId != Guid.Empty)
                    crmResult = srv.cvt_vistaintegrationresultSet.FirstOrDefault(wr =>
                        wr.cvt_ServiceActivity.Id == saId
                        && wr.cvt_PersonId == result.PersonId
                        && wr.cvt_FacilityCode == result.FacilityCode
                        && wr.cvt_ClinicIEN == result.ClinicIen);
                else if (apptId != Guid.Empty)
                    crmResult = srv.cvt_vistaintegrationresultSet.FirstOrDefault(wr =>
                        wr.cvt_Appointment.Id == apptId
                        && wr.cvt_PersonId == result.PersonId
                        && wr.cvt_FacilityCode == result.FacilityCode
                        && wr.cvt_ClinicIEN == result.ClinicIen);
            }
            if (crmResult == null)
            {
                logger.WriteDebugMessage(string.Format("No Write Result found, creating new result for {0}", result.Name.ToString()));
                var record = new cvt_vistaintegrationresult
                {
                    cvt_ClinicIEN = result.ClinicIen,
                    cvt_ClinicName = result.ClinicName,
                    cvt_DateTime = result.DateTime,
                    cvt_FacilityCode = result.FacilityCode,
                    cvt_FacilityName = result.FacilityName,
                    cvt_PatientName = result.Name.ToString(),
                    cvt_PersonId = result.PersonId,
                    cvt_Reason = reason,
                    cvt_VistAStatus = result.VistaStatus.ToString(),
                    cvt_ParentResult = new EntityReference { Id = IntegrationResultId, LogicalName = mcs_integrationresult.EntityLogicalName },
                    cvt_name = string.Format("{0} - at {1} ({2})", result.Name.ToString(), result.FacilityName, result.FacilityCode),
                    cvt_VistaReasonCode = result.VistaCancelReason == "UNKNOWN" ? "" : result.VistaCancelReason,
                    cvt_VistaStatusCode = result.VistaCancelCode == "UNKNOWN" ? "" : result.VistaCancelCode,
                    cvt_Veteran = patient
                };
                if (saId != Guid.Empty) 
                    record.cvt_ServiceActivity = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, saId);

                if (apptId != Guid.Empty)
                    record.cvt_Appointment = new EntityReference(DataModel.Appointment.EntityLogicalName, apptId);
                try
                {
                    organizationService.Create(record);
                }
                catch(Exception ex)
                {
                    logger.WriteToFile("Unable to create Vista Integration Result.  Error: " + CvtHelper.BuildExceptionMessage(ex));
                }
            }
            else
            {
                logger.WriteDebugMessage(string.Format("Found existing Write Result for {0} at {1}, updating Vista Integration record {2}", result.Name.ToString(), result.FacilityCode, crmResult.cvt_name));
                if (crmResult.cvt_VistAStatus != result.VistaStatus.ToString())
                {
                    if ((crmResult.cvt_VistAStatus == VistaStatus.SCHEDULED.ToString() && result.VistaStatus.ToString() == VistaStatus.FAILED_TO_SCHEDULE.ToString()) || 
                        (crmResult.cvt_VistAStatus == VistaStatus.CANCELED.ToString() && result.VistaStatus.ToString() == VistaStatus.FAILED_TO_CANCEL.ToString()) )
                    {
                        logger.WriteToFile(string.Format("Write Result for {2} not updated, old status: {0}, new status: {1}", crmResult.cvt_VistAStatus, result.VistaStatus, crmResult.cvt_name));
                    }
                    else
                    {
                        var status = result.VistaStatus.ToString();
                        var updateResult = new cvt_vistaintegrationresult
                        {
                            Id = crmResult.Id,
                            cvt_VistAStatus = status,
                            cvt_Reason = reason,
                            cvt_VistaStatusCode = result.VistaCancelCode,
                            cvt_VistaReasonCode = result.VistaCancelReason
                        };
                        try
                        {
                            organizationService.Update(updateResult);
                        }
                        catch (Exception ex)
                        {
                            logger.WriteToFile(string.Format("Unable to update Vista Integration Result for {0}.  Error: {1}", crmResult.cvt_name, CvtHelper.BuildExceptionMessage(ex)));
                        }
                    }
                }
                else
                {
                    logger.WriteDebugMessage(string.Format("Status of Vista Integration Result did not change, skipping update of {0}", crmResult.cvt_name));
                }
            }
        }

        public static int ReturnBookStatusFromWriteResults(int failures, int writeResultCount)
        {
            if (failures == 0)
                return (int)serviceappointment_statuscode.ReservedScheduled;
            if (failures > 0 && failures < writeResultCount)
                return (int)serviceappointment_statuscode.PartialVistaFailure;
            if (failures > 0 && failures == writeResultCount)
                return (int)serviceappointment_statuscode.VistaFailure;
            throw new InvalidPluginExecutionException(string.Format("Unexpected number of Vista Books/Cancels.  {0} Failures, {1} Write Results(attempts)", failures, writeResultCount));
        }

        public static int ReturnCancelStatusFromWriteResults(int failures, int writeResultCount, int saStatus)
        {
            var status = 0;
            if (failures == 0)
                status = saStatus;
            else if (failures > 0 && failures < writeResultCount)
                status = (int)serviceappointment_statuscode.CancelFailure;
            else if (failures > 0 && failures == writeResultCount)
                status = (int)serviceappointment_statuscode.CancelFailure;
            else
                throw new InvalidPluginExecutionException(string.Format("Unexpected number of Vista Cancels.  {0} Failures, {1} Write Results(attempts)", failures, writeResultCount));
            return status;
        }

        internal static Guid GetPatIdFromIcn(string icn, IOrganizationService organizationService)
        {
            Guid id;
            using (var srv = new Xrm(organizationService))
            {
                var personIdentifier = srv.mcs_personidentifiersSet.FirstOrDefault(i => i.mcs_identifier == icn);
                if (personIdentifier != null && personIdentifier.mcs_patient != null)
                {
                    var pat = srv.ContactSet.FirstOrDefault(c => c.Id == personIdentifier.mcs_patient.Id);

                    if (pat == null)
                        throw new InvalidPluginExecutionException(string.Format("Person Identifier {0} is not linked to a patient.  ", icn));
                    id = pat.Id;
                }
                else
                {
                    throw new InvalidPluginExecutionException("No patient could be found with ICN = " + icn);
                }
            }
            return id;
        }

        public static List<Guid> GetPatientsFromActivityPartyList(List<ActivityParty> currentPatientsApList)
        {
            var patients = new List<Guid>();
            foreach (var ap in currentPatientsApList)
            {
                patients.Add(ap.PartyId.Id);
            }
            return patients;
        }

        /// <summary>
        /// Builds an error message from an exception recursively.
        /// </summary>
        /// <param name="ex">Exception.</param>
        /// <returns>Exception message.</returns>
        public static string BuildErrorMessage(Exception ex)
        {
            var errorMessage = ex.Message;

            if (ex.InnerException == null) return errorMessage;

            //errorMessage += string.Format("\n\n{0}\n", ex.InnerException.Message);
            errorMessage += "\n" + BuildErrorMessage(ex.InnerException);

            return errorMessage;
        }

        public static int GetVimtTimeout(Xrm srv, MCSLogger Logger, string type)
        {
            var timeout = 0;
            var timeoutRecord = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VimtTimeout_" + type);
            if (timeoutRecord == null)//if there is no timeout specific to the plugin calling this, then grab the default VimtTimeout setting
            {
                timeoutRecord = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VimtTimeout");
                Logger.WriteDebugMessage($"No Integration Specific Timeout with the name VimtTimeout_{type} has been set, using System Wide VIMT Timeout.");
            }
            if (timeoutRecord == null)
                Logger.WriteToFile("No VIMT Timeout Setting with the name VimtTimeout has be set.  Using default timeout period");
            else
            {
                try
                {
                    timeout = Convert.ToInt32(timeoutRecord.mcs_value);
                }
                catch(Exception ex)
                {
                    Logger.WriteToFile("Invalid Format for Vimt Timeout Setting: " + BuildErrorMessage(ex));
                }
                if (timeout < 0)
                    timeout = 0;
            }
            return timeout;
        }

        public static void ChangeEntityStatus(IOrganizationService service, Entity entity, int statusValue, bool useUpdateNotSsr = false)
        {
            var attributeRequest = new RetrieveAttributeRequest
            {
                EntityLogicalName = entity.LogicalName,
                LogicalName = "statuscode",
                RetrieveAsIfPublished = true
            };
            var attributeResponse = (RetrieveAttributeResponse)service.Execute(attributeRequest);
            var statusMetadata = (StatusAttributeMetadata)attributeResponse.AttributeMetadata;
            var status = (StatusOptionMetadata)statusMetadata.OptionSet.Options.FirstOrDefault(option => option.Value == statusValue);
            if (status == null)
                throw new InvalidPluginExecutionException(string.Format("{0} is an invalid status for Changing Entity State of {1} with id {2}", statusValue, entity.LogicalName, entity.Id));
            if (status.State == ((OptionSetValue)(entity.Attributes["statecode"])).Value && useUpdateNotSsr)
            {
                var updateEntity = new Entity { Id = entity.Id, LogicalName = entity.LogicalName };
                if (status.Value == null)
                    throw new InvalidPluginExecutionException(string.Format("Invalid Status Reason List: unable to Change Status for record: {0}; id={1}", entity.LogicalName, entity.Id));
                updateEntity.Attributes.Add(new KeyValuePair<string, object>("statuscode", new OptionSetValue(status.Value.Value)));
                try
                {
                    service.Update(updateEntity);
                }
                catch (Exception ex)
                {
                    throw new InvalidPluginExecutionException(string.Format("Failed to Update status of {0} with id {1}.  Error: {2}", entity.LogicalName, entity.Id, ex.Message));
                }
            }
            else
            {
                var stateRequest = new SetStateRequest
                {
                    State = new OptionSetValue((int)status.State),
                    Status = new OptionSetValue((int)status.Value),
                    EntityMoniker = new EntityReference
                    {
                        LogicalName = entity.LogicalName,
                        Id = entity.Id
                    }
                };
                try
                {
                    service.Execute(stateRequest);
                }
                catch (Exception ex)
                {
                    throw new InvalidPluginExecutionException(string.Format("Set State Request failed for {0} with id {1}.  Error: {2}", entity.LogicalName, entity.Id, ex.Message));
                }
            }
        }

        internal static string BoundString(string input, int length)
        {
            return string.IsNullOrEmpty(input) ? "" : input.Length > length ? input.Substring(0, length) : input;
        }

        /// <summary>
        /// Gets the Veteran Id for all new or removed veterans
        /// </summary>
        /// <returns></returns>
        internal static Dictionary<Guid, bool> GetListOfNewPatients(IOrganizationService OrganizationService, Entity PrimaryEntity, MCSLogger Logger, IPluginExecutionContext PluginExecutionContext)
        {
            List<Guid> currentCustomers;
            string key = string.Empty;
            if (PrimaryEntity.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
            {
                currentCustomers = PrimaryEntity.ToEntity<DataModel.ServiceAppointment>().Customers?.Select(ap => ap.PartyId.Id).ToList() ?? new List<Guid>();
                key = "customers";
            }
            else if (PrimaryEntity.LogicalName == DataModel.Appointment.EntityLogicalName)
            {
                currentCustomers = PrimaryEntity.ToEntity<DataModel.Appointment>().OptionalAttendees?.Select(ap => ap.PartyId.Id).ToList() ?? new List<Guid>();
                key = "optionalattendees";
            }
            else
                throw new InvalidPluginExecutionException("Failed to Retrieve current List of Veterans for Proxy Add");

            var previousCustomers = new List<Guid>();

            if (PluginExecutionContext.MessageName != "Create")
            {
                ////previously using images to determine who was added.  Doesn't work for retry, Int Results handles retry or direct additions of new patients
                //var pre = PluginExecutionContext.PreEntityImages["pre"];
                //var patientsAtt = pre.Attributes.FirstOrDefault(k => k.Key == key);
                //if (patientsAtt.Value != null)
                //{
                //    EntityCollection ec = (EntityCollection)patientsAtt.Value;
                //    foreach (var entity in ec.Entities)
                //    {
                //        previousCustomers.Add(entity.ToEntity<ActivityParty>().PartyId.Id);
                //    }
                //}
                using (var srv = new Xrm(OrganizationService))
                {
                    //Name of the XML Node where the veteran ID is stored in VIMT Request, which is ALWAYS tracked in any integration result
                    var vetIdString = "//VeteranPartyId";
                    var pastProxyAdds = srv.mcs_integrationresultSet.Where(ir => ir.mcs_VimtMessageRegistryName == MessageRegistry.ProxyAddRequestMessage && (ir.mcs_serviceappointmentid.Id == PrimaryEntity.Id || ir.mcs_appointmentid.Id == PrimaryEntity.Id) && ir.mcs_status.Value != (int)mcs_integrationresultmcs_status.Error).ToList();
                    foreach(var ir in pastProxyAdds)
                    {
                        //Get the VeteranId
                        try
                        {
                            XmlDocument xmlDoc = new XmlDocument();
                            xmlDoc.LoadXml(ir.mcs_vimtrequest);
                            var vetId = xmlDoc.SelectSingleNode(vetIdString)?.InnerText;
                            if (!string.IsNullOrEmpty(vetId))
                                previousCustomers.Add(new Guid(vetId));
                            else
                                throw new Exception("No Veteran Id Found");
                        }
                        catch(Exception ex)
                        {
                            Logger.WriteToFile(string.Format("Unable to Find Veteran Id in integration result: {0}.  Error: {1}", ir.Id, CvtHelper.BuildExceptionMessage(ex)));
                        }                            
                    }
                }
            }
            var veteransAdded = new List<Guid>();
            var veteransRemoved = new List<Guid>();

            if (previousCustomers.Count != 0 && currentCustomers.Count != 0)
            {
                veteransAdded = currentCustomers.Except(previousCustomers).ToList();
                veteransRemoved = previousCustomers.Except(currentCustomers).ToList();
            }
            else
                veteransAdded = currentCustomers.ToList();

            var veterans = new Dictionary<Guid, bool>();
            veteransAdded.ForEach(a => veterans.Add(a, true));
            veteransRemoved.ForEach(r => veterans.Add(r, false));
            return veterans;
        }

        private static decimal ConvertSecondsToMilliseconds(int time)
        {
            return (decimal)time / 1000;
        }

        /// <summary>
        /// This method ranks the list of integration results based on their sequence in the process and returns the 1st one in the sequence
        /// </summary>
        /// <param name="irs"></param>
        /// <returns></returns>
        /// <remarks>arbitrarily picks if there is both cancel and book for same integration endpoint (aka vista book and cancel)</remarks>
        internal static mcs_integrationresult FindEarliestFailure(List<mcs_integrationresult> irs, MCSLogger logger)
        {
            var dictionary = new Dictionary<mcs_integrationresult, int>();
            foreach (var failure in irs)
            {
                switch (failure.mcs_VimtMessageRegistryName)
                {
                    case MessageRegistry.ProxyAddRequestMessage:
                        dictionary.Add(failure, 1);
                        break;
                    case MessageRegistry.HealthShareMakeCancelOutboundRequestMessage:
                        dictionary.Add(failure, 3);
                        break;
                    case MessageRegistry.VirtualMeetingRoomCreateRequestMessage:
                        dictionary.Add(failure, 2);
                        break;
                    case MessageRegistry.VideoVisitCreateRequestMessage:
                        dictionary.Add(failure, 4);
                        break;
                    case MessageRegistry.VirtualMeetingRoomDeleteRequestMessage:
                        dictionary.Add(failure, 2);
                        break;
                    case MessageRegistry.VideoVisitDeleteRequestMessage:
                        dictionary.Add(failure, 4);
                        break;
                    default:
                        logger.WriteDebugMessage($"Unknown Message Type: {failure.mcs_VimtMessageRegistryName}, skipping retry");
                        break;
                }
            }

            // Order the dictionary by the "values" (aka sort by the rankings 1-3 listed above), then get the IR (key) from the first item in the resulting dictionary
            var orderedList = dictionary.OrderBy(x => x.Value);
            
            // From entry in dictionary order by entry.Value ascending select entry;
            return orderedList.FirstOrDefault().Key;
        }
    }
}