﻿using Microsoft.Xrm.Sdk.Query;
using System.Linq;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.VideoVisit.StateObject;
using VIMT.VideoVisit.Messages;

namespace VA.TMP.Integration.VIMT.Shared
{
    internal class BusinessRuleValidator
    {
        internal string CheckPatientCount(VIMTTmpPatients[] patients)
        {
            return patients.Count() >= 0 && patients.Count() <= 50 ? string.Empty : "The maximum number of patients is 50";
        }

        internal string CheckPatientAssigningAuthorityAndName(VIMTTmpPatients[] patients)
        {
            var errors = new StringBuilder();
            foreach (var pat in patients)
            {
                if (pat.Id == null) errors.Append(string.Format("Patient {0} should have an identifier|", GetPatientName(pat)));
                else if (pat.Id.AssigningAuthority != "ICN") errors.Append(string.Format("Patient {0}'s identifier should be ICN|", GetPatientName(pat)));
                errors.Append(CheckPatientName(pat));
            }
            return errors.ToString();
        }

        internal string CheckFacilityAndClinic(VIMTTmpLocation location)
        {
            var errors = new StringBuilder();
            if (location.Type != VIMTTmpLocationType.VA) return errors.ToString();
            if (string.IsNullOrEmpty(location.Facility?.SiteCode) || string.IsNullOrEmpty(location.Facility.Name))
                errors.Append("For VA Location Type, Facility name and number are required|");
            if (string.IsNullOrEmpty(location.Clinic?.Ien) || string.IsNullOrEmpty(location.Clinic.Name))
                errors.Append("For VA Location Type, Clinic Name and IEN are required|");
            return errors.ToString();
        }

        internal string CheckPatientEmail(VIMTTmpPatients[] patients, VIMTTmpAppointmentKind appointmentKind)
        {
            if (appointmentKind == VIMTTmpAppointmentKind.MOBILE_ANY)
            {
                var errors = new StringBuilder();
                foreach (var pat in patients)
                {
                    if (string.IsNullOrEmpty(pat.ContactInformation?.PreferredEmail))
                        errors.Append(string.Format("Patient Email is Required for Mobile BYOD appointments for {0}|", GetPatientName(pat)));
                }
                return errors.ToString();
            }

            return string.Empty;
        }

        internal string CheckProviderEmail(VIMTTmpProviders[] providers)
        {
            var errors = new StringBuilder();
            foreach (var pro in providers)
            {
                if (string.IsNullOrEmpty(pro.ContactInformation?.PreferredEmail))
                    errors.Append(string.Format("Provider {0} must have an email address|", Convert(pro.Name)));
            }
            return errors.ToString();
        }

        internal string CheckPatientName(VIMTTmpPatients patient)
        {
            return patient.Name == null ? "Patient name cannot be null" : string.IsNullOrEmpty(GetPatientName(patient).Trim()) ? "Patient name cannot be empty" : string.Empty;
        }

        internal string CheckAppointmentKind(VIMTTmpCreateAppointmentRequestData createRequest)
        {
            return string.IsNullOrEmpty(createRequest.AppointmentKind.ToString()) ? "Appointment Kind cannot be null|" : string.Empty;
        }

        internal string CheckAppointmentKind(VIMTTmpUpdateAppointmentRequestData updateRequest)
        {
            return string.IsNullOrEmpty(updateRequest.AppointmentKind.ToString()) ? "Appointment Kind cannot be null|" : string.Empty;
        }

        internal string CheckPatLocation(VIMTTmpPatients patient, VIMTTmpAppointmentKind appointmentKind)
        {
            var errors = new StringBuilder();
            if (patient.Location == null) errors.Append("Patient Location information must be specified for all appointments|");
            else
            {
                switch (appointmentKind)
                {
                    case VIMTTmpAppointmentKind.CLINIC_BASED:
                    case VIMTTmpAppointmentKind.STORE_FORWARD:
                        if (patient.Location.Type != VIMTTmpLocationType.VA)
                            errors.Append(string.Format("Patient Location Type must be VA for {0} appointments|", appointmentKind));
                        errors.Append(CheckFacilityAndClinic(patient.Location));
                        break;
                    case VIMTTmpAppointmentKind.MOBILE_ANY:
                    case VIMTTmpAppointmentKind.MOBILE_GFE:
                        if (patient.Location.Type == VIMTTmpLocationType.VA)
                            errors.Append(string.Format("Patient Location Type must be NON-VA for {0} appointments|", appointmentKind));
                        break;
                }
            }
            return errors.ToString();
        }

        internal string CheckProvLocation(VIMTTmpProviders provider, VIMTTmpAppointmentKind appointmentKind)
        {
            var errors = new StringBuilder();
            switch (appointmentKind)
            {
                case VIMTTmpAppointmentKind.CLINIC_BASED:
                case VIMTTmpAppointmentKind.MOBILE_ANY:
                case VIMTTmpAppointmentKind.MOBILE_GFE:
                    if (provider.Location == null || provider.Location.Type != VIMTTmpLocationType.VA)
                        errors.Append("Provider Location type must be VA for all appointments except for Store Forward|");
                    errors.Append(CheckFacilityAndClinic(provider.Location));
                    break;
                case VIMTTmpAppointmentKind.STORE_FORWARD:
                    errors.Append("Providers should not be booked on Store Forward appointments|");
                    break;
            }

            return errors.ToString();
        }

        internal string CheckLocationTypes(VIMTTmpUpdateAppointmentRequestData updateRequest)
        {
            var errors = new StringBuilder();
            foreach (var pat in updateRequest.Patients) errors.Append(CheckPatLocation(pat, updateRequest.AppointmentKind));
            foreach (var prov in updateRequest.Providers) errors.Append(CheckProvLocation(prov, updateRequest.AppointmentKind));
            return errors.ToString();
        }

        internal string CheckLocationTypes(VIMTTmpCreateAppointmentRequestData createRequest)
        {
            var errors = new StringBuilder();
            foreach (var pat in createRequest.Patients) errors.Append(CheckPatLocation(pat, createRequest.AppointmentKind));
            foreach (var prov in createRequest.Providers) errors.Append(CheckProvLocation(prov, createRequest.AppointmentKind));
            return errors.ToString();
        }

        internal string CheckVmrs(VIMTTmpCreateAppointmentRequestData createRequest)
        {
            var errors = new StringBuilder();
            if (createRequest.AppointmentKind != VIMTTmpAppointmentKind.MOBILE_ANY) return errors.ToString();
            foreach (var pat in createRequest.Patients)
            {
                errors.Append(CheckForVmr(pat));
            }
            return errors.ToString();
        }

        internal string CheckVmrs(VIMTTmpUpdateAppointmentRequestData updateRequest)
        {
            var errors = new StringBuilder();
            if (updateRequest.AppointmentKind != VIMTTmpAppointmentKind.MOBILE_ANY) return errors.ToString();
            foreach(var pat in updateRequest.Patients)
            {
                errors.Append(CheckForVmr(pat));
            }
            return errors.ToString();
        }

        private static string CheckForVmr(VIMTTmpPatients patient)
        {
            if (patient.VirtualMeetingRoom == null) return "Patient VMR cannot be null|";
            if (string.IsNullOrEmpty(patient.VirtualMeetingRoom.Conference) || string.IsNullOrEmpty(patient.VirtualMeetingRoom.Pin) || string.IsNullOrEmpty(patient.VirtualMeetingRoom.Url))
                return "Patient VMR details must all be populated|";
            return string.Empty;
        }

        internal string ValidateCancelRequest(VIMTTmpCancelAppointmentRequestData request)
        {
            var errors = new StringBuilder();
            if (string.IsNullOrEmpty(request.Id))
                errors.Append("Appointment Id must be populated for a Cancel Request|");
            if (request.PatientBookingStatuses == null || request.PatientBookingStatuses.PersonBookingStatus.Count() == 0)
            {
                errors.Append("At least one patient cancellation must be sent in all Cancel Requests|");
                return errors.ToString();
            }

            foreach (var booking in request.PatientBookingStatuses.PersonBookingStatus)
            {
                if (string.IsNullOrEmpty(booking.Id?.AssigningAuthority) || booking.Id.AssigningAuthority != "ICN" || string.IsNullOrEmpty(booking.Id.UniqueId))
                    errors.Append("All Patients being canceled must have an ID and an ICN|");
                if (booking.Status == null) errors.Append("All cancellations must have a cancel status and reason|");

                if (booking.Status == null) errors.Append("Cancel Booking Status cannot be null");
                if (booking.Status != null && booking.Status.Reason != VIMTTmpReasonCode.OTHER) errors.Append("Cancel reason should = 'Other'|");
            }
            return errors.ToString();
        }

        internal string CheckAppointmentDateUnchanged(VIMTTmpUpdateAppointmentRequestData updateRequest, VideoVisitUpdateStateObject state)
        {
            var errors = new StringBuilder();
            using (var srv = new Xrm(state.OrganizationServiceProxy))
            {
                var results = !state.IsGroup
                    ? srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_ServiceActivity.Id == state.ServiceAppointment.Id && vir.cvt_VistAStatus == VIMTTmpVistaStatus.RECEIVED.ToString())
                    : srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_Appointment.Id == state.CrmAppointment.Id && vir.cvt_VistAStatus == VIMTTmpVistaStatus.RECEIVED.ToString());
                foreach(var result in results)
                {

                if (result == null)
                    return "";
                if (result.cvt_DateTime.Replace(" GMT","") != state.Appointment.DateTime)
                    errors.Append("Cannot change time of appointment once it has been booked in vista without canceling the Vista Appointment First|");
                    if (state.Appointment?.Providers?.FirstOrDefault() == null || state.Appointment.Providers.FirstOrDefault().Location == null || state.Appointment.Providers.FirstOrDefault().Location.Facility.SiteCode == null)
                    {
                        if (updateRequest.AppointmentKind != VIMTTmpAppointmentKind.STORE_FORWARD)
                            errors.Append("Cannot have a null provider|");
                    }
                    //if (state)
                    if (updateRequest.AppointmentKind != VIMTTmpAppointmentKind.STORE_FORWARD && result.cvt_FacilityCode != state.Appointment.Providers.FirstOrDefault().Location.Facility.SiteCode)
                        errors.Append("Cannot change the facility for the provider or patient|");

                }
            }

            state.OrganizationServiceProxy.Retrieve(state.ServiceAppointment.LogicalName, state.ServiceAppointment.Id, new ColumnSet(true)).ToEntity<ServiceAppointment>();
            return errors.ToString();
        }

        internal string GetPatientName(VIMTTmpPatients patient)
        {
            return patient.Name == null ? "unknown name" : Convert(patient.Name);
        }

        internal string Convert(VIMTTmpPersonName name)
        {
            return string.Format("{0}, {1} {2}", name.LastName, name.FirstName, name.MiddleInitial);
        }
    }
}