﻿using MCSUtilities2011;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace MCSShared
{
    public static partial class CvtHelper
    {
        //1)mcs_resource type = VistA Clinic - Status Reason change; create/update with a Primary Stop Code and a Default Provider and IEN and site
        //2)SystemUser - Status Reason change/isDisabled = Yes/No; change in one of the three VC fields
        //3)cvt_participatingsite -  Status Reason change; To Be Scheduled = Yes or No
        //4)cvt_facilityapproval - Status Reason change; Into or Out of Approved Status

        #region Helper Functions
        /// <summary>
        /// Pass in the  Participating Site and retrieve the type of Scheduling Package
        /// </summary>
        /// <param name="PS"></param>
        /// <param name="OrganizationService"></param>
        /// <param name="Logger"></param>
        /// <returns>'VVC', 'Group', 'Individual'</returns>
        public static string parentSP(cvt_participatingsite PS, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                #region Get the SP and validate the correct fields exist
                Logger.WriteDebugMessage("Get this PS's parent SP.");

                if (PS.cvt_resourcepackage == null)
                {
                    Logger.WriteDebugMessage("No Scheduling Package was listed as the parent of this Participating Site. PSId: " + PS.Id);
                    return string.Empty;
                }

                var parentSP = srv.cvt_resourcepackageSet.FirstOrDefault(rp => rp.Id == PS.cvt_resourcepackage.Id);
                if (parentSP == null)
                {
                    Logger.WriteDebugMessage("No Scheduling Package was retrieved as the parent of this Participating Site. PSId: " + PS.Id);
                    return string.Empty;
                }

                if (parentSP.cvt_patientlocationtype == null)
                {
                    Logger.WriteDebugMessage("No Patient Location Type is listed for this Scheduling Package. SPId: " + parentSP.Id);
                    return string.Empty;
                }

                if (parentSP.cvt_groupappointment == null)
                {
                    Logger.WriteDebugMessage("No Group/Individual indicator is listed for this Scheduling Package. SPId: " + parentSP.Id);
                    return string.Empty;
                }
                #endregion

                #region Return Type of parent SP
                if (parentSP.cvt_patientlocationtype.Value == (int)cvt_resourcepackagecvt_patientlocationtype.HomeMobile)
                {
                    return "VVC";
                }
                if (parentSP.cvt_groupappointment.Value == true)
                {
                    return "Group";
                }
                else
                    return "Individual";
                #endregion
            }
        }
        /// <summary>
        /// Pass in the Facility Approval and retrieve the type of Scheduling Package
        /// </summary>
        /// <param name="providerPS"></param>
        /// <param name="OrganizationService"></param>
        /// <param name="Logger"></param>
        /// <returns>'VVC', 'Group', 'Individual'</returns>
        public static string parentSP(cvt_facilityapproval FA, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                #region Get the SP and validate the correct fields exist
                Logger.WriteDebugMessage("Get this FA's parent SP.");

                if (FA.cvt_resourcepackage == null)
                {
                    Logger.WriteDebugMessage("No Scheduling Package was listed as the parent of this Facility Approval. FAId: " + FA.Id);
                    return string.Empty;
                }

                var parentSP = srv.cvt_resourcepackageSet.FirstOrDefault(rp => rp.Id == FA.cvt_resourcepackage.Id);
                if (parentSP == null)
                {
                    Logger.WriteDebugMessage("No Scheduling Package was retrieved as the parent of this Facility Approval. FAId: " + FA.Id);
                    return string.Empty;
                }

                if (parentSP.cvt_patientlocationtype == null)
                {
                    Logger.WriteDebugMessage("No Patient Location Type is listed for this Scheduling Package. SPId: " + parentSP.Id);
                    return string.Empty;
                }

                if (parentSP.cvt_groupappointment == null)
                {
                    Logger.WriteDebugMessage("No Group/Individual indicator is listed for this Scheduling Package. SPId: " + parentSP.Id);
                    return string.Empty;
                }
                #endregion

                #region Return Type of parent SP
                if (parentSP.cvt_patientlocationtype.Value == (int)cvt_resourcepackagecvt_patientlocationtype.HomeMobile)
                {
                    return "VVC";
                }
                if (parentSP.cvt_groupappointment.Value == true)
                {
                    return "Group";
                }
                else
                    return "Individual";
                #endregion
            }
        }
        public static List<EntityReference> getOppositePSFromPS(cvt_participatingsite inputPS, Guid VCSiteId, bool returningPatient, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("Starting");
            List<EntityReference> result = new List<EntityReference>();

            using (var srv = new Xrm(OrganizationService))
            {
                #region Get the SP and then opposite PS
                Logger.WriteDebugMessage("Get this PS's parent SP.");

                if (inputPS.cvt_resourcepackage == null)
                {
                    Logger.WriteDebugMessage("No Scheduling Package was listed as the parent of this Participating Site. PSId: " + inputPS.Id);
                    return result;
                }

                int locationType = (returningPatient) ? (int)cvt_participatingsitecvt_locationtype.Patient : (int)cvt_participatingsitecvt_locationtype.Provider;
                var siblingOppositePS = srv.cvt_participatingsiteSet.Where(ps => ps.cvt_resourcepackage.Id == inputPS.cvt_resourcepackage.Id
                        && ps.cvt_locationtype.Value == locationType
                        && ps.cvt_scheduleable.Value == true
                        && ps.statuscode.Value == (int)cvt_participatingsite_statuscode.Active);

                if (siblingOppositePS == null || siblingOppositePS.ToList().Count == 0)
                {
                    Logger.WriteDebugMessage("No opposing Participating Sites were retrieved as relating to this Participating Site. PSId: " + inputPS.Id);
                    return result;
                }

                #endregion

                #region Create List of PS; match site if VCSiteId value is present
                foreach (cvt_participatingsite ps in siblingOppositePS)
                {
                    if (VCSiteId != Guid.Empty)
                    {
                        if (ps.cvt_site.Id == VCSiteId)
                        {
                            result.Add(new EntityReference(cvt_participatingsite.EntityLogicalName, ps.Id));
                        }
                    }
                    else
                        result.Add(new EntityReference(cvt_participatingsite.EntityLogicalName, ps.Id));
                }
                #endregion
                //Return list of Participating Sites as Entity References
                return result;
            }
        }

        /// <summary>
        /// Build Constraint Based Group
        /// </summary>
        public static Guid BuildOutforSP(cvt_resourcepackage schPackage, StringBuilder strBuilder, IOrganizationService OrganizationService, int ReqCount, int constraintBasedGroupTypeCode, Boolean Site)
        {
            //constraintBasedGroupTypeCode
            //        Static = 0,
            //        Dynamic = 1,
            //        Implicit = 2

            //Correctly tag the XML
            var mainStrBuilder = BuildConstraintsXML(strBuilder);

            var constraintBasedGroupSetup = new ConstraintBasedGroup
            {
                BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, schPackage.OwningBusinessUnit.Id),
                Constraints = mainStrBuilder.ToString(),
                Name = schPackage.cvt_name,
                GroupTypeCode = new OptionSetValue(constraintBasedGroupTypeCode)
            };
            //Create the new Constraint Based Group
            var newConstraintGroup = OrganizationService.Create(constraintBasedGroupSetup);

            var newSpec = new ResourceSpec
            {
                BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, schPackage.OwningBusinessUnit.Id),
                ObjectiveExpression = @"<Expression><Body>udf ""Random""(factory,resource,appointment,request,leftoffset,rightoffset)</Body><Parameters><Parameter name=""factory"" /><Parameter name=""resource"" /><Parameter name=""appointment"" /><Parameter name=""request"" /><Parameter name=""leftoffset"" /><Parameter name=""rightoffset"" /></Parameters><Properties EvaluationInterval=""P0D"" evaluationcost=""0"" /></Expression>",
                RequiredCount = ReqCount,
                Name = "Selection Rule",
                GroupObjectId = newConstraintGroup,
                SameSite = Site
            };
            var specId = OrganizationService.Create(newSpec);
            return specId;
        }
        /// <summary>
        /// Build Constraint Based Group
        /// </summary>
        public static Guid CreateSpecId(Guid BuId, StringBuilder strBuilder, IOrganizationService OrganizationService, int ReqCount, int constraintBasedGroupTypeCode, Boolean Site)
        {
            //constraintBasedGroupTypeCode
            //        Static = 0,
            //        Dynamic = 1,
            //        Implicit = 2

            //Correctly tag the XML
            var mainStrBuilder = BuildConstraintsXML(strBuilder);

            var constraintBasedGroupSetup = new ConstraintBasedGroup
            {
                BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, BuId),
                Constraints = mainStrBuilder.ToString(),
                Name = "SystemGenerated CBG",
                GroupTypeCode = new OptionSetValue(constraintBasedGroupTypeCode)
            };
            //Create the new Constraint Based Group
            var newConstraintGroup = OrganizationService.Create(constraintBasedGroupSetup);

            var newSpec = new ResourceSpec
            {
                BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, BuId),
                ObjectiveExpression = @"<Expression><Body>udf ""Random""(factory,resource,appointment,request,leftoffset,rightoffset)</Body><Parameters><Parameter name=""factory"" /><Parameter name=""resource"" /><Parameter name=""appointment"" /><Parameter name=""request"" /><Parameter name=""leftoffset"" /><Parameter name=""rightoffset"" /></Parameters><Properties EvaluationInterval=""P0D"" evaluationcost=""0"" /></Expression>",
                RequiredCount = ReqCount,
                Name = "Selection Rule",
                GroupObjectId = newConstraintGroup,
                SameSite = Site
            };
            var specId = OrganizationService.Create(newSpec);
            return specId;
        }
        public static string GetResSpecFromRes(Guid ResourceId, IOrganizationService OrgService)
        {
            using (var srv = new Xrm(OrgService))
            {
                var resource = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == ResourceId && r.statuscode.Value == (int)mcs_resourcegroup_statuscode.Active);
                if (resource != null && resource.mcs_resourcespecguid != null)
                {
                    var builder = "resource[\"Id\"] == ";
                    builder += new Guid(resource.mcs_resourcespecguid).ToString("B");
                    builder += " || ";
                    return builder;
                }
                else
                {
                    //No ResSpec found
                    return string.Empty;
                }
            }
        }

        public static string GetProviderFullName(SystemUser UserRecord)
        {
            if (UserRecord != null)
            {
                string fullName = "";
                if (UserRecord.FirstName != null)
                    fullName = UserRecord.FirstName;
                if (UserRecord.LastName != null)
                    fullName += " " + UserRecord.LastName;
                if (UserRecord.cvt_suffix != null)
                    fullName += " " + UserRecord.cvt_suffix;

                return fullName;
            }
            else
                return string.Empty;
        }

        /// <summary>
        /// Split the Valid Provider site into an array, Join to the SP list, Sort Alphabetically, Get distinct
        /// </summary>
        /// <param name="newValues">The newly generated Values</param>
        /// <param name="existingValues">The existing Values</param>
        /// <returns>The updated list in the form of comma separated string</returns>
        private static string ConsolidateDistinctList(string newValues, string existingValues)
        {
            string consolidatedString = string.Empty;

            if (string.IsNullOrWhiteSpace(existingValues))
                consolidatedString = newValues;
            else if (string.IsNullOrWhiteSpace(newValues))
                consolidatedString = existingValues;
            else
            {
                var separator = ';';
                var newList = newValues.Split(separator).Concat(existingValues.Split(separator)).ToList();
                consolidatedString = string.Join(separator.ToString(), newList.OrderBy(q => q).Distinct());
            }

            return consolidatedString;
        }

        /// <summary>
        /// Method to update the Scheduling package with the Provider Sites, Patient Sites and Provider Details
        /// </summary>
        /// <param name="thisSP">The Scheduling Package</param>
        /// <param name="validProviderSites">The list of comma delimited provider sites</param>
        /// <param name="validPatientSites">The list of comma delimited Patient sites</param>
        /// <param name="providers">The list of comma delimited providers</param>
        /// <param name="serviceLog">The Service log</param>
        /// <param name="Logger">Logger object for logging puposes</param>
        /// <param name="OrganizationService">The Organization Service Object</param>
        /// <param name="srv">Xrm Service Context Object</param>
        /// <param name="newServiceId">The Service record Id</param>
        public static void UpdateSP(cvt_resourcepackage thisSP, string validProviderSites, String validPatientSites, String providers, String serviceLog, MCSLogger Logger, IOrganizationService OrganizationService, Xrm srv, Guid newServiceId)
        {
            Logger.WriteDebugMessage("starting");
            cvt_resourcepackage updateSP = new cvt_resourcepackage
            {
                Id = thisSP.Id
            };

            var updateCount = 0;
            Guid formerService = Guid.Empty;

            char[] charsToTrim = { ' ', ';' };

            if (newServiceId == Guid.Empty) //No Valid Service
            {
                //Set to null
                formerService = (thisSP.cvt_relatedservice != null) ? thisSP.cvt_relatedservice.Id : Guid.Empty;
                if (formerService != Guid.Empty)
                {
                    Logger.WriteDebugMessage("Service Reference is now being set to null, update the SP.");
                    updateSP.cvt_relatedservice = null;
                    updateCount += 1;
                }
            }
            else //Valid Service - always use the new id if present
            {
                formerService = (thisSP.cvt_relatedservice != null) ? thisSP.cvt_relatedservice.Id : Guid.Empty;
                updateSP.cvt_relatedservice = new EntityReference(Service.EntityLogicalName, newServiceId);
                updateCount += 1;
                //serviceLog += "Updating service.";
            }

            if (!string.IsNullOrWhiteSpace(serviceLog))
            {
                if (thisSP.cvt_servicedetails != serviceLog)
                {
                    updateSP.cvt_servicedetails = serviceLog;
                    updateCount += 1;
                }
            }
            else
            {
                updateSP.cvt_servicedetails = string.Empty;
                updateCount += 1;
            }

            //TODO: Review with Mark if we need two different fields in PS for patient/Provider sites or just have one cvt_oppositesites here.
            if (!string.IsNullOrWhiteSpace(validProviderSites))
            {
                //string consolidatedProviderSites = ConsolidateDistinctList(validProviderSites, thisSP.cvt_providersites);
                string consolidatedProviderSites = ConsolidateDistinctList(validProviderSites, "");
                var providerSiteString = ValidateLength(consolidatedProviderSites, 2500).TrimEnd(charsToTrim);

                //Compare it with the existing SP list to see if update is needed
                if (thisSP.cvt_providersites != providerSiteString)
                {
                    updateSP.cvt_providersites = providerSiteString;
                    updateCount += 1;
                }
            }
            else
            {
                updateSP.cvt_providersites = string.Empty;
                updateCount += 1;
            }

            if (!string.IsNullOrWhiteSpace(validPatientSites))
            {
                //string consolidatedPatientSites = ConsolidateDistinctList(validPatientSites, thisSP.cvt_patientsites);
                string consolidatedPatientSites = ConsolidateDistinctList(validPatientSites, "");
                var patientSiteString = ValidateLength(consolidatedPatientSites, 2500).TrimEnd(charsToTrim);

                //Compare it with the existing SP list to see if update is needed
                if (thisSP.cvt_patientsites != patientSiteString)
                {
                    updateSP.cvt_patientsites = patientSiteString;
                    updateCount += 1;
                }
            }
            else
            {
                updateSP.cvt_patientsites = string.Empty;
                updateCount += 1;
            }

            if (!string.IsNullOrWhiteSpace(providers))
            {
                //string consolidatedProvider = ConsolidateDistinctList(providers, thisPS.cvt_providers);
                string consolidatedProvider = ConsolidateDistinctList(providers, "");
                var prositeUsersToUpdate = CvtHelper.ValidateLength(consolidatedProvider, 2500).TrimEnd(charsToTrim);
                if (thisSP.cvt_providers != prositeUsersToUpdate)
                {
                    //Todo: Review whether the below condition is needed
                    if ((thisSP.cvt_providers != null || thisSP.cvt_providers != "") && (prositeUsersToUpdate != null || prositeUsersToUpdate != ""))
                    {
                        updateSP.cvt_providers = prositeUsersToUpdate;
                        updateCount += 1;
                    }
                }
            }
            else
            {
                updateSP.cvt_providers = string.Empty;
                updateCount += 1;
            }

            if (updateCount > 0)
            {
                OrganizationService.Update(updateSP);
                Logger.WriteDebugMessage("SP Updated");
                if (formerService != Guid.Empty)
                {
                    //Search on the prior service and attempt to delete it.
                    var ServiceActivity = srv.ServiceAppointmentSet.FirstOrDefault(i => i.ServiceId.Id == formerService);
                    //If an associated SA doesn't exist, delete
                    if (ServiceActivity == null)
                    {
                        OrganizationService.Delete(Service.EntityLogicalName, formerService);
                        Logger.WriteDebugMessage("Prior Service successfully deleted.");
                    }
                    else
                        Logger.WriteDebugMessage("Prior Service could not be deleted, associated Service Activities exist.");
                }
            }
            else
                Logger.WriteDebugMessage("No new systematically built values to update on SP");
        }

        public static void UpdatePS(cvt_participatingsite thisPS, string validOppositeSites, string providers, string serviceLog, MCSLogger Logger, IOrganizationService OrganizationService, Xrm srv, Guid newServiceId, string groupPatVariable)
        {
            Logger.WriteDebugMessage("starting");
            cvt_participatingsite updatePS = new cvt_participatingsite();
            updatePS.Id = thisPS.Id;

            var updateCount = 0;
            Guid formerService = Guid.Empty;

            char[] charsToTrim = { ' ', ';' };

            if (newServiceId == Guid.Empty) //No Valid Service
            {
                //Set to null
                formerService = (thisPS.cvt_relatedservice != null) ? thisPS.cvt_relatedservice.Id : Guid.Empty;
                if (formerService != Guid.Empty)
                {
                    Logger.WriteDebugMessage("Service Reference is now being set to null, update the PS.");
                    updatePS.cvt_relatedservice = null;
                    updateCount += 1;
                }
            }
            else //Valid Service - always use the new id if present
            {
                formerService = (thisPS.cvt_relatedservice != null) ? thisPS.cvt_relatedservice.Id : Guid.Empty;
                updatePS.cvt_relatedservice = new EntityReference(Service.EntityLogicalName, newServiceId);
                updateCount += 1;
                //serviceLog += "Updating service.";
            }

            if (!string.IsNullOrWhiteSpace(serviceLog))
            {
                if (thisPS.cvt_servicedetails != serviceLog)
                {
                    updatePS.cvt_servicedetails = serviceLog;
                    updateCount += 1;
                }
            }
            else
            {
                updatePS.cvt_servicedetails = string.Empty;
                updateCount += 1;
            }

            if (!string.IsNullOrWhiteSpace(validOppositeSites))
            {
                //string consolidatedOppositeSites = ConsolidateDistinctList(validOppositeSites, thisPS.cvt_oppositevalidsites);
                string consolidatedOppositeSites = ConsolidateDistinctList(validOppositeSites, "");
                var siteString = ValidateLength(consolidatedOppositeSites, 2500);
                if (thisPS.cvt_oppositevalidsites != siteString)
                {
                    updatePS.cvt_oppositevalidsites = siteString;
                    updateCount += 1;
                }
            }
            else
            {
                updatePS.cvt_oppositevalidsites = string.Empty;
                updateCount += 1;
            }

            if (!string.IsNullOrWhiteSpace(providers))
            {
                //string consolidatedProvider = ConsolidateDistinctList(providers, thisPS.cvt_providers);
                string consolidatedProvider = ConsolidateDistinctList(providers, "");
                var prositeUsersToUpdate = CvtHelper.ValidateLength(consolidatedProvider, 2500).TrimEnd(charsToTrim);
                if (thisPS.cvt_providers != prositeUsersToUpdate)
                {
                    //Todo: Review whether the below condition is needed
                    if ((thisPS.cvt_providers != null || thisPS.cvt_providers != "") && (prositeUsersToUpdate != null || prositeUsersToUpdate != ""))
                    {
                        updatePS.cvt_providers = prositeUsersToUpdate;
                        updateCount += 1;
                    }
                }
            }
            else
            {
                updatePS.cvt_providers = string.Empty;
                updateCount += 1;
            }

            if (!string.IsNullOrWhiteSpace(groupPatVariable))
            {
                if (thisPS.cvt_grouppatientbranch != groupPatVariable)
                {
                    updatePS.cvt_grouppatientbranch = groupPatVariable;
                    updateCount += 1;
                }
            }
            else
            {
                updatePS.cvt_grouppatientbranch = string.Empty;
                updateCount += 1;
            }
            if (updateCount > 0)
            {
                OrganizationService.Update(updatePS);
                Logger.WriteDebugMessage("Participating Site Updated. Name: " + thisPS.cvt_name);
                if (formerService != Guid.Empty)
                {
                    //Search on the prior service and attempt to delete it.
                    var ServiceActivity = srv.ServiceAppointmentSet.FirstOrDefault(i => i.ServiceId.Id == formerService);
                    //If an associated SA doesn't exist, delete
                    if (ServiceActivity == null)
                    {
                        OrganizationService.Delete(Service.EntityLogicalName, formerService);
                        Logger.WriteDebugMessage("Prior Service successfully deleted.");
                    }
                    else
                        Logger.WriteDebugMessage("Prior Service could not be deleted, associated Service Activities exist.");
                }

                //var schedulingPackage = srv.cvt_resourcepackageSet.FirstOrDefault(sp => sp.Id == thisPS.cvt_resourcepackage.Id);

                ////Update the Scheduling Package with the Patient Site, Provider Site and Providers string fields
                //if (schedulingPackage != null)
                //    UpdateSP(schedulingPackage, null, validOppositeSites, providers, serviceLog, Logger, OrganizationService, srv, Guid.Empty);
            }
            else
                Logger.WriteDebugMessage("No new systematically built values to update on PS");
        }

        public static string AssessProviderValidity(SystemUser providerRecord, cvt_participatingsite ProviderPS, cvt_participatingsite PatientPS, List<cvt_participatingsite> groupPatSites, bool isVVC, bool isIntrafacility, bool isGroup, MCSLogger Logger, IOrganizationService OrganizationService, out string outServiceLog)
        {
            Logger.WriteDebugMessage("Starting");
            Logger.WriteDebugMessage("isVVC: " + isVVC + ". isIntrafacility: " + isIntrafacility + ". isGroup: " + isGroup);
            var validProvider = new System.Text.StringBuilder("");
            string proVCstring = string.Empty;
            string patVCstring = string.Empty;
            Guid providerVC = Guid.Empty;

            //Determine if they have the required VCs to be included
            using (var srv = new Xrm(OrganizationService))
            {
                //Add the Provider to the Constraint Based Group              
                validProvider.Append(AddResourceToConstraintGroup(providerRecord.Id.ToString("B")));
                Logger.WriteDebugMessage("Added Provider to builder: " + (GetProviderFullName(providerRecord)));
                
                //Verify that the Provider has the correct provider side VC
                if (isVVC)
                {
                    if (providerRecord.cvt_HomeMobileVistaClinic == null)
                    {
                        proVCstring = "Missing VVC VistA Clinic on the user record. ";
                        Logger.WriteDebugMessage("Missing VVC VistA Clinic on the user record. ");
                    }
                    else
                    {
                        providerVC = providerRecord.cvt_HomeMobileVistaClinic.Id;
                        proVCstring = "Provider VVC VistA Clinic found";
                        Logger.WriteDebugMessage("Provider VVC VC identified. ");
                    }
                }
                else if (isIntrafacility)
                {
                    if (providerRecord.cvt_IntrafacilityVistaClinic == null)
                    { 
                        proVCstring = "Missing intrafacility VistA Clinic on the user record. ";
                        Logger.WriteDebugMessage("Missing Intrafacility VistA Clinic on the user record. ");
                    }
                    else
                    {
                        providerVC = providerRecord.cvt_IntrafacilityVistaClinic.Id;
                        proVCstring = "Provider Intrafacility VC identified. ";
                        Logger.WriteDebugMessage("Provider Intrafacility VC found");
                    }
                }
                else if (!isIntrafacility) //interfacility
                {
                    if (providerRecord.cvt_InterfacilityVistaClinic == null)
                    { 
                        proVCstring = "Missing interfacility VistA Clinic on the user record. ";
                        Logger.WriteDebugMessage("Missing Interfacility VistA Clinic on the user record. ");
                    }
                    else
                    {
                        providerVC = providerRecord.cvt_InterfacilityVistaClinic.Id;
                        proVCstring = "Provider Interfacility VC identified. ";
                        Logger.WriteDebugMessage("Provider Interfacility VC found");
                    }
                }
                //Validate Pro VC is active, then add the Provider VC to the Stringbuilder as a Constraint Based Group
                if (providerVC != Guid.Empty)
                {
                    var provVC = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == providerVC && r.statuscode.Value == (int)mcs_resource_statuscode.Active);
                    if (provVC == null)
                        proVCstring += ", but not active. ";
                    else
                    {
                        proVCstring = "";
                        
                        validProvider.Append(AddResourceToConstraintGroup(provVC.mcs_relatedResourceId.Id.ToString("B")));
                        Logger.WriteDebugMessage("Added Prov VC to builder");
                    }
                }
                Logger.WriteDebugMessage("isVVC: " + isVVC + ". isIntrafacility: " + isIntrafacility + ". isGroup: " + isGroup);
                //if not VVC then evaluate Patient VC
                if (!isVVC && !isGroup && PatientPS?.Id != Guid.Empty)
                {
                    Logger.WriteDebugMessage("Not VVC AND not group.");
                    //Validate Patient VC exists
                    var patVC = srv.mcs_resourceSet.FirstOrDefault(r => r.mcs_RelatedSiteId.Id == PatientPS.cvt_site.Id && r.cvt_primarystopcode == "690" && r.cvt_defaultprovider.Id == providerRecord.Id
                    && r.statuscode.Value == (int)mcs_resource_statuscode.Active);

                    //Add valid Patient side VC to the builder as a Constraint Based Group
                    if (patVC == null)
                        patVCstring = "Missing/Inactive Patient VistA Clinic at: " + PatientPS.cvt_site.Name + ". ";
                    else
                    {
                        validProvider.Append(AddResourceToConstraintGroup(patVC.mcs_relatedResourceId.Id.ToString("B")));
                        Logger.WriteDebugMessage("Added Pat VC to builder");
                    }
                }
                else if (!isVVC && isGroup)
                {
                    Logger.WriteDebugMessage("Not VVC AND is group. groupPatSites count: " + groupPatSites.Count);
                    foreach (cvt_participatingsite patSite in groupPatSites)
                    {
                        //Validate Patient VC exists
                        if (patSite != null && patSite?.cvt_site?.Id != Guid.Empty)
                        {
                            var patVC = srv.mcs_resourceSet.FirstOrDefault(r => r.mcs_RelatedSiteId.Id == patSite.cvt_site.Id && r.cvt_primarystopcode == "690" && r.cvt_defaultprovider.Id == providerRecord.Id
                            && r.statuscode.Value == (int)mcs_resource_statuscode.Active);

                            //Add valid Patient side VC to the builder as a Constraint Based Group
                            if (patVC == null)
                                patVCstring += "Missing/Inactive Patient VistA Clinic at: " + patSite.cvt_site.Name + ". ";
                            else
                            {
                                validProvider.Append(AddResourceToConstraintGroup(patVC.mcs_relatedResourceId.Id.ToString("B")));
                                Logger.WriteDebugMessage("Added Pat VC to builder");
                            }
                        }
                        else
                        {
                            Logger.WriteDebugMessage("patSite within groupPatSites list is null.");
                            patVCstring += "Missing/Inactive Patient VistA Clinic. ";
                        }
                    }
                }
                else
                    Logger.WriteDebugMessage("VVC, so no Patient VC needed. ");

                outServiceLog = String.Format("Assessed {0}: ", GetProviderFullName(providerRecord));
                if (proVCstring + patVCstring != string.Empty)
                {
                    outServiceLog += "Failed: " + proVCstring + patVCstring;
                    Logger.WriteDebugMessage("Provider Requirements failed. ");
                    return string.Empty;
                }
                else
                {
                    outServiceLog += "Passed. ";
                    Logger.WriteDebugMessage("Provider Requirements passed. ");
                    return validProvider.ToString();
                }
            }
        }

        public static EntityReference GetProviderSite(ServiceAppointment serviceAppointment, IOrganizationService organizationService, MCSLogger logger)
        {
            logger.WriteDebugMessage("Getting Provider Site");
            EntityReference providerSite = serviceAppointment.mcs_relatedprovidersite;

            if (providerSite == null)
            {
                using (var srv = new Xrm(organizationService))
                {
                    var equips = serviceAppointment.Resources.Where(ap => ap.PartyId.LogicalName == Equipment.EntityLogicalName).ToList();

                    logger.WriteDebugMessage("Getting Vista Clinics");
                    foreach (var equipment in equips)
                    {
                        var e = (Equipment)organizationService.Retrieve(Equipment.EntityLogicalName, equipment.PartyId.Id, new ColumnSet("mcs_relatedresource"));
                        if (e.mcs_relatedresource == null)
                        {
                            logger.WriteDebugMessage($"Orphaned Resource has been scheduled: {e.Name} with Id: {e.Id}. Please fix this resource and rebuild the Scheduling Package (or just re-link the equipment with the TMP Resource).");
                            break;
                        }
                        var resource = (mcs_resource)organizationService.Retrieve(mcs_resource.EntityLogicalName, e.mcs_relatedresource.Id, new ColumnSet("mcs_name", "mcs_type", "cvt_primarystopcode", "mcs_RelatedSiteId"));
                        if (resource.mcs_Type != null && resource.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic)
                        {
                            if (resource.cvt_primarystopcode == "179" || resource.cvt_primarystopcode == "693" || resource.cvt_primarystopcode == "692")
                            {
                                var providerParticipatingSites = srv.cvt_participatingsiteSet.Where(ps => ps.cvt_resourcepackage.Id == serviceAppointment.cvt_relatedschedulingpackage.Id
                                    && ps.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider
                                    && ps.cvt_scheduleable.Value == true && ps.cvt_site.Id == resource.mcs_RelatedSiteId.Id
                                    && ps.statuscode.Value == (int)cvt_participatingsite_statuscode.Active);

                                if (providerParticipatingSites == null || providerParticipatingSites.ToList().Count == 0)
                                {
                                    logger.WriteToFile($"Provider site for the Appointment {serviceAppointment.Id} is not populated. No provider Participating Sites were retrieved related to this Site Id: {resource.mcs_RelatedSiteId.Id} - {resource.mcs_RelatedSiteId.Name} and scheduling package {serviceAppointment.cvt_relatedschedulingpackage.Id}.");
                                }
                                else
                                {
                                    providerSite = resource.mcs_RelatedSiteId;
                                    break;
                                }
                            }
                        }
                    }
                }
            }
            return providerSite;
        }
        #endregion

        //Should be called from the mcs_resource UpdatePreStage or UpdatePostStageSync if the Default Provider or Primary Stop Code changes
        public static void VcProviderCheck(mcs_resource VistaClinic, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            string msg = "The following providers have Vista Clinic " + VistaClinic.mcs_name + " listed as an InterFacility, IntraFacility, or Home/Mobile clinic for which services are provided:\n";
            Logger.WriteDebugMessage("Begin ProviderCheck...");
            using (var srv = new Xrm(OrganizationService))
            {
                var relatedProviderVC = (from rp in srv.SystemUserSet
                                         where ((rp.cvt_IntrafacilityVistaClinic.Id == VistaClinic.Id) ||
                                                (rp.cvt_IntrafacilityVistaClinic.Id == VistaClinic.Id) || //This is the same as above, should it be Inter not Intra
                                                (rp.cvt_HomeMobileVistaClinic.Id == VistaClinic.Id))

                                         select new { rp }).ToList();

                Logger.WriteDebugMessage("Found " + relatedProviderVC.ToArray().Length + " providers associated with Vista Clinic " + VistaClinic.mcs_name);
                foreach (var p in relatedProviderVC)
                {
                    Logger.WriteDebugMessage("Checking the next provider...");
                    msg += p.rp.FullName + "\n";

                }
            }
            Logger.WriteDebugMessage("ProviderCheck complete!");
            msg += "Please update each listed provider prior to changing the VistaClinic.";
            //Why are we throwing here?
            throw new InvalidOperationException(msg);
        }


        #region Step 1 Validation 
        private static bool DoesUpdateProviderRequireRebuildPackage(SystemUser Provider, SystemUser ProviderPreImage, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("Starting");
            var requiresPackageRebuild = false;

            if (Provider.cvt_HomeMobileVistaClinic != null && Provider.cvt_HomeMobileVistaClinic?.Id != ProviderPreImage?.cvt_HomeMobileVistaClinic?.Id)
            {
                Logger.WriteDebugMessage("cvt_HomeMobileVistaClinic new/changed.");
                requiresPackageRebuild = true;
            }
            else if (Provider.cvt_HomeMobileVistaClinic == null && ProviderPreImage?.cvt_HomeMobileVistaClinic?.Id != null)
            {
                Logger.WriteDebugMessage("cvt_HomeMobileVistaClinic removed.");
                requiresPackageRebuild = true;
            }

            if (Provider.cvt_InterfacilityVistaClinic != null && Provider.cvt_InterfacilityVistaClinic?.Id != ProviderPreImage?.cvt_InterfacilityVistaClinic?.Id)
            {
                Logger.WriteDebugMessage("cvt_InterfacilityVistaClinic new/changed.");
                requiresPackageRebuild = true;
            }
            else if (Provider.cvt_InterfacilityVistaClinic == null && ProviderPreImage?.cvt_InterfacilityVistaClinic?.Id != null)
            {
                Logger.WriteDebugMessage("cvt_InterfacilityVistaClinic removed.");
                requiresPackageRebuild = true;
            }

            if (Provider.cvt_IntrafacilityVistaClinic != null && Provider.cvt_IntrafacilityVistaClinic?.Id != ProviderPreImage?.cvt_IntrafacilityVistaClinic?.Id)
            {
                Logger.WriteDebugMessage("cvt_IntrafacilityVistaClinic new/changed.");
                requiresPackageRebuild = true;
            }
            else if (Provider.cvt_IntrafacilityVistaClinic == null && ProviderPreImage?.cvt_IntrafacilityVistaClinic?.Id != null)
            {
                Logger.WriteDebugMessage("cvt_IntrafacilityVistaClinic removed.");
                requiresPackageRebuild = true;
            }

            Logger.WriteDebugMessage("requiresPackageRebuild: " + requiresPackageRebuild);
            return requiresPackageRebuild;
        }

        private static bool DoesUpdateVistaClinicRequireRebuildPackage(mcs_resource preImageVistaClinic, mcs_resource updateVistaClinic, MCSLogger Logger)
        {
            var requiresPackageRebuild = false;

            if (updateVistaClinic.cvt_defaultprovider != null && updateVistaClinic.cvt_defaultprovider?.Id != preImageVistaClinic?.cvt_defaultprovider?.Id)
            {
                Logger.WriteDebugMessage("cvt_defaultprovider new/changed.");
                requiresPackageRebuild = true;
            }
            else if (updateVistaClinic.cvt_defaultprovider == null && preImageVistaClinic?.cvt_defaultprovider?.Id != null)
            {
                Logger.WriteDebugMessage("cvt_defaultprovider removed.");
                requiresPackageRebuild = true;
            }
            if (updateVistaClinic.cvt_primarystopcode != preImageVistaClinic.cvt_primarystopcode)
            {
                Logger.WriteDebugMessage("cvt_primarystopcode new/changed.");
                requiresPackageRebuild = true;
            }
            else if ((updateVistaClinic.cvt_primarystopcode == null || updateVistaClinic.cvt_primarystopcode == string.Empty) 
                && preImageVistaClinic.cvt_primarystopcode != string.Empty)
            {
                Logger.WriteDebugMessage("cvt_primarystopcode removed.");
                requiresPackageRebuild = true;

            }
            //TODO: Should we restrict site update on UI?
            return requiresPackageRebuild;
        }
        #endregion

        #region Step 2 Service Building Entry Point
        public static void EntryFromFacilityApproval(cvt_facilityapproval fa, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                //Evaluate SP
                var type = parentSP(fa, OrganizationService, Logger);
                int locationType = 0;
                switch (type)
                {
                    //If non-group, get list of Patient Sites that belong to that Facility Approval
                    case "Individual":
                        locationType = (int)cvt_participatingsitecvt_locationtype.Patient;
                            break;
                    //If group, get list of Provider Sites that belong to that Facility Approval
                    case "Group":
                        locationType = (int)cvt_participatingsitecvt_locationtype.Provider;
                        break;
                    //VVC has no FA
                }
                if (locationType != 0)
                {
                    var participatingSites = srv.cvt_participatingsiteSet.Where(ps => ps.cvt_resourcepackage.Id == fa.cvt_resourcepackage.Id
                            && ps.cvt_locationtype.Value == locationType
                            && ps.cvt_scheduleable.Value == true
                            && ps.statuscode.Value == (int)cvt_participatingsite_statuscode.Active);

                    if (participatingSites == null || participatingSites.ToList().Count == 0)
                    {
                        Logger.WriteDebugMessage("No Participating Sites were retrieved as relating to this Facility Approval. FAId: " + fa.Id);
                    }
                    else
                    {
                        Logger.WriteDebugMessage("Found participating sites.  Passing records to ServiceBuildFromPSList.");
                        ServiceBuildFromPSList(participatingSites.ToList<cvt_participatingsite>(), Logger, OrganizationService);
                        
                    }
                }

                else
                    Logger.WriteDebugMessage("Error: VVC Scheduling Package should not have a Facility Approval record.");
            }
        }

        public static void EntryFromParticipatingSite(cvt_participatingsite ps, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                var psRetrieve = srv.cvt_participatingsiteSet.FirstOrDefault(p => p.Id == ps.Id);
                List<EntityReference> erList = GetERFromPS(psRetrieve, Guid.Empty, OrganizationService, Logger);

                Logger.WriteDebugMessage("Submitting erList with a count of " + erList.Count + " to ServiceBuildFromERList function.");
                //Submit erList to BuildService functions
                ServiceBuildFromERList(erList, Logger, OrganizationService);
            }
        }

        //Entry Point for Service Generation from the Provider
        //isVCNew = Yes if changed or added, No if removed
        //TODO - Determine which 3 fields changed
        public static void EntryFromProvider(SystemUser Provider, SystemUser ProviderPreImage,  bool isVCNew,int typeToBuild, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");
            //Do we need VistaClinicType, isVCNew parameters?
            bool rebuildRequired = DoesUpdateProviderRequireRebuildPackage(Provider, ProviderPreImage, Logger);

            if (rebuildRequired)
            {
                //Should we call this function from here?
                List<Guid> provider = new List<Guid>
                {
                    Provider.Id
                };

                PSFromProvider(provider, isVCNew,typeToBuild, Logger, OrganizationService);
            }

            {
                ///Do we need to perform any action to the previously selected VCs?

                #region Commented Code (delete if not used)
                //Do we need the below code?
                //using (var srv = new Xrm(OrganizationService))
                //{

                //    //Determine what has changed
                //    if (isVCNew) //Then VC is new (either added or changed)
                //    {
                //        Guid providerVC = new Guid();
                //        mcs_resource providerVistAClinic = new mcs_resource();
                //        switch (VistaClinicType.ToLower())
                //        {
                //            case "vvc":
                //                if (Provider.cvt_HomeMobileVistaClinic != null)
                //                {
                //                    providerVistAClinic = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == Provider.cvt_HomeMobileVistaClinic.Id);
                //                    if (providerVistAClinic != null)
                //                    {
                //                        //Verify that the Primary Stop Code matches the lookup field filled.
                //                        if (providerVistAClinic.cvt_primarystopcode == "179")
                //                        {
                //                            //Matching Primary Stop Code to Type of lookup field
                //                            ///Find every Provider PS, where Default Provider(Scheduling Resource on a Provider Participating Site) where SP is VVC
                //                            var participatingSites = new List<EntityReference>();

                //                            //Loop this list
                //                            if (participatingSites != null)
                //                            {
                //                                ServiceBuildFromERList(participatingSites, Logger, OrganizationService);
                //                            }
                //                        }
                //                        else
                //                        {

                //                        }
                //                    }
                //                    else
                //                    {
                //                        //ERROR
                //                    }

                //                }
                //                break;
                //            case "interfacility":
                //                if (Provider.cvt_InterfacilityVistaClinic != null)
                //                    providerVC = Provider.cvt_InterfacilityVistaClinic.Id;
                //                break;
                //            case "intrafacility":
                //                if (Provider.cvt_IntrafacilityVistaClinic != null)
                //                    providerVC = Provider.cvt_IntrafacilityVistaClinic.Id;
                //                break;
                //            default:
                //                //No Type passed in
                //                break;
                //        }

                //    }
                //    else //VC is removed
                //    {

                //    }

                //    //if one of the three VC lookups changed, then initate the check for that particular type of SP.
                //    if (Provider.Contains(""))
                //    {
                //        var relatedProviderPS = (from ps in srv.cvt_participatingsiteSet
                //                                 join schResource in srv.cvt_schedulingresourceSet on ps.Id equals schResource.cvt_participatingsite.Id
                //                                 where schResource.cvt_user.Id == Provider.Id && ps.cvt_scheduleable.Value == true

                //                                 select new
                //                                 {
                //                                     ps
                //                                 }).ToList();
                //        foreach (var ps in relatedProviderPS)
                //        {
                //            if (ps != null)
                //            {
                //                ServiceBuild(ps.ps, Logger, OrganizationService);
                //            }
                //        }

                //    }
                //}
                #endregion
            }
        }

        public static void EntryFromVistaClinic(mcs_resource preImageVistaClinic, mcs_resource updateVistaClinic, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            ///If Default Provider is changed
            ///Use the old provider to generate the list of PS
            ///Use the new provider to generate the list of PS

            ///SCENARIO cannot happen if the Default Provider field is required
            ///If Default Provider had a value, and is now null
            ///Use the old provider to generate the list of PS
            ///If the Site changed
            ///If the Facility Changed

            ///Unlikely:
            ///If the Primary Stop Code has changed
            Logger.WriteDebugMessage("Starting");

            using (var srv = new Xrm(OrganizationService))
            {
                ///Compare preImageVistaClinic with updateVistaClinic to determine what has changed.
                bool rebuildRequired = DoesUpdateVistaClinicRequireRebuildPackage(preImageVistaClinic, updateVistaClinic, Logger);

                if (rebuildRequired)
                {
                    Logger.WriteDebugMessage("Rebuild is required.");
                    bool hasDefaultProviderChanged = preImageVistaClinic.cvt_defaultprovider != null && updateVistaClinic.cvt_defaultprovider != null &&
                        updateVistaClinic.cvt_defaultprovider?.Id != preImageVistaClinic?.cvt_defaultprovider?.Id;

                    List<Guid> provList = new List<Guid>();
                    Logger.WriteDebugMessage("Use the current provider to generate a list of PS.");
                    provList.Add(updateVistaClinic.cvt_defaultprovider.Id);

                    if (hasDefaultProviderChanged)
                    {
                        Logger.WriteDebugMessage("Use the old provider to generate a list of PS.");
                        provList.Add(preImageVistaClinic.cvt_defaultprovider.Id);
                    }

                    if (provList.Count > 0)
                    {
                        bool isNew = (preImageVistaClinic != null);
                        Logger.WriteDebugMessage("Number of providers to evaluate. Count: " + provList.Count);
                        Logger.WriteDebugMessage("Passing in isNew: " + isNew);
                        PSFromProvider(provList, isNew, 0, Logger, OrganizationService);
                    }
                    else
                        Logger.WriteDebugMessage("No Participating Sites to evaluate.");
                }
            }
        }

        public static void EntryFromVistaClinic( mcs_resource VistaClinic, bool? isNew, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            ///If Default Provider is changed
            ///Use the old provider to generate the list of PS
            ///Use the new provider to generate the list of PS

            ///SCENARIO cannot happen if the Default Provider field is required
            ///If Default Provider had a value, and is now null
            ///Use the old provider to generate the list of PS
            ///If the Site changed
            ///If the Facility Changed

            ///Unlikely:
            ///If the Primary Stop Code has changed
            Logger.WriteDebugMessage("Starting Overload Method 'Entry From VistaClinic' - no preimage");

            using (var srv = new Xrm(OrganizationService))
            {
                ///Compare preImageVistaClinic with updateVistaClinic to determine what has changed.
                //bool rebuildRequired = DoesUpdateVistaClinicRequireRebuildPackage(preImageVistaClinic, updateVistaClinic, Logger);
                //it's a new clinic: rebuildRequired = TRUE
                bool rebuildRequired = true;

                if (rebuildRequired)
                {
                    Logger.WriteDebugMessage("Rebuild is required for new VC.");
                    //bool hasDefaultProviderChanged = preImageVistaClinic.cvt_defaultprovider != null && updateVistaClinic.cvt_defaultprovider != null &&
                    //    updateVistaClinic.cvt_defaultprovider?.Id != preImageVistaClinic?.cvt_defaultprovider?.Id;
                    bool hasDefaultProviderChanged = true;

                    List <Guid> provList = new List<Guid>();
                    Logger.WriteDebugMessage("Use the current provider to generate a list of PS.");
                    provList.Add(VistaClinic.cvt_defaultprovider.Id);

                    //if (hasDefaultProviderChanged)
                    //{
                    //    Logger.WriteDebugMessage("New Clinic - no 'old' provider to generate a list of PS.");
                    //    //provList.Add(preImageVistaClinic.cvt_defaultprovider.Id);
                    //}

                    if (provList.Count > 0)
                    {
                        //bool isNew = (preImageVistaClinic != null);
                        if (isNew == null)
                        { isNew = true; }

                        //need a bool for PSFromProvider, not a 'nullable' bool, so:
                        bool isNewVal = isNew.Value;

                        Logger.WriteDebugMessage("Number of providers to evaluate. Count: " + provList.Count);
                        Logger.WriteDebugMessage("Passing in isNew: " + isNew);
                        PSFromProvider(provList, isNewVal,0, Logger, OrganizationService);
                    }
                    else
                        Logger.WriteDebugMessage("No Participating Sites to evaluate.");
                }
            }
        }

        //Pass in the list of PSs, and then go generate the service for each PS
        public static void ServiceBuildFromPSList(List<cvt_participatingsite> psList, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            foreach (cvt_participatingsite ps in psList)
            {
                ServiceBuild(ps, Logger, OrganizationService);
            }
        }

        #endregion

        #region Step 3 Determining the PS/SP to check
        //Build a service for each Patient PS that has 1+ Provider PS AND FA or no FA but is intrafacility
        //If the checks pass in the Patient PS, then we can check for opposing Provider PS and verify FAs
        /// <summary>
        /// Pass in the list of Entity References, and then go generate the service for each PS/SP depending
        /// </summary>
        /// <param name="list"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        public static void ServiceBuildFromERList(List<EntityReference> list, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                if (list.Count != 0)
                {
                    foreach (EntityReference i in list)
                    {
                        switch (i.LogicalName)
                        {
                            case cvt_participatingsite.EntityLogicalName:
                                Logger.WriteDebugMessage(String.Format("ER ({0}) is a PS.", i.Id));
                                var ps = srv.cvt_participatingsiteSet.FirstOrDefault(x => x.Id == i.Id);
                                if (ps != null)
                                {
                                    ServiceBuild(ps, Logger, OrganizationService);
                                }
                                break;
                            case cvt_resourcepackage.EntityLogicalName:
                                Logger.WriteDebugMessage(String.Format("ER ({0}) is a SP.", i.Id));
                                var sp = srv.cvt_resourcepackageSet.FirstOrDefault(x => x.Id == i.Id && x.statecode.Value ==(int)cvt_resourcepackageState.Active);
                                if (sp != null)
                                {
                                    ServiceBuild(sp, Logger, OrganizationService);
                                }
                                else
                                {
                                    Logger.WriteDebugMessage("SP can't be found or is not active.");
                                }
                                break;
                        }
                    }
                }
                else
                    Logger.WriteDebugMessage("No ERs to build service from.");
            }
        }

        public static string ValidatePatientPS(cvt_resourcepackage thisSP, cvt_participatingsite patientPS, MCSLogger Logger, IOrganizationService OrganizationService, out string outServiceLog, out string patActivityParty)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                #region Declaration of Variables
                outServiceLog = "";
                patActivityParty = string.Empty;
                //Build the service rules for this Provider PS
                var builder = new System.Text.StringBuilder("");
                var builderAND = new System.Text.StringBuilder("");
                var builderPairedOR = new System.Text.StringBuilder("");
                #endregion

                #region Retrieve Patient PS Resources
                //Retrieving the Active resources listed on the Patient PS.
                var patientSchedulingResources = srv.cvt_schedulingresourceSet.Where(sr => sr.cvt_participatingsite.Id == patientPS.Id && sr.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active);
                int numberPatSR = (patientSchedulingResources != null ? patientSchedulingResources.ToList().Count : 0);
                outServiceLog += String.Format("\nReviewed Patient Site ({0}). Found {1} Patient Resources.\n", patientPS.cvt_site.Name, numberPatSR.ToString());
                Logger.WriteDebugMessage(String.Format("Found {0} Patient Scheduling Resources. ", numberPatSR.ToString()));
                //If 0, then verify SFT or throw error
                if (numberPatSR == 0)
                {
                    if (thisSP.cvt_availabletelehealthmodality.Value == (int)cvt_resourcepackagecvt_availabletelehealthmodality.StoreandForward)
                    {
                        var patSFTVC = srv.mcs_resourceSet.FirstOrDefault(r => r.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic
                                                                            && r.mcs_RelatedSiteId.Id == patientPS.Id
                                                                            && r.cvt_primarystopcode == "694"
                                                                            && r.statecode.Value == (int)mcs_resourceState.Active);
                        if (patSFTVC != null)
                        {
                            Logger.WriteDebugMessage("Adding Patient SFT VC to AND Builder");
                            outServiceLog += String.Format("Added SFT VC: {0}.\n", patSFTVC.mcs_name);
                            builderAND.Append(AddResourceToConstraintGroup(patSFTVC.mcs_relatedResourceId.Id.ToString("B")));
                        }
                        else
                        {
                            Logger.WriteDebugMessage("No SFT VC (694) at Patient Site found.");
                            outServiceLog += ("No SFT VC (694) at Patient Site found.\n");
                            return string.Empty;
                        }
                    }
                    else
                    {
                        Logger.WriteDebugMessage("No Resources needed. Finished patient side evaluation.");
                        outServiceLog += "No Resources needed.\n";
                        return string.Empty;
                    }
                }
                #endregion

                #region Loop through all of the Patient Site resources and add them to the builder
                //Resource/Equipment/User is listed as resources within the Constraints attribute of a Constraint Based Group
                //The Constraint Based Group is listed within the Resource Spec as the GroupObjectId
                //The Resource Spec is listed on the Service
                //Resource Spec contains required count 1/all
                //outServiceLog += "Evaluating Scheduling Resources.\n";


                foreach (var item in patientSchedulingResources)
                {
                    Logger.WriteDebugMessage("Starting loop for " + item.cvt_name);
                    if (item.cvt_schedulingresourcetype != null)
                    {
                        Guid sysResId = Guid.Empty;
                        var schedulingResourceRGBuilder = new System.Text.StringBuilder("");

                        switch (item.cvt_schedulingresourcetype.Value)
                        {
                            case (int)cvt_tsaresourcetype.ResourceGroup:
                                Logger.WriteDebugMessage("Resource Group");
                                outServiceLog += "Reviewed TMP Resource Group: " + item.cvt_name + ".\n";

                                //query RG and find out if it is paired
                                var resourceGroup = srv.mcs_resourcegroupSet.FirstOrDefault(rg => rg.Id == item.cvt_tmpresourcegroup.Id && rg.statuscode.Value == (int)mcs_resourcegroup_statuscode.Active);
                                if (resourceGroup != null)
                                {
                                    //if Paired, GROUP all Paired as ORs
                                    //if Paired, GROUP all Paired as ORs
                                    if (resourceGroup.mcs_Type.Value == (int)mcs_resourcetype.PairedResourceGroup)
                                    {
                                        //outServiceLog += "Reviewed TMP Resource Group: " + resourceGroup.mcs_name + ". ";

                                        //Get all Group Resources
                                        var groupResources = srv.mcs_groupresourceSet.Where(gr => gr.mcs_relatedResourceGroupId.Id == resourceGroup.Id && gr.statuscode.Value == (int)mcs_groupresource_statuscode.Active);

                                        foreach (mcs_groupresource linker in groupResources)
                                        {
                                            if (linker.mcs_RelatedResourceId != null)
                                            {
                                                //Get the Resource
                                                var resource = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == linker.mcs_RelatedResourceId.Id && r.statuscode.Value == (int)mcs_resource_statuscode.Active);
                                                if (resource != null)
                                                {
                                                    //if Group Resource is VC ignore
                                                    if (resource.mcs_Type != null && resource.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic)
                                                    {
                                                        outServiceLog += "--Ignored Vista Clinic: " + resource.mcs_name + "\n";
                                                    }
                                                    //if Group Resource is Tech or Room add it
                                                    else
                                                    {
                                                        outServiceLog += "--Added TMP Resource: " + resource.mcs_name + "\n";
                                                        schedulingResourceRGBuilder.Append(AddResourceToConstraintGroup(resource.mcs_resourcespecguid));
                                                        if (String.IsNullOrEmpty(patActivityParty))
                                                            patActivityParty = resource.mcs_relatedResourceId.Id + "|" + Equipment.EntityLogicalName + "|" + resource.mcs_relatedResourceId.Id;
                                                    }
                                                }
                                                else
                                                {
                                                    //Can't find active resource
                                                    Logger.WriteDebugMessage("Resource can't be found or is disabled.");
                                                }
                                            }
                                            else if (linker.mcs_RelatedUserId != null)
                                            {
                                                var groupUserRecord = srv.SystemUserSet.FirstOrDefault(su => su.Id == item.cvt_user.Id && su.IsDisabled.Value == false);
                                                if (groupUserRecord != null)
                                                {
                                                    builderAND.Append(AddResourceToConstraintGroup(groupUserRecord.Id.ToString("B")));
                                                    outServiceLog += "Added Patient Side User.\n";
                                                    if (String.IsNullOrEmpty(patActivityParty))
                                                        patActivityParty = groupUserRecord.Id + "|" + SystemUser.EntityLogicalName + "|" + groupUserRecord.Id;
                                                }
                                                else
                                                    Logger.WriteDebugMessage("Failed. " + item.cvt_user.Name + " is disabled. Did not add User.\n");
                                            }
                                        }
                                        if (schedulingResourceRGBuilder.ToString() != string.Empty)
                                        {
                                            //Create Resource Spec
                                            Guid specId = CreateSpecId(item.OwningBusinessUnit.Id, schedulingResourceRGBuilder, OrganizationService, -1, 0, false);

                                            //Add Resource Spec to SR
                                            //Do we need to?  We won't ever reuse it, unless we can unravel it to check it

                                            //Add SpecId to OR Builder
                                            builderPairedOR.Append(AddResourceToConstraintGroup(specId.ToString()));
                                        }
                                    }
                                    //if not paired, Add as AND to the service
                                    else if (resourceGroup.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic)
                                    {
                                        outServiceLog += "Ignored Group of VistA Clinics.\n";
                                    }
                                    //Room, Tech, Telepresenter
                                    else
                                    {
                                        //Add SpecId to AND Builder
                                        builderAND.Append(AddResourceToConstraintGroup(resourceGroup.mcs_resourcespecguid));
                                        outServiceLog += "Added Group.\n";
                                    }
                                }
                                else
                                {
                                    Logger.WriteDebugMessage("Record could not be found or is inactive.");
                                    outServiceLog += "Record could not be found or is inactive.\n";
                                }
                                break;
                            case (int)cvt_tsaresourcetype.SingleResource:
                                Logger.WriteDebugMessage("Single Resource");
                                //if this is a VC, ignore
                                var returnTMPResource = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == item.cvt_tmpresource.Id);
                                if (returnTMPResource != null)
                                //Add as AND to the service
                                {
                                    if (returnTMPResource.mcs_Type.Value != (int)mcs_resourcetype.VistaClinic)
                                    {
                                        Logger.WriteDebugMessage("Adding Equipment to AND Builder");
                                        outServiceLog += String.Format("Added TMP Resource: {0}.\n", returnTMPResource.mcs_name);
                                        builderAND.Append(AddResourceToConstraintGroup(returnTMPResource.mcs_relatedResourceId.Id.ToString("B")));
                                    }
                                    else
                                    {
                                        outServiceLog += String.Format("Ignored VistA Clinic: {0}.\n", returnTMPResource.mcs_name);
                                    }
                                }
                                else
                                {
                                    Logger.WriteDebugMessage("Record could not be found or is inactive.");
                                    outServiceLog += "Record could not be found or is inactive.\n";
                                }
                                break;
                            case (int)cvt_tsaresourcetype.SingleTelepresenter:
                            //case (int)cvt_tsaresourcetype.SingleProvider:
                                Logger.WriteDebugMessage("Single Telepresenter");
                                //Add as AND to the service
                                var userRecord = srv.SystemUserSet.FirstOrDefault(su => su.Id == item.cvt_user.Id && su.IsDisabled.Value == false);
                                if (userRecord != null)
                                {
                                    builderAND.Append(AddResourceToConstraintGroup(userRecord.Id.ToString("B")));
                                    outServiceLog += "Added Patient Side User.\n";
                                }
                                else
                                    Logger.WriteDebugMessage("Failed. " + item.cvt_user.Name + " is disabled. Did not add User.\n");
                                break;
                        }
                    }
                    else //It is a Required Field
                    {
                        outServiceLog += item.cvt_name + " missing schedulingresourcetype, was not added.\n";
                        Logger.WriteDebugMessage("Error: Following Resource has no schedulingresourcetype: " + item.cvt_name);
                        //Could query against all three lookups to try to categorize the Scheduling Resource
                    }
                }
                #endregion

                #region Sorted Successfully
                //Nest the Paired RGs as a choose 1 of ORs to join the other resources
                if (builderPairedOR.Length > 0)
                {
                    Logger.WriteDebugMessage("Adding rule of ORs for Paired Resource Groups.");
                    builderAND.Append(AddResourceToConstraintGroup(BuildOutforSP(thisSP, builderPairedOR, OrganizationService, 1, 0, false).ToString("B")));
                }
                Logger.WriteDebugMessage("Finished Sorting through Patient Site Resources BuilderAND: " + builderAND.ToString());

                var AndBranchId = BuildOutforSP(thisSP, builderAND, OrganizationService, -1, 0, false);
                Logger.WriteDebugMessage("AndBranchId:" + AndBranchId.ToString("B"));
                //Create the Choose All with the nested Paired
                builder.Append(AddResourceToConstraintGroup(AndBranchId.ToString("B")));

                //Return the Service Rules for this Patient Site if applicable
                if (builder.Length > 0)
                {
                    outServiceLog += "Added this Patient PS to the Service.\n";        
                    Logger.WriteDebugMessage("outServiceLog: " + outServiceLog);
                    Logger.WriteDebugMessage("Exiting function. Builder: " + builder.ToString());
                    return builder.ToString();
                }
                else
                {
                    outServiceLog += "No Valid Patient Resources assessed, did not add this Patient PS to the Service.\n";
                    Logger.WriteDebugMessage("outServiceLog: " + outServiceLog);
                    Logger.WriteDebugMessage("Exiting function. Builder: " + builder.ToString());
                    return string.Empty;
                }
                #endregion
            }
        }

        /// <summary>
        /// /Foreach provider on the provider site - make sure that a VC that correlates to the Patient PS, then we can use that site/provider.  if not, then skip
        /// </summary>
        /// <param name="providerPS"></param>
        /// <param name="patientPSId"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <param name="outProviders"></param>
        /// <param name="outProviderSite"></param>
        /// <param name="outServiceLog"></param>
        /// <returns></returns>
        public static string ValidateProviderPS(cvt_resourcepackage thisSP, cvt_participatingsite providerPS, Guid patientPSId, List<cvt_participatingsite> groupPatSites, MCSLogger Logger, IOrganizationService OrganizationService, out string outProviders, out string outProviderSite, out string outServiceLog)
        {
            {/*
            outServiceLog Clinic Individual example:
            Evaluating Provider Site: {name}, Patient Site: {name}.
            -Provider/Patient relationship is {intrafacility/interfacility}. Facility Approval: {Failed/Passed/Not Needed}.
            -SFT: {True/False}.
            -Found 0 Provider Scheduling Resources. No Resources found, only valid if SFT. Is SFT, {found/created} {SFT Technology @ Site}. Adding it to Provider side of Service. Finished evaluation.
            -Found 0 Provider Scheduling Resources. No Resources found, only valid if SFT. Is not SFT, Failed resource requirement check. Finished evaluation.
            -Found {n} Provider Scheduling Resources. Evaluating Scheduling Resources.

            --Due to VC Automation, skipped manually added TMP Resource VC: {0}.\n
            --Scheduling Resource had no schedulingresourcetype, was skipped: {name}


            outServiceLog VVC example:
            Evaluating Provider Site: {name}, No Patient Site: VVC.
            -Found 0 Provider Scheduling Resources. No Resources found, only valid if SFT. Is not SFT, Failed resource requirement check. Finished evaluation. 
            -Found {n} Provider Scheduling Resources. Evaluating Scheduling Resources.

             */
            }
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                #region Declaration of Variables
                bool isVVC = false;
                bool isSFT = false;
                bool isIntrafacility = true;
                bool isGroup = false;

                cvt_participatingsite patientPS = new cvt_participatingsite();

                outProviders = "";
                outProviderSite = "";
                outServiceLog = "";
                var outPatientSites = "";

                outServiceLog += String.Format("\nReviewed Provider Site: {0} ", providerPS.cvt_site.Name);
                #endregion
                

                #region Retrieve Patient Participating Site for Individual
                
                if (patientPSId != Guid.Empty)
                {
                    patientPS = srv.cvt_participatingsiteSet.FirstOrDefault(ps => ps.Id == patientPSId && ps.statuscode.Value == (int)cvt_participatingsite_statuscode.Active);
                    outServiceLog += String.Format("to Patient Site: {0}. ", patientPS.cvt_site.Name);
                }
                else if (thisSP.cvt_patientlocationtype.Value == (int)cvt_resourcepackagecvt_patientlocationtype.HomeMobile)
                {
                    isVVC = true;
                    outServiceLog += "to VVC. ";
                }
                #endregion

                #region Determine if Group and Intrafacility vs Interfacility
                if (thisSP.cvt_groupappointment.Value)
                {
                    Logger.WriteDebugMessage("Group SP detected.");
                    isGroup = true;
                    outServiceLog += "Group. ";
                    if (groupPatSites.ToList().Count > 0)
                    {

                        foreach (cvt_participatingsite patPS in groupPatSites)
                        {
                            if (providerPS.cvt_facility != null && patPS.cvt_facility != null)
                                if (providerPS.cvt_facility.Id != patPS.cvt_facility.Id)
                                    isIntrafacility = false;

                            outPatientSites += patPS.cvt_site.Name + "; ";
                        }
                        outServiceLog += outPatientSites;
                        Logger.WriteDebugMessage("Group Patient Sites: " + outPatientSites);
                        Logger.WriteDebugMessage("isGroup: " + isGroup + ". isIntrafacility: " + isIntrafacility);
                    }
                }
                #endregion
                #region Determine Intrafacility vs InterFacility for Individual
                else
                {
                    Logger.WriteDebugMessage("Individual SP");
                    //Is this an intrafacility or interfacility relationship? Evaluate if not VVC
                    if (isVVC != true)
                    {
                        if (providerPS.cvt_facility != null && patientPS.cvt_facility != null)
                            if (providerPS.cvt_facility.Id != patientPS.cvt_facility.Id)
                                isIntrafacility = false;
                        outServiceLog += isIntrafacility ? "Intrafacility. " : "Interfacility. ";

                        #region Determine SFT
                        if (thisSP.cvt_availabletelehealthmodality != null && thisSP.cvt_availabletelehealthmodality.Value == (int)cvt_resourcepackagecvt_availabletelehealthmodality.StoreandForward)
                            isSFT = true;

                        //outServiceLog += (isSFT) ? "SFT. " : "";
                        #endregion
                    }
                }
                #endregion

                #region Retrieve Provider PS Resources/SFT
                //Retrieving the Active resources listed on the Provider PS.
                var providerSchedulingResources = srv.cvt_schedulingresourceSet.Where(sr => sr.cvt_participatingsite.Id == providerPS.Id && sr.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active);
                int numberProvSR = (providerSchedulingResources != null ? providerSchedulingResources.ToList().Count : 0);

                outServiceLog += String.Format("Found {0} Provider Resources.\n", numberProvSR.ToString());
                Logger.WriteDebugMessage(String.Format("Found {0} Provider Resources. ", numberProvSR.ToString()));
                
                //If 0, then verify SFT or throw error
                if (numberProvSR == 0 || isSFT)
                {
                    Logger.WriteDebugMessage("No Resources found or is SFT. ");
                    //Could be SFT Provider PS, in that case this is valid - continue, don't exit.
                    if (isSFT)
                    {
                        Logger.WriteDebugMessage("Is SFT");
                        outServiceLog += "SFT: ";
                        outProviderSite = providerPS.cvt_site.Name + ";";
                        //so add the phantom provider SFT resource
                        var shadowSFTResource = srv.EquipmentSet.FirstOrDefault(e => e.Name == "SFT Technology @ " + providerPS.cvt_site.Name);
                        Logger.WriteDebugMessage("Shadow Resource");
                        if (shadowSFTResource != null)
                        {
                            Logger.WriteDebugMessage("Found Shadow Resource. Added it to Service.");
                            outServiceLog += String.Format("Ignored Provider Resources and added {0}.\nAdded this Provider PS to the Service.\n", shadowSFTResource.Name);
                            return AddResourceToConstraintGroup(shadowSFTResource.Id.ToString("B"));
                        }
                        else
                        {
                            Logger.WriteDebugMessage("Didn't Find Shadow Resource, need to create it.");
                            var siteRecord = srv.mcs_siteSet.FirstOrDefault(s => s.Id == providerPS.cvt_site.Id);
                            if (siteRecord != null)
                            {
                                Equipment newSFT = new Equipment
                                {
                                    Name = "SFT Technology @ " + providerPS.cvt_site.Name,
                                    BusinessUnitId = siteRecord.mcs_BusinessUnitId,                                 
                                    cvt_capacity = 20,
                                    SiteId = siteRecord.mcs_RelatedActualSiteId,
                                    TimeZoneCode = siteRecord.mcs_TimeZone
                                    //mcs_relatedresource = new EntityReference(mcs_resource.EntityLogicalName, ResourceId)
                                    //TODO set the work hours?
                                    //cvt_type = ThisResource.mcs_Type
                                };

                                Guid newSFTId = OrganizationService.Create(newSFT);
                                Logger.WriteDebugMessage("Created Shadow Resource. Added it to Service.");
                                outServiceLog += String.Format("Ignored Provider Resources and added {0}.\nAdded this Provider PS to the Service.\n", newSFT.Name);
                                return AddResourceToConstraintGroup(newSFTId.ToString("B"));
                            }
                        }
                    }
                    else
                    {
                        //Return nothing to be added to the scheduling rules
                        //return String.Empty;
                        outServiceLog += "Failed: No Resources found.\n";
                        return string.Empty;
                    }
                }
                #endregion

                #region Loop through all of the Provider Site resources and add them to the builder
                //Resource/Equipment/User is listed as resources within the Constraints attribute of a Constraint Based Group
                //The Constraint Based Group is listed within the Resource Spec as the GroupObjectId
                //The Resource Spec is listed on the Service
                //Resource Spec contains required count 1/all

                Logger.WriteDebugMessage("Evaluating Provider Site Resources");
                //Build the service rules for this Provider PS
                var builder = new System.Text.StringBuilder("");
                var builderProviderAND = new System.Text.StringBuilder("");
                var builderProviderPairedOR = new System.Text.StringBuilder("");
                foreach (var item in providerSchedulingResources)
                {
                    Logger.WriteDebugMessage("Starting loop for " + item.cvt_name);
                    if (item.cvt_schedulingresourcetype != null)
                    {
                        Guid sysResId = Guid.Empty;
                        var schedulingResourceRGBuilder = new System.Text.StringBuilder("");
                        switch (item.cvt_schedulingresourcetype.Value)
                        {
                            case (int)cvt_tsaresourcetype.ResourceGroup:
                                Logger.WriteDebugMessage("Resource Group");
                                outServiceLog += "Reviewed TMP Resource Group: " + item.cvt_name + ".\n";
                                //query RG and find out if it is paired
                                var resourceGroup = srv.mcs_resourcegroupSet.FirstOrDefault(rg => rg.Id == item.cvt_tmpresourcegroup.Id && rg.statuscode.Value == (int)mcs_resourcegroup_statuscode.Active);

                                if (resourceGroup != null)
                                {
                                    //if Paired, GROUP all Paired as ORs
                                    if (resourceGroup.mcs_Type.Value == (int)mcs_resourcetype.PairedResourceGroup)
                                    {
                                        //outServiceLog += "Reviewed TMP Resource Group: " + resourceGroup.mcs_name + ". ";

                                        //Get all Group Resources
                                        var groupResources = srv.mcs_groupresourceSet.Where(gr => gr.mcs_relatedResourceGroupId.Id == resourceGroup.Id && gr.statuscode.Value == (int)mcs_groupresource_statuscode.Active);

                                        foreach (mcs_groupresource linker in groupResources)
                                        {
                                            if (linker.mcs_RelatedResourceId != null)
                                            {
                                                //Get the Resource
                                                var resource = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == linker.mcs_RelatedResourceId.Id && r.statuscode.Value == (int)mcs_resource_statuscode.Active);
                                                if (resource != null)
                                                {
                                                    //if Group Resource is VC ignore
                                                    if (resource.mcs_Type != null && resource.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic)
                                                    {
                                                        outServiceLog += "--Ignored VC: " + resource.mcs_name + "\n";
                                                    }
                                                    //if Group Resource is Tech or Room add it
                                                    else
                                                    {
                                                        outServiceLog += "--Added Resource: " + resource.mcs_name + "\n";
                                                        schedulingResourceRGBuilder.Append(AddResourceToConstraintGroup(resource.mcs_resourcespecguid));
                                                    }
                                                }
                                                else
                                                {
                                                    //Can't find active resource
                                                    Logger.WriteDebugMessage("Resource can't be found or is disabled.");
                                                }
                                            }
                                            else if (linker.mcs_RelatedUserId != null)
                                            {
                                                var providerRecord = srv.SystemUserSet.FirstOrDefault(su => su.Id == linker.mcs_RelatedUserId.Id && su.IsDisabled.Value == false);
                                                if (providerRecord != null)
                                                {
                                                    //Find correct pat VC to pair           
                                                    var result = AssessProviderValidity(providerRecord, providerPS, patientPS, groupPatSites, isVVC, isIntrafacility, isGroup, Logger, OrganizationService, out string resultLog);
                                                    outServiceLog += "--" + resultLog;
                                                    if (result == string.Empty || result == null)
                                                        outServiceLog += "Did not add Provider/VC(s).\n";
                                                    else
                                                    {
                                                        outProviders += providerRecord.FullName + ";";
                                                        outProviderSite += providerPS.cvt_site.Name + ";";
                                                        outServiceLog += "Added Provider/VC(s).\n";
                                                        var provSetSB = new System.Text.StringBuilder("");
                                                        provSetSB.Append(result);
                                                        Logger.WriteDebugMessage("Adding Provider SET as choose all to schedulingResourceRGBuilder");
                                                        schedulingResourceRGBuilder.Append(AddResourceToConstraintGroup(CreateSpecId(item.OwningBusinessUnit.Id, provSetSB, OrganizationService, -1, 0, false).ToString()));
                                                    }
                                                }
                                                else
                                                    Logger.WriteDebugMessage("Failed. " + linker.mcs_RelatedUserId.Name + " is disabled. Did not add Provider or VC(s).\n");
                                            }
                                        }
                                        if (schedulingResourceRGBuilder.ToString() != string.Empty)
                                        {
                                            //Create Resource Spec
                                            Guid specId = CreateSpecId(item.OwningBusinessUnit.Id, schedulingResourceRGBuilder, OrganizationService, 1, 0, false);

                                            //Add Resource Spec to SR
                                            //Do we need to?  We won't ever reuse it, unless we can unravel it to check it

                                            //Add SpecId to OR Builder
                                            builderProviderPairedOR.Append(AddResourceToConstraintGroup(specId.ToString()));
                                        }
                                    }
                                    //if not paired, Add as AND to the service
                                    else if (resourceGroup.mcs_Type.Value == (int)mcs_resourcetype.Provider)
                                    {
                                        outServiceLog += "Reviewed TMP Resource Group of Providers: " + resourceGroup.mcs_name + ".\n";
                                        //if there is an invalid provider, then skip that one in this Paired Resource Group

                                        //Get all Group Resources
                                        var groupResources = srv.mcs_groupresourceSet.Where(gr => gr.mcs_relatedResourceGroupId.Id == resourceGroup.Id && gr.statuscode.Value == (int)mcs_groupresource_statuscode.Active);

                                        foreach (mcs_groupresource linker in groupResources)
                                        {
                                            if (linker.mcs_RelatedUserId != null)
                                            {
                                                var providerRecord = srv.SystemUserSet.FirstOrDefault(su => su.Id == linker.mcs_RelatedUserId.Id && su.IsDisabled.Value == false);
                                                if (providerRecord != null)
                                                {
                                                    //Find correct pat VC to pair           
                                                    var result = AssessProviderValidity(providerRecord, providerPS, patientPS, groupPatSites, isVVC, isGroup, isIntrafacility, Logger, OrganizationService, out string resultLog);
                                                    //Logger.WriteDebugMessage("Finished AssessProviderValidity function 2");
                                                    outServiceLog += "--" + resultLog;
                                                    if (result == string.Empty || result == null)
                                                        outServiceLog += "Did not add Provider/VC(s).\n";
                                                    else
                                                    {
                                                        outProviders += providerRecord.FullName + ";";
                                                        outProviderSite += providerPS.cvt_site.Name + ";";
                                                        outServiceLog += "Added Provider/VC(s).\n";
                                                        var provSetSB = new System.Text.StringBuilder("");
                                                        provSetSB.Append(result);
                                                        Logger.WriteDebugMessage("Adding Provider SET as choose all to schedulingResourceRGBuilder");
                                                        schedulingResourceRGBuilder.Append(AddResourceToConstraintGroup(CreateSpecId(item.OwningBusinessUnit.Id, provSetSB, OrganizationService, -1, 0, false).ToString()));
                                                    }
                                                }
                                                else
                                                    Logger.WriteDebugMessage("Failed. " + linker.mcs_RelatedUserId.Name + " is disabled. Did not add Provider or VC(s).\n");
                                            }
                                        }
                                        Logger.WriteDebugMessage("schedulingResourceRGBuilder.ToString(): " + schedulingResourceRGBuilder.ToString());
                                        if (schedulingResourceRGBuilder.ToString() != string.Empty)
                                        {
                                            //Create Resource Spec
                                            Guid specId = CreateSpecId(item.OwningBusinessUnit.Id, schedulingResourceRGBuilder, OrganizationService, 1, 0, false);

                                            //Add Resource Spec to SR
                                            //Do we need to?  We won't ever reuse it, unless we can unravel it to check it

                                            //Add SpecId to OR Builder
                                            builderProviderAND.Append(AddResourceToConstraintGroup(specId.ToString()));
                                        }
                                    }
                                    else if (resourceGroup.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic)
                                    {
                                        outServiceLog += "Ignored Group of VistA Clinics.\n";
                                    }
                                    //Room, Tech
                                    else
                                    {
                                        //Add SpecId to AND Builder
                                        builderProviderAND.Append(AddResourceToConstraintGroup(resourceGroup.mcs_resourcespecguid));
                                        outServiceLog += "Added Group.\n";
                                    }
                                }
                                else
                                {
                                    Logger.WriteDebugMessage("Record could not be found or is inactive.");
                                    outServiceLog += "Record could not be found or is inactive. \n";
                                }
                                break;
                            case (int)cvt_tsaresourcetype.SingleResource:
                                Logger.WriteDebugMessage("Single Resource");
                                //if this is a VC, ignore
                                var returnTMPResource = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == item.cvt_tmpresource.Id);
                                if (returnTMPResource != null)
                                //Add as AND to the service
                                {
                                    if (returnTMPResource.mcs_Type.Value != (int)mcs_resourcetype.VistaClinic)
                                    {
                                        Logger.WriteDebugMessage("Adding Equipment to AND Builder");
                                        outServiceLog += String.Format("Added TMP Resource: {0}.\n", returnTMPResource.mcs_name);
                                        builderProviderAND.Append(AddResourceToConstraintGroup(returnTMPResource.mcs_relatedResourceId.Id.ToString("B")));
                                    }
                                    else
                                    {
                                        outServiceLog += String.Format("Ignored VistA Clinic: {0}.\n", returnTMPResource.mcs_name);
                                    }
                                }
                                else
                                {
                                    Logger.WriteDebugMessage("Record could not be found or is inactive.");
                                    outServiceLog += "Record could not be found or is inactive. \n";
                                }
                                break;
                            case (int)cvt_tsaresourcetype.SingleProvider:
                                Logger.WriteDebugMessage("Single Provider");
                                //case (int)cvt_tsaresourcetype.SingleTelepresenter:
                                //Add as AND to the service
                                var proRecord = srv.SystemUserSet.FirstOrDefault(su => su.Id == item.cvt_user.Id && su.IsDisabled.Value == false);
                                if (proRecord != null)
                                {
                                    //Find correct pat VC to pair           
                                    var result = AssessProviderValidity(proRecord, providerPS, patientPS, groupPatSites, isVVC, isIntrafacility, isGroup, Logger, OrganizationService, out string resultLog);
                                    outServiceLog += resultLog;
                                    if (result == string.Empty || result == null)
                                    {
                                        Logger.WriteDebugMessage("Did not add Provider or VC(s)");
                                        outServiceLog += "Did not add Provider/VC(s).\n";
                                    }
                                    else
                                    {
                                        outProviders += proRecord.FullName + ";";
                                        outProviderSite += providerPS.cvt_site.Name + ";";
                                        outServiceLog += "Added Provider/VC(s).\n";
                                        builderProviderAND.Append(result);
                                        Logger.WriteDebugMessage("Added Provider/VC(s)");
                                    }
                                }
                                else
                                    Logger.WriteDebugMessage("Failed. " + item.cvt_user.Name + " is disabled. Did not add Provider or VC(s).\n");
                                break;
                        }
                    }
                    else //It is a Required Field
                    {
                        outServiceLog += item.cvt_name + " missing schedulingresourcetype, was not added.\n";
                        Logger.WriteDebugMessage("Error: Following Resource has no schedulingresourcetype: " + item.cvt_name);
                        //Could query against all three lookups to try to categorize the Scheduling Resource
                    }
                }
                #endregion

                #region Sorted Successfully
                //Buildout Prov Only Paired here
                if (builderProviderPairedOR.Length > 0)
                {
                    Logger.WriteDebugMessage("Adding rule of ORs for Paired Resource Groups.");
                    //Nest the Paired RGs as a choose 1 of ORs to join the other provider resources
                    builderProviderAND.Append(AddResourceToConstraintGroup(BuildOutforSP(thisSP, builderProviderPairedOR, OrganizationService, 1, 0, false).ToString("B")));
                }
                Logger.WriteDebugMessage("Finished Sorting through Provider Site Resources");

                //Set variables to be output back to main function to later Update the Patient PS with the string of valid Providers, valid Provider PS, 'Service build log'
                outProviderSite = providerPS.cvt_site.Name;

                //Create the Choose All with the nested Paired
                builder.Append(AddResourceToConstraintGroup(BuildOutforSP(thisSP, builderProviderAND, OrganizationService, -1, 0, false).ToString("B")));

                //Return the Service Rules for this Provider Site if applicable
                if (outProviders == "")
                {
                    outProviderSite = string.Empty;
                    outServiceLog += "No Valid Providers assessed, did not add this Provider PS to the Service. \n";
                    Logger.WriteDebugMessage("Exiting function. No Valid Providers assessed, not returning any CBG");
                    Logger.WriteDebugMessage("outServiceLog: " + outServiceLog);
                    return string.Empty;
                }
                else
                {
                    if (isGroup)
                        outProviderSite = outPatientSites;
                    else
                        outProviderSite = providerPS.cvt_site.Name + ";";

                    outServiceLog += "Added this Provider PS to the Service. \n";
                    Logger.WriteDebugMessage("outServiceLog: " + outServiceLog);
                    Logger.WriteDebugMessage("Exiting function. Builder: " + builder.ToString());
                    return builder.ToString();
                }
                #endregion
            }
        }

        //VC only
        //non VC only
        //all
        public static List<cvt_participatingsite> GetPSFromProvider(Guid providerId, int typetobuild, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                List<cvt_participatingsite> providerPS = new List<cvt_participatingsite>();
                List<cvt_participatingsite> providerPSSecondLevel = new List<cvt_participatingsite>();


                switch (typetobuild)
                {
                    case 0: //ALL
                        //Find all Provider PS where the default provider is this provider
                        //First look at Providers directly added as Scheduling Resources
                        providerPS = (from provPS in srv.cvt_participatingsiteSet
                                          join schResource in srv.cvt_schedulingresourceSet on provPS.Id equals schResource.cvt_participatingsite.Id
                                          where provPS.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider
                                          && provPS.statuscode.Value == (int)cvt_participatingsite_statuscode.Active
                                          && schResource.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active
                                          && schResource.cvt_user.Id == providerId
                                          select provPS).ToList();

                        Logger.WriteDebugMessage("Participating Sites where Provider is added directly as a Resource: " + providerPS.Count);
                        //Next look at Providers added as Scheduling Resources as part of a Resource Group
                        providerPSSecondLevel = (from provPS in srv.cvt_participatingsiteSet
                                                     join schPackage in srv.cvt_resourcepackageSet on provPS.cvt_resourcepackage.Id equals schPackage.cvt_resourcepackageId
                                                     join schResource in srv.cvt_schedulingresourceSet on provPS.Id equals schResource.cvt_participatingsite.Id
                                                     join resourceGroup in srv.mcs_resourcegroupSet on schResource.cvt_tmpresourcegroup.Id equals resourceGroup.Id
                                                     join groupResource in srv.mcs_groupresourceSet on resourceGroup.mcs_resourcegroupId equals groupResource.mcs_relatedResourceGroupId.Id
                                                     where provPS.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider
                                                     && provPS.statuscode.Value == (int)cvt_participatingsite_statuscode.Active
                                                     && schResource.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active
                                                     && resourceGroup.statuscode.Value == (int)mcs_resourcegroup_statuscode.Active
                                                     && groupResource.statuscode.Value == (int)mcs_groupresource_statuscode.Active
                                                     && groupResource.mcs_RelatedUserId.Id == providerId
                                                     && schPackage.cvt_patientlocationtype.Value == (int)cvt_resourcepackagecvt_patientlocationtype.HomeMobile
                                                     select provPS).ToList();
                        break;
                    case 1: //VVC only
                        providerPS = (from provPS in srv.cvt_participatingsiteSet
                                          join schResource in srv.cvt_schedulingresourceSet on provPS.Id equals schResource.cvt_participatingsite.Id
                                          join schPackage in srv.cvt_resourcepackageSet on provPS.cvt_resourcepackage.Id equals schPackage.cvt_resourcepackageId
                                          where provPS.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider
                                          && provPS.statuscode.Value == (int)cvt_participatingsite_statuscode.Active
                                          && schResource.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active
                                          && schResource.cvt_user.Id == providerId
                                           && schPackage.cvt_patientlocationtype.Value == (int)cvt_resourcepackagecvt_patientlocationtype.HomeMobile
                                          select provPS).ToList();

                        providerPSSecondLevel = (from provPS in srv.cvt_participatingsiteSet
                                                     join schPackage in srv.cvt_resourcepackageSet on provPS.cvt_resourcepackage.Id equals schPackage.cvt_resourcepackageId
                                                     join schResource in srv.cvt_schedulingresourceSet on provPS.Id equals schResource.cvt_participatingsite.Id
                                                     join resourceGroup in srv.mcs_resourcegroupSet on schResource.cvt_tmpresourcegroup.Id equals resourceGroup.Id
                                                     join groupResource in srv.mcs_groupresourceSet on resourceGroup.mcs_resourcegroupId equals groupResource.mcs_relatedResourceGroupId.Id
                                                     where provPS.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider
                                                     && provPS.statuscode.Value == (int)cvt_participatingsite_statuscode.Active
                                                     && schResource.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active
                                                     && resourceGroup.statuscode.Value == (int)mcs_resourcegroup_statuscode.Active
                                                     && groupResource.statuscode.Value == (int)mcs_groupresource_statuscode.Active
                                                     && groupResource.mcs_RelatedUserId.Id == providerId
                                                     && schPackage.cvt_patientlocationtype.Value == (int)cvt_resourcepackagecvt_patientlocationtype.HomeMobile
                                                     select provPS).ToList();
                        break;

                    case 2: //Non VVC
                        providerPS = (from provPS in srv.cvt_participatingsiteSet
                                          join schResource in srv.cvt_schedulingresourceSet on provPS.Id equals schResource.cvt_participatingsite.Id
                                          where provPS.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider
                                          && provPS.statuscode.Value == (int)cvt_participatingsite_statuscode.Active
                                          && schResource.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active
                                          && schResource.cvt_user.Id == providerId
                                          select provPS).ToList();

                        Logger.WriteDebugMessage("Participating Sites where Provider is added directly as a Resource: " + providerPS.Count);
                        //Next look at Providers added as Scheduling Resources as part of a Resource Group
                        providerPSSecondLevel = (from provPS in srv.cvt_participatingsiteSet
                                                     join schPackage in srv.cvt_resourcepackageSet on provPS.cvt_resourcepackage.Id equals schPackage.cvt_resourcepackageId
                                                     join schResource in srv.cvt_schedulingresourceSet on provPS.Id equals schResource.cvt_participatingsite.Id
                                                     join resourceGroup in srv.mcs_resourcegroupSet on schResource.cvt_tmpresourcegroup.Id equals resourceGroup.Id
                                                     join groupResource in srv.mcs_groupresourceSet on resourceGroup.mcs_resourcegroupId equals groupResource.mcs_relatedResourceGroupId.Id
                                                     where provPS.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider
                                                     && provPS.statuscode.Value == (int)cvt_participatingsite_statuscode.Active
                                                     && schResource.statuscode.Value == (int)cvt_schedulingresource_statuscode.Active
                                                     && resourceGroup.statuscode.Value == (int)mcs_resourcegroup_statuscode.Active
                                                     && groupResource.statuscode.Value == (int)mcs_groupresource_statuscode.Active
                                                     && groupResource.mcs_RelatedUserId.Id == providerId
                                                     && schPackage.cvt_patientlocationtype.Value == (int)cvt_resourcepackagecvt_patientlocationtype.HomeMobile
                                                     select provPS).ToList();
                        break;

                }
                //Find all Provider PS where the default provider is this provider
                //First look at Providers directly added as Scheduling Resources
                
                Logger.WriteDebugMessage("Participating Sites where Provider is added through a TMP Resource Group: " + providerPSSecondLevel.Count);
                //Add second list to first list
                providerPS.AddRange(providerPSSecondLevel);

                //returning relevant Provider Participating Sites
                Logger.WriteDebugMessage("Participating Sites returned: " + providerPS.Count);
                return providerPS;
            }
        }

        public static List<EntityReference> GetERFromPS(cvt_participatingsite ps, Guid VCSiteId, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                List<EntityReference> erList = new List<EntityReference>();

                string spType = parentSP(ps, OrganizationService, Logger);

                //Determine if the ProviderPS has a PatientPS that is the Site listed on the 690 VC, and if so, if parent SP is group, then record the Provider PS into one list, if non - group, record the Patient PS in another list.
                Logger.WriteDebugMessage("Retrieved Scheduling Package type.");
                if (ps.cvt_locationtype != null && ps.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider)
                {
                    Logger.WriteDebugMessage("PS being assessed is Provider.");
                    //Determine if VVC, non-VVC Group or non-VVC non-Group
                    switch (spType)
                    {
                        case "VVC": //SP is VVC (group or ind), then record the parent SP as the guid
                            Logger.WriteDebugMessage("SP is VVC");
                            erList.Add(new EntityReference(cvt_resourcepackage.EntityLogicalName, ps.cvt_resourcepackage.Id));
                            break;
                        case "Individual": //SP is non-group, then record the opposite Patient PSs
                            Logger.WriteDebugMessage("SP is Individual");
                            erList.AddRange(getOppositePSFromPS(ps, VCSiteId, true, OrganizationService, Logger));
                            break;
                        case "Group": //SP is non VVC group, then record the Provider PS
                            Logger.WriteDebugMessage("SP is Group");
                            erList.Add(new EntityReference(cvt_participatingsite.EntityLogicalName, ps.Id));
                            break;
                    }
                }
                else //Patient
                {
                    Logger.WriteDebugMessage("PS being assessed is Patient.");
                    //Determine if VVC, non-VVC Group or non-VVC non-Group
                    switch (spType)
                    {
                        case "Individual": //SP is non-group, then record this Patient PS
                            erList.Add(new EntityReference(cvt_participatingsite.EntityLogicalName, ps.Id));
                            break;
                        case "Group": //SP is non VVC group, then record the opposite Provider PS
                            erList.AddRange(getOppositePSFromPS(ps, VCSiteId, false, OrganizationService, Logger));
                            break;
                    }
                }
                //Return distinct list of Entity References
                return erList.Distinct().ToList();
            }
        }

        //TODO what is the isVCNew for?
        public static void PSFromProvider(List<Guid> ProviderIds, bool isVCNew,int buildType, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");
            List<EntityReference> erList = new List<EntityReference>();

            foreach (Guid ProviderId in ProviderIds)
            {
                List<cvt_participatingsite> psList = GetPSFromProvider(ProviderId, buildType, OrganizationService, Logger);
                
                foreach (cvt_participatingsite ps in psList)
                {
                    erList.AddRange(GetERFromPS(ps, Guid.Empty, OrganizationService, Logger));
                }
            }
            //Submit erList to BuildService functions
            ServiceBuildFromERList(erList, Logger, OrganizationService);
        }

        #endregion

        #region Step 4 Building the Service

        //Can pass in either patient or provider Participating Site? Yes
        //Overload: ServiceBuild for Participating Site - this is for Clinic based Individual or Group - non VVC
        public static void ServiceBuild(cvt_participatingsite psVar, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");

            using (var srv = new Xrm(OrganizationService))
            {
                List<EntityReference> buildingServiceFromPS = new List<EntityReference>();
                #region Record and Field validation
                var ps = (cvt_participatingsite)psVar;

                //Check for scheduleable
                if (ps.cvt_scheduleable == null || ps.cvt_scheduleable.Value == false)
                {
                    Logger.WriteDebugMessage("This Participating Site 'Can Be Scheduled' = No. Skipping build process.");
                    UpdatePS(psVar, string.Empty, string.Empty, "This Participating Site 'Can Be Scheduled' = No. Skipping build process.", Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                    return;
                }

                //Check for parent Scheduling package
                var schPackage = srv.cvt_resourcepackageSet.FirstOrDefault(sp => sp.Id == ps.cvt_resourcepackage.Id);
                if (schPackage == null)
                {
                    Logger.WriteDebugMessage("SP could not be found, skipping build process.");
                    UpdatePS(psVar, string.Empty, string.Empty, "SP could not be found, skipping build process.", Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                    return;
                }
                Logger.WriteDebugMessage("Finished Record and Field Validation section.");
                #endregion

                //Defaulted to Patient
                int oppLocationType = (int)cvt_participatingsitecvt_locationtype.Patient; 
                
                if (ps.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Patient)
                {
                    Logger.WriteDebugMessage("Participating Site is Patient. Gather interfacility Provider PS where the FA is approved and/or all of the intrafacility Provider PS.");
                    oppLocationType = (int)cvt_participatingsitecvt_locationtype.Provider;
                }
                else //For PROVIDER PS, so must be Group. Get List of Opposing Patient PS
                    Logger.WriteDebugMessage("Participating Site is Provider. Gather interfacility Patient PS where the FA is approved and/or all of the intrafacility Patient PS.");

                Logger.WriteDebugMessage("Querying for the related PS");
                List<cvt_participatingsite> relatedPS = new List<cvt_participatingsite>();
                var relatedInterPS = srv.cvt_participatingsiteSet.Where(oppPS =>
                                         oppPS.cvt_facility.Id != ps.cvt_facility.Id 
                                         && oppPS.cvt_locationtype.Value == oppLocationType 
                                         && oppPS.cvt_resourcepackage.Id == ps.cvt_resourcepackage.Id 
                                         && oppPS.cvt_scheduleable.Value == true 
                                         && oppPS.statecode.Value == (int)cvt_participatingsiteState.Active).ToList();
                Logger.WriteDebugMessage("Finished querying for the opposing Interfacility PS. Count: " + relatedInterPS.Count);

                foreach (cvt_participatingsite interPS in relatedInterPS)
                {
                    Guid provFacilityId = Guid.Empty;
                    Guid patFacilityId = Guid.Empty;

                    switch (ps.cvt_locationtype.Value)
                    {
                        case (int)cvt_participatingsitecvt_locationtype.Provider:
                            provFacilityId = ps.cvt_facility.Id;
                            patFacilityId = interPS.cvt_facility.Id;
                            break;
                        case (int)cvt_participatingsitecvt_locationtype.Patient:
                            provFacilityId = interPS.cvt_facility.Id;
                            patFacilityId = ps.cvt_facility.Id;
                            break;
                    }
                    //Query for FA
                    var fa = srv.cvt_facilityapprovalSet.FirstOrDefault(thisFA => thisFA.cvt_providerfacility.Id == provFacilityId && thisFA.cvt_patientfacility.Id == patFacilityId && thisFA.statecode.Value == (int)cvt_facilityapprovalState.Active && thisFA.statuscode.Value == (int)cvt_facilityapproval_statuscode.Approved);
                    if (fa != null)
                    {
                        relatedPS.Add(interPS);
                        Logger.WriteDebugMessage("Passed FA check, adding interfacility PS to list: " + interPS.cvt_name);
                    }
                    else
                        Logger.WriteDebugMessage("Failed FA check, skipping interfacility PS: " + interPS.cvt_name);
                }

                var relatedIntraPS = srv.cvt_participatingsiteSet.Where(oppPS => 
                                        oppPS.cvt_facility.Id == ps.cvt_facility.Id 
                                        && oppPS.cvt_locationtype.Value == oppLocationType 
                                        && oppPS.cvt_resourcepackage.Id == ps.cvt_resourcepackage.Id 
                                        && oppPS.cvt_scheduleable.Value == true 
                                        && oppPS.statecode.Value == (int)cvt_participatingsiteState.Active).ToList();
                Logger.WriteDebugMessage("Finished querying for the opposing Intrafacility PS. Count: " + relatedIntraPS.Count);
                relatedPS.AddRange(relatedIntraPS);
                Logger.WriteDebugMessage("Total PS Count: " + relatedPS.Count);
                if (relatedPS == null || relatedPS.Count == 0)
                {
                    Logger.WriteDebugMessage("No opposite PS in a state to build the service, exiting service build.");
                    UpdatePS(psVar, string.Empty, string.Empty, "No opposite PS in a state to build the service, exiting service build.", Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                    return;
                }
                switch (ps.cvt_locationtype.Value)
                {
                    case (int)cvt_participatingsitecvt_locationtype.Provider:
                        //Loop through List, passing in Provider PS and List of Patient PS
                        CreateUpdateGroupService(schPackage, ps, relatedPS, Logger, OrganizationService);
                        break;
                    case (int)cvt_participatingsitecvt_locationtype.Patient:
                        //Loop through List, passing in Patient PS and List of Provider PS
                        CreateUpdateIndividualService(schPackage, relatedPS, ps, Logger, OrganizationService);
                        break;
                }
            }
        }

        //Overload: ServiceBuild for SchedulingPackage - this is for VVC
        public static void ServiceBuild(cvt_resourcepackage schPackage, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");

            using (var srv = new Xrm(OrganizationService))
            {
                List<EntityReference> buildingServiceFromSP = new List<EntityReference>();

                Logger.WriteDebugMessage("Querying for the related provider PS");
                var relatedPS = srv.cvt_participatingsiteSet.Where(p => p.cvt_locationtype.Value == (int)cvt_participatingsitecvt_locationtype.Provider 
                                && p.cvt_resourcepackage.Id == schPackage.Id 
                                && p.cvt_scheduleable.Value == true
                                && p.statecode.Value == (int)cvt_participatingsiteState.Active).ToList();
                Logger.WriteDebugMessage("Finished querying for the PS. Count: " + relatedPS.Count);

                if (relatedPS == null || relatedPS.Count == 0)
                {
                    Logger.WriteDebugMessage("No child PS in a state to build the service, exiting service build.");
                    UpdateSP(schPackage, string.Empty, string.Empty, string.Empty, "VVC Scheduling Package: No Provider Participating Sites in a state to build the service, exiting service build.", Logger, OrganizationService, srv, Guid.Empty);
                    return;
                }
                else
                {
                    Logger.WriteDebugMessage("Continuing into the function CreateUpdateVVCService");
                    CreateUpdateVVCService(schPackage, relatedPS, Logger, OrganizationService);
                }
            }
        }
       
        public static void CreateUpdateIndividualService(cvt_resourcepackage thisSP, List<cvt_participatingsite> provSites, cvt_participatingsite patientPS, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");

            Guid resourceSpecId = Guid.Empty;
            string providerSites = "";
            string providers = "";
            string serviceLog = "Clinic:Clinic Service:";

            using (var srv = new Xrm(OrganizationService))
            {
                try
                {
                    if (provSites == null)
                    {
                        Logger.WriteDebugMessage("No Provider Sites, exiting.");
                        UpdatePS(patientPS, string.Empty, string.Empty, "No Provider Sites, no service generated.", Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                        return;
                    }

                    if (patientPS.cvt_scheduleable == false)
                    {
                        Logger.WriteDebugMessage("This Patient PS is not scheduleable, no service generated.");
                        UpdatePS(patientPS, string.Empty, string.Empty, "This Patient PS is not scheduleable, no service generated.", Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                        return;
                    }
                    #region PROV SITE RESOURCES
                    Logger.WriteDebugMessage("Starting PROV SITE RESOURCES region.");
                    //Build a service for each Patient PS that has 1+ Provider PS AND FA
                    //loop through the Participating Provider Sites. If Prov PS has at least one Provider that has a VC that correlates to the Patient PS, then we can use that site/provider.  if no, then skip
                    var builderProviderSites = new System.Text.StringBuilder("");
                    foreach (var prov in provSites)
                    {
                        var singleProviderSiteCBG = ValidateProviderPS(thisSP, prov, patientPS.Id, new List<cvt_participatingsite>(), Logger, OrganizationService, out string validProviders, out string outValidSites, out string outLogOutput);
                        Logger.WriteDebugMessage("Individual. Did any Provider Site CBG get written? singleProviderSiteCBG: " + singleProviderSiteCBG);
                        serviceLog += outLogOutput;
                        if (singleProviderSiteCBG != String.Empty)
                        {
                            providerSites += outValidSites;
                            providers += validProviders;
                            builderProviderSites.Append(singleProviderSiteCBG);
                            Logger.WriteDebugMessage("Added CBG");
                        }
                        else
                            Logger.WriteDebugMessage("ValidateProviderPS for " + prov.cvt_name + " returned no valid CBG.");
                    }

                    #endregion

                    #region PATIENT SITE RESOURCES
                    //After collecting all of the patient data, if builder is not empty, then add all patient
                    var builderPat = new System.Text.StringBuilder("");
                    //Retrieve all the patient site resources
                    Logger.WriteDebugMessage("Start Sorting through Patient Site Resources");
                    var singlePatientSiteCBG = ValidatePatientPS(thisSP, patientPS, Logger, OrganizationService, out string logOutput, out string groupPatVariable);
                    serviceLog += logOutput;

                    if (singlePatientSiteCBG != string.Empty)
                    {
                        Logger.WriteDebugMessage("Patient returned something.");
                        builderPat.Append(singlePatientSiteCBG);
                    }
                    
                    Logger.WriteDebugMessage("Finished Sorting through Patient Site Resources");
                    #endregion

                    #region Build the Service Components
                    Logger.WriteDebugMessage("Starting Build the Service Components region");
                    //Validation
                    Logger.WriteDebugMessage("providerSites: " + providerSites);
                    Logger.WriteDebugMessage("providers: " + providers);
                    Logger.WriteDebugMessage("serviceLog: " + serviceLog);
                    Logger.WriteDebugMessage("builderProviderSites.Length:" + builderProviderSites.Length);
                    if (builderProviderSites.Length == 0)
                    {
                        serviceLog += "\nNo Valid Provider Participating Sites assessed, no service constructed.\n";
                        UpdatePS(patientPS, string.Empty, string.Empty, serviceLog, Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                        // throw new InvalidPluginExecutionException("Scheduling Resources must be listed on the Provider Side in order to build this service.");
                        return;
                    }

                    if (builderPat.Length == 0 && thisSP.cvt_availabletelehealthmodality.Value == (int)cvt_resourcepackagecvt_availabletelehealthmodality.StoreandForward)
                    {
                        serviceLog += "\nNo Valid Patient Participating Site assessed, no service constructed.\n";
                        UpdatePS(patientPS, string.Empty, string.Empty, serviceLog, Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                        // throw new InvalidPluginExecutionException("SFT SP must have a valid Patient SFT VC in order to build this service.");
                        return;
                    }
                    //Defaulted to 'Clinic' initially
                    var builder = new System.Text.StringBuilder("");

                    //If Ind C:C, then All Providers to 1 Patient
                    // Clinic Based 

                    if (builderProviderSites.Length > 0)
                    {
                        builder.Append(AddResourceToConstraintGroup(BuildOutforSP(thisSP, builderProviderSites, OrganizationService, 1, 0, true).ToString("B")));
                        Logger.WriteDebugMessage("Adding Provider Builder to Main Builder.");
                    }

                    if (builderPat.Length > 0)
                    {
                        builder.Append(builderPat);
                        Logger.WriteDebugMessage("Adding Patient Builder to Main Builder.");
                    }
                    #endregion

                    #region Logic - Constructing the service
                    Logger.WriteDebugMessage("Constructing the service");
                    resourceSpecId = BuildOutforSP(thisSP, builder, OrganizationService, -1, 2, false);
                    
                    int initialstatus = (int)service_initialstatuscode.Pending;
                    var duration = (thisSP.cvt_AppointmentLength != null) ? thisSP.cvt_AppointmentLength.Value : 60;
                    var startEvery = (thisSP.cvt_StartAppointmentsEvery != null) ? thisSP.cvt_StartAppointmentsEvery.ToString() : "60";

                    Service newService = new Service
                    {
                        Name = patientPS.cvt_name.ToString(),
                        AnchorOffset = 480,
                        Duration = duration,
                        InitialStatusCode = new OptionSetValue(initialstatus),
                        Granularity = "FREQ=MINUTELY;INTERVAL=" + startEvery + ";",
                        ResourceSpecId = new EntityReference(ResourceSpec.EntityLogicalName, resourceSpecId)
                    };

                    //create the service
                    Logger.WriteDebugMessage("Creating new Service");
                    var newServiceId = OrganizationService.Create(newService);
                    serviceLog += "\nConstructed the service.\n";
                    Logger.WriteDebugMessage("Updating the Participating Site.");
                    UpdatePS(patientPS, providerSites, providers, serviceLog,  Logger, OrganizationService, srv, newServiceId, string.Empty);
                    #endregion

                    #region Catch
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Logger.WriteToFile(ex.Message);
                    throw new InvalidPluginExecutionException(ex.Message);
                }
                catch (Exception ex)
                {
                    if (ex.Message.StartsWith("custom"))
                    {
                        Logger.WriteDebugMessage(ex.Message.Substring(6));

                        throw new InvalidPluginExecutionException(ex.Message.Substring(6));
                    }
                    else
                    {
                        Logger.setMethod = "Execute";
                        Logger.WriteToFile(ex.Message);
                        throw new InvalidPluginExecutionException(ex.Message);
                    }
                }
                #endregion
            }
        }

        public static void CreateUpdateVVCService(cvt_resourcepackage SchedulingPackage, List<cvt_participatingsite> provSites, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");

            Guid resourceSpecId = Guid.Empty;
            string providerSites = "";
            string providers = "";
            string serviceLog = "VA Video Connect Service:";

            using (var srv = new Xrm(OrganizationService))
            {
                try
                {
                    Logger.WriteDebugMessage("# prov sites: " + provSites.Count);
                    //loop through the Participating Provider Sites. 
                    var builderProviderSites = new System.Text.StringBuilder("");
                    foreach (cvt_participatingsite prov in provSites)
                    {
                        var singleProviderSiteCBG = ValidateProviderPS(SchedulingPackage, prov, Guid.Empty, new List<cvt_participatingsite>(), Logger, OrganizationService, out string validProviders, out string outValidSites, out string outLogOutput);

                        serviceLog += outLogOutput;
                        //If Prov PS has at least one Provider that has a VC, then we can use that site / provider.  if no, then skip
                        if (singleProviderSiteCBG != string.Empty)
                        {
                            providerSites += outValidSites;
                            providers += validProviders;
                            builderProviderSites.Append(singleProviderSiteCBG);
                            Logger.WriteDebugMessage("Added CBG");
                        }
                        else
                            Logger.WriteDebugMessage("ValidateProviderPS for " + prov.cvt_name + " returned no valid CBG.");
                    }

                    //Validation
                    Logger.WriteDebugMessage("providerSites: " + providerSites);
                    Logger.WriteDebugMessage("providers: " + providers);
                    Logger.WriteDebugMessage("serviceLog: " + serviceLog);
                    Logger.WriteDebugMessage("builderProviderSites.Length:" + builderProviderSites.Length);
                    if (builderProviderSites.Length == 0)
                    {
                        serviceLog += "\nNo Valid Provider Participating Sites assessed, no service generated.\n";
                        UpdateSP(SchedulingPackage, string.Empty, string.Empty, string.Empty, serviceLog, Logger, OrganizationService, srv, Guid.Empty);
                        throw new InvalidPluginExecutionException("Scheduling Resources must be listed on the Provider Side in order to build this service.");
                    }

                    if (builderProviderSites.Length > 0)
                        resourceSpecId = BuildOutforSP(SchedulingPackage, builderProviderSites, OrganizationService, 1, 2, false);

                    Logger.WriteDebugMessage("Constructing the service");

                    int initialstatus = (int)service_initialstatuscode.Pending;
                    var duration = (SchedulingPackage.cvt_AppointmentLength != null) ? SchedulingPackage.cvt_AppointmentLength.Value : 60;
                    var startEvery = (SchedulingPackage.cvt_StartAppointmentsEvery != null) ? SchedulingPackage.cvt_StartAppointmentsEvery.ToString() : "60";

                    Service newService = new Service
                    {
                        Name = SchedulingPackage.cvt_name.ToString(),
                        AnchorOffset = 480,
                        Duration = duration,
                        InitialStatusCode = new OptionSetValue(initialstatus),
                        Granularity = "FREQ=MINUTELY;INTERVAL=" + startEvery + ";",
                        ResourceSpecId = new EntityReference(ResourceSpec.EntityLogicalName, resourceSpecId)
                    };

                    var newServiceId = OrganizationService.Create(newService);
                    Logger.WriteDebugMessage("Created new Service. Updating the Scheduling Package.");
                    serviceLog += "\nConstructed the service.";
                    UpdateSP(SchedulingPackage, providerSites, string.Empty, providers, serviceLog, Logger, OrganizationService, srv, newServiceId);

                    #region Catch
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Logger.WriteToFile(ex.Message);
                    throw new InvalidPluginExecutionException(ex.Message);
                }
                catch (Exception ex)
                {
                    if (ex.Message.StartsWith("custom"))
                    {
                        Logger.WriteDebugMessage(ex.Message.Substring(6));

                        throw new InvalidPluginExecutionException(ex.Message.Substring(6));
                    }
                    else
                    {
                        Logger.setMethod = "Execute";
                        Logger.WriteToFile(ex.Message);
                        throw new InvalidPluginExecutionException(ex.Message);
                    }
                }
                #endregion
            }
        }

        public static void CreateUpdateGroupService(cvt_resourcepackage thisSP, cvt_participatingsite providerPS, List<cvt_participatingsite> patSites, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.WriteDebugMessage("Starting");
            var builder = new System.Text.StringBuilder("");
            var builderProvider = new System.Text.StringBuilder("");
            var builderProvider2 = new System.Text.StringBuilder("");
            var builderPatient = new System.Text.StringBuilder("");
            var builderBoth = new System.Text.StringBuilder("");
            //ALL has two branches, one of just PROVIDER, and one of BOT provider and patient
            Guid resourceSpecId = Guid.Empty;

            string patientSites = "";
            string providers = "";
            string serviceLog = "Group Service:";

            using (var srv = new Xrm(OrganizationService))
            {
                try
                {
                    if (patSites == null)
                    {
                        Logger.WriteDebugMessage("No Patient Sites, exiting.");
                        UpdatePS(providerPS, string.Empty, string.Empty, "No Patient Sites, no service generated.", Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                        return;
                    }

                    if (providerPS.cvt_scheduleable == false)
                    {
                        Logger.WriteDebugMessage("This Patient PS is not scheduleable, no service generated.");
                        UpdatePS(providerPS, string.Empty, string.Empty, "This Provider PS is not scheduleable, no service generated.", Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                        return;
                    }

                    #region PROV SITE RESOURCES
                    Logger.WriteDebugMessage("Starting PROV SITE RESOURCES region.");
                    var singleProviderSiteCBG = ValidateProviderPS(thisSP, providerPS, Guid.Empty, patSites, Logger, OrganizationService, out string validProviders, out string outValidSites, out string outLogOutput);
                    var anotherProviderSiteCBG = ValidateProviderPS(thisSP, providerPS, Guid.Empty, patSites, Logger, OrganizationService, out string xvalidProviders, out string xoutValidSites, out string xoutLogOutput);
                    Logger.WriteDebugMessage("Group. Provider Site CBG: " + singleProviderSiteCBG);
                    serviceLog += outLogOutput;

                    if (singleProviderSiteCBG != String.Empty)
                    {
                        //patientSites += outValidSites;
                        providers += validProviders;
                        builderProvider.Append(singleProviderSiteCBG);
                        builderProvider2.Append(anotherProviderSiteCBG);
                        Logger.WriteDebugMessage("Added CBG");
                    }
                    else
                        Logger.WriteDebugMessage("ValidateProviderPS for Provider PS" + providerPS.cvt_name + " returned no valid CBG.");


                    #endregion

                    #region PATIENT SITE RESOURCES
                    string groupPatVariable = string.Empty;
                    foreach (var record in patSites)
                    {
                        //Retrieve all the patient site resources
                        Logger.WriteDebugMessage("Start Sorting through Patient Site Resources");
                        var singlePatientSiteCBG = ValidatePatientPS(thisSP, record, Logger, OrganizationService, out string logOutput, out string outGroupPatVariable);
                        serviceLog += logOutput;
                        builderPatient.Append(singlePatientSiteCBG);
                        if (outGroupPatVariable != string.Empty)
                            groupPatVariable = outGroupPatVariable;
                    }
                        Logger.WriteDebugMessage("Finished Sorting through Patient Site Resources");
                    #endregion

                    #region Build the Service Components
                    Logger.WriteDebugMessage("Starting Build the Service Components region");
                    //Validation
                    Guid providerSpec = Guid.Empty;

                    if (builderProvider.Length == 0)
                    {
                        serviceLog += "\nNo Valid Provider Participating Sites assessed, no service generated.\n";
                        UpdatePS(providerPS, string.Empty, string.Empty, serviceLog, Logger, OrganizationService, srv, Guid.Empty, string.Empty);
                        Logger.WriteDebugMessage("Scheduling Resources must be listed on the Provider Side in order to build this service.");
                        return;
                    }
                    else if (builderProvider.Length > 0)
                    {
                        providerSpec = BuildOutforSP(thisSP, builderProvider2, OrganizationService, 1, 0, true);
                        builder.Append(AddResourceToConstraintGroup(providerSpec.ToString("B")));
                        providerSpec = BuildOutforSP(thisSP, builderProvider, OrganizationService, 1, 0, true);
                        builderBoth.Append(AddResourceToConstraintGroup(providerSpec.ToString("B")));
                        Logger.WriteDebugMessage("Adding Provider Builder to Main Builder.");
                    }

                    if (builderPatient.Length > 0)
                    {
                        builderBoth.Append(builderPatient);
                        Logger.WriteDebugMessage("Adding Patient Builder to Both Builder.");
                        resourceSpecId = BuildOutforSP(thisSP, builderBoth, OrganizationService, -1, 0, false);
                        builder.Append(AddResourceToConstraintGroup(resourceSpecId.ToString("B")));
                        Logger.WriteDebugMessage("Adding Both Builder to Main Builder.");
                    }
                    #endregion

                    #region Logic - Constructing the service
                    Logger.WriteDebugMessage("Constructing the group service.");
                    resourceSpecId = BuildOutforSP(thisSP, builder, OrganizationService, 1, 2, false);

                    int initialstatus = (int)service_initialstatuscode.Pending;
                    var duration = (thisSP.cvt_AppointmentLength != null) ? thisSP.cvt_AppointmentLength.Value : 60;
                    var startEvery = (thisSP.cvt_StartAppointmentsEvery != null) ? thisSP.cvt_StartAppointmentsEvery.ToString() : "60";

                    Service newService = new Service
                    {
                        Name = providerPS.cvt_name.ToString(),
                        AnchorOffset = 480,
                        Duration = duration,
                        InitialStatusCode = new OptionSetValue(initialstatus),
                        Granularity = "FREQ=MINUTELY;INTERVAL=" + startEvery + ";",
                        ResourceSpecId = new EntityReference(ResourceSpec.EntityLogicalName, resourceSpecId)
                    };

                    //create the service
                    Logger.WriteDebugMessage("Creating new Service");
                    var newServiceId = OrganizationService.Create(newService);
                    serviceLog += "\nConstructed the service.\n";
                    Logger.WriteDebugMessage("Updating the Provider Participating Site.");
                    UpdatePS(providerPS, patientSites, providers, serviceLog, Logger, OrganizationService, srv, newServiceId, groupPatVariable);
                    #endregion

                    #region Catch
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Logger.WriteToFile(ex.Message);
                    throw new InvalidPluginExecutionException(ex.Message);
                }
                catch (Exception ex)
                {
                    if (ex.Message.StartsWith("custom"))
                    {
                        Logger.WriteDebugMessage(ex.Message.Substring(6));

                        throw new InvalidPluginExecutionException(ex.Message.Substring(6));
                    }
                    else
                    {
                        Logger.setMethod = "Execute";
                        Logger.WriteToFile(ex.Message);
                        throw new InvalidPluginExecutionException(ex.Message);
                    }
                }
                #endregion
            }
        }
        #endregion
    }
}