﻿using MCSUtilities2011;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace MCSShared
{
    public static partial class CvtHelper
    {

        #region Email Related Functions
        //Overloaded method: creates a list from an Activty Party to populate a From, To, or CC field
        public static List<ActivityParty> SetPartyList(ActivityParty user)
        {
            List<ActivityParty> userList = new List<ActivityParty>();
            //user if exists
            if (user != null)
                userList.Add(user);
            return userList;
        }

        //Overloaded method: return empty list if userRef is null, otherwise get Activity party and call first SetEmailProperties method
        public static List<ActivityParty> SetPartyList(EntityReference userRef)
        {
            return userRef == null ? new List<ActivityParty>() :
                CvtHelper.SetPartyList(new ActivityParty() { PartyId = new EntityReference(userRef.LogicalName, userRef.Id) });
        }

        //Overloaded method: creates a list from an EntityCollection to populate a From, To, or CC field
        public static List<ActivityParty> SetPartyList(EntityCollection userCollection)
        {
            List<ActivityParty> userList = new List<ActivityParty>();

            foreach (var record in userCollection.Entities.Select(e => e).Distinct())
            {
                var activityParty = new ActivityParty() { PartyId = new EntityReference(record.LogicalName, record.Id) };
                userList.Add(activityParty);
            }
            return userList;
        }

        public static List<ActivityParty> SetPartyList(List<SystemUser> userList)
        {
            List<ActivityParty> users = new List<ActivityParty>();

            foreach (SystemUser u in userList.Select(e => e).Distinct())
            {
                var activityParty = new ActivityParty() { PartyId = new EntityReference(u.LogicalName, u.Id) };
                users.Add(activityParty);
            }
            return users;
        }

        //This method takes the email passed in and sends it
        public static void SendEmail(Email email, IOrganizationService OrganizationService)
        {
            //check for pssc in the server url, if it is true, then do nothing
            var serverURL = CvtHelper.getServerURL(OrganizationService);
            if (!serverURL.Contains("pssc"))
            {
                SendEmailRequest requestObj = new SendEmailRequest()
                {
                    EmailId = (Guid)email.ActivityId,
                    IssueSend = true,
                    TrackingToken = ""
                };
                SendEmailResponse response = (SendEmailResponse)OrganizationService.Execute(requestObj);
            }
        }

        //Saves and Sends the email
        public static void UpdateSendEmail(Email email, IOrganizationService OrganizationService, MCSLogger logger)
        {
            try
            {
                OrganizationService.Update(email);
                SendEmail(email, OrganizationService);
            }
            catch (Exception ex)
            {
                logger.WriteToFile($"Error occured while updating/sending the email with id: {email.ActivityId.Value}\nDetails:{ex.Message} {ex}");
            }
        }

        public static string GenerateNeedHelpSection(bool isCancelled)
        {
            var testurl = "https://care.DNS   /vvc-app?name=TestPatient&conference=TestWaitingRoom@care.DNS   &pin=5678";
            var needHelpSection = (isCancelled) ? string.Empty : $"<b>Need Help?</b><br/><ul><li>If you would like to test your connection to VA Video Connect prior to your appointment, {CvtHelper.buildHTMLUrlAlt(testurl, "Click Here to Test")}"
                + $"<li>If you would like more information about VA Video Connect, please review VA Video Connect information at the following link: {CvtHelper.buildHTMLUrlAlt("https://mobile.DNS   /app/va-video-connect", "Additional VA Video Connect Information")}</ li > "
                + "<li>If you need technical assistance or want to do a test call with a VA help desk technician, please call the National Telehealth Technology Help Desk at (866) 651-3180 or (703) 234-4483 Monday through Saturday, 7 a.m. through 11 p.m. EST</li></ul><br/><br/>";
            needHelpSection += "<b>Need to Reschedule?</b><br/>Do not reply to this message. This message is sent from an unmonitored mailbox.  For any questions or concerns please contact your VA Facility or VA Clinical Team.<br/>"
                + "-----------------------------------------------------------------------------------------------------------------------------------------<br/>"
                + "Please do not reply to this message.It comes from an unmonitored mailbox.";
            return needHelpSection;
        }

        public static string EmailFooter()
        {
            return "<br/><br/>Please Do Not Reply to this message.  It comes from an unmonitored mailbox.  For any questions or concerns, please contact your VA Facility or VA Clinical Team.";
        }

        public static string TechnicalAssistanceEmailFooter()
        {
            return "<br/><b>Technical Assistance</b><br/>For technical assistance, clinicians should contact the National Telehealth Technology Help Desk (NTTHD) (866) 651 - 3180 or (703) 234 - 4483, Monday through Saturday, 7 a.m. through 11 p.m. EST."
                + "<br/>-----------------------------------------------------------------------------------------------------------------------------------------<br/>"
                + "Please do not reply to this message.It comes from an unmonitored mailbox.";
        }

        public static bool ShouldGenerateVeteranEmail(Guid patientId, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            bool shouldGenerateVeteranEmail = false;
            using (var srv = new Xrm(OrganizationService))
            {
                var patient = srv.ContactSet.FirstOrDefault(p => p.Id == patientId);
                if (patient != null && patient.cvt_TabletType != null)
                {
                    switch (patient.cvt_TabletType.Value)
                    {
                        case (int)Contactcvt_TabletType.PersonalVAVideoConnectDevice:
                            shouldGenerateVeteranEmail = true;
                            break;
                        case (int)Contactcvt_TabletType.VAIssuediOSDevice:
                            if (patient.DoNotEMail != null && !patient.DoNotEMail.Value)
                            {
                                shouldGenerateVeteranEmail = true;
                            }
                            break;
                    }
                }
                Logger.WriteDebugMessage($"ShouldGenerateVeteranEmail for patient with name: {patient.FullName} and ID: {patientId} returned {shouldGenerateVeteranEmail}");
            }
            return shouldGenerateVeteranEmail;
        }


        ///// <summary>
        ///// Overload for basic generateEmailBody - displays the url as the message for "Click Here"
        ///// </summary>
        ///// <param name="record">ID of the email</param>
        ///// <param name="entityStringName">string name of the entity - to retrieve object type code</param>
        ///// <param name="customMessage">The message</param>
        ///// <returns></returns>
        //internal static string GenerateEmailBody(Guid record, string entityStringName, string customMessage, IOrganizationService OrganizationService)
        //{
        //    return GenerateEmailBody(record, entityStringName, customMessage, OrganizationService);
        //}

        /// <summary>
        /// Standard "boilerplate" E-Mail body
        /// </summary>
        /// <param name="record">ID of the email record</param>
        /// <param name="entityStringName">The string name of the object type code</param>
        /// <param name="customMessage">The custom string that goes into the email body</param>
        /// <param name="clickHereMessage">The message that is used as the display for the hyperlink</param>
        /// <returns>the body of the email</returns>
        internal static string GenerateEmailBody(Guid record, string entityStringName, string customMessage, IOrganizationService OrganizationService, string clickHereMessage = "")
        {
            string body;
            body = GetRecordLink(new EntityReference(entityStringName, record), OrganizationService, clickHereMessage);
            body += "<br/><br/>" + customMessage;
            body += EmailFooter();

            return body;
        }

        internal static string GetRecordLink(Entity record, IOrganizationService orgService, string clickHereText = "")
        {
            return GetRecordLink(new EntityReference(record.LogicalName, record.Id), orgService, clickHereText);
        }

        internal static string GetRecordLink(EntityReference record, IOrganizationService OrganizationService, string clickHereText = "")
        {
            var etc = CvtHelper.GetEntityTypeCode(OrganizationService, record.LogicalName);
            string servernameAndOrgname = CvtHelper.getServerURL(OrganizationService);
            string url = servernameAndOrgname + "/userDefined/edit.aspx?etc=" + etc + "&id=" + record.Id;
            return String.Format("<a href=\"{0}\">{1}</a>", url, !string.IsNullOrEmpty(clickHereText) ? clickHereText : url);
        }

        internal static string ProviderSafetyChecks()
        {
            var safetyChecks = "During your initial assessment be sure to verify the following: ";
            safetyChecks += "<ul><li>Do you have any concerns about suicide?</li>";
            safetyChecks += "<li>The Patient verbally consents to the telehealth visit?</li>";
            safetyChecks += "<li>If the line drops, what number can I call you at?</li>";
            safetyChecks += "<li>What is the name, phone number, and relationship of the person we should contact in the case of an emergency?</li>";
            safetyChecks += "<li>What is your local 10 digit phone number for law enforcement in your community?</li>";
            safetyChecks += "<li>What is the address of your location during this visit?</li></ul>";
            safetyChecks += "<li>Are you in a safe and private place?</li></ul>";
            return safetyChecks;
        }

        internal static string VvcAppointmentInstructions(bool isCvtTablet)
        {
            var apptInstructions = $"<br/><b>{(isCvtTablet ? "VVC Tablet SIP Address dialing" : "VA Video Connect (VVC)")} Appointment Instructions:</b>"
                              //+ $"<ol><li>{(isCvtTablet ? "To dial a patient’s VVC SIP Tablet, you must add SIP and colon as a prefix to any SIP Address, plus tablet number, as well as .bltablet to the end of each SIP VVC Tablet Address." : "The \"Click Here\" link above is unique to this visit.  You will need to click it to enter the Virtual Medical Room at the time of the appointment.")}</li>"
                              + $"<li>{(isCvtTablet ? "To dial a patient’s VVC SIP Tablet, you must add SIP and colon as a prefix to any SIP Address, plus tablet number, as well as .bltablet to the end of each SIP VVC Tablet Address." : "The \"Click Here\" link above is unique to <b>this visit</b>.  You will need to click it to enter the Virtual Medical Room at the time of the appointment.")}</li>"
                              + $"{(isCvtTablet ? "<li>To more easily find the VVC tablet SIP address at the time of the visit, we recommend you do one of the following:</li>" : "<li>To more easily find this unique link at the time of the visit, we recommend you do one of the following:</li>")} ";
            apptInstructions += $"{(isCvtTablet ? "<ul><li>Save the attached invitation to your Outlook Calendar</li>" : "<ul><li>Save the attached invitation to your Outlook Calendar</li>")}";
            apptInstructions += $"{(isCvtTablet ? "<li> Copy this invitation into your Outlook calendar and create a calendar entry at the time of the visit, and/ or</li>" : "<li> Copy this invitation into your Outlook calendar and create a calendar entry at the time of the visit, and/ or</li>")}";
            apptInstructions += $"{(isCvtTablet ? "< li>Search your Outlook e-mail for the date or Last Initial / Last 4, on the day of the visit.</li></ul>" : "<li>Search your Outlook e-mail for the date or Last Initial / Last 4, on the day of the visit.</li></ul>")}";
            apptInstructions += $"{(isCvtTablet ? "<li>At the start of your VVC Appointment , remember (CAPS):<br/><i>*Some information can be obtained prior to the VVC appointment by another member of the care team.</i></li>" : "<li>At the start of your VVC Appointment , remember (CAPS-Lock):<br/><i>*Some information can be obtained prior to the VVC appointment by another member of the care team.</i></li>")}";
            apptInstructions += $"{(isCvtTablet ? "<ul><li><b><u>C</u></b>onsent*: Obtain or confirm that verbal consent for telehealth has been documented (Note: This is a one-time requirement for your service)</li>" : "<ul><li><b><u>C</u></b>onsent*: Obtain or confirm that verbal consent for telehealth has been documented (Note: This is a one-time requirement for your service)</li>")}";
            apptInstructions += $"{(isCvtTablet ? "<li><b><u>A</u></b>ddress: Obtain or confirm the location and address of the patient to ensure they are in a safe place and for use in case of an emergency.</li>" : "<li><b><u>A</u></b>ddress: Obtain or confirm the location and address of the patient to ensure they are in a safe place and for use in case of an emergency.</li>")}";
            apptInstructions += $"{(isCvtTablet ? "<li><b><u>P</u></b>hone Numbers:*<br/>Patient’s current phone number – for use if disconnected.<br/>Emergency contact’s phone number – for use in an emergency.</li>" : "<li><b><u>P</u></b>hone Numbers:*<br/>Patient’s current phone number – for use if disconnected.<br/>Emergency contact’s phone number – for use in an emergency.</li>")}";
            apptInstructions += $"{(isCvtTablet ? "<li><b><u>S</u></b>urvey the environment and identify all participants.</li>" : "<li><b><u>S</u></b>urvey the environment and identify all participants.</li><br/><br/>")}";
            apptInstructions += $"{(isCvtTablet ? string.Empty : "<li><b><u>L</u></b>ock the virtual conference room once all participants have joined.</li>")}";
            apptInstructions += $"</ul></ol>";

            return apptInstructions;
        }

        /// <summary>
        /// This method creates the .ics attachment and appends it to the email
        /// </summary>
        /// <param name="email">This is the email that the attachment is attaching to</param>
        /// <param name="sa">The service appointment which </param>
        /// <param name="statusCode">The status of the email - which sets the status of the attachment as well as the subject of the email</param>
        internal static void CreateCalendarAppointmentAttachment(Email email, ServiceAppointment sa, int statusCode, string stethIP, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            bool group = false;
            if (sa.mcs_groupappointment != null)
            {
                group = sa.mcs_groupappointment.Value;
            }
            Logger.WriteTxnTimingMessage("Begin Creating Calendar Appointment");
            string schLocation = "See Description";
            string subjectPrefix = "Telehealth Visit";
            if (email.Subject.Contains("VA Video Connect"))
            {
                subjectPrefix = "VA Video Connect";
                schLocation = "VA Video Connect";
            }
            else if (email.Subject.Contains("VVC SIP Tablet"))
            {
                subjectPrefix = "VVC SIP Tablet";
                schLocation = "VVC SIP Tablet";
            }
            string schSubject = group == true ? $"{subjectPrefix}- Group Appointment" :
                $"{subjectPrefix}-Single Appointment";
            string schDescription = email.Description;
            System.DateTime schBeginDate = (System.DateTime)sa.ScheduledStart;
            System.DateTime schEndDate = (System.DateTime)sa.ScheduledEnd;
            string sequence = "";
            string status = "CONFIRMED";
            string method = "";
            //if the appointment is canceled, send a cancellation notice based on the UID of the previous entry sent
            if (statusCode == (int)serviceappointment_statuscode.PatientCanceled || statusCode == (int)serviceappointment_statuscode.ClinicCanceled)
            {
                method = "METHOD:CANCEL\n";
                sequence = "SEQUENCE:1\n";
                status = "CANCELLED";
                schSubject = $"Canceled: {subjectPrefix} : Do Not Reply";
            }

            //attach a ClearSteth CVL file if a steth is in the components
            string cvlAtttachment = string.IsNullOrEmpty(stethIP) ? "" :
                "ATTACH;ENCODING=BASE64;VALUE=BINARY;X-FILENAME=invitation.cvl:" + Convert.ToBase64String(new ASCIIEncoding().GetBytes("<?xml version=\"1.0\" encoding=\"UTF-8\"?><CVL><IP>" + stethIP + "</IP ><Port>9005</Port><ConferenceId>12345</ConferenceId></CVL>")) + "\n";

            string att = "BEGIN:VCALENDAR\n" +
                              "PRODID:-//VA//Veterans Affairs//EN\n" +
                              method +
                              "BEGIN:VEVENT\n" +
                              cvlAtttachment +
                              "UID:" + sa.Id + "\n" + sequence +
                              "DTSTART:" + schBeginDate.ToUniversalTime().ToString("yyyyMMdd\\THHmmss\\Z") + "\n" +
                              "DTEND:" + schEndDate.ToUniversalTime().ToString("yyyyMMdd\\THHmmss\\Z") + "\n" +
                              "LOCATION:" + schLocation +
                              //Use Description tag for email clients that cant handle x-alt-desc tag with HTML
                              "\nDESCRIPTION;ENCODING=QUOTED-PRINTABLE:" + schDescription.Replace("<br/>", "").Replace("<b>", "").Replace("</b>", "").Replace("<u>", "").Replace("</u>", "") +
                              "\nSUMMARY:" + schSubject + "\nPRIORITY:3\n" +
                              "STATUS:" + status + "\n" +
                              //Include alternate description if the calendar client can handle html x-alt-desc tag
                              "X-ALT-DESC;FMTTYPE=text/html:<html>" + schDescription.Replace("\n", "<br/>") + "</html>" + "\n" +
                          "END:VEVENT\n" + "END:VCALENDAR\n";

            ActivityMimeAttachment calendarAttachment = new ActivityMimeAttachment()
            {
                ObjectId = new EntityReference(Email.EntityLogicalName, email.Id),
                ObjectTypeCode = Email.EntityLogicalName,
                Subject = $"{subjectPrefix} Appointment",
                Body = Convert.ToBase64String(
                        new ASCIIEncoding().GetBytes(att)),
                FileName = string.Format(CultureInfo.CurrentCulture, $"{subjectPrefix}-Appointment.ics")
            };
            OrganizationService.Create(calendarAttachment);
            Logger.WriteTxnTimingMessage("Finished Creating Calendar Appointment");
            return;
        }

        #endregion


        #region Commonly Used Functions
        //TODO TO-DO: Consolidate with Email Automation function, should we add check for User's Email existing before adding to the AP List?
        /// <summary>
        /// Gets the list of team members and returns them as an activity party list to be added to whatever email we are using
        /// </summary>
        /// <param name="email"></param>
        /// <param name="TeamId"></param>
        /// <param name="originalParty"></param>
        /// <returns></returns>
        internal static List<ActivityParty> RetrieveFacilityTeamMembers(Email email, Guid TeamId, IEnumerable<ActivityParty> originalParty, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("starting RetrieveFacilityTeamMembers");
            using (var srv = new Xrm(OrganizationService))
            {
                var teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == TeamId).ToList());
                var recipientList = new List<ActivityParty>();

                if (originalParty != null)
                    recipientList.AddRange(originalParty);

                Logger.WriteDebugMessage("About to add members of team.");
                foreach (var member in teamMembers)
                {
                    var party = new ActivityParty()
                    {
                        ActivityId = new EntityReference(email.LogicalName, email.Id),
                        PartyId = new EntityReference(SystemUser.EntityLogicalName, member.SystemUserId.Value)
                    };
                    recipientList.Add(party);
                }
                Logger.WriteDebugMessage("Finished adding members of team.");
                return recipientList;
            }
        }
        #endregion
    }
}