﻿using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.Vista.StateObject;
using VA.TMP.OptionSets;
using VIMT.VIASchedulingService.Messages;
using VA.TMP.Integration.VIMT.Vista.PipelineSteps.MakeAppointment;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.Vista.Mappers
{
    public static class VistaMapperHelper
    {
        public static VIMTVIASchdMakeApptReqqueryBean GetQueryBean(MakeAppointmentStateObject state, Side side)
        {
            var siteCode = GetLoginSiteCode(state, side);
            var currentUser = state.OrganizationServiceProxy.Retrieve(SystemUser.EntityLogicalName, state.UserId, new Microsoft.Xrm.Sdk.Query.ColumnSet(true));
            var queryBeanAppSettings = GetSecuritySettings(state.OrganizationServiceProxy);
            if (currentUser == null)
                throw new Exception("No Executing User passed into Integration pipeline (or user could not be found): " + state.UserId);
            return new VIMTVIASchdMakeApptReqqueryBean
            {
                mcs_requestingApp = queryBeanAppSettings.RequestingApp,
                mcs_consumingAppToken = queryBeanAppSettings.ConsumingAppToken,
                mcs_consumingAppPassword = queryBeanAppSettings.ConsumingAppPassword,
                mcs_multiSiteQuery = false,
                VIMTVIASchdMakeApptReqproviderInfo = new VIMTVIASchdMakeApptReqprovider
                {
                    mcs_name = GetProviderName(state),
                    mcs_loginSiteCode = siteCode,
                    mcs_vistaLocations = siteCode
                    //mcs_userId = GetViaLoginId(state, side) //Now handled in a separate pipeline step
                },
                VIMTVIASchdMakeApptReqpatientInfo = new VIMTVIASchdMakeApptReqpatient
                {
                    mcs_mpiPid = GetPatientICN(state),
                    mcs_localPid = GetPatientLocalPid(state, siteCode),
                    mcs_localSiteId = siteCode,
                    mcs_vistaLocations = siteCode,
                    
                },
                mcs_name = currentUser.ToEntity<SystemUser>().FullName,
                
            };
        }

        public static string GetViaLoginId(VIMTVIAScheLIloginVIARequest request, Side side)
        {
            Logger.Instance.Debug(string.Format("Calling LoginVIA EC for {0} Side", side.ToString()));
            var loginResponse = request.SendReceive<VIMTVIAScheLIloginVIAResponse>(MessageProcessType.Local);
            Logger.Instance.Debug(string.Format("Exiting LoginVIA EC for {0} Side", side.ToString()));

            if (loginResponse == null)
                throw new Exception(string.Format("VIA Login failed for {0} Site, no response was returned", side.ToString()));
            if (loginResponse.VIMTVIAScheLIuserTOInfo == null)
                throw new Exception(string.Format("VIA Login failed for {0} Site, an empty response was returned", side.ToString()));
            if (loginResponse.ExceptionOccured)
                throw new Exception(string.Format("VIA Login Failed for {0} Site. Error: {1}.  Additional Details: {2}", side.ToString(), loginResponse.ExceptionMessage, loginResponse.VIMTVIAScheLIuserTOInfo?.VIMTVIAScheLIfault2Info?.mcs_message));
            if (string.IsNullOrEmpty(loginResponse.VIMTVIAScheLIuserTOInfo.mcs_DUZ))
                throw new Exception(string.Format("No User Id was returned from the Login Request for the {0} Site", side.ToString()));
            return loginResponse.VIMTVIAScheLIuserTOInfo.mcs_DUZ;
        }

        public static string GetPatientLocalPid(MakeAppointmentStateObject state, string siteCode)
        {
            var personIdentifier = new mcs_personidentifiers();
            var patId = state.MakeAppointmentRequestMessage.AddedPatients.FirstOrDefault();
            if (patId == null)
                throw new Exception("Patient Cannot be null");

            using (var srv = new Xrm(state.OrganizationServiceProxy))
            {
                var personId = srv.mcs_personidentifiersSet.FirstOrDefault(p => p.mcs_patient.Id == patId && p.mcs_assigningfacility == siteCode);
                if (personId == null)
                    throw new Exception(string.Format("Patient was not successfully registered to the Vista Station {0}, please confirm Proxy Add ran successfully", siteCode));
            }
            return personIdentifier.mcs_identifier;
        }

        public static mcs_facility GetFacility(MakeAppointmentStateObject state, Side side)
        {
            EntityReference siteLookup;
            if (side == Side.Provider)
            {
                if (state.AppointmentType != AppointmentType.STORE_FORWARD)
                    siteLookup = state.ServiceAppointment.mcs_relatedprovidersite;
                else
                    throw new Exception("No Provider Side Encounter should be booked for a Store/Forward Appointment");
            }
            else
            {
                if (state.AppointmentType == AppointmentType.GROUP)
                    siteLookup = state.CrmAppointment.cvt_Site;
                else if (state.AppointmentType == AppointmentType.HOME_MOBILE)
                    throw new Exception("No Patient Side Encounter should be booked for a Home/Mobile Appointment");
                else
                    siteLookup = state.ServiceAppointment.mcs_relatedsite;
            }
            return GetFacilityFromSite(state.OrganizationServiceProxy, siteLookup);
        }

        public static string GetLoginSiteCode(MakeAppointmentStateObject state, Side side)
        {
            return GetFacility(state, side).mcs_StationNumber;
        }

        public static mcs_facility GetFacilityFromSite(IOrganizationService orgService, EntityReference siteLookup)
        {
            using (var srv = new Xrm(orgService))
            {
                var facility = (from f in srv.mcs_facilitySet
                                join s in srv.mcs_siteSet
                                on f.Id equals s.mcs_FacilityId.Id
                                where s.Id == siteLookup.Id
                                select f).FirstOrDefault();
                return facility;
            }
        }

        public static string GetProviderName(MakeAppointmentStateObject state)
        {
            var user = state.SystemUsers.FirstOrDefault();
            if (user != null)
                return user.FullName.ToUpper();
            else
                return string.Empty;
        }

        public static string GetPatientICN(MakeAppointmentStateObject state)
        {
            var personIdentifier = new mcs_personidentifiers();
            var patId = state.MakeAppointmentRequestMessage.AddedPatients.FirstOrDefault();
            if (patId == null)
                throw new Exception("Patient Cannot be null");
            using (var srv = new Xrm(state.OrganizationServiceProxy))
            {
                personIdentifier = srv.mcs_personidentifiersSet.FirstOrDefault(x => x.mcs_identifiertype.Value == (int)mcs_personidentifiersmcs_identifiertype.NationalIdentifier_NI && x.mcs_assigningauthority == "USVHA" && x.mcs_patient.Id == patId);
            }
            if (personIdentifier == null)
                throw new Exception("Patient has no ICN: " + patId + " - SA/ApptId = " + state.AppointmentId);
            return personIdentifier.mcs_identifier;
        }

        public static QueryBeanSecuritySettings GetSecuritySettings(IOrganizationService orgService)
        {
            QueryBeanSecuritySettings settings = new QueryBeanSecuritySettings();
            using (var srv = new Xrm(orgService))
            {
                var errorString = string.Empty;
                var ViaSettings = srv.mcs_integrationsettingSet.Where(s => s.mcs_name.Contains("VIA")).ToList();
                var RequestingAppRecord = ViaSettings.FirstOrDefault(s => s.mcs_name == "VIA Requesting App");
                var ConsumingAppRecord = ViaSettings.FirstOrDefault(s => s.mcs_name == "VIA Consuming App Token");
                var ConsumingAppPasswordRecord = ViaSettings.FirstOrDefault(s => s.mcs_name == "VIA Consuming App Password");

                if (RequestingAppRecord == null)
                    errorString += "Requesting App";
                if (ConsumingAppRecord == null)
                    errorString += string.IsNullOrEmpty(errorString) ? "Consuming App Token" : ", Consuming App Token";
                if (ConsumingAppPasswordRecord == null)
                    errorString += string.IsNullOrEmpty(errorString) ? "Consuming App Password" : ", Consuming App Password";
                if (!string.IsNullOrEmpty(errorString))
                    throw new Exception("No VIA Consuming App Password Setting was found, unable to process Vista Request");

                settings.RequestingApp = RequestingAppRecord.mcs_value;
                settings.ConsumingAppToken = ConsumingAppRecord.mcs_value;
                settings.ConsumingAppPassword = ConsumingAppPasswordRecord.mcs_value;
            }
            return settings;
        }

    }
}
