﻿using MCSUtilities2011;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle updating a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentVvsUpdatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentVvsUpdatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            var sa = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();
            if (sa == null) throw new InvalidPluginExecutionException(string.Format("Unable to Find Service Appointment with id {0}", PrimaryEntity.Id));
            if (sa.cvt_VMRCompleted == null || !sa.cvt_VMRCompleted.Value)
            {
                Logger.WriteDebugMessage("VMR Completed is false or null, exiting VVS");
                return;
            }

            using (var context = new Xrm(OrganizationService))
            {
                if (VistaPluginHelpers.RunVVS(sa, context, Logger))
                {
                    var vimtUrlObj = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                    if (vimtUrlObj == null) throw new InvalidPluginExecutionException("VIMT Url cannot be null, please ensure there is a setting called \"VIMT URL\"");

                    VimtUrl = vimtUrlObj.mcs_value;
                    // Ensure this is a Home/Mobile Service Appointment of the correct status.
                    //if (sa.cvt_Type == null || !sa.cvt_Type.Value) return;
                    if (sa.StateCode == null || sa.StateCode != ServiceAppointmentState.Scheduled) throw new InvalidPluginExecutionException("Service Appointment is not in Scheduled State.");

                    if (sa.StatusCode == null 
                        || sa.StatusCode.Value == (int)serviceappointment_statuscode.Pending 
                        || sa.StatusCode.Value == (int)serviceappointment_statuscode.InterfaceVIMTFailure
                        || sa.StatusCode.Value == (int)serviceappointment_statuscode.ReservedScheduled)
                    {
                        var videoVisitCreateResponseMessage = VistaPluginHelpers.CreateAndSendVideoVisitService(sa, PluginExecutionContext.InitiatingUserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger);
                        if (videoVisitCreateResponseMessage == null)
                        {
                            Logger.WriteDebugMessage("Null Response from VVS, exiting handling of response");
                            return;
                        }
                        // Process the Response from VIMT
                        ProcessVistaCreateResponse(videoVisitCreateResponseMessage);
                    }
                    else
                        Logger.WriteDebugMessage("Service Activity not in Proper 'Pending' or 'Interface VIMT Failure' status for writing Vista Results back into TMP, skipping processVistaCreateResponse");
                }
                else
                {
                    Logger.WriteDebugMessage("VVS Integration Bypassed, Updating Service Activity Status");
                    IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.ReservedScheduled);
                }
            }
        }

        /// <summary>
        /// Handle the response - write integration result, Vista "Write Results", and update Status of SA if applicable
        /// </summary>
        /// <param name="response"></param>
        private void ProcessVistaCreateResponse(VideoVisitCreateResponseMessage response)
        {
            if (response == null) return;

            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;

            var integrationResultId = IntegrationPluginHelpers.CreateIntegrationResult("Book Vista Appointment", response.ExceptionOccured, errorMessage, response.VimtRequest,
                response.SerializedInstance, response.VimtResponse, typeof(VideoVisitCreateRequestMessage).FullName, typeof(VideoVisitCreateResponseMessage).FullName,
                MessageRegistry.VideoVisitCreateRequestMessage, PrimaryEntity.Id, OrganizationService);

            if (response.ExceptionOccured)
            {
                IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.InterfaceVIMTFailure);
                Logger.WriteDebugMessage("VVS Create Failed");
                return;
            }
            var status = IntegrationPluginHelpers.WriteVistaResults(response.WriteResults, integrationResultId, Guid.Empty, PrimaryEntity.Id, "book", OrganizationService, Logger);

            IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, (serviceappointment_statuscode)status);
        }
    }
}