﻿using System;
using System.ServiceModel;
using System.Linq;
using Microsoft.Xrm.Sdk;
using VA.TMP.CRM;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;
using Microsoft.Xrm.Sdk.Query;
using VRMRest;
using VA.TMP.Integration.Plugins.Messages;
using System.Collections.Generic;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle creating a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentCreatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentCreatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        /// <summary>
        /// Gets or sets the ProxyAddFakeResponseType.
        /// </summary>
        private string ProxyAddFakeResponseType { get; set; }

        /// <summary>
        /// Gets or sets ProcessingCode.
        /// </summary>
        private string ProcessingCode { get; set; }

        /// <summary>
        /// Gets or sets ReturnMviMessagesInResponse.
        /// </summary>
        private bool ReturnMviMessagesInResponse { get; set; }

        /// <summary>
        /// Gets or sets PatientVeteran.
        /// </summary>
        private bool PatientVeteran { get; set; }

        /// <summary>
        /// Gets or sets PatientServiceConnected.
        /// </summary>
        private bool PatientServiceConnected { get; set; }

        /// <summary>
        /// Gets or sets PatientType.
        /// </summary>
        private int PatientType { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            try
            {
                var saRecord = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();

                if (saRecord.mcs_groupappointment == null || !saRecord.mcs_groupappointment.Value || saRecord.cvt_Type.Value)
                {
                    if (saRecord.StatusCode.Value == (int)serviceappointment_statuscode.Pending || saRecord.StatusCode.Value == (int)serviceappointment_statuscode.ReservedScheduled)
                    {
                        var orgName = PluginExecutionContext.OrganizationName;
                        var user = OrganizationService.Retrieve(SystemUser.EntityLogicalName, PluginExecutionContext.UserId, new ColumnSet(true)).ToEntity<SystemUser>();

                        using (var context = new Xrm(OrganizationService))
                        {
                            Logger.WriteDebugMessage("Starting GetAndSetIntegrationSettings");
                            GetAndSetIntegrationSettings(context);
                            Logger.WriteDebugMessage("Finished GetAndSetIntegrationSettings");

                            var anyFailures = RunProxyAdd(orgName, user);

                            if (!anyFailures)
                                TriggerNextIntegration();
                        }
                    }
                    else
                    {
                        Logger.WriteDebugMessage(string.Format("ServiceAppointment is not scheduled (status is {1}), Proxy Add Canceled: {0}", PrimaryEntity.Id, saRecord.StatusCode.Value));
                    }
                }
                else
                {
                    Logger.WriteDebugMessage("Proxy Add is executed on the Appointment, not the Service Activity for Group Appointments that are Clinic Based.");
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(string.Format("ERROR in ServiceAppointmentCreatePostStageRunner: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)));
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        /// <summary>
        /// Gets and sets the Integration Settings needed for Proxy Add.
        /// </summary>
        /// <param name="context">CRM context.</param>
        private void GetAndSetIntegrationSettings(Xrm context)
        {
            var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
            if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
            else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");

            var proxyAddFakeResponseType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "MVI ProxyAdd Fake Response Type");
            if (proxyAddFakeResponseType != null) ProxyAddFakeResponseType = proxyAddFakeResponseType.mcs_value;
            else throw new InvalidPluginExecutionException("No ProxyAddFakeResponseType listed in Integration Settings.  Please contact the Help Desk to add ProxyAddFakeResponseType.  Proxy Add Canceled.");
            
            var processingCode = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "ProcessingCode");
            if (processingCode != null) ProcessingCode = processingCode.mcs_value;
            else throw new InvalidPluginExecutionException("No ProcessingCode listed in Integration Settings.  Please contact the Help Desk to add ProcessingCode.  Proxy Add Canceled.");

            var returnMviMessagesInResponse = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "ReturnMviMessagesInResponse");
            if (returnMviMessagesInResponse != null) ReturnMviMessagesInResponse = Convert.ToBoolean(returnMviMessagesInResponse.mcs_value);
            else throw new InvalidPluginExecutionException("No ReturnMviMessagesInResponse listed in Integration Settings.  Please contact the Help Desk to add ReturnMviMessagesInResponse.  Proxy Add Canceled.");

            var patientVeteran = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientVeteran");
            if (patientVeteran != null) PatientVeteran = Convert.ToBoolean(patientVeteran.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientVeteran listed in Integration Settings.  Please contact the Help Desk to add PatientVeteran.  Proxy Add Canceled.");

            var patientServiceConnected = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientServiceConnected");
            if (patientServiceConnected != null) PatientServiceConnected = Convert.ToBoolean(patientServiceConnected.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientServiceConnected listed in Integration Settings.  Please contact the Help Desk to add PatientServiceConnected.  Proxy Add Canceled.");

            var patientType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientType");
            if (patientType != null) PatientType = Convert.ToInt32(patientType.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientType listed in Integration Settings.  Please contact the Help Desk to add PatientType.  Proxy Add Canceled.");
        }

        /// <summary>
        /// Gets the Veteran Activity Party for all matches between
        /// </summary>
        /// <returns></returns>
        private List<Guid> GetListOfPatients()
        {
            DataModel.ServiceAppointment sa = new DataModel.ServiceAppointment();
            using (var srv = new Xrm(OrganizationService))
            {
                try
                {
                    sa = srv.ServiceAppointmentSet.FirstOrDefault(s => s.Id == PrimaryEntity.Id);
                    srv.LoadProperty(sa, "serviceappointment_activity_parties");
                }
                catch (Exception ex) { Logger.WriteDebugMessage(ex.Message); }
            }
            var newCustomers = sa.serviceappointment_activity_parties.Where(ap => ap.ParticipationTypeMask.Value == (int)ActivityPartyParticipationTypeMask.Customer).Select(ap => ap.PartyId.Id).ToList();
            var customers = new List<Guid>();

            if (PluginExecutionContext.MessageName != "Create")
            {
                var pre = PluginExecutionContext.PreEntityImages["pre"];
                var patientsAtt = pre.Attributes.FirstOrDefault(k => k.Key == "customers");
                if (patientsAtt.Value != null)
                {
                    EntityCollection ec = (EntityCollection)patientsAtt.Value;
                    foreach (var entity in ec.Entities)
                    {
                        customers.Add(entity.ToEntity<ActivityParty>().PartyId.Id);
                    }
                }
            }
            var veterans = new List<Guid>();

            if (customers.Count != 0 && sa.Customers.ToList().Count != 0)
                veterans = newCustomers.Except(customers).ToList();
            else
                veterans = newCustomers.ToList();
            return veterans;
        }

        /// <summary>
        /// Sends the Proxy Add Request to VIMT.
        /// </summary>
        /// <param name="orgName">Organization Service.</param>
        /// <param name="user">CRM User.</param>
        /// <returns>ProxyAddResponseMessage.</returns>
        private bool RunProxyAdd(string orgName, SystemUser user)
        {
            var anyfailures = false;
            //Use the PrimaryEntity so that you can only send Customers that were modified (added or removed)
            var veterans = GetListOfPatients();

            foreach (var veteranParty in veterans)
            {
                var request = new ProxyAddRequestMessage
                {
                    LogRequest = true,
                    ServiceAppointmentId = PrimaryEntity.Id,
                    OrganizationName = orgName,
                    FakeResponseType = ProxyAddFakeResponseType,
                    UserFirstName = user.FirstName,
                    UserLastName = user.LastName,
                    UserId = user.Id,
                    ProcessingCode = ProcessingCode,
                    ReturnMviMessagesInResponse = ReturnMviMessagesInResponse,
                    PatientVeteran = PatientVeteran,
                    PatientServiceConnected = PatientServiceConnected,
                    PatientType = PatientType,
                    VeteranPartyId = veteranParty
                };

                var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);
                try
                {
                    Logger.WriteDebugMessage(string.Format("Sending Proxy Add Message to VIMT: {0}.", vimtRequest));
                    var response = Utility.SendReceive<ProxyAddResponseMessage>(new Uri(VimtUrl), MessageRegistry.ProxyAddRequestMessage, request, null);
                    Logger.WriteDebugMessage(string.Format("Finished Sending Proxy Add Message to VIMT for {0}", veteranParty));
                    ProcessProxyAddToVistaResponse(response);
                }
                catch (Exception ex)
                {
                    var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                    IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Create Proxy Add to Vista", errorMessage, vimtRequest, typeof(ProxyAddRequestMessage).FullName,
                        typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage, PrimaryEntity.Id, OrganizationService);
                    Logger.WriteToFile(errorMessage);
                    anyfailures = true;
                }
            }
            return anyfailures;
        }

        /// <summary>
        /// Process Proxy Add to VistA response.
        /// </summary>
        /// <param name="response">Proxy Add Response.</param>
        private void ProcessProxyAddToVistaResponse(ProxyAddResponseMessage response)
        {
            if (response == null) return;

            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            IntegrationPluginHelpers.CreateIntegrationResult("Proxy Add to VistA", response.ExceptionOccured, errorMessage,
                response.VimtRequest, response.SerializedInstance, response.VimtResponse,
                typeof(ProxyAddRequestMessage).FullName, typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage,
                PrimaryEntity.Id, OrganizationService);

        }

        private void TriggerNextIntegration()
        {
            Logger.WriteDebugMessage("Updating ProxyAddCompleted flag");
            if (PluginExecutionContext.MessageName != "Create")
            {
                var runVvsAgainObject = new ServiceAppointmentVvsUpdatePostStage();
                runVvsAgainObject.Execute(ServiceProvider);
            }
            else
            {
                var updateServiceAppointment = new DataModel.ServiceAppointment
                {
                    Id = PrimaryEntity.Id,
                    cvt_ProxyAddCompleted = true
                };
                try
                {
                    OrganizationService.Update(updateServiceAppointment);
                }
                catch (Exception ex)
                {
                    throw new InvalidPluginExecutionException(string.Format("Failed to Update Proxy Add Completed Flag to initiate VVC/VVS plugin: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)), ex);
                }
            }
        }
    }
}
