﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VRMRest;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle canceling a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentCancelPostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentCancelPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        /// <summary>
        /// Gets the primary entity.
        /// </summary>
        /// <returns>Returns the primary entity.</returns>
        public override Entity GetPrimaryEntity()
        {
            var primaryReference = (EntityReference)PluginExecutionContext.InputParameters["EntityMoniker"];

            return new Entity(primaryReference.LogicalName) { Id = primaryReference.Id };
        }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            try
            {
                using (var context = new Xrm(OrganizationService))
                {
                    VimtUrl = context.mcs_integrationsettingSet.First(x => x.mcs_name == "VIMT URL").mcs_value;
                    var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
                    if (settings == null) throw new InvalidPluginExecutionException("Active Settings Cannot be Null");

                    var serviceAppointment = context.ServiceAppointmentSet.FirstOrDefault(x => x.Id == PrimaryEntity.Id);
                    if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null.");

                    if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Canceled)
                    {
                        Logger.WriteDebugMessage("Service Activity Not in Canceled status, exiting Cancel Integrations");
                        return;
                    }

                    if (settings.cvt_accenturevyopta != null && settings.cvt_accenturevyopta.Value)
                    {
                        // Call Service to cancel the Virtual Meeting Room.
                        if (serviceAppointment.cvt_Type != null && serviceAppointment.cvt_Type.Value)
                        {
                            var virtualMeetingRoomDeleteResponseMessage = CancelAndSendVirtualMeetingRoom(serviceAppointment.Id, PluginExecutionContext.InitiatingUserId, PluginExecutionContext.OrganizationName);
                            if (virtualMeetingRoomDeleteResponseMessage == null) return;
                            ProcessVirtualMeetingRoomCreateResponseMessage(virtualMeetingRoomDeleteResponseMessage);

                            // If we fail canceling a virtual meeting room do not call the video visit service.
                            if (virtualMeetingRoomDeleteResponseMessage.ExceptionOccured) return;
                        }
                    }
                    else
                    {
                        Logger.WriteDebugMessage("Bypassed VMR Cancel");
                    }

                    // Call Service to cancel the Video Visit.
                    if (VistaPluginHelpers.RunVVS(serviceAppointment, context, Logger))
                    {
                        var videoVisitDeleteResponseMessage = VistaPluginHelpers.CancelAndSendVideoVisitServiceSa(serviceAppointment, PluginExecutionContext.InitiatingUserId, PluginExecutionContext.OrganizationName, VimtUrl, OrganizationService, Logger);
                        if (videoVisitDeleteResponseMessage == null) return;
                        ProcessVideoVisitDeleteResponseMessage(videoVisitDeleteResponseMessage);
                    }
                    else
                        Logger.WriteDebugMessage("Bypassed VVS Cancel");
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(string.Format("ERROR in ServiceAppointmentCancelPostStageRunner: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)));
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        /// <summary>
        /// Create an instance of the Virtual Meeting Room request and send to VIMT.
        /// </summary>
        /// <returns>VirtualMeetingRoomDeleteResponseMessage.</returns>
        private VirtualMeetingRoomDeleteResponseMessage CancelAndSendVirtualMeetingRoom(Guid appointmentId, Guid userId, string organizationName)
        {
            var request = new VirtualMeetingRoomDeleteRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                AppointmentId = appointmentId,
                MiscData = string.Empty
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

            try
            {
                Logger.WriteDebugMessage("Begin: Sending Cancel to VVS");
                var response = Utility.SendReceive<VirtualMeetingRoomDeleteResponseMessage>(new Uri(VimtUrl), MessageRegistry.VirtualMeetingRoomDeleteRequestMessage, request, null);
                Logger.WriteDebugMessage("End: Sending Cancel to VVS");
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Cancel Virtual Meeting Room", errorMessage, vimtRequest, typeof(VirtualMeetingRoomDeleteRequestMessage).FullName,
                    typeof(VirtualMeetingRoomDeleteResponseMessage).FullName, MessageRegistry.VirtualMeetingRoomDeleteRequestMessage, PrimaryEntity.Id, OrganizationService, true);
                Logger.WriteToFile(errorMessage);

                return null;
            }
        }

        /// <summary>
        /// Create an Integration Result.
        /// </summary>
        /// <param name="virtualMeetingRoomDeleteResponseMessage">Virtual Meeting Room Delete Response Message.</param>
        private void ProcessVirtualMeetingRoomCreateResponseMessage(VirtualMeetingRoomDeleteResponseMessage virtualMeetingRoomDeleteResponseMessage)
        {
            var errorMessage = virtualMeetingRoomDeleteResponseMessage.ExceptionOccured ? virtualMeetingRoomDeleteResponseMessage.ExceptionMessage : null;
            IntegrationPluginHelpers.CreateIntegrationResult("Cancel Virtual Meeting Room", virtualMeetingRoomDeleteResponseMessage.ExceptionOccured, errorMessage,
                virtualMeetingRoomDeleteResponseMessage.VimtRequest, virtualMeetingRoomDeleteResponseMessage.SerializedInstance, virtualMeetingRoomDeleteResponseMessage.VimtResponse,
                typeof(VirtualMeetingRoomDeleteRequestMessage).FullName, typeof(VirtualMeetingRoomDeleteResponseMessage).FullName, MessageRegistry.VirtualMeetingRoomDeleteRequestMessage,
                PrimaryEntity.Id, OrganizationService);
        }

        /// <summary>
        /// Create an Integration Result.
        /// </summary>
        /// <param name="videoVisitDeleteResponseMessage">Video Visit Delete Response Message.</param>
        private void ProcessVideoVisitDeleteResponseMessage(VideoVisitDeleteResponseMessage videoVisitDeleteResponseMessage)
        {
            Logger.WriteDebugMessage("Processing VVS Cancel Response");
            var errorMessage = videoVisitDeleteResponseMessage.ExceptionOccured ? videoVisitDeleteResponseMessage.ExceptionMessage : string.Empty;
            var intResultId = IntegrationPluginHelpers.CreateIntegrationResult("Cancel Video Visit", videoVisitDeleteResponseMessage.ExceptionOccured, errorMessage,
                videoVisitDeleteResponseMessage.VimtRequest, videoVisitDeleteResponseMessage.SerializedInstance, videoVisitDeleteResponseMessage.VimtResponse,
                typeof(VideoVisitDeleteRequestMessage).FullName, typeof(VideoVisitDeleteResponseMessage).FullName, MessageRegistry.VideoVisitDeleteRequestMessage,
                PrimaryEntity.Id, OrganizationService);

            var sa = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();
            var status = IntegrationPluginHelpers.WriteVistaResults(videoVisitDeleteResponseMessage.WriteResults, intResultId, Guid.Empty, PrimaryEntity.Id, "cancel", OrganizationService, Logger, sa.StatusCode.Value);

            if (sa.StatusCode.Value != status)
                IntegrationPluginHelpers.ChangeEntityStatus(OrganizationService, sa, status);
            else
                Logger.WriteDebugMessage("All Vista Cancels were successful.  No need to update status, response processing complete");
            Logger.WriteDebugMessage("Finished Processing VVS Cancel Response");
        }
    }
}