﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Appointment
{
    public class AppointmentUpdateVVSPostStageRunner : PluginRunner
    {

        public AppointmentUpdateVVSPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        public override string McsSettingsDebugField
        {
            get { return "mcs_appointmentplugin"; }
        }

        public string VimtUrl { get; private set; }

        public override void Execute()
        {
            if (!PrimaryEntity.Attributes.Contains("cvt_triggervvs") || (bool)PrimaryEntity.Attributes["cvt_triggervvs"] == false)
            {
                Logger.WriteDebugMessage("Not triggering VVS, exiting plugin");
                return;
            }
            ClearVVSTrigger();
            var callVVS = false;
            var appt = OrganizationService.Retrieve(DataModel.Appointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.Appointment>();
            using (var srv = new Xrm(OrganizationService))
            {
                var vimtUrlSetting = srv.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
                else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");
                var sa = srv.ServiceAppointmentSet.FirstOrDefault(s => s.Id == appt.cvt_serviceactivityid.Id);
                if (sa == null) throw new InvalidPluginExecutionException("No Service Activity found for this appointment");
                callVVS = VistaPluginHelpers.RunVVS(sa, srv, Logger);
            }
            if (callVVS)
            {
                SendVistaMessage(appt);
            }
            else
            {
                Logger.WriteDebugMessage("VVS Integration Bypassed, Updating Service Activity Status");
            }
        }

        #region Vista Booking and Cancelation
        private void SendVistaMessage(DataModel.Appointment appointment)
        {
            Logger.WriteDebugMessage("Beginning VistA Book/Cancel");
            bool isBookRequest;
            bool isWholeApptCanceled;
            var changedPatients = GetChangedPatients(appointment, out isBookRequest, out isWholeApptCanceled);
            if (isBookRequest)
            {
                bool isUpdate = false;
                using (var srv = new Xrm(OrganizationService))
                {
                    var pastVistaBookings = srv.mcs_integrationresultSet.Where(
                        i => i.mcs_appointmentid.Id == appointment.Id
                        && i.mcs_status.Value != (int)mcs_integrationresultmcs_status.Error).ToList();
                    //Removed this check from where clause because string functions aren't allowed in LINQ conditions
                    foreach (var vir in pastVistaBookings)
                    {
                        if (vir.mcs_VimtMessageRegistryName.ToLower().Contains("accenture"))
                        {
                            isUpdate = true;
                            break;
                        }
                    }
                }

                if (!isUpdate) {
                    var createRequest = new VideoVisitCreateRequestMessage
                    {
                        AppointmentId = appointment.Id,
                        LogRequest = true,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        AddedPatients = changedPatients
                    };

                    Logger.WriteDebugMessage("Sending Create Booking to Vista");
                    var response = Utility.SendReceive<VideoVisitCreateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitCreateRequestMessage, createRequest, null);
                    ProcessVistaCreateResponse(response, typeof(VideoVisitCreateRequestMessage), typeof(VideoVisitCreateResponseMessage), appointment);
                } else {
                    var updateRequest = new VideoVisitUpdateRequestMessage
                    {
                        AppointmentId = appointment.Id,
                        LogRequest = true,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        Contacts = changedPatients
                    };

                    Logger.WriteDebugMessage("Sending Update Booking to Vista");
                    var response = Utility.SendReceive<VideoVisitUpdateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitUpdateRequestMessage, updateRequest, null);
                    ProcessVistaUpdateResponse(response, typeof(VideoVisitUpdateRequestMessage), typeof(VideoVisitUpdateResponseMessage), appointment);
                }
            }
            else
            {
                if (isWholeApptCanceled)
                {
                    var cancelRequest = new VideoVisitDeleteRequestMessage
                    {
                        AppointmentId = appointment.Id,
                        OrganizationName = PluginExecutionContext.OrganizationName,
                        UserId = PluginExecutionContext.UserId,
                        LogRequest = true,
                        CanceledPatients = changedPatients,
                        WholeAppointmentCanceled = isWholeApptCanceled
                    };
                    Logger.WriteDebugMessage("Sending Cancel Booking to Vista");
                    var response = Utility.SendReceive<VideoVisitDeleteResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitDeleteRequestMessage, cancelRequest, null);
                    ProcessVistaCancelResponse(response, typeof(VideoVisitDeleteRequestMessage), typeof(VideoVisitDeleteResponseMessage), appointment, isWholeApptCanceled);
                }
                else
                {
                    Logger.WriteDebugMessage("Individual patient was canceled through Cancel Dialog and has already been sent to VVS in previous plugin instance.  No action needed here, ending thread.");
                }
            }
        }

        private void ProcessVistaBookResponse(bool isCreate, DataModel.Appointment appointment, string errorMessage, bool exceptionOccured, string vimtRequest, string serializedInstance, string vimtResponse, WriteResults writeResults)
        {

            var name = isCreate ? "Group Book to Vista" : "Group Booking Update to Vista";
            var reqType = isCreate ? typeof(VideoVisitCreateRequestMessage).FullName : typeof(VideoVisitUpdateRequestMessage).FullName;
            var respType = isCreate ? typeof(VideoVisitCreateResponseMessage).FullName : typeof(VideoVisitUpdateResponseMessage).FullName;
            var regName = isCreate ? MessageRegistry.VideoVisitCreateRequestMessage : MessageRegistry.VideoVisitUpdateRequestMessage;
            var integrationResultId = IntegrationPluginHelpers.CreateAppointmentIntegrationResult(name, exceptionOccured, errorMessage, vimtRequest, serializedInstance, vimtResponse, reqType, respType, regName, appointment.Id, OrganizationService);

            if (exceptionOccured)
                Logger.WriteToFile("Exception Occurred in Group Booking: " + errorMessage);

            IntegrationPluginHelpers.WriteVistaResults(writeResults, integrationResultId, appointment.Id, Guid.Empty, "book", OrganizationService, Logger);
        }

        private void ProcessVistaCreateResponse(VideoVisitCreateResponseMessage response, Type requestType, Type responseType, DataModel.Appointment appointment)
        {
            if (response == null) return;
            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            ProcessVistaBookResponse(true, appointment, errorMessage, response.ExceptionOccured, response.VimtRequest, response.SerializedInstance, response.VimtResponse, response.WriteResults);
        }

        private void ProcessVistaUpdateResponse(VideoVisitUpdateResponseMessage response, Type requestType, Type responseType, VA.TMP.DataModel.Appointment appointment)
        {
            if (response == null) return;
            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            ProcessVistaBookResponse(false, appointment, errorMessage, response.ExceptionOccured, response.VimtRequest, response.SerializedInstance, response.VimtResponse, response.WriteResults);
        }

        private void ProcessVistaCancelResponse(VideoVisitDeleteResponseMessage response, Type requestType, Type responseType, VA.TMP.DataModel.Appointment appointment, bool wholeAppointmentCanceled)
        {
            if (response == null) return;

            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;

            var integrationResultId = IntegrationPluginHelpers.CreateAppointmentIntegrationResult("Group Cancel to Vista", response.ExceptionOccured, errorMessage, response.VimtRequest,
                response.SerializedInstance, response.VimtResponse, requestType.FullName, responseType.FullName,
                MessageRegistry.VideoVisitDeleteRequestMessage, PrimaryEntity.Id, OrganizationService, false);

            int status = wholeAppointmentCanceled ? appointment.cvt_IntegrationBookingStatus.Value : (int)Appointmentcvt_IntegrationBookingStatus.PatientCanceled;
            if (response.ExceptionOccured)
                status = (int)Appointmentcvt_IntegrationBookingStatus.CancelFailure;
            else
                status = IntegrationPluginHelpers.WriteVistaResults(response.WriteResults, integrationResultId, PrimaryEntity.Id, Guid.Empty, "cancel", OrganizationService, Logger, status);
            if (!wholeAppointmentCanceled)
            {
                Logger.WriteDebugMessage("Individual Cancelation, not updating entire appointment status to " + ((Appointmentcvt_IntegrationBookingStatus)status).ToString());
                return;
            }
            if (appointment.cvt_IntegrationBookingStatus.Value != status)
                IntegrationPluginHelpers.UpdateAppointment(OrganizationService, appointment.Id, (Appointmentcvt_IntegrationBookingStatus)status);
            else
                Logger.WriteDebugMessage("Appointment Booking Status has not changed, no need to update appointment on cancel");
        }

        internal List<Guid> GetChangedPatients(VA.TMP.DataModel.Appointment appointment, out bool isBook, out bool isWholeAppointmentCanceled)
        {
            isBook = false;
            isWholeAppointmentCanceled = false;
            var apptWasCanceled = FullAppointmentCanceled(appointment);
            var currentPatients = VistaPluginHelpers.GetPatientsFromActivityPartyList(appointment.OptionalAttendees.ToList());

            if (apptWasCanceled) {
                Logger.WriteDebugMessage("Full Appointment Canceled, cancelling all patients with clinic canceled status");
                isWholeAppointmentCanceled = true;
                return currentPatients;
            } else {
                //This means that specific patients were added or removed, but the overall appointment is still booked.  Need to find delta in booked vs current to get list of cancels or adds.  Individual Patient Cancels are handled by Vista Integration Result, so no action is needed here unless new patients are booked
                var previouslyBookedPatients = new List<Guid>();
                Logger.WriteDebugMessage("Individual Patient(s) added or removed");
                using (var srv = new Xrm(OrganizationService)) {
                    var pastBookings = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_Appointment.Id == appointment.Id && vir.cvt_VistAStatus != VistaStatus.CANCELLED.ToString());
                    foreach (var book in pastBookings) {
                        var contactId = IntegrationPluginHelpers.GetPatIdFromIcn(book.cvt_PersonId, OrganizationService);
                        //Only want distinct patients since WriteResults (aka vista integration results) will return 2 copies for a clinic based (1 for pat side and 1 for pro side with the same patient)
                        if (!previouslyBookedPatients.Contains(contactId))
                            previouslyBookedPatients.Add(contactId);
                    }
                }
                //var cancels = previouslyBookedPatients.Except(currentPatients).ToList();
                var adds = currentPatients.Except(previouslyBookedPatients).ToList();
                if (adds.Count > 0) {
                    isBook = true;
                    return currentPatients;
                }

                Logger.WriteDebugMessage("No patients were added in this Appointment");
                return new List<Guid>();
            }
        }

        /// <summary>
        /// Determines if the overall appointment got canceled, or if it was just individual patients getting removed
        /// </summary>
        /// <param name="appt"></param>
        /// <returns></returns>
        private bool FullAppointmentCanceled(DataModel.Appointment appt)
        {
            var isCanceled = IsCanceledBookingStatus(appt.cvt_IntegrationBookingStatus);

            // TO DO determine the best way to derive if the appointment itself is being canceled
            if (appt.StateCode == null)
                return true;
            if (isCanceled || appt.StateCode.Value == AppointmentState.Canceled)
                return true;

            return false;
        }

        private static bool IsCanceledBookingStatus(OptionSetValue statusCode)
        {
            if (statusCode == null)
                return false;
            switch (statusCode.Value) {
                case (int)Appointmentcvt_IntegrationBookingStatus.ClinicCancelled:
                case (int)Appointmentcvt_IntegrationBookingStatus.PatientCanceled:
                case (int)Appointmentcvt_IntegrationBookingStatus.TechnologyFailure:
                case (int)Appointmentcvt_IntegrationBookingStatus.SchedulingError: 
                case (int)Appointmentcvt_IntegrationBookingStatus.PatientNoShow: return true;
                default: return false;
            }
        }

        #endregion

        private void ClearVVSTrigger()
        {
            var updateAppt = new DataModel.Appointment
            {
                Id = PrimaryEntity.Id,
                cvt_TriggerVVS = false
            };
            OrganizationService.Update(updateAppt);
            Logger.WriteDebugMessage("Cleared Trigger flag so future appt calls can trigger VVS");
        }
    }
}
