using CRM007.CRM.SDK.Core;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.Xrm.Sdk.Discovery;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Formatting;
using System.Net.Http.Headers;
using System.Security.Cryptography.X509Certificates;
using System.ServiceModel.Description;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using System.Xml.Schema;
using System.Xml.Serialization;
using VA.TMP.DataModel;
using VRM.Integration.Servicebus.Core;
using XmlSerializer = System.Xml.Serialization.XmlSerializer;

namespace VA.TMP.Integration.VIMT.Shared
{
    /// <summary>
    /// Static class helpers for the VIMT pipeline steps.
    /// </summary>
    public static class PipelineUtilities
    {
        /// <summary>
        /// Enum used to determine whether to send a given Web Service JSON or XML.
        /// </summary>
        public enum ServiceMediaType
        {
            Json,
            Xml
        }

        /// <summary>
        /// Enum used to determine additional authentication if the default doesn't work.
        /// </summary>
        public enum CrmAuthenticationMode
        {
            Ad,
            Adfs
        }

        /// <summary>
        /// Connects to CRM and returns the connection.
        /// </summary>
        /// <param name="organizationName">CRM organization name.</param>
        /// <param name="authenticationMode">Leave null for normal connection; otherwise specify AD or ADFS.</param>
        /// <returns>CRM connection.</returns>
        public static OrganizationServiceProxy ConnectToCrm(string organizationName, CrmAuthenticationMode? authenticationMode = null)
        {
            var parms = CrmConnectionConfiguration.Current.GetCrmConnectionParmsByName(organizationName);

            try
            {
                try
                {
                    var connection = CrmConnection.Connect(parms, typeof(mcs_integrationsetting).Assembly);
                    connection.Authenticate();
                    
                    return connection;
                }
                catch (Exception innerException)
                {
                    Logger.Instance.Debug(string.Format("TMP: Unable to connect to CRM Organization {0} using standard approach. Trying alternate method using AD/ADFS. Error was {1}", organizationName, innerException));

                    var credentials = new ClientCredentials();
                    credentials.UserName.UserName = parms.Domain + "\\" + parms.UserName;
                    credentials.UserName.Password = parms.Password;
                    OrganizationServiceProxy connection;

                    if (authenticationMode != null && (authenticationMode == CrmAuthenticationMode.Adfs))
                    {
                        var discoConfig = ServiceConfigurationFactory.CreateConfiguration<IDiscoveryService>(new Uri(parms.DiscoveryServiceUrl));
                        var userWrapper = discoConfig.Authenticate(credentials);
                        var svcConfig = ServiceConfigurationFactory.CreateConfiguration<IOrganizationService>(new Uri(parms.OrganizationServiceUrl));
                        connection = new OrganizationServiceProxy(svcConfig, userWrapper);
                    }
                    else if (authenticationMode != null && (authenticationMode == CrmAuthenticationMode.Ad))
                    {
                        connection = new OrganizationServiceProxy(new Uri(parms.OrganizationServiceUrl), null, credentials, null);
                    }
                    else
                    {
                        throw new ArgumentException(string.Format("Unknown authentication mode {0}", authenticationMode));
                    }

                    connection.EnableProxyTypes(typeof(mcs_integrationsetting).Assembly);
                    return connection;
                }
            }
            catch (Exception outerException)
            {
                throw new Exception(string.Format("TMP: Unable to connect to CRM Organization {0}. Error was {1}", organizationName, outerException));
            }
        }

        /// <summary>
        /// Serializes an instance of a class to a string.
        /// </summary>
        /// <typeparam name="T">The Type of class.</typeparam>
        /// <param name="classInstance">The instance of the class.</param>
        /// <returns>Serialized instance of class.</returns>
        public static string SerializeInstance<T>(T classInstance)
        {
            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(T));
                serializer.Serialize(stream, classInstance);

                return Encoding.UTF8.GetString(stream.ToArray());
            }
        }

        /// <summary>
        /// Serializes an instance of a class to a string.
        /// </summary>
        /// <typeparam name="T">The Type of class.</typeparam>
        /// <param name="classInstance">The instance of the class.</param>
        /// <param name="prefixes">List of namespace prefixes.</param>
        /// <param name="namespaces">List of namespaces.</param>
        /// <returns>Serialized instance of class.</returns>
        public static string SerializeInstance<T>(T classInstance, List<string> prefixes, List<string> namespaces)
        {
            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(T));

                var ns = new XmlSerializerNamespaces();

                for (var i = 0; i < prefixes.Count; i++)
                {
                    ns.Add(prefixes[i], namespaces[i]);
                }

                serializer.Serialize(stream, classInstance, ns);
                return Encoding.ASCII.GetString(stream.ToArray());
            }
        }

        /// <summary>
        /// Validate schema with namespaces.
        /// </summary>
        /// <param name="area">Functional area of schema being validated.</param>
        /// <param name="schemaPath">The file path to the schema.</param>
        /// <param name="namespaces">The list of namespaces for the schema.</param>
        /// <param name="schemaFileNames">The list of schema files.</param>
        /// <param name="xml">XML represenation of the class instance.</param>
        public static void ValidateSchema(string area, string schemaPath, List<string> namespaces, List<string> schemaFileNames, string xml)
        {
            var schemas = new XmlSchemaSet();

            for (var i = 0; i < namespaces.Count; i++)
            {
                schemas.Add(namespaces[i], Path.Combine(schemaPath, schemaFileNames[i]));
            }

            var examStatusUpdateXml = XDocument.Parse(xml);
            var schemaValidationMessage = string.Empty;
            examStatusUpdateXml.Validate(schemas, (o, err) => { schemaValidationMessage = err.Message; });

            if (string.IsNullOrEmpty(schemaValidationMessage)) return;

            var validationError = string.Format("{0} Schema Validation Error: {1}", area, schemaValidationMessage);
            Logger.Instance.Error(validationError);
            throw new Exception(validationError);
        }

        /// <summary>
        /// Post a given entity.
        /// </summary>
        /// <typeparam name="T">Type of entity to POST.</typeparam>
        /// <param name="area">Entity name.</param>
        /// <param name="useCertificate">Whether to use a certificate in the request.</param>
        /// <param name="certificateSubjectName">Name of the certificate.</param>
        /// <param name="baseUri">Base URI.</param>
        /// <param name="uri">Remaining URI.</param>
        /// <param name="serviceMediaType">Use Xml or Json.</param>
        /// <param name="payload">Data to send.</param>
        /// <returns>HTTP Response.</returns>
        public static async Task<HttpResponseMessage> PostToRestService<T>(string area, bool useCertificate, string certificateSubjectName, string baseUri, string uri, ServiceMediaType serviceMediaType, T payload)
        {
            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;

            using (var client = GetHttpClient(useCertificate, certificateSubjectName))
            {
                client.BaseAddress = new Uri(baseUri);
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(serviceMediaType == ServiceMediaType.Xml
                    ? new MediaTypeWithQualityHeaderValue("application/xml")
                    : new MediaTypeWithQualityHeaderValue("application/json"));

                HttpResponseMessage response;

                try
                {
                    if (serviceMediaType == ServiceMediaType.Xml) response = await client.PostAsync(uri, payload, new XmlMediaTypeFormatter { UseXmlSerializer = true });
                    else response = await client.PostAsync(uri, payload, new JsonMediaTypeFormatter { UseDataContractJsonSerializer = false });
                }
                catch (Exception ex)
                {
                    throw new Exception(string.Format("Failed POST {0} to Service. Reason : {1}", area, ex.Message), ex.InnerException);
                }

                if (!response.IsSuccessStatusCode)
                {
                    throw new Exception(string.Format("Failed POST {0} to Service. Reason : {1}", area, response.StatusCode));
                }

                return response;
            }
        }

        /// <summary>
        /// Put a given entity.
        /// </summary>
        /// <typeparam name="T">Type of entity to PUT.</typeparam>
        /// <param name="area">Entity name.</param>
        /// <param name="useCertificate">Whether to use a certificate in the request.</param>
        /// <param name="certificateSubjectName">Name of the certificate.</param>
        /// <param name="baseUri">Base URI.</param>
        /// <param name="uri">Remaining URI.</param>
        /// <param name="serviceMediaType">Use Xml or Json.</param>
        /// <param name="payload">Data to send.</param>
        /// <returns>HTTP Response.</returns>
        public static async Task<HttpResponseMessage> PutToRestService<T>(string area, bool useCertificate, string certificateSubjectName, string baseUri, string uri, ServiceMediaType serviceMediaType, T payload)
        {
            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;

            using (var client = GetHttpClient(useCertificate, certificateSubjectName))
            {
                client.BaseAddress = new Uri(baseUri);
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(serviceMediaType == ServiceMediaType.Xml
                    ? new MediaTypeWithQualityHeaderValue("application/xml")
                    : new MediaTypeWithQualityHeaderValue("application/json"));

                HttpResponseMessage response;

                try
                {
                    if (serviceMediaType == ServiceMediaType.Xml) response = await client.PutAsync(uri, payload, new XmlMediaTypeFormatter { UseXmlSerializer = true });
                    else response = await client.PutAsync(uri, payload, new JsonMediaTypeFormatter { UseDataContractJsonSerializer = false });
                }
                catch (Exception ex)
                {
                    throw new Exception(string.Format("Failed PUT {0} to Service. Reason : {1}", area, ex.Message), ex.InnerException);
                }

                if (!response.IsSuccessStatusCode)
                {
                    throw new Exception(string.Format("Failed PUT {0} to Service. Reason : {1}", area, response.StatusCode));
                }

                return response;
            }
        }
        
        /// <summary>
        /// Gets an HttpClient instance in order to make a REST call.
        /// </summary>
        /// <param name="useCertificate">Whether to use a certificate in the request.</param>
        /// <param name="certificateSubjectName">Name of the certificate.</param>
        /// <returns>Http Client used for REST calls.</returns>
        private static HttpClient GetHttpClient(bool useCertificate, string certificateSubjectName)
        {
            if (!useCertificate)
            {
                return new HttpClient();
            }

            var store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);

            var cert = store.Certificates.Find(X509FindType.FindBySubjectName, certificateSubjectName, true);

            if (cert.Count < 1)
            {
                throw new Exception(string.Format("Could not find a valid client certificate with subject {0}", certificateSubjectName));
            }

            var clientHandler = new WebRequestHandler();
            clientHandler.ClientCertificates.Add(cert[0]);

            return new HttpClient(clientHandler);
        }

        /// <summary>
        /// Builds an error message from an exception recursively.
        /// </summary>
        /// <param name="ex">Exception.</param>
        /// <returns>Exception message.</returns>
        public static string BuildErrorMessage(Exception ex)
        {
            var errorMessage = string.Empty;

            if (ex.InnerException == null) return errorMessage;

            errorMessage += string.Format("\n\n{0}\n", ex.InnerException.Message);
            errorMessage += BuildErrorMessage(ex.InnerException);

            return errorMessage;
        }

        /// <summary>
        /// Trims the given string from the begining.
        /// </summary>
        /// <param name="value">String to trim.</param>
        /// <param name="toTrim">Value to trim.</param>
        /// <returns></returns>
        public static string TrimStart(this string value, string toTrim)
        {
            var result = value;
            while (TrimStart(result, toTrim, out result)) { }
            return result;
        }

        /// <summary>
        /// Trims the given string from the begining.
        /// </summary>
        /// <param name="value">String to trim.</param>
        /// <param name="toTrim">Value to trim.</param>
        /// <param name="result">Result of the trim start.</param>
        /// <returns></returns>
        private static bool TrimStart(this string value, string toTrim, out string result)
        {
            result = value;
            if (!value.StartsWith(toTrim)) return false;

            var startIndex = toTrim.Length;
            result = value.Substring(startIndex);
            return true;
        }

        /// <summary>
        /// Determines if it is a GFE Service Appointment.
        /// </summary>
        /// <param name="sa">Service Appointment.</param>
        /// <param name="organizationService">Organization Service.</param>
        /// <returns>Whether the Service Appointment is a GFE.</returns>
        public static bool IsGfeServiceActivity(ServiceAppointment sa, IOrganizationService organizationService)
        {
            if (sa.Customers?.FirstOrDefault() == null) throw new Exception(string.Format("No Patients are listed on the service activity {0} named {1}", sa.Id, sa.Subject));

            var patientAp = sa.Customers.FirstOrDefault();
            if (patientAp == null) throw new Exception("Unable to find Customer.");

            Contact patient;
            using (var srv = new Xrm(organizationService))
                patient = srv.ContactSet.FirstOrDefault(c => c.Id == patientAp.PartyId.Id);

            if (patient == null) throw new Exception(string.Format("No patient was found with Id: {0}, AP.ActivityId: {1}", patientAp.PartyId.Id, patientAp.ActivityId.Id));
            var hasTablet = !string.IsNullOrEmpty(patient.cvt_BLTablet);
            Logger.Instance.Debug(string.Format("SaId: {0}, PatientName: {1} Patient hasTablet (Service Activity is GFE): {2}", sa.Id, patient.FullName, hasTablet));

            return hasTablet;
        }
    }
}