﻿using MCSShared;
using MCSUtilities2011;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace VA.TMP.CRM
{
    public class ServiceAppointmentEmail
    {
        #region Constructor/Data Model for Service Appointment Emails
        IOrganizationService OrganizationService;
        MCSLogger Logger;
        Email Email;
        ServiceAppointment ServiceAppointment;
        mcs_services Tsa;

        List<SystemUser> patTCTList = new List<SystemUser>();
        List<SystemUser> proTCTList = new List<SystemUser>();
        string patTCTs;
        string proTCTs;

        private string SiteLocal911Phone;
        private string SiteMainPhone;
        private string ProRoom;
        private string PatRoom;
        private string stethIP;
        private string PatientVirtualMeetingSpace;
        private string ProviderVirtualMeetingSpace;
        private cvt_component VirtualMeetingSpaceComponent;
        private bool isCvtTablet;

        public ServiceAppointmentEmail(IOrganizationService organizationService, MCSLogger logger, Email email, ServiceAppointment serviceAppointment, mcs_services tsa)
        {
            OrganizationService = organizationService;
            Logger = logger;
            Email = email;
            ServiceAppointment = serviceAppointment;
            Tsa = tsa;
        }
        #endregion

        #region Choose which Email to send
        /// <summary>
        /// Determine which email to populate and send, then call the appropriate function(s)
        /// </summary>
        public void Execute()
        {
            if ((ServiceAppointment.StateCode.Value == ServiceAppointmentState.Canceled ||  ServiceAppointment.StatusCode.Value == (int)serviceappointment_statuscode.ReservedScheduled) && Email.RegardingObjectId == null)
            {
                if (Email.Subject.StartsWith("TMP Scheduler Action:"))
                {
                    Logger.WriteDebugMessage("Beginning Vista Reminder Email");
                    SendVistaReminder();
                    Logger.WriteDebugMessage("Completed Vista Reminder Email");
                }
                else if (Email.Subject.Contains("Telehealth Appointment Notification for"))
                {
                    Logger.WriteDebugMessage("Beginning Service Activity Notification Email");
                    NotifyParticipantsOfAppointment(true, "");
                    Logger.WriteDebugMessage("Completed Service Activity Notification Email");
                }
                else if (Email.Subject.Contains("Patients have been"))
                {
                    Logger.WriteDebugMessage("Beginning Provider email notification of change in patients");
                    NotifyProviderOfPatientChange();
                    Logger.WriteDebugMessage("Completed Provider email notification of change in patients");
                }
                else if (Email.Subject.Contains("Your Video Visit has been "))
                {
                    Logger.WriteDebugMessage("Beginning Patient email notification of addition/removal from H/M Group SA");
                    NotifyPatientOfAdditionOrRemovalFromGroup(Email.Subject.Contains("Cancelled"));
                    Logger.WriteDebugMessage("Completed Patient email notification of addition/removal from H/M Group SA");
                }
            }
            else if (Email.RegardingObjectId != null && Email.RegardingObjectId.LogicalName == Contact.EntityLogicalName)
            {
                Logger.WriteDebugMessage("Beginning Patient Email");
                CvtHelper.CreateCalendarAppointmentAttachment(Email, ServiceAppointment, ServiceAppointment.StatusCode.Value, "", OrganizationService, Logger);
                CvtHelper.UpdateSendEmail(Email, OrganizationService);
                Logger.WriteDebugMessage("Completed Patient Email");
            }
            else
                return;
        }
        #endregion

        #region Vista Reminder Email
        internal void SendVistaReminder()
        {
            Logger.WriteDebugMessage("Beginning Vista Reminder");
            var provTeamMembers = new List<TeamMembership>();
            var patTeamMembers = new List<TeamMembership>();
            var provFacilityId = Tsa.cvt_ProviderFacility.Id;
            var patFacilityId = Tsa.cvt_PatientFacility != null ? Tsa.cvt_PatientFacility.Id : Guid.Empty;
            var intraFacility = (provFacilityId == patFacilityId) ? true : false;

            using (var srv = new Xrm(OrganizationService))
            {
                var provTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == provFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290005);
                if (provTeam != null)
                    provTeamMembers = srv.TeamMembershipSet.Where(TM => TM.TeamId == provTeam.Id).ToList();
                else
                    Logger.WriteToFile("The provider side Scheduler Team was unable to be found for Service Activity: " + ServiceAppointment.Id);

                var patTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290005);
                if (patTeam != null)
                    patTeamMembers = srv.TeamMembershipSet.Where(TM => TM.TeamId == patTeam.Id).ToList();
                else
                    Logger.WriteToFile("The patient side Scheduler Team was unable to be found for Service Activity: " + ServiceAppointment.Id);
            }

            Logger.WriteDebugMessage(string.Format("Retrieved {0} Pat Team Members and {1} Pro Team Members", patTeamMembers.Count, provTeamMembers.Count));
            bool provCheck = false;
            bool patCheck = false;
            EntityCollection provMembers = new EntityCollection();
            EntityCollection patMembers = new EntityCollection();
            var subSpecialty = ServiceAppointment.mcs_servicesubtype != null ? ServiceAppointment.mcs_servicesubtype.Id : Guid.Empty;

            if (provTeamMembers.Count == 0)
                Logger.WriteToFile("There are no members of the Scheduler team at " + Tsa.cvt_ProviderFacility.Name + ".  Please contact the FTC and ensure this is corrected.");
            else
            {
                foreach (TeamMembership tm in provTeamMembers)
                {
                    if (tm.SystemUserId != null)
                    {
                        if (FilterMembersBySpecialty(tm.SystemUserId.Value, ServiceAppointment.mcs_servicetype.Id, subSpecialty))
                        {
                            ActivityParty p = new ActivityParty()
                            {
                                PartyId = new EntityReference(SystemUser.EntityLogicalName, tm.SystemUserId.Value)
                            };
                            provMembers.Entities.Add(p);
                        }
                        if (ServiceAppointment.CreatedBy.Id == tm.SystemUserId.Value)
                            provCheck = true;
                    }
                }
            }
            if (patTeamMembers.Count == 0 && !Tsa.cvt_Type.Value)
                Logger.WriteToFile(string.Format("There are no members of the Scheduler team at {0}.  Please contact the FTC and ensure this is corrected.", Tsa.cvt_PatientFacility != null ? Tsa.cvt_PatientFacility.Name : "\"No Facility Listed\""));
            else
            {
                foreach (TeamMembership tm in patTeamMembers)
                {
                    if (tm.SystemUserId != null)
                    {
                        if (FilterMembersBySpecialty(tm.SystemUserId.Value, ServiceAppointment.mcs_servicetype.Id, subSpecialty))
                        {
                            ActivityParty p = new ActivityParty()
                            {
                                PartyId = new EntityReference(SystemUser.EntityLogicalName, tm.SystemUserId.Value)
                            };
                            patMembers.Entities.Add(p);
                        }
                        if (ServiceAppointment.CreatedBy.Id == tm.SystemUserId.Value)
                            patCheck = true;
                    }
                }
            }
            //If TSA is Store Forward and the scheduler is on the patient Scheduler team side OR if the scheduler is on both Scheduler Teams, then don't send email
            if ((Tsa.cvt_AvailableTelehealthModalities != null && Tsa.cvt_AvailableTelehealthModalities.Value == (int)mcs_servicescvt_AvailableTelehealthModalities.StoreandForward && patCheck) || (patCheck && provCheck))
            {
                DeleteVistaReminder("No need to send out email, Deleting Email.  TSA is SFT and Scheduler is on Pat Team OR Scheduler is on Both Pat and Prov Team");
            }
            else
            {
                SetupVistaReminderEmail(provMembers, patMembers, provCheck, patCheck);
            }
        }

        internal void DeleteVistaReminder(string debugMessage)
        {
            Logger.WriteDebugMessage(debugMessage);
            try
            {
                OrganizationService.Delete(Email.LogicalName, Email.Id);
                Logger.WriteDebugMessage("Email Deleted");
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to Delete Email " + ex.Message + ".  Leaving email as is.");
            }
        }

        /// <summary>
        /// returns false if the user is not associated with the specialty on SA (or sub-specialty if listed)
        /// </summary>
        /// <param name="userId">id of user to check for specialties</param>
        /// <param name="specialty">specialty to check</param>
        /// <param name="subSpecialty">sub-specialty to check</param>
        /// <returns>true if user is associated with specialty/sub-specialty or false if not</returns>
        internal bool FilterMembersBySpecialty(Guid userId, Guid specialty, Guid subSpecialty)
        {
            var userAssociatedWithSpecialty = false;
            var userAssociatedWithSubSpecialty = false;
            var specialties = new List<mcs_servicetype>();
            var subSpecialties = new List<mcs_servicesubtype>();
            using (var srv = new Xrm(OrganizationService))
            {
                var user = srv.SystemUserSet.FirstOrDefault(u => u.Id == userId);

                //Retrieve related records through N:N association - CRM doesn't eager load, so have to call this.  It returns null if no items are in the list, so also need to null check before converting to List
                srv.LoadProperty(user, "cvt_systemuser_mcs_servicetype");
                var specialtyRelated = user.cvt_systemuser_mcs_servicetype;
                if (specialtyRelated != null)
                    specialties = specialtyRelated.ToList();

                if (subSpecialty != Guid.Empty)
                {
                    //Same comment as above
                    srv.LoadProperty(user, "cvt_systemuser_mcs_servicesubtype");
                    var subSpecialtyRelated = user.cvt_systemuser_mcs_servicesubtype;
                    if (subSpecialtyRelated != null)
                        subSpecialties = subSpecialtyRelated.ToList();
                    if (subSpecialties.Count == 0)
                    {
                        userAssociatedWithSubSpecialty = true;
                        Logger.WriteDebugMessage(string.Format("{0} has no sub-specialties listed, checking specialties", user.FullName));
                    }
                    else
                    {
                        var subMatch = subSpecialties.FirstOrDefault(s => s.Id == subSpecialty);
                        userAssociatedWithSubSpecialty = subMatch != null;
                        Logger.WriteDebugMessage(String.Format("{0} {1} {2} as a sub-specialty", user.FullName, userAssociatedWithSubSpecialty ? "has" : "does not have", subSpecialty));
                        return userAssociatedWithSubSpecialty;
                    }
                }
                if (specialties.Count == 0)
                {
                    userAssociatedWithSpecialty = true;
                    Logger.WriteDebugMessage("User has no specialties listed, auto-opting in " + user.FullName + " to emails");
                }
                else
                {
                    var match = specialties.FirstOrDefault(s => s.Id == specialty);
                    userAssociatedWithSpecialty = match != null;
                    Logger.WriteDebugMessage(String.Format("{0} {1} {2} as a specialty", user.FullName, userAssociatedWithSpecialty ? "has" : "does not have", specialty));
                }
            }
            return userAssociatedWithSpecialty;
        }

        internal void SetupVistaReminderEmail(EntityCollection provMembers, EntityCollection patMembers, bool provCheck, bool patCheck)
        {
            Logger.WriteDebugMessage("Beginning SetupVistaReminderEmail");
            mcs_facility patFacility = null;
            if (Tsa.cvt_PatientFacility != null)
                patFacility = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, Tsa.cvt_PatientFacility.Id, new ColumnSet("mcs_stationnumber"));
            var patStation = patFacility == null ? string.Empty : " (" + patFacility.mcs_StationNumber + ")";
            mcs_facility proFacility = null;
            if (Tsa.cvt_ProviderFacility != null)
                proFacility = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, Tsa.cvt_ProviderFacility.Id, new ColumnSet("mcs_stationnumber"));
            var proStation = proFacility == null ? string.Empty : " (" + proFacility.mcs_StationNumber + ")";

            Email.From = CvtHelper.SetPartyList(ServiceAppointment.CreatedBy);
            int timeZone = 0;
            List<ActivityParty> To = new List<ActivityParty>();
            Logger.WriteDebugMessage(string.Format("Retrieved pat {0} and pro {1} facilities and set the email sender {2}", patStation, proStation, Email.From.ToString()));
            if (provCheck == false)
            {
                //Add Prov Scheduler Team Members to To Line
                foreach (ActivityParty ap in provMembers.Entities)
                {
                    To.Add(ap);
                }
                //To = provMembers.Entities.ToList<ActivityParty>();
                Entity proSite; //Either the site or facility of the provider
                if (Tsa.cvt_relatedprovidersiteid != null)
                    proSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, Tsa.cvt_relatedprovidersiteid.Id, new ColumnSet(true));
                else
                    proSite = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, Tsa.cvt_ProviderFacility.Id, new ColumnSet(true));
                timeZone = (int)proSite.Attributes["mcs_timezone"];
            }
            if (patCheck == false)
            {
                //Add Pat Scheduler Team Members to To Line
                foreach (ActivityParty ap in patMembers.Entities)
                {
                    To.Add(ap);
                }
                To = To.GroupBy(A => A.PartyId.Id).Select(g => g.First()).ToList<ActivityParty>(); //Method to select distinct recipients based on recipient ID (since entire Activity Party may not be duplicate)
                Entity patSite;
                if (Tsa.cvt_relatedprovidersiteid != null)
                    patSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, Tsa.cvt_relatedprovidersiteid.Id, new ColumnSet(true));
                else
                    patSite = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, Tsa.cvt_ProviderFacility.Id, new ColumnSet(true));
                timeZone = (int)patSite.Attributes["mcs_timezone"];
            }
            Email.To = To;
            
            string fullDateTime = CvtHelper.GetTimeZoneString(timeZone, ServiceAppointment.ScheduledStart.Value, OrganizationService, Logger);
            var equips = ServiceAppointment.Resources.Where(ap => ap.PartyId.LogicalName == Equipment.EntityLogicalName).ToList();

            //Added to ensure that resources from child appointments (for Group SAs) are also retrieved and included in the list of resources
            var childEquips = GetApptResources(Equipment.EntityLogicalName);
            equips.AddRange(childEquips);
            var vistaClinics = "Vista Clinic(s): ";
            Logger.WriteDebugMessage("Getting Vista Clinics");
            foreach (var equipment in equips)
            {
                var e = (Equipment)OrganizationService.Retrieve(Equipment.EntityLogicalName, equipment.PartyId.Id, new ColumnSet("mcs_relatedresource"));
                if (e.mcs_relatedresource == null)
                {
                    Logger.WriteDebugMessage("Orphaned Resource has been scheduled: " + e.Name + ".  Please fix this resource and rebuild the TSA (or just re-link the equipment with the TMP Resource).");
                    break;
                }
                var resource = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName, e.mcs_relatedresource.Id, new ColumnSet("mcs_name", "mcs_type"));
                if (resource.mcs_Type != null && resource.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic)
                    vistaClinics += resource.mcs_name + "; ";
            }
            Logger.WriteDebugMessage("Added vista clinics to Scheduler Action email: " + vistaClinics);
            var displayTime = "Appointment Start Time: " + fullDateTime + "; <br/>";
            var body = CvtHelper.GenerateEmailBody(ServiceAppointment.Id, ServiceAppointment.EntityLogicalName, displayTime + vistaClinics, OrganizationService, "Click Here to open the Service Activity in TMP");
            var serviceType = Tsa.cvt_servicetype.Name;
            if (Tsa.cvt_servicesubtype != null)
                serviceType += " : " + Tsa.cvt_servicesubtype.Name;
            var status = ServiceAppointment.StatusCode.Value == 4 ? "scheduled" : "canceled";
            var proFacName = Tsa.cvt_ProviderFacility == null ? string.Empty : Tsa.cvt_ProviderFacility.Name + proStation;
            var patFacName = Tsa.cvt_PatientFacility == null ? string.Empty : Tsa.cvt_PatientFacility.Name + patStation;

            if (Tsa.cvt_Type != null && Tsa.cvt_Type.Value)
                patFacName = "Home/Mobile";

            Email.Description = string.Format("A {0} telehealth appointment has been {1} at a remote facility, please {2} this patient in VistA. The provider facility is: {3}. The patient facility is: {4}. {5}",
                serviceType,
                status,
                status == "scheduled" ? "schedule" : "cancel",
                proFacName,
                patFacName,
                body);

            if (Tsa.cvt_relatedpatientsiteid == null)
                Email.Subject += patFacName;

            CvtHelper.UpdateSendEmail(Email, OrganizationService);
        }

        internal List<ActivityParty> GetApptResources(string filter = "")
        {
            var childResources = new List<ActivityParty>();
            var childAppts = new List<Appointment>();
            using (var srv = new Xrm(OrganizationService))
            {
                childAppts = srv.AppointmentSet.Where(a => a.cvt_serviceactivityid.Id == ServiceAppointment.Id && a.ScheduledStart.Value == ServiceAppointment.ScheduledStart.Value).ToList();
            }
            foreach (var appt in childAppts)
            {
                //If there is no entityType filter listed, then just add all members of appointment requiredAttendees
                if (string.IsNullOrEmpty(filter))
                    childResources.AddRange(appt.RequiredAttendees);
                else
                {
                    foreach (var resource in appt.RequiredAttendees)
                    {
                        //PartyID should never be null, but added null check just in case.  
                        if (resource.PartyId != null && resource.PartyId.LogicalName == filter)
                            childResources.Add(resource);
                    }
                }
            }
            Logger.WriteDebugMessage("Appointment Resources retrieved for Service Activity: " + ServiceAppointment.Id);
            return childResources;
        }
        #endregion
        internal string returnTeamMemberPhones(Guid teamId, out List<SystemUser> outTCTList )
        {
            Logger.setMethod = "returnTeamMemberPhones";
            Logger.WriteGranularTimingMessage("Starting returnTeamMemberPhones");

            string AllTctPhones = string.Empty;
            List<SystemUser> TCTList = new List<SystemUser>();
            using (var srv = new Xrm(OrganizationService))
            {
                var teamMembers = srv.TeamMembershipSet.Where(tm => tm.TeamId.Value == teamId).ToList();

                if (teamMembers.Count != 0)
                {
                    Logger.WriteDebugMessage("Found Team Members: " + teamMembers.Count);
                    foreach (TeamMembership result in teamMembers)
                    {
                        SystemUser tct = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, result.SystemUserId.Value, new ColumnSet("mobilephone", "cvt_officephone", "firstname", "lastname", "internalemailaddress"));

                        var TCTName = tct.FirstName + " " + tct.LastName;
                        var TCTPhone = tct.MobilePhone;  //Use the TCT number
                                                                      
                        if (TCTPhone == null)
                            TCTPhone = tct.cvt_officephone;  //Use the TCT number
                        //TCTEmail = tct.InternalEMailAddress;
                        TCTList.Add(tct);

                        //Add TCT Phone to string if exists
                        if (TCTPhone != null)
                        {
                            if (AllTctPhones != String.Empty)
                                AllTctPhones += " OR ";
                            AllTctPhones += TCTName + " at " + TCTPhone;
                        }
                    }
                }
                outTCTList = TCTList;
                return AllTctPhones;
            }
           

        }
        #region Scheduled/Canceled Service Appointment Notification Email
        /// <summary>
        /// Primary function which generates the Service Activity notification (including ical)
        /// </summary>
        /// <param name="isHomeMobileGroup">Home Mobile Groups have different types of notifications, so this is sent on subsequent emails (also causing iCals to not get sent to Provider)</param>
        /// <param name="action">Is either Addition, Cancelation, or Addition and Cancelation</param>
        internal void NotifyParticipantsOfAppointment(bool isCreateOrCancel, string action)
        {
            //Get Pro TCTs
            if (Tsa.cvt_providersitetctteam != null)
                proTCTs = returnTeamMemberPhones(Tsa.cvt_providersitetctteam.Id, out proTCTList);
            else if (Tsa.cvt_relatedprovidersiteid.Id != null)
            {
                //get the TSA's related provider site
                mcs_site relatedProSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, Tsa.cvt_relatedprovidersiteid.Id, new ColumnSet(true));
                proTCTs = returnTeamMemberPhones(relatedProSite.cvt_tctteam.Id, out proTCTList);
            }

            //If CVT to Home, there is no patient site
            if ((Tsa.cvt_Type != true) && (Tsa.cvt_groupappointment != true))
            {
                //get the TSA's related patient site
                mcs_site relatedPatSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, Tsa.cvt_relatedpatientsiteid.Id, new ColumnSet(true));

                //Get Pat TCTs
                if (Tsa.cvt_patientsitetctteam != null)
                    patTCTs = returnTeamMemberPhones(Tsa.cvt_patientsitetctteam.Id, out patTCTList);
                else if (Tsa.cvt_relatedpatientsiteid != null)
                    patTCTs = returnTeamMemberPhones(relatedPatSite.cvt_tctteam.Id, out patTCTList);

                //Get the Site's Emergency Contact Info
                if (relatedPatSite.cvt_Local911 != null)
                    SiteLocal911Phone = relatedPatSite.cvt_Local911; //Use the Local 911
                if (relatedPatSite.cvt_phone != null)
                    SiteMainPhone = relatedPatSite.cvt_phone; //Use the Site Phone
            }

            //Get the resources listed on the service activity
            var resources = ServiceAppointment.GetAttributeValue<EntityCollection>("resources");
            EntityCollection users = new EntityCollection();
            EntityCollection equipmentResources = new EntityCollection();
            resources.Entities.AddRange(GetApptResources(string.Empty));

            //Get the users from the resource list (filter out equipment)
            foreach (var res in resources.Entities)
            {
                var party = res.ToEntity<ActivityParty>();
                if (party.PartyId.LogicalName == SystemUser.EntityLogicalName)
                {
                    ActivityParty p = new ActivityParty()
                    {
                        PartyId = new EntityReference(SystemUser.EntityLogicalName, party.PartyId.Id)
                    };
                    users.Entities.Add(p);
                }
                else
                {
                    Equipment e = (Equipment)OrganizationService.Retrieve(Equipment.EntityLogicalName, party.PartyId.Id, new ColumnSet("equipmentid", "mcs_relatedresource"));
                    mcs_resource equip = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName, e.mcs_relatedresource.Id, new ColumnSet(true));
                    equipmentResources.Entities.Add(equip);
                }
            }
            Logger.WriteDebugMessage("Split Resources into Users and Equipment");

            //Get the rooms and techs and segment them by patient/provider site
            var tsaProResources = getPRGs("provider");
            var tsaPatResources = getPRGs("patient");
            var providerTechs = ClassifyResources(equipmentResources, tsaProResources, (int)mcs_resourcetype.Technology);
            var patientTechs = ClassifyResources(equipmentResources, tsaPatResources, (int)mcs_resourcetype.Technology);
            var providerRooms = ClassifyResources(equipmentResources, tsaProResources, (int)mcs_resourcetype.Room);
            var patientRooms = ClassifyResources(equipmentResources, tsaPatResources, (int)mcs_resourcetype.Room);

            Logger.WriteDebugMessage("Classified Resources into types and sides");

            //select which users to send the email to (providers/patients): null means provider side, 1 means patient side (and 0 means both)
            var providerResources = GetRecipients(users, tsaProResources);
            var patientResources = GetRecipients(users, tsaPatResources);

            Logger.WriteDebugMessage("Classified Users by side");

            //Get providers so we can pass in string for list of clinicians to patient email
            var clinicians = providerResources;

            //format body of the email (telepresenters can be duplicated)
            var patSite = Tsa.cvt_relatedpatientsiteid;
            var patSiteString = patSite != null ? patSite.Name : string.Empty;
            if (patSiteString == string.Empty)
                patSiteString = Tsa.cvt_groupappointment.Value && !Tsa.cvt_Type.Value ? Tsa.cvt_PatientFacility.Name : "Home/Mobile";

            var timeZone = CvtHelper.GetSiteTimeZoneCode(ServiceAppointment, OrganizationService, Logger);
            var fullDateTime = CvtHelper.GetTimeZoneString(timeZone, ServiceAppointment.ScheduledStart.Value, OrganizationService, Logger);

            Email.Description = formatNotificationEmailBody(providerTechs, patientTechs, providerRooms, patientRooms, patientResources, providerResources, isCreateOrCancel, action, fullDateTime);
            Email.Subject = Email.Subject.Trim() + " " + patSiteString + " " + fullDateTime;
            //Combine the lists and then add them as the email recipients
            providerResources.AddRange(patientResources);

            //Add the Pro and Pat TCT to the .To
            providerResources.AddRange(proTCTList);
            providerResources.AddRange(patTCTList);

            Email.To = CvtHelper.SetPartyList(providerResources);

            //Get the owner of the workflow for the From field    
            if (Email.From.Count() == 0)
                Email.From = CvtHelper.GetWorkflowOwner("Service Activity Notification", OrganizationService);
            //OrganizationService.Update(email);

            //Send a Calendar Appointment if the appointment is scheduled (if canceled, send cancellation update)
            //If just a patient addition or removal, dont attach calendar or send additional patient email.  
            if (isCreateOrCancel)
            {
                CvtHelper.CreateCalendarAppointmentAttachment(Email, ServiceAppointment, ServiceAppointment.StatusCode.Value, stethIP, OrganizationService, Logger);
                CvtHelper.UpdateSendEmail(Email, OrganizationService);

                // Only send patient email for Home/Mobile TSAs (but not for groups on update)
                // Future Phase TODO: use static VMRs and notify patient to hit desktop icon
                if (Tsa.cvt_Type.Value)
                {
                    //Create and Send Email to Patient/Veteran (copy sender from Provider Email)
                    bool isCancelled = (ServiceAppointment.StatusCode.Value == 9 || ServiceAppointment.StatusCode.Value == 917290000) ? true : false; //Cancellation
                    SendPatientEmail(clinicians, isCancelled);
                }
            }
        }

        /// <summary>
        /// Take the various input lists and format the Email body
        /// </summary>
        /// <param name="providerTechs">List of Technology type resources for the provider side</param>
        /// <param name="patientTechs">List of Technology type resources for the patient side</param>
        /// <param name="providerRooms">List of Room type resources for the provider side</param>
        /// <param name="patientRooms">List of Room type resources for the patient side</param>
        /// <param name="telepresenters">List of Users for the patient side</param>
        /// <param name="providers">List of Users for the provider side</param>
        /// <param name="isCreateOrCancel">indicates whether this is the initial notification or a modification (addition/removal of patient)</param>
        /// <remarks>if this is just a patient addition/removal, then there is a separate email for the patient and the provider doesn't get an iCal change</remarks>
        /// <param name="action">string value indicating whether it is an addition, removal, or both</param>
        /// <param name="convertedDate"></param>
        /// <returns></returns>
        internal string formatNotificationEmailBody(List<mcs_resource> providerTechs, List<mcs_resource> patientTechs, List<mcs_resource> providerRooms,
            List<mcs_resource> patientRooms, List<SystemUser> telepresenters, List<SystemUser> providers, bool isCreateOrCancel, string action, string fullDateTime)
        {
            Logger.WriteDebugMessage("Starting Formatting Email Body");
            string emailBody = "";
            string providerTechsString = "";
            string patientTechsString = null;
            string providerRoomsString = null;
            string patientRoomsString = null;
            string telepresentersString = null;
            string providersString = null;
            string DEALicensed = "";

            //DEA Licensed
            DEALicensed = (ServiceAppointment.cvt_Type.Value == true) ? "This is a Home/Mobile visit; consider Ryan Haight regulations prior to prescribing any controlled medications." : "";

            foreach (mcs_resource r in providerTechs)
            {
                providerTechsString += r.mcs_name;
                if (r.cvt_relateduser != null)
                {
                    SystemUser poc = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, r.cvt_relateduser.Id, new ColumnSet("fullname", "mobilephone", "cvt_officephone", "cvt_teleworkphone"));
                    providerTechsString += "; POC Name: " + poc.FullName + "; ";
                    var phone = poc.MobilePhone == null ? poc.cvt_officephone : poc.MobilePhone;

                    //If TSA is telework (true), then add that number here as well.
                    providerTechsString += ((Tsa.cvt_ProviderLocationType != null) && (Tsa.cvt_ProviderLocationType.Value == true) && (poc.cvt_TeleworkPhone != null)) ? "POC Telework Phone #: " + poc.cvt_TeleworkPhone + ";" : "";
                    providerTechsString += (phone != null) ? "POC Phone #: " + phone : "";
                }
                providerTechsString += "<br/>";
                providerTechsString += getComponents(r, ServiceAppointment);
            }

            foreach (mcs_resource r in patientTechs)
            {
                patientTechsString += r.mcs_name;
                if (r.cvt_relateduser != null)
                {
                    SystemUser poc = (SystemUser)OrganizationService.Retrieve(
                        SystemUser.EntityLogicalName, r.cvt_relateduser.Id, new ColumnSet("fullname", "mobilephone", "cvt_officephone"));
                    patientTechsString += "; POC Name: " + poc.FullName + "; ";
                    var phone = poc.MobilePhone == null ? poc.cvt_officephone : poc.MobilePhone;
                    patientTechsString += "POC Phone #: " + phone;
                }
                patientTechsString += "<br/>";
                patientTechsString += getComponents(r, ServiceAppointment);
            }

            foreach (mcs_resource r in providerRooms)
            {
                providerRoomsString += "<b><u>Room:</u></b> " + r.mcs_name;
                if (r.cvt_phone != null)
                    ProRoom += (ProRoom == null) ? r.cvt_phone : ", " + r.cvt_phone;

                providerRoomsString += "<br/>";
            }

            foreach (mcs_resource r in patientRooms)
            {
                patientRoomsString += "<b><u>Room:</u></b> " + r.mcs_name;
                if (r.cvt_phone != null)
                {
                    PatRoom += (PatRoom == null) ? r.cvt_phone : ", " + r.cvt_phone;
                }
                if (DEALicensed == "" && r.mcs_RelatedSiteId != null)
                {
                    var resourceSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, r.mcs_RelatedSiteId.Id, new ColumnSet(true));

                    if (resourceSite.cvt_DEALicensed != null && resourceSite.cvt_DEALicensed.Value == true)
                        patientRoomsString += ";  <u><b>Note: The patient care site is DEA registered.</u></b>";
                    else
                        patientRoomsString += ";  <u><b>Note: The patient care site is NOT DEA registered.</u></b>";
                }
                patientRoomsString += "<br/>";
            }

            foreach (SystemUser t in telepresenters)
            {
                var phone = t.cvt_officephone != null ? t.cvt_officephone : t.MobilePhone;
                telepresentersString += "<b><u>Telepresenter:</u></b> " + t.FullName + ": " + phone + "<br/>";
            }
            foreach (SystemUser t in providers)
            {
                var phone = t.cvt_officephone != null ? t.cvt_officephone : t.MobilePhone;
                providersString += "<b><u>Provider:</u></b> " + t.FullName;
                providersString += (phone != null) ? "; Phone: " + phone : "";

                //If TSA is telework (true), then add that number here as well.
                if ((Tsa.cvt_ProviderLocationType != null) && (Tsa.cvt_ProviderLocationType.Value == true))
                {
                    //Check user for telework number                  
                    providersString += (t.cvt_TeleworkPhone != null) ? "; Telework Phone: " + t.cvt_TeleworkPhone + ";" : "";

                }
                providersString += "<br/>";
            }

            if (!isCreateOrCancel)
            {
                var numPatients = ServiceAppointment.Customers == null ? 0 : ServiceAppointment.Customers.ToList().Count;
                emailBody += string.Format("This is an automated Message to notify you that a there has been a patient {0} to your Group Telehealth Appointment scheduled for today at {1}. You now have {2} patients scheduled for this appointment.  The details are listed below: <br /><br />", action, fullDateTime, numPatients);
            }
            if (ServiceAppointment.StateCode.Value == ServiceAppointmentState.Canceled)
            {
                emailBody += "This is an automated Message to notify you that a Telehealth Appointment previously scheduled for " + fullDateTime +
                "has been <font color='red'><u>Canceled</u></font>.  Please open the attachment and click \"Remove from Calendar\" to remove this event from your calendar.  " +
                "The details are listed below: <br/><br/>";
            }
            else if (ServiceAppointment.StatusCode.Value == 4)
            {
                emailBody += "This is an automated Message to notify you that a Telehealth Appointment has been <font color='green'><u>Scheduled</u></font> " +
                    "for " + fullDateTime + ".  " +
                    "Please open the attachment and click \"Save and Close\" to add this event to your calendar.  " +
                    "The details are listed below: <br/><br/>";
            }

            //Provider Info
            emailBody += "<br/><font size='5' color='blue'>Provider Site Information:</font><br/>";
            emailBody += providerRoomsString;
            emailBody += (Tsa.cvt_provsitevistaclinics != null) ? "<b><u>Vista Clinic:</u></b> " + Tsa.cvt_provsitevistaclinics.ToString() : ""; //Needs to be specific Prov VC, not all
            emailBody += (!String.IsNullOrEmpty(providerTechsString)) ? "<br/><b><u>Technologies: </u></b><br/> " + providerTechsString + "<br/>" : "";
            emailBody += providersString;

            if (ProRoom != null || proTCTs != String.Empty)
                {
                emailBody += "<u><b>Telephone Contact Information:</u></b><br/> <ul>";
                emailBody += (ProRoom != null) ? "<li>To direct dial the room: " + ProRoom + "</li>" : "";
                emailBody += (proTCTs != String.Empty) ? "<li>To contact the TCTs at the provider site, call " + proTCTs + ".</li><br/><br/>" : "<li>No Team Members listed on TCT Site Team.</li>";
                emailBody += "</ul><br/><br/>";
            }

            //Patient Info
            if (Tsa.cvt_Type != true)
            {
                emailBody += "<br/><font size='5' color='blue'>Patient Site Information:</font><br/>";
                emailBody += patientRoomsString;
                emailBody += (Tsa.cvt_patsitevistaclinics != null) ? "<b><u>Vista Clinic:</u></b> " + Tsa.cvt_patsitevistaclinics.ToString() + "<br/>" : "";
                emailBody += (!String.IsNullOrEmpty(patientTechsString)) ? "<b><u>Technologies: </u></b><br/>" + patientTechsString + "<br/>" : "";
                emailBody += telepresentersString;

                if (PatRoom != null || SiteMainPhone != null || SiteLocal911Phone != null || patTCTs != String.Empty)
                {
                    emailBody += "<u><b>Telephone Contact Information:</u></b><br/> <ul>";
                    emailBody += (PatRoom != null) ? "<li>To direct dial the room: " + PatRoom + "</li>" : "";
                    emailBody += (SiteMainPhone != null) ? "<li>To reach the main phone number for the patient side clinic: " + SiteMainPhone + "</li>" : "";
                    emailBody += (SiteLocal911Phone != null) ? "<li>If you are out of area, this is the number to reach emergency services: " + SiteLocal911Phone + "</li>" : "";
                    emailBody += (patTCTs != String.Empty) ? "<li>To contact the TCTs at the patient site, call " + patTCTs + ".</li><br/><br/>" : "<li>No Team Members listed on TCT Site Team.</li>";
                    emailBody += "</ul>";
                }
            }
            else  //Condition for CVT to Home
            {
                bool? patient = null;
                var meetingSpace = getPatientVirtualMeetingSpace(out patient);

                emailBody += "<br/><br/><font size='5' color='blue'>Home/Mobile Information:</font><br/>";
                emailBody += (DEALicensed != "") ? "<u><b>Site DEA Licensed:</u></b><br/> " + DEALicensed + "<br/>" : "";
                emailBody += (patient == false) ? "<br/> Patient CVT Tablet: <br/>" : "<br/> Virtual Meeting Space: <br/>";

                if (meetingSpace == string.Empty)
                    meetingSpace = "Please Contact Your Clinician for Web Meeting Details";

                //Change to read the ProviderVirtualMeetingSpace on the patient record.
                // https://pexipdemo.com/px/vatest/#/?name=ProviderName&join=1&media=&escalate=1&conference=vatest@pexipdemo.com&pin=1234  
                if (patient == true || ProviderVirtualMeetingSpace != string.Empty)
                {
                    emailBody += "From your Web browser: " + CvtHelper.buildHTMLUrl(ProviderVirtualMeetingSpace, "Click Here to Join Virtual Medical Room") + "<br/>";

                    var conf = getParamValue(ProviderVirtualMeetingSpace, "conference=");
                    var cid = getParamValue(ProviderVirtualMeetingSpace, "pin=");

                    if (!string.IsNullOrEmpty(conf) && !string.IsNullOrEmpty(cid))
                        emailBody += String.Format("<br/>And if you wanted to dial from your VTC device:<br/><br/>From any VTC device: {0}<br/>Host CID: {1}", conf, cid);
                }
                else
                    emailBody += meetingSpace + "<br/>";

                emailBody += "<br />" + CvtHelper.ProviderSafetyChecks();
            }
            emailBody += CvtHelper.EmailFooter();

            Logger.WriteDebugMessage("Finishing Formatting Email Body");
            return emailBody;
        }
        #endregion

        #region PatientEmail
        internal void SendPatientEmail(List<SystemUser> provs, bool isCancelled, Guid? patientId = null)
        {
            //Modify caller to populate Virtual Meeting Space variables
            if (VirtualMeetingSpaceComponent != null && VirtualMeetingSpaceComponent.Id != Guid.Empty || PatientVirtualMeetingSpace.IndexOf("Please Contact Your") == -1)
            // last check is only relevant if we do not want to generate email at all given the scenario where no virtual meeting space is provided
            {
                //Get the Patient and their timezone
                List<ActivityParty> patientAPs = new List<ActivityParty>();
                if (patientId != null)
                    patientAPs.Add(new ActivityParty { PartyId = new EntityReference(Contact.EntityLogicalName, patientId.Value) });
                else
                    patientAPs = ServiceAppointment.Customers.ToList();
                if (patientAPs == null || patientAPs.ToList().Count == 0)
                    Logger.WriteToFile("No Patient was found to receive the email for following Service Activity: " + ServiceAppointment.Id);
                else
                {
                    //Getting the Subject Specialty, Specialty Sub Type
                    var serviceText = (Tsa.cvt_servicetype != null) ? "<b>Specialty:</b> " + Tsa.cvt_servicetype.Name + "<br />" : "";
                    serviceText += (Tsa.cvt_servicesubtype != null) ? "<b>Specialty Sub Type:</b> " + Tsa.cvt_servicesubtype.Name + "<br />" : "";

                    var clinicians = string.Empty;
                    foreach (SystemUser user in provs)
                    {
                        if (!string.IsNullOrEmpty(clinicians))
                            clinicians += "; ";
                        clinicians += user.FullName;
                    }
                    if (provs.Count == 1)
                        clinicians = "<b>Clinician:</b> " + clinicians + "<br />";
                    else if (provs.Count > 1)
                        clinicians = "<b>Clinicians:</b> " + clinicians + "<br />";

                    //if you can't find "Please Contact Your" that means a real url was entered, so use it as a hyperlink, otherwise, display the "Please Contact Your..." message as it comes across
                    var meetingSpace = PatientVirtualMeetingSpace.IndexOf("Please Contact Your") == -1 ?
                        CvtHelper.buildHTMLUrl(PatientVirtualMeetingSpace, "Click Here to Join the Virtual Medical Room")
                        : "<b>Your virtual meeting room was not found, " + PatientVirtualMeetingSpace + "</b>";

                    var dynamicBody = isCvtTablet ? "Your provider will call your CVT tablet for the appointment." : "Please click the following link to access the virtual medical room.  This will take you into the virtual waiting room until your provider joins.<br />";

                    //Set up difference in Scheduled vs Cancelation text
                    var descrStatus = "reminder of your";
                    var attachmentText = "<br /><br />A calendar appointment is attached to this email; you can open the attachment and save it to your calendar.";
                    var desktopLink = "https://vaots.blackboard.com/bbcswebdav/institution/CVT/TSS/vmr-pat-desktop/index.htm";
                    var iosLink = "https://vaots.blackboard.com/bbcswebdav/institution/CVT/TSS/vmr-pat-ios/index.htm";
                    var trainingLink = "<br /><br />For information on how to use VMRs from Desktop and Android tablet devices, please <a href=" + desktopLink + ">Click Here</a>" +
                        "<br />For information on how to use VMRs from iOS/Apple devices (e.g. iPad, iPhone, etc.), please <a href=" + iosLink + ">Click Here</a>";

                    foreach (var patientAP in patientAPs)
                    {
                        var recipient = new ActivityParty()
                        {
                            PartyId = new EntityReference(Contact.EntityLogicalName, patientAP.PartyId.Id)
                        };
                        Logger.WriteDebugMessage("Sending Patient Email to " + patientAP.PartyId.Name);
                        var patient = (Contact)OrganizationService.Retrieve(Contact.EntityLogicalName, recipient.PartyId.Id, new ColumnSet(true));

                        //Setup variables to get timeZone conversion properly
                        DateTime timeConversion = ServiceAppointment.ScheduledStart.Value;
                        //TODO - This is no longer the source of sending patient a cancelation - if patient was removed, it was canceled for that particular patient
                        

                        var fullDate = CvtHelper.GetTimeZoneString(patient.cvt_TimeZone, timeConversion, OrganizationService, Logger);
                        //Creating the Subject text
                        var subject = "Your VA Video Visit has been ";
                        subject += (isCancelled) ? "canceled" : "scheduled";
                        subject += " for " + fullDate.Trim();
                        Logger.WriteDebugMessage("Local Time: " + fullDate);
                        if (isCancelled) //Canceled
                        {
                            descrStatus = "cancelation notice for your previously scheduled";
                            attachmentText = "<br /><br />A calendar appointment cancelation is attached to this email, you can open the attachment and click \"Remove from Calendar\" to remove this event from your calendar.";
                            dynamicBody = "";
                            meetingSpace = "";
                            trainingLink = "";
                        }

                        var description = String.Format(
                            "This is a {0} Video Visit with a VA clinician on <b>{1}</b>. {2}{3}<br /><br />{4}<br /><br />{5}{6}<br />If you have any questions or concerns, please contact your clinic. <br />{7}{8}{9}",
                            descrStatus,
                            fullDate,
                            dynamicBody,
                            (!isCvtTablet ? meetingSpace : ""),
                            PatientSafetyChecks() + "<br />",
                            serviceText,
                            clinicians,
                            attachmentText,
                            trainingLink,
                            "<br /><br />" + CvtHelper.EmailFooter());
                        List<ActivityParty> sender = new List<ActivityParty>();
                        if (Email.From == null || Email.From.ToList().Count == 0)
                            sender = CvtHelper.GetWorkflowOwner("Service Activity Notification", OrganizationService);
                        else
                        {
                            foreach(var item in Email.From)
                            {
                                sender.Add(new ActivityParty { PartyId = new EntityReference(ActivityParty.EntityLogicalName, item.PartyId.Id) });
                            }
                        }
                        Email patientEmail = new Email()
                        {
                            Subject = subject,
                            Description = description,
                            mcs_RelatedServiceActivity = new EntityReference(ServiceAppointment.EntityLogicalName, ServiceAppointment.Id),
                            RegardingObjectId = new EntityReference(Contact.EntityLogicalName, patientAP.PartyId.Id),
                            From = sender,
                            To = CvtHelper.SetPartyList(recipient)
                        };

                        OrganizationService.Create(patientEmail);
                        Logger.WriteDebugMessage("Patient Email Created Successfully");
                    }
                }
            }
            else
                Logger.WriteToFile("No VMR information could be found");
        }

        internal string PatientSafetyChecks()
        {
            var safetyChecks = "Prior to your visit, ensure the place you will be in is private and safe, and have the following information available:";
            safetyChecks += "<ul><li>At what phone number should we contact you if the call drops?</li>";
            safetyChecks += "<li>What is your local 10 digit phone number for law enforcement in your community?</li>";
            safetyChecks += "<li>What is the name, phone number, and relationship of the person we should contact in the case of an emergency?</li>";
            safetyChecks += "<li>What is the address of your location during this visit?</li></ul>";
            return safetyChecks;
        }
        #endregion

        #region H/M Groups
        public void NotifyProviderOfPatientChange()
        {
            var action = string.Empty;
            if (Email.Subject.Contains("added") && Email.Subject.Contains("removed"))
                action = "Addition and Cancelation";
            else if (Email.Subject.Contains("added"))
                action = "Addition";
            else if (Email.Subject.Contains("removed"))
                action = "Cancelation";
            NotifyParticipantsOfAppointment(false, action);

            CvtHelper.UpdateSendEmail(Email, OrganizationService);
            Logger.WriteDebugMessage(string.Format("{0} email sent to providers/telepresenters"));
        }

        public void NotifyPatientOfAdditionOrRemovalFromGroup(bool isCancelled)
        {
            //TODO - work on bug with adding patients to existing appt - Null Reference error occuring here
            Logger.WriteDebugMessage("TESTING DEBUGGER");
            var bookedUsers = ServiceAppointment.Resources.Where(ap => ap.PartyId.LogicalName == SystemUser.EntityLogicalName).ToList<Entity>();
            var ec = new EntityCollection(bookedUsers);
            var tsaProResources = getPRGs("provider");

            var clinicians = GetRecipients(ec, tsaProResources);
            bool? isPatSpace = null;
            getPatientVirtualMeetingSpace(out isPatSpace);
            SendPatientEmail(clinicians, isCancelled);
        }
        #endregion

        #region SA Notification Helpers
        internal List<Entity> getPRGs(string location)
        {
            QueryByAttribute qa = new QueryByAttribute("cvt_" + location + "resourcegroup");
            qa.ColumnSet = new ColumnSet("cvt_tsaresourcetype", "cvt_relateduserid", "cvt_relatedresourcegroupid", "cvt_relatedresourceid");
            qa.AddAttributeValue("cvt_relatedtsaid", Tsa.Id);
            var results = OrganizationService.RetrieveMultiple(qa);
            return results.Entities.ToList();
        }

        /// <summary>
        /// looks to find a group resource record for the group and user listed
        /// </summary>
        /// <param name="user">user Id of the group resource</param>
        /// <param name="group">group id of the group resource</param>
        /// <returns>true if a record exists for the group passed in and the user passed it</returns>
        internal bool MatchResourceToGroup(Guid userId, Guid groupId)
        {
            using (var srv = new Xrm(OrganizationService))
                return srv.mcs_groupresourceSet.FirstOrDefault(gr => gr.mcs_relatedResourceGroupId.Id == groupId && gr.mcs_RelatedUserId.Id == userId) != null;
        }

        /// <summary>
        /// looks to find a group resource record for the group and resource listed
        /// </summary>
        /// <param name="resource">resource record of the group resource</param>
        /// <param name="group">resource record of the group resource</param>
        /// <returns>true if a record exists for the group passed in and the resource passed it</returns>

        internal bool MatchResourceToGroup(mcs_resource resource, mcs_resourcegroup group)
        {
            using (var srv = new Xrm(OrganizationService))
                return srv.mcs_groupresourceSet.FirstOrDefault(gr => gr.mcs_relatedResourceGroupId.Id == group.Id && gr.mcs_RelatedResourceId.Id == resource.Id) != null;
        }

        internal List<SystemUser> GetRecipients(List<ActivityParty> users, List<Entity> prgs)
        {
            var userEc = new EntityCollection();
            foreach(var user in users)
            {
                userEc.Entities.Add(user);
            }
            return GetRecipients(userEc, prgs);
        }

        /// <summary>
        /// returns the list of users who are being booked as either single resources or in part of a group
        /// </summary>
        /// <param name="users"></param>
        /// <param name="prgs"></param>
        /// <returns></returns>
        internal List<SystemUser> GetRecipients(EntityCollection users, List<Entity> prgs)
        {
            List<SystemUser> recipients = new List<SystemUser>();
            var singles = prgs.Where(p => p.Attributes.Contains("cvt_tsaresourcetype") && ((OptionSetValue)p.Attributes["cvt_tsaresourcetype"]).Value == (int)cvt_tsaresourcetype.SingleProvider).ToList();
            var groups = prgs.Where(p => p.Attributes.Contains("cvt_tsaresourcetype") && ((OptionSetValue)p.Attributes["cvt_tsaresourcetype"]).Value == (int)cvt_tsaresourcetype.ResourceGroup).ToList();

            foreach (var singlePRG in singles)
            {
                if (singlePRG.Attributes.Contains("cvt_relateduserid") && singlePRG.Attributes["cvt_relateduserid"] != null)
                {
                    SystemUser singleUser = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName,
                        ((EntityReference)singlePRG.Attributes["cvt_relateduserid"]).Id, new ColumnSet(true));
                    if (singleUser != null && singleUser.Id != Guid.Empty)
                    {
                        foreach (ActivityParty u in users.Entities)
                        {
                            if (singleUser.Id == u.PartyId.Id)
                            {
                                recipients.Add(singleUser);
                                break;
                            }
                        }
                    }
                }
            }
            foreach (var groupPRG in groups)
            {
                if (groupPRG.Attributes.Contains("cvt_relatedresourcegroupid") && groupPRG.Attributes["cvt_relatedresourcegroupid"] != null)
                {
                    mcs_resourcegroup group = (mcs_resourcegroup)OrganizationService.Retrieve(
                            mcs_resourcegroup.EntityLogicalName, ((EntityReference)groupPRG.Attributes["cvt_relatedresourcegroupid"]).Id, new ColumnSet(true));
                    //if group type value is any of the "user-type" groups (provider or all required or telepresenter)
                    if ((group.mcs_Type.Value == (int)mcs_resourcetype.Provider) || (group.mcs_Type.Value == (int)mcs_resourcetype.AllRequired) ||
                        (group.mcs_Type.Value == (int)mcs_resourcetype.TelepresenterImager))
                    {
                        //if the user selected is in the resource group, return true and add the user to the entitycollection
                        foreach (ActivityParty u in users.Entities)
                        {
                            var user = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, u.PartyId.Id, new ColumnSet(true));
                            if (MatchResourceToGroup(user.Id, group.Id))
                                recipients.Add(user);
                        }
                    }
                }
            }
            return recipients;
        }

        /// <summary>
        /// filters down the list of all equipment on SA based on criteria provided
        /// </summary>
        /// <param name="equipment">collection of mcs_resources that correspond to the equipment in the resources field on the sa</param>
        /// <param name="tsa">tsa for the Service Activity</param>
        /// <param name="prgs">cvt_patientresourcegroup or cvt_providerresourcegroup for all resources on the tsa</param>
        /// <param name="equipType">type of mcs_resource (room, tech, vista clinic, etc.)</param>
        /// <returns>the list of mcs_resources based on the filters listed (pro or pat location and equipment type)</returns>
        internal List<mcs_resource> ClassifyResources(EntityCollection equipment, List<Entity> prgs, int? equipType)
        {
            List<mcs_resource> relevantResources = new List<mcs_resource>();

            var singles = prgs.Where(prg => ((OptionSetValue)(prg.Attributes["cvt_tsaresourcetype"])).Value == (int)cvt_tsaresourcetype.SingleResource).ToList();
            var groups = prgs.Where(prg => ((OptionSetValue)(prg.Attributes["cvt_tsaresourcetype"])).Value == (int)cvt_tsaresourcetype.ResourceGroup).ToList();

            foreach (Entity singlePRG in singles)
            {
                if (singlePRG.Attributes.Contains("cvt_relatedresourceid") && singlePRG.Attributes["cvt_relatedresourceid"] != null)
                {
                    foreach (mcs_resource r in equipment.Entities)
                    {
                        if (r.mcs_Type.Value != equipType && equipType != null)
                            continue;

                        mcs_resource resource = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName,
                                ((EntityReference)singlePRG.Attributes["cvt_relatedresourceid"]).Id, new ColumnSet(true));
                        if (resource != null && resource.Id == r.Id)
                            relevantResources.Add(r);
                    }
                }
            }
            foreach (Entity groupPRG in groups)
            {
                if (groupPRG.Attributes.Contains("cvt_relatedresourcegroupid") && groupPRG.Attributes["cvt_relatedresourcegroupid"] != null)
                {
                    mcs_resourcegroup group = (mcs_resourcegroup)OrganizationService.Retrieve(mcs_resourcegroup.EntityLogicalName,
                                ((EntityReference)groupPRG.Attributes["cvt_relatedresourcegroupid"]).Id, new ColumnSet(true));
                    if (group.mcs_Type.Value == (int)mcs_resourcetype.Room || group.mcs_Type.Value == (int)mcs_resourcetype.Technology || group.mcs_Type.Value == (int)mcs_resourcetype.AllRequired)
                    {
                        foreach (mcs_resource r in equipment.Entities)
                        {
                            if (r.mcs_Type.Value != equipType && equipType != null)
                                continue;

                            if (MatchResourceToGroup(r, group))
                            {
                                relevantResources.Add(r);
                                break;
                            }
                        }
                    }
                }
            }

            return relevantResources;
        }

        internal string getPatientVirtualMeetingSpace(out bool? patientSpace)
        {
            Logger.WriteDebugMessage("Getting Virtual Meeting Space");
            patientSpace = null;
            if (ServiceAppointment.mcs_PatientUrl != null && ServiceAppointment.mcs_providerurl != null)
            {
                PatientVirtualMeetingSpace = ServiceAppointment.mcs_PatientUrl;
                ProviderVirtualMeetingSpace = ServiceAppointment.mcs_providerurl;
                Logger.WriteDebugMessage("Virtual Meeting Space is from Service Activity Record: " + PatientVirtualMeetingSpace + ", " + ProviderVirtualMeetingSpace);
            }
            else
            {
                var patientAP = ServiceAppointment.Customers.FirstOrDefault();
                if (patientAP == null || patientAP.PartyId == null)
                    return string.Empty;
                Logger.WriteDebugMessage(ServiceAppointment.Customers.ToList().Count().ToString() + " patients " + patientAP.PartyId.Name.ToString());
                var patient = (Contact)OrganizationService.Retrieve(Contact.EntityLogicalName, patientAP.PartyId.Id, new ColumnSet(true));
                Logger.WriteDebugMessage("Contact: " + patient.FullName + " VMR: " + patient.cvt_PatientVirtualMeetingSpace + " and Tablet: " + patient.cvt_BLTablet);
                if (patient != null && patient.cvt_PatientVirtualMeetingSpace != null)
                {
                    patientSpace = true;
                    PatientVirtualMeetingSpace = patient.cvt_PatientVirtualMeetingSpace;
                    ProviderVirtualMeetingSpace = patient.cvt_ProviderVirtualMeetingSpace;
                }
                else if (patient != null && patient.cvt_BLTablet != null)
                {
                    patientSpace = false;
                    PatientVirtualMeetingSpace = patient.cvt_BLTablet;
                    isCvtTablet = true;
                }
                else if (VirtualMeetingSpaceComponent.Id != new Guid())
                    PatientVirtualMeetingSpace = VirtualMeetingSpaceComponent.cvt_webinterfaceurl;
                else
                    PatientVirtualMeetingSpace = "Please Contact Your TCT for Web Meeting Details";
                Logger.WriteDebugMessage(PatientVirtualMeetingSpace + ": Virtual Meeting Space is from Patient record = " + patientSpace.ToString());
            }
            return PatientVirtualMeetingSpace;
        }

        internal string getParamValue(string url, string key)
        {
            var result = string.Empty;
            var parameter = url.Split('&').LastOrDefault(s => s.ToLower().Contains(key));
            var parameterKeyValue = parameter != null ? parameter.Split('=') : null;
            if (parameterKeyValue != null && parameterKeyValue.Count() == 2)
                result = parameterKeyValue[1];
            return result;
        }

        internal string getComponents(mcs_resource technology, ServiceAppointment SA)
        {
            //Get all the components and include the CEVN Alias and IP Addresses for each.  Return the formatted string with a line for each Component
            //virtualMeetingSpace = null;
            string components = null;
            using (var context = new Xrm(OrganizationService))
            {
                var compList = context.cvt_componentSet.Where(c => c.cvt_relatedresourceid.Id == technology.Id);
                foreach (cvt_component c in compList)
                {
                    if (components == null)
                        components += "<ul>";
                    components += "<li>" + c.cvt_name;
                    switch (c.cvt_name)
                    {
                        case "Codec, Hardware":
                            if (c.cvt_cevnalias != null)
                                components += "; CEVN Alias: " + c.cvt_cevnalias;
                            break;
                        case "Telemedicine Encounter Management":
                        case "Telemed Encounter Management":
                        case "TEMS (Telemedicine Encounter Management Software)":
                        case "TEMS (Telemed Encounter Management Software)":
                            if (c.cvt_ipaddress != null)
                                components += "; IP Address: " + CvtHelper.buildHTMLUrl(c.cvt_ipaddress);
                            break;
                        case "CVT Patient Tablet":
                            if (c.cvt_serialnumber != null)
                                components += "; Serial Number: " + c.cvt_serialnumber;
                            break;
                        case "Virtual Meeting Space":
                            VirtualMeetingSpaceComponent = c;
                            break;
                        case "Digital Stethoscope Peripheral":
                            stethIP = c.cvt_ipaddress;
                            break;
                    }
                    //Send URL
                    var url = "";
                    var contact = getDummyContact();
                    var secondaryEntities = new List<Entity>();
                    secondaryEntities.Add(SA);
                    secondaryEntities.Add(contact);
                    if (UrlBuilder.TryGetUrl(OrganizationService, this.GetType().ToString(), c, secondaryEntities, out url))
                        components += "; <a href=" + url + ">" + url + "</a>";
                    components += "</li>";
                }
                if (components != null)
                    components += "</ul>";
            }
            return components;
        }

        internal Contact getDummyContact()
        {
            Contact c = new Contact();
            using (var srv = new Xrm(OrganizationService))
            {
                c = srv.ContactSet.FirstOrDefault();
            }
            return c;
        }
        #endregion

    }
}
