﻿using MCSShared;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace VA.TMP.CRM
{
    public class EmailAutomationCreatePostStageRunner : PluginRunner
    {        
        public EmailAutomationCreatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider) { }
        //Declare global variables
        public int ClosedTimeSpan;

        #region Implementation
        /// <summary>
        /// Called by PluginRunner - Decide which email to send out (aka which branch of the plugin to run)
        /// </summary>
        public override void Execute()
        {

            var Name = PrimaryEntity.Attributes["cvt_name"].ToString();

            if (Name.Contains("System Automated"))
            {
                Logger.WriteDebugMessage("Starting PPE Review automatic emails");
                PPEReviewSummary();
                //Should we try to send all PPEFeedbacks through this process as well?
            }
        }

        #endregion

        #region Commonly Used Functions
        internal List<ActivityParty> RetrieveFacilityTeamMembers(Email email, Guid TeamId, out string unapprovedUsers)
        {
            Logger.WriteDebugMessage("Starting RetrieveFacilityTeamMembers function");
            unapprovedUsers = "";

            using (var srv = new Xrm(OrganizationService))
            {
                var teamMembers = srv.TeamMembershipSet.Where(t => t.TeamId == TeamId).ToList();
                var recipientList = new List<ActivityParty>();

                foreach (var member in teamMembers)
                {
                    var user = srv.SystemUserSet.FirstOrDefault(u => u.Id == member.SystemUserId);

                    if (user == null)
                    {
                        Logger.WriteDebugMessage("Team member is not a user.");
                        break;

                    }
                    else
                    {
                        Logger.WriteDebugMessage("Checking for the User's Email Address.");
                        if ((!String.IsNullOrEmpty(user.InternalEMailAddress)))
                        {
                            var party = new ActivityParty()
                            {
                                ActivityId = new EntityReference(email.LogicalName, email.Id),
                                PartyId = new EntityReference(SystemUser.EntityLogicalName, user.Id)
                            };
                            recipientList.Add(party);
                        }
                        else
                        {
                            if (unapprovedUsers != "")
                                unapprovedUsers += "; ";

                            unapprovedUsers += user.FullName;
                        }
                    }
                }
                return recipientList;
            }
        }
        #endregion

        public int GetBusinessClosuresInPastXDays(int timespan)
        {
            var numberOfBusinessClosures = 0;
            using (var srv = new Xrm(OrganizationService))
            {
                var calendar = srv.CalendarSet.FirstOrDefault(c => c.Name == "Business Closure Calendar");
                if (calendar == null)
                    return 0;
                srv.LoadProperty(calendar, "CalendarRules");
                if (calendar.CalendarRules == null || calendar.CalendarRules.ToList().Count() == 0)
                    return 0;
                var holidays = calendar.CalendarRules.Select(r => r.StartTime).Distinct().Where(startTime =>
                    { return startTime.Value.AddDays(timespan) > DateTime.Now; }
                    ).ToList();
                numberOfBusinessClosures = holidays.Count;
            }
            return numberOfBusinessClosures;
        }

        #region PPE Review/Feedback
        internal void PPEReviewSummary()
        {
            Logger.setMethod = "FindOpenPPEReviews";
            Logger.WriteDebugMessage("Starting");
            string error = "";
            int successCount = 0;

            ClosedTimeSpan = DateTime.Today.DayOfWeek == DayOfWeek.Monday ? 3 : 1;
            ClosedTimeSpan += GetBusinessClosuresInPastXDays(ClosedTimeSpan);
            using (var srv = new Xrm(OrganizationService))
            {
                //Contains facilities with both open or recently closed PPE Reviews
                var activePpeFacilities = (from facility in srv.mcs_facilitySet
                                            join review in srv.cvt_ppereviewSet on facility.Id equals review.cvt_facility.Id
                                            join feedback in srv.cvt_ppefeedbackSet on review.Id equals feedback.cvt_ppereview.Id
                                            where review.cvt_outstandingppefeedbacks.Value > 0
                                            || (review.cvt_outstandingppefeedbacks == 0
                                                && review.ModifiedOn.Value.Date.AddDays(ClosedTimeSpan) > DateTime.Today
                                                && feedback.cvt_anythingtoreport.Value == (int)cvt_ppefeedbackcvt_anythingtoreport.Yes)
                                            select new
                                            {
                                                review.cvt_facility,
                                                review.cvt_specialty
                                            }).Distinct().ToList();
                Logger.WriteDebugMessage("Checking for Facilities with Open PPEReviews.");
                if (activePpeFacilities == null || activePpeFacilities.Count == 0)
                {
                    UpdateEmailAutomation(PrimaryEntity.Id, "No Open or Recently Closed PPE Reviews. No emails sent.", "");
                    return;
                }
                Logger.WriteDebugMessage("Found Facility/Specialty Combinations: " + activePpeFacilities.Count);
                foreach (var item in activePpeFacilities)
                {
                    var team = srv.TeamSet.FirstOrDefault(t => t.cvt_Type.Value == (int)Teamcvt_Type.ServiceChief && t.cvt_ServiceType.Id == item.cvt_specialty.Id && t.cvt_Facility.Id == item.cvt_facility.Id);

                    if (team == null)
                    {
                        error += String.Format("Could Not Find {0} Service Chief Team for {1}.", item.cvt_specialty.Name, item.cvt_facility.Name);
                        break;
                    }
                    Logger.WriteDebugMessage("Found SC Team at the Facility.");
                    
                    //Create the emails
                    Email newEmail = new Email()
                    {
                        RegardingObjectId = new EntityReference(cvt_emailautomation.EntityLogicalName, PrimaryEntity.Id),
                        From = CvtHelper.GetWorkflowOwner("Privileging: PPE Submitted", OrganizationService),
                        Subject = "Daily PPE feedback tracking summary"
                    };
                    Guid newEmailID = OrganizationService.Create(newEmail);
                    Logger.WriteDebugMessage("Created the email object.");

                    Email newPPEReviewSummary = new Email()
                    {
                        Id = newEmailID,
                        To = null
                    };
                    var unapprovedUsers = "";
                    newPPEReviewSummary.To = RetrieveFacilityTeamMembers(newPPEReviewSummary, team.Id, out unapprovedUsers);

                    
                    if (unapprovedUsers != "")
                        error += String.Format("No Email addresses for the following users {0} on the team: {1}", unapprovedUsers, team.Name);
                    else
                        Logger.WriteDebugMessage("All Users listed on the team have approved emails.");

                    Logger.WriteDebugMessage("Count for " + item.cvt_facility.Name + ", TO Count: " + newPPEReviewSummary.To.Count().ToString());
                    if (newPPEReviewSummary.To.Count() == 0)
                    {
                        error += String.Format("No Team members for {0}", team.Name);
                        break;
                    }
                   
                    //Edit the E-mail body with the summary grid
                    var etc = CvtHelper.GetEntityTypeCode(OrganizationService, cvt_ppereview.EntityLogicalName);
                    string servernameAndOrgname = CvtHelper.getServerURL(OrganizationService);
                    string url = servernameAndOrgname + "/userDefined/edit.aspx?etc=" + etc + "&id=";

                    Logger.WriteDebugMessage("Building the email body.");
                    var openPPETable = BuildOpenPpeReviewTable(item.cvt_facility, item.cvt_specialty, url, srv);
                    var recentlyClosedPPETable = BuildRecentlyClosedPPEReviewTable(item.cvt_facility, item.cvt_specialty, url, srv);

                    var customMessage = !string.IsNullOrEmpty(openPPETable) ? "<b>Active PPE Reviews</b><br/>" + openPPETable + "<br/><br/>" : "No Active PPE Reviews";
                    customMessage += !string.IsNullOrEmpty(recentlyClosedPPETable) ? "<b>Recently Closed PPE Reviews with Something to Report</b><br/>" + recentlyClosedPPETable : "No Recently Closed PPE Reviews with Anything to Report";
                    customMessage += "<br/><br/>This is an automated notification from the Telehealth Management Platform.";
                    newPPEReviewSummary.Description = customMessage;
                    
                    //Only Send email if there are either recently closed or currently open PPE Reviews
                    if (!string.IsNullOrEmpty(openPPETable + recentlyClosedPPETable))
                    {
                        CvtHelper.UpdateSendEmail(newPPEReviewSummary, OrganizationService);
                        successCount++;
                    }
                }

                string summary = "Successfully sent out " + successCount + " PPE Review summaries.";
                //Update the Email Automation record
                UpdateEmailAutomation(PrimaryEntity.Id, summary, error);

            }
        }

        internal string BuildOpenPpeReviewTable(EntityReference facility, EntityReference specialty, string url, Xrm srv)
        {
            var tableString = string.Empty;
            Logger.WriteDebugMessage("Building the table string for open PPE Reviews.");
            tableString = "<table style = 'width:100%'><tr>";
            //Table Headings
            tableString += HtmlTableHeaderItem("Provider Name");
            tableString += HtmlTableHeaderItem("# of Received Feedback");
            tableString += HtmlTableHeaderItem("# of Requested Feedback");
            tableString += HtmlTableHeaderItem("Initiation Date");
            tableString += HtmlTableHeaderItem("Due Date");
            tableString += HtmlTableHeaderItem("Requests Escalated");
            tableString += HtmlTableHeaderItem("PPE Review record");

            tableString += "</tr >";

            var openReviews = srv.cvt_ppereviewSet.Where(r =>
                    r.cvt_facility.Id == facility.Id &&
                    r.cvt_specialty.Id == specialty.Id &&
                    r.cvt_outstandingppefeedbacks != 0 &&
                    r.cvt_outstandingppefeedbacks != null &&
                    r.cvt_submittedppefeedbacks != null).ToList();
            foreach (cvt_ppereview open in openReviews)
            {
                var escalated = (open.cvt_escalated.Value == true) ? "Yes" : "No";
                tableString += "<tr>";
                tableString += createHtmlTableDataElement(open.cvt_provider.Name);
                tableString += createHtmlTableDataElement(open.cvt_submittedppefeedbacks.ToString());
                tableString += createHtmlTableDataElement(open.cvt_requestedppefeedbacks.ToString());
                tableString += createHtmlTableDataElement(open.cvt_initiateddate.Value.ToString("MM/dd/yyyy"));
                tableString += createHtmlTableDataElement(open.cvt_duedate.Value.ToString("MM/dd/yyyy"));
                tableString += createHtmlTableDataElement(escalated);
                tableString += createHtmlTableDataElement("<a href=\"" + url + open.Id + "\">View Record</a>"); //clickable URL
                tableString += "</tr>";
            }

            tableString += "</table >";
            if (openReviews.Count == 0)
            {
                Logger.WriteDebugMessage("No Open PPE Reviews Found, skipping 'Active PPE Reviews' table");
                return string.Empty;
            }
            else
                return tableString;
        }

        internal string BuildRecentlyClosedPPEReviewTable(EntityReference facility, EntityReference specialty, string url, Xrm srv)
        {
            Logger.WriteDebugMessage("Building Closed PPE Review Table");
            var recentlyClosedReviews = (from review in srv.cvt_ppereviewSet
                                         join feedback in srv.cvt_ppefeedbackSet on review.Id equals feedback.cvt_ppereview.Id
                                         where 
                                             review.cvt_facility.Id == facility.Id &&
                                             review.cvt_specialty.Id == specialty.Id &&
                                             review.cvt_submittedppefeedbacks != null &&
                                             review.cvt_outstandingppefeedbacks != null &&
                                             review.cvt_outstandingppefeedbacks == 0 &&
                                             review.ModifiedOn.Value.AddDays(ClosedTimeSpan) > DateTime.Today &&
                                             feedback.cvt_anythingtoreport.Value == (int)cvt_ppefeedbackcvt_anythingtoreport.Yes
                                         select review).Distinct().ToList();
            
            if (recentlyClosedReviews == null || recentlyClosedReviews.Count == 0)
            {
                Logger.WriteDebugMessage("No Recently Closed PPE Reviews Reporting Feedback Found, skipping 'Recently Closed PPE Reviews' table");
                return string.Empty;
            }

            var tableString = "<table style = 'width:100%'><tr>";
            //Table Headings
            tableString += HtmlTableHeaderItem("Provider Name");
            tableString += HtmlTableHeaderItem("# of Received Feedback");
            tableString += HtmlTableHeaderItem("Initiation Date");
            tableString += HtmlTableHeaderItem("Due Date");
            tableString += HtmlTableHeaderItem("Requests Escalated");
            tableString += HtmlTableHeaderItem("Closed Date");
            tableString += HtmlTableHeaderItem("PPE Review record");
            tableString += "</tr >";
            foreach(var review in recentlyClosedReviews)
            {
                var escalated = (review.cvt_escalated.Value == true) ? "Yes" : "No";
                tableString += "<tr>";
                tableString += createHtmlTableDataElement(review.cvt_provider.Name);
                tableString += createHtmlTableDataElement(review.cvt_submittedppefeedbacks.ToString());
                tableString += createHtmlTableDataElement(review.cvt_initiateddate.Value.ToString("MM/dd/yyyy"));
                tableString += createHtmlTableDataElement(review.cvt_duedate.Value.ToString("MM/dd/yyyy"));
                tableString += createHtmlTableDataElement(escalated);
                tableString += createHtmlTableDataElement(review.ModifiedOn.Value.ToString("MM/dd/yyyy"));
                tableString += createHtmlTableDataElement("<a href=\"" + url + review.Id + "\">View Record</a>");
                tableString += "</tr>";
            }
            tableString += "</table>";
            return tableString;
        }

        internal string HtmlTableHeaderItem(string element)
        {
            return "<th>" + element + "</th>";
        }

        /// <summary>
        /// Using Existing Formatting from V1
        /// </summary>
        /// <param name="element"></param>
        /// <returns></returns>
        internal string createHtmlTableDataElement(string element)
        {
            //Using existing formatting from V1 - commented out what would normally be expected for Table Data Elements
            return HtmlTableHeaderItem(element);
            //return "<td>" + element + "</td>";
        }

        internal void UpdateEmailAutomation(Guid recordId, string summary, string error)
        {
            Logger.WriteDebugMessage("About to update the Email Automation record.");

            var now = DateTime.Today;

            cvt_emailautomation updateRecord = new cvt_emailautomation()
            {
                Id = recordId,
                cvt_summary = summary,
                cvt_errors = error,
                cvt_name = "System Automated Emails " + now.ToString("yyyy/MM/dd")
            };
            OrganizationService.Update(updateRecord);
            Logger.WriteDebugMessage("Updated Email Automation record.");
        }

        #endregion

        #region Implementing additional interface methods
        public override string McsSettingsDebugField
        {
            get { return "cvt_ppereview"; }
        }
        #endregion
    }
}
