﻿using MCSUtilities2011;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace MCSShared
{
    public static partial class CvtHelper
    {

        #region Email Related Functions
        //Overloaded method: creates a list from an Activty Party to populate a From, To, or CC field
        public static List<ActivityParty> SetPartyList(ActivityParty user)
        {
            List<ActivityParty> userList = new List<ActivityParty>();
            //user if exists
            if (user != null)
                userList.Add(user);
            return userList;
        }

        //Overloaded method: return empty list if userRef is null, otherwise get Activity party and call first SetEmailProperties method
        public static List<ActivityParty> SetPartyList(EntityReference userRef)
        {
            return userRef == null ? new List<ActivityParty>() :
                CvtHelper.SetPartyList(new ActivityParty() { PartyId = new EntityReference(userRef.LogicalName, userRef.Id) });
        }

        //Overloaded method: creates a list from an EntityCollection to populate a From, To, or CC field
        public static List<ActivityParty> SetPartyList(EntityCollection userCollection)
        {
            List<ActivityParty> userList = new List<ActivityParty>();

            foreach (var record in userCollection.Entities.Select(e => e).Distinct())
            {
                var activityParty = new ActivityParty() { PartyId = new EntityReference(record.LogicalName, record.Id) };
                userList.Add(activityParty);
            }
            return userList;
        }

        public static List<ActivityParty> SetPartyList(List<SystemUser> userList)
        {
            List<ActivityParty> users = new List<ActivityParty>();

            foreach (SystemUser u in userList.Select(e => e).Distinct())
            {
                var activityParty = new ActivityParty() { PartyId = new EntityReference(u.LogicalName, u.Id) };
                users.Add(activityParty);
            }
            return users;
        }

        //This method takes the email passed in and sends it
        public static void SendEmail(Email email, IOrganizationService OrganizationService)
        {
            SendEmailRequest requestObj = new SendEmailRequest()
            {
                EmailId = (Guid)email.ActivityId,
                IssueSend = true,
                TrackingToken = ""
            };
            SendEmailResponse response = (SendEmailResponse)OrganizationService.Execute(requestObj);
        }

        //Saves and Sends the email
        public static void UpdateSendEmail(Email email, IOrganizationService OrganizationService)
        {
            OrganizationService.Update(email);
            SendEmail(email, OrganizationService);
        }

        public static string EmailFooter()
        {
            return "<br/><br/>Please Do Not Reply to this message.  It comes from an unmonitored mailbox.  For any questions or concerns, please contact your VA Facility or VA Clinical Team.";
        }


        ///// <summary>
        ///// Overload for basic generateEmailBody - displays the url as the message for "Click Here"
        ///// </summary>
        ///// <param name="record">ID of the email</param>
        ///// <param name="entityStringName">string name of the entity - to retrieve object type code</param>
        ///// <param name="customMessage">The message</param>
        ///// <returns></returns>
        //internal static string GenerateEmailBody(Guid record, string entityStringName, string customMessage, IOrganizationService OrganizationService)
        //{
        //    return GenerateEmailBody(record, entityStringName, customMessage, OrganizationService);
        //}

        /// <summary>
        /// Standard "boilerplate" E-Mail body
        /// </summary>
        /// <param name="record">ID of the email record</param>
        /// <param name="entityStringName">The string name of the object type code</param>
        /// <param name="customMessage">The custom string that goes into the email body</param>
        /// <param name="clickHereMessage">The message that is used as the display for the hyperlink</param>
        /// <returns>the body of the email</returns>
        internal static string GenerateEmailBody(Guid record, string entityStringName, string customMessage, IOrganizationService OrganizationService, string clickHereMessage = "")
        {
            string body;
            body = GetRecordLink(new EntityReference(entityStringName, record), OrganizationService, clickHereMessage);
            body += "<br/><br/>" + customMessage;
            body += EmailFooter();

            return body;
        }

        internal static string GetRecordLink(Entity record, IOrganizationService orgService, string clickHereText = "")
        {
            return GetRecordLink(new EntityReference(record.LogicalName, record.Id), orgService, clickHereText);
        }

        internal static string GetRecordLink(EntityReference record, IOrganizationService OrganizationService, string clickHereText = "")
        {
            var etc = CvtHelper.GetEntityTypeCode(OrganizationService, record.LogicalName);
            string servernameAndOrgname = CvtHelper.getServerURL(OrganizationService);
            string url = servernameAndOrgname + "/userDefined/edit.aspx?etc=" + etc + "&id=" + record.Id;
            return String.Format("<a href=\"{0}\">{1}</a>", url, !string.IsNullOrEmpty(clickHereText) ? clickHereText : url);
        }

        internal static string ProviderSafetyChecks()
        {
            var safetyChecks = "During your initial assessment be sure to verify the following: ";
            safetyChecks += "<ul><li>Do you have any concerns about suicide?</li>";
            safetyChecks += "<li>The Patient verbally consents to the telehealth visit?</li>";
            safetyChecks += "<li>If the line drops, what number can I call you at?</li>";
            safetyChecks += "<li>What is the name, phone number, and relationship of the person we should contact in the case of an emergency?</li>";
            safetyChecks += "<li>What is your local 10 digit phone number for law enforcement in your community?</li>";
            safetyChecks += "<li>What is the address of your location during this visit?</li></ul>";
            safetyChecks += "<li>Are you in a safe and private place?</li></ul>";
            return safetyChecks;
        }


        /// <summary>
        /// This method creates the .ics attachment and appends it to the email
        /// </summary>
        /// <param name="email">This is the email that the attachment is attaching to</param>
        /// <param name="sa">The service appointment which </param>
        /// <param name="statusCode">The status of the email - which sets the status of the attachment as well as the subject of the email</param>
        internal static void CreateCalendarAppointmentAttachment(Email email, ServiceAppointment sa, int statusCode, string stethIP, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            bool group = false;
            if (sa.mcs_groupappointment != null)
            {
                group = sa.mcs_groupappointment.Value;
            }
            Logger.WriteTxnTimingMessage("Begin Creating Calendar Appointment");
            string schLocation = "See Description";
            string schSubject = group == true ? "Telehealth Visit-Group Appointment: Do Not Reply" :
                "Telehealth Visit-Single Appointment: Do Not Reply";
            string schDescription = email.Description;
            System.DateTime schBeginDate = (System.DateTime)sa.ScheduledStart;
            System.DateTime schEndDate = (System.DateTime)sa.ScheduledEnd;
            string sequence = "";
            string status = "CONFIRMED";
            string method = "";
            //if the appointment is canceled, send a cancellation notice based on the UID of the previous entry sent
            if (statusCode == (int)serviceappointment_statuscode.PatientCanceled || statusCode == (int)serviceappointment_statuscode.ClinicCanceled)
            {
                method = "METHOD:CANCEL\n";
                sequence = "SEQUENCE:1\n";
                status = "CANCELLED";
                schSubject = "Canceled: Telehealth Visit: Do Not Reply";
            }

            //attach a ClearSteth CVL file if a steth is in the components
            string cvlAtttachment = string.IsNullOrEmpty(stethIP) ? "" :
                "ATTACH;ENCODING=BASE64;VALUE=BINARY;X-FILENAME=invitation.cvl:" + Convert.ToBase64String(new ASCIIEncoding().GetBytes("<?xml version=\"1.0\" encoding=\"UTF-8\"?><CVL><IP>" + stethIP + "</IP ><Port>9005</Port><ConferenceId>12345</ConferenceId></CVL>")) + "\n";

            string att = "BEGIN:VCALENDAR\n" +
                              "PRODID:-//VA//Veterans Affairs//EN\n" +
                              method +
                              "BEGIN:VEVENT\n" +
                              cvlAtttachment +
                              "UID:" + sa.Id + "\n" + sequence +
                              "DTSTART:" + schBeginDate.ToUniversalTime().ToString("yyyyMMdd\\THHmmss\\Z") + "\n" +
                              "DTEND:" + schEndDate.ToUniversalTime().ToString("yyyyMMdd\\THHmmss\\Z") + "\n" +
                              "LOCATION:" + schLocation +
                              //Use Description tag for email clients that cant handle x-alt-desc tag with HTML
                              "\nDESCRIPTION;ENCODING=QUOTED-PRINTABLE:" + schDescription.Replace("<br/>", "").Replace("<b>", "").Replace("</b>", "").Replace("<u>", "").Replace("</u>", "") +
                              "\nSUMMARY:" + schSubject + "\nPRIORITY:3\n" +
                              "STATUS:" + status + "\n" +
                              //Include alternate description if the calendar client can handle html x-alt-desc tag
                              "X-ALT-DESC;FMTTYPE=text/html:<html>" + schDescription.Replace("\n", "<br/>") + "</html>" + "\n" +
                          "END:VEVENT\n" + "END:VCALENDAR\n";

            ActivityMimeAttachment calendarAttachment = new ActivityMimeAttachment()
            {
                ObjectId = new EntityReference(Email.EntityLogicalName, email.Id),
                ObjectTypeCode = Email.EntityLogicalName,
                Subject = string.Format("Telehealth Visit"),
                Body = Convert.ToBase64String(
                        new ASCIIEncoding().GetBytes(att)),
                FileName = string.Format(CultureInfo.CurrentCulture, "Telehealth-Appointment.ics")
            };
            OrganizationService.Create(calendarAttachment);
            Logger.WriteTxnTimingMessage("Finished Creating Calendar Appointment");
            return;
        }

        #endregion


        #region Commonly Used Functions
        //TODO TO-DO: Consolidate with Email Automation function, should we add check for User's Email existing before adding to the AP List?
        /// <summary>
        /// Gets the list of team members and returns them as an activity party list to be added to whatever email we are using
        /// </summary>
        /// <param name="email"></param>
        /// <param name="TeamId"></param>
        /// <param name="originalParty"></param>
        /// <returns></returns>
        internal static List<ActivityParty> RetrieveFacilityTeamMembers(Email email, Guid TeamId, IEnumerable<ActivityParty> originalParty, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.WriteDebugMessage("starting RetrieveFacilityTeamMembers");
            using (var srv = new Xrm(OrganizationService))
            {
                var teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == TeamId).ToList());
                var recipientList = new List<ActivityParty>();

                if (originalParty != null)
                    recipientList.AddRange(originalParty);

                Logger.WriteDebugMessage("About to add members of team.");
                foreach (var member in teamMembers)
                {
                    var party = new ActivityParty()
                    {
                        ActivityId = new EntityReference(email.LogicalName, email.Id),
                        PartyId = new EntityReference(SystemUser.EntityLogicalName, member.SystemUserId.Value)
                    };
                    recipientList.Add(party);
                }
                Logger.WriteDebugMessage("Finished adding members of team.");
                return recipientList;
            }
        }
        #endregion
    }
}