﻿using MCSUtilities2011;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace MCSShared
{
    public static partial class CvtHelper
    {
        #region Team
        /// <summary>
        /// Passing in either a SystemUser or a Team, associate security roles
        /// </summary>
        /// <param name="thisId">Accepts either Team or SystemUser Guids</param>
        /// <param name="EntityLogicalName"></param>
        /// <param name="OrganizationService"></param>
        /// <param name="Logger"></param>
        internal static void UpdateSecurityRoles(Guid thisId, string EntityLogicalName, IOrganizationService OrganizationService, MCSLogger Logger, out int addRoles, out int removeRoles)
        {
            using (var srv = new Xrm(OrganizationService))
            {
                //Vars to Track Counts
                bool addedRoles = false;
                bool removedRoles = false;
                addRoles = 0;
                removeRoles = 0;

                IQueryable currentRoleResults = null;
                string rolesToAdd = "";
                int totalRoles = 0;
                Guid businessUnitId = Guid.Empty;
                string thisRecordName = "";
                string displayMessage = "";
                EntityReferenceCollection RolesToAssociate = new EntityReferenceCollection();
                EntityReferenceCollection RolesToDisassociate = new EntityReferenceCollection();

                if (EntityLogicalName == Team.EntityLogicalName)
                {
                    var thisTeam = srv.TeamSet.First(t => t.Id == thisId);
                    if (thisTeam.BusinessUnitId == null)
                    {
                        Logger.WriteDebugMessage("No BU set on Team, so no roles were added.");
                        //Maybe check For Facility value and align to that BU.
                        return; //no BU, so exit                      
                    }
                    businessUnitId = thisTeam.BusinessUnitId.Id;
                    thisRecordName = thisTeam.Name;
                    rolesToAdd = "TMP User";
                    if (thisTeam.cvt_Type != null)
                    {
                        switch (thisTeam.cvt_Type.Value)
                        {
                            case (int)Teamcvt_Type.FTC:
                                //User + Approver + FTC
                                rolesToAdd += "|TMP TSA Approver|TMP TSA Manager";
                                break;
                            case (int)Teamcvt_Type.ServiceChief:
                                rolesToAdd += "|TMP TSA Approver|TMP PPE Feedback";
                                break;
                            case (int)Teamcvt_Type.ChiefofStaff:
                                rolesToAdd += "|TMP TSA Approver";
                                break;
                            case (int)Teamcvt_Type.CredentialingandPrivileging:
                                rolesToAdd += "|TMP TSA Approver|TMP Privileging";
                                break;
                            //case "917290004": //TSA Notification
                            //    break;
                            case (int)Teamcvt_Type.Scheduler:
                                //User + Scheduler
                                rolesToAdd += "|TMP Scheduler";
                                break;
                            case (int)Teamcvt_Type.DataAdministrator:
                                rolesToAdd += "|TMP Field Application Administrator";
                                break;
                            case (int)Teamcvt_Type.TCT:
                                rolesToAdd += "|TMP TSA Manager";
                                break;
                        }
                    }
                    else
                    {
                        if (thisTeam.Name.Contains("("))
                            rolesToAdd += "|TMP Site Team";
                        else
                            rolesToAdd += "|TMP BU Team";
                    }

                    displayMessage = "Missing Default Role(s) systematically associated (based on Type) to Team: ";
                    currentRoleResults = srv.TeamRolesSet.Where(tr => tr.TeamId == thisId);
                }
                else if (EntityLogicalName == SystemUser.EntityLogicalName)
                {
                    var thisUser = srv.SystemUserSet.First(u => u.Id == thisId);
                    if (thisUser.cvt_SecurityRolesString == null)
                        return; //Future: Auto add Roles based on Type field on User Record (like Team)
                    rolesToAdd = thisUser.cvt_SecurityRolesString;
                    businessUnitId = thisUser.BusinessUnitId.Id;
                    thisRecordName = thisUser.FullName;
                    displayMessage = " Former Roles re-associated in new Business Unit for user: ";
                    currentRoleResults = srv.SystemUserRolesSet.Where(u => u.SystemUserId == thisId);
                }

                //Common Code
                if (rolesToAdd == "TMP User")
                {
                    //Check for Site Team, Facility Team, VISN Team.
                    //Could query for site or facility or visn for exact match of name and if so, add team depending on which result?
                    Logger.WriteDebugMessage(String.Format("No roles to assign for {0}: {1}", EntityLogicalName, thisRecordName));
                    return;
                }
                var SecRoles = rolesToAdd.Contains('|') ? rolesToAdd.Split('|') : new string[] { rolesToAdd }; //split or only one value
                totalRoles = SecRoles.Count();

                try
                {
                    //Get the Roles to Add
                    foreach (var defaultRole in SecRoles)
                    {
                        Boolean matchToIgnoreAdd = false;
                        //If no current roles, do not try to match
                        if (currentRoleResults != null)
                        {
                            foreach (Entity role in currentRoleResults)
                            {
                                if (matchToIgnoreAdd)
                                    break;

                                Guid roleGuid = Guid.Empty;
                                if (role != null && role.Attributes != null && role.Attributes["roleid"] != null)
                                    roleGuid = (Guid)(role.Attributes["roleid"]);
                                var roleRecord = srv.RoleSet.FirstOrDefault(r => r.Id == roleGuid);
                                if ((roleRecord != null) && (roleRecord.Name != null))
                                    matchToIgnoreAdd = (defaultRole == roleRecord.Name) ? true : false;
                            }
                        }
                        if (!matchToIgnoreAdd) //No Match, so add
                        {
                            var newRole = srv.RoleSet.FirstOrDefault(sr => sr.BusinessUnitId.Id == businessUnitId && sr.Name == defaultRole.ToString());
                            RolesToAssociate.Add(new EntityReference(Role.EntityLogicalName, newRole.Id));
                            addRoles++;
                            addedRoles = true;
                        }
                    }

                    //If no current roles, do not try to match
                    if (currentRoleResults != null)
                    {
                        //Get the Roles to Remove
                        foreach (Entity role in currentRoleResults)
                        {
                            Guid roleGuid = Guid.Empty;
                            if (role != null && role.Attributes != null && role.Attributes["roleid"] != null)
                                roleGuid = (Guid)(role.Attributes["roleid"]);
                            var roleRecord = srv.RoleSet.FirstOrDefault(r => r.Id == roleGuid);
                            Boolean matchToRemove = SecRoles.FirstOrDefault(r => r == roleRecord.Name) == null;
                            if (matchToRemove)
                            {
                                RolesToDisassociate.Add(new EntityReference(Role.EntityLogicalName, roleGuid));
                                removeRoles++;
                                removedRoles = true;
                            }
                        }
                    }

                    if (addedRoles)
                    {
                        OrganizationService.Associate(EntityLogicalName, thisId, new Relationship(EntityLogicalName.ToLower() + "roles_association"), RolesToAssociate);
                        Logger.WriteDebugMessage(String.Format("{0}/{1} {2}{3}", addRoles, totalRoles, displayMessage, thisRecordName));
                    }

                    if (removedRoles)
                    {
                        OrganizationService.Disassociate(EntityLogicalName, thisId, new Relationship(EntityLogicalName.ToLower() + "roles_association"), RolesToDisassociate);
                        Logger.WriteDebugMessage(String.Format("{0} role(s) removed from {1}", removeRoles, thisRecordName));
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Logger.WriteToFile(ex.Message);
                }
                catch (Exception ex)
                {
                    Logger.WriteToFile(ex.Message);
                }
            }
        }

        /// <summary>
        /// Update Site Team to the specified Business Unit and associate the TMP Site Team role from that new Business Unit
        /// </summary>
        /// <param name="teamId"></param>
        /// <param name="buId"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        internal static void UpdateSiteTeam(Guid teamId, Guid buId, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.setMethod = "CvtHelper.UpdateSiteTeam";
            try
            {
                var req = new SetParentTeamRequest()
                {
                    TeamId = teamId,
                    BusinessId = buId
                };

                OrganizationService.Execute(req);
                Logger.WriteDebugMessage("Site team record updated to new BU");

                //Need to reassign the TMP Site Team security role
                //Logger.WriteDebugMessage("About to query for appropriate security role.");
                var siteTeamrole = new Role();
                using (var srv = new Xrm(OrganizationService))
                {
                    siteTeamrole = srv.RoleSet.FirstOrDefault(r => r.Name == "TMP Site Team" && r.BusinessUnitId.Id == buId);
                }
                if (siteTeamrole != null)
                {
                    // Associate the user with the role.
                    OrganizationService.Associate(Team.EntityLogicalName, teamId, new Relationship("teamroles_association"),
                        new EntityReferenceCollection() { new EntityReference(Role.EntityLogicalName, siteTeamrole.Id) });
                    Logger.WriteDebugMessage("Associated TMP Site Team role to team.");
                }
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Exception with UpdateSiteTeam for " + teamId + ". Message: " + ex.Message);
            }
        }

        /// <summary>
        /// Update the Primary Team field to match the VISN on the user record.
        /// </summary>
        /// <param name="thisId"></param>
        internal static void UpdatePrimaryTeam(Guid thisId, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.setMethod = "UpdatePrimaryTeam";
            Logger.WriteDebugMessage("Starting UpdatePrimaryTeam");
            using (var srv = new Xrm(OrganizationService))
            {
                var thisUser = srv.SystemUserSet.FirstOrDefault(su => su.Id == thisId);
                Logger.WriteDebugMessage("Retrieved User Set.");
                if (thisUser.cvt_PrimaryTeam == null || (thisUser.cvt_PrimaryTeam.Name != thisUser.BusinessUnitId.Name))
                {
                    Logger.WriteDebugMessage("Mismatched Primary Team and VISN name, update team.");
                    //Need to update the team to match the VISN
                    var VISNTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisUser.BusinessUnitId.Name);
                    if (VISNTeam != null)
                    {
                        Logger.WriteDebugMessage("Found Team of user's current VISN. About to Update.");
                        SystemUser userUpdate = new SystemUser()
                        {
                            Id = thisUser.Id,
                            cvt_PrimaryTeam = new EntityReference() { LogicalName = Team.EntityLogicalName, Id = VISNTeam.Id, Name = VISNTeam.Name }
                        };
                        OrganizationService.Update(userUpdate);
                        Logger.WriteDebugMessage("Update successful.");
                    }
                }
            }
        }

        #endregion

        #region AssignOwner
        /// <summary>
        /// Assigns Ownership of the record to the correct team. 
        /// </summary>
        /// <param name="thisRecord"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns>Assigned - true/false</returns>
        internal static bool AssignOwner(Entity thisRecord, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            #region Setup
            Logger.setMethod = "AssignOwner";
            Logger.WriteDebugMessage("Starting AssignOwner");

            EntityReference assignOwner = new EntityReference()
            {
                LogicalName = Team.EntityLogicalName,
            };
            Entity findTeam = new Entity();
            #endregion

            switch (thisRecord.LogicalName)
            {
                #region TSS Site
                case mcs_site.EntityLogicalName:
                    var thisSite = (mcs_site)thisRecord;
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisSite.mcs_name);
                    break;
                #endregion
                #region TSS Resource
                case mcs_resource.EntityLogicalName:

                    var thisResource = (mcs_resource)thisRecord;
                    //Check that the TSS Resource's owner is the same name as the site
                    if (thisResource.mcs_RelatedSiteId == null || thisResource.OwnerId.Name == thisResource.mcs_RelatedSiteId.Name)
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisResource.mcs_RelatedSiteId.Name);

                    break;
                #endregion
                #region Component
                case cvt_component.EntityLogicalName:

                    var thisComponent = (cvt_component)thisRecord;
                    if (thisComponent.cvt_relatedresourceid == null)
                        return false;

                    mcs_resource parentResource = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName, thisComponent.cvt_relatedresourceid.Id, new ColumnSet(true));
                    if (parentResource.mcs_RelatedSiteId == null)
                        return false;

                    //Check that the TSS Resource's owner is the same name as the site, if so, pass along or query for correct one.
                    if (parentResource.OwnerId.Name == parentResource.mcs_RelatedSiteId.Name)
                        findTeam.Id = parentResource.OwnerId.Id;
                    else
                    {
                        //Get the correct team
                        using (var srv = new Xrm(OrganizationService))
                            findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == parentResource.mcs_RelatedSiteId.Name);
                        if (findTeam == null)
                            return false;

                        //Also Reassign TSS Resource
                        AssignRequest assignTSSResource = new AssignRequest()
                        {
                            Assignee = findTeam.ToEntityReference(),
                            Target = new EntityReference(mcs_resource.EntityLogicalName, parentResource.Id)
                        };

                        OrganizationService.Execute(assignTSSResource);
                        Logger.WriteDebugMessage("TSS Resource reassigned to Site Team.");
                    }
                    //No need to reassign, corrrect owner
                    if (thisComponent.OwnerId.Id == findTeam.Id)
                        return false;
                    break;
                #endregion
                #region TSS Resource Group
                case mcs_resourcegroup.EntityLogicalName:

                    var thisResourceGroup = (mcs_resourcegroup)thisRecord;
                    if (thisResourceGroup.mcs_relatedSiteId == null || thisResourceGroup.OwnerId.Name == thisResourceGroup.mcs_relatedSiteId.Name)
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisResourceGroup.mcs_relatedSiteId.Name);
                    break;
                #endregion
                #region Group Resource
                case mcs_groupresource.EntityLogicalName:

                    var thisGroupResource = (mcs_groupresource)thisRecord;
                    if (thisGroupResource.mcs_relatedSiteId == null || thisGroupResource.OwnerId.Name == thisGroupResource.mcs_relatedSiteId.Name)
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisGroupResource.mcs_relatedSiteId.Name);
                    break;
                #endregion
                #region MTSA
                case cvt_mastertsa.EntityLogicalName:

                    var thisMTSA = (cvt_mastertsa)thisRecord;
                    //Check the Provider Facility vs OwnerId
                    if ((thisMTSA.OwnerId.Name.Contains("FTC") && (thisMTSA.OwnerId.Name.Contains(thisMTSA.cvt_ProviderFacility.Name))))
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility == thisMTSA.cvt_ProviderFacility && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.FTC);

                    if (findTeam == null)
                    {
                        string name = (thisMTSA.cvt_ProviderFacility != null) ? thisMTSA.cvt_ProviderFacility.Name : "No Facility Listed";
                        Logger.WriteDebugMessage("FTC Team not found for Facility: " + name + " Creating team.");
                        Team newTeam = new Team()
                        {
                            cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, thisMTSA.cvt_ProviderFacility.Id),
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.FTC),
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisMTSA.OwningBusinessUnit.Id),
                            Name = "FTC Approval Group @ " + thisMTSA.cvt_ProviderFacility.Name
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }
                    else
                    {
                        //Check that the TSS Resource's owner is the same name as the site, if so, pass along or query for correct one.
                        if (thisMTSA.OwnerId != null && thisMTSA.OwnerId.Id == findTeam.Id)
                            return false;
                    }
                    break;
                #endregion
                #region TSA
                case mcs_services.EntityLogicalName:

                    var thisTSA = (mcs_services)thisRecord;

                    //Check the Provider Facility vs OwnerId
                    if ((thisTSA.OwnerId.Name.Contains("FTC") && (thisTSA.OwnerId.Name.Contains(thisTSA.cvt_ProviderFacility.Name))))
                        return false;

                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility == thisTSA.cvt_ProviderFacility && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.FTC);

                    if (findTeam == null)
                    {
                        string name = (thisTSA.cvt_ProviderFacility != null) ? thisTSA.cvt_ProviderFacility.Name : "No Facility Listed";
                        Logger.WriteDebugMessage("FTC Team not found for Facility: " + name + " Creating team.");
                        Team newTeam = new Team()
                        {
                            cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, thisTSA.cvt_ProviderFacility.Id),
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.FTC),
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisTSA.OwningBusinessUnit.Id),
                            Name = "FTC Approval Group @ " + thisTSA.cvt_ProviderFacility.Name
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }

                    if (thisTSA.OwnerId != null && thisTSA.OwnerId.Id == findTeam.Id)
                        return false;
                    break;
                #endregion
                #region Telehealth Privileging
                case cvt_tssprivileging.EntityLogicalName:

                    var thisPrivileging = (cvt_tssprivileging)thisRecord;

                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == thisPrivileging.cvt_PrivilegedAtId.Id && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.CredentialingandPrivileging);

                    //Can't find team or already corrrect owner
                    if (findTeam == null)
                    {
                        Logger.WriteToFile("C&P Team Not Found for Facility: " + thisPrivileging.cvt_PrivilegedAtId.Name);
                        return false;
                    }
                    else
                    {
                        if (thisPrivileging.OwnerId.Id == findTeam.Id)
                            return false;
                    }

                    break;
                #endregion
                #region FPPE/OPPE
                case cvt_qualitycheck.EntityLogicalName:

                    var thisPPE = (cvt_qualitycheck)thisRecord;
                    Logger.WriteDebugMessage("Querying for the parent priv.");
                    var privilegeParent = (cvt_tssprivileging)OrganizationService.Retrieve(cvt_tssprivileging.EntityLogicalName, thisPPE.cvt_TSSPrivilegingId.Id, new ColumnSet(true));
                    Logger.WriteDebugMessage("Querying for the Service Chief Team for the facility.");

                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == privilegeParent.cvt_PrivilegedAtId.Id
                            && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.ServiceChief
                            && t.cvt_ServiceType != null && t.cvt_ServiceType.Id == privilegeParent.cvt_ServiceTypeId.Id);

                    //Can't find team or already correct owner
                    if (findTeam == null)
                    {
                        Logger.WriteDebugMessage("Can't find Service Chief team for Facility: " + privilegeParent.cvt_PrivilegedAtId.Name + " with Specialty: " + privilegeParent.cvt_ServiceTypeId.Name + ". Creating team.");
                        Team newTeam = new Team()
                        {
                            cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, privilegeParent.cvt_PrivilegedAtId.Id),
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.ServiceChief),
                            Name = privilegeParent.cvt_ServiceTypeId.Name + " Service Chief Approval Group @ " + privilegeParent.cvt_PrivilegedAtId.Name,
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            cvt_ServiceType = privilegeParent.cvt_ServiceTypeId,
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, privilegeParent.OwningBusinessUnit.Id)
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }
                    else
                    {
                        if (thisPPE.OwnerId.Id == findTeam.Id)
                            return false;
                    }
                    break;

                #endregion
                #region Prov/Pat Site Resources
                case cvt_providerresourcegroup.EntityLogicalName:
                case cvt_patientresourcegroup.EntityLogicalName:
                    var owner = (EntityReference)thisRecord["ownerid"];
                    var resourceSite = OrganizationService.Retrieve(mcs_site.EntityLogicalName, ((EntityReference)thisRecord["cvt_relatedsiteid"]).Id, new ColumnSet(true)).ToEntity<mcs_site>();
                    if (resourceSite.mcs_FacilityId == null)
                        throw new InvalidPluginExecutionException(string.Format("Site {0} does not have a facility", resourceSite.mcs_name));
                    if (owner.Name.Contains("FTC") && owner.Name.Contains(resourceSite.mcs_FacilityId.Name))
                        return false;
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name.Contains("FTC") && t.Name.Contains(resourceSite.mcs_FacilityId.Name));
                    if (findTeam == null)
                    {
                        var facility = OrganizationService.Retrieve(mcs_facility.EntityLogicalName, resourceSite.mcs_FacilityId.Id, new ColumnSet(true)).ToEntity<mcs_facility>();
                        Team newTeam = new Team()
                        {
                            cvt_Facility = resourceSite.mcs_FacilityId,
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.FTC),
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, facility.OwningBusinessUnit.Id),
                            Name = "FTC Approval Group @ " + facility.mcs_name
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }
                    break;
                #endregion
                #region Service Activity
                case ServiceAppointment.EntityLogicalName:
                    var thisServiceAppointment = (ServiceAppointment)thisRecord;
                    using (var srv = new Xrm(OrganizationService))
                    {
                        if (thisServiceAppointment.cvt_Type.HasValue)
                        {
                            var site = (thisServiceAppointment.cvt_Type.Value) ? thisServiceAppointment.mcs_relatedprovidersite : thisServiceAppointment.mcs_relatedsite;

                            if (site != null && site.Id != Guid.Empty)
                            {
                                findTeam = (from t in srv.TeamSet
                                            join f in srv.mcs_facilitySet on t.cvt_Facility.Id equals f.mcs_facilityId.Value
                                            join s in srv.mcs_siteSet on f.mcs_facilityId.Value equals s.mcs_FacilityId.Id
                                            where s.mcs_siteId.Value == site.Id && t.cvt_Type.Value == Teamcvt_Type.Scheduler.GetHashCode()
                                            select t).FirstOrDefault();

                                if (findTeam == null || findTeam.Id == Guid.Empty)
                                    Logger.WriteDebugMessage(string.Format("Scheduler Team associated to the site on the Service Activity {0} with Id {1} is null or empty. Hence the Service appointment is not assigned to the Patient Facility Scheduler Team.", site.Name, site.Id));
                                else
                                    Logger.WriteDebugMessage("Initiating the assignment to the Patient Facility Scheduler Team: " + findTeam.Id + ((Team)findTeam).Name);
                            }
                            else
                            {
                                Logger.WriteDebugMessage(string.Format("Associated site on the Service Activity is null or empty. Service Activity Type: {0}", (thisServiceAppointment.cvt_Type.Value) ? "Home/Mobile" : "Clinic Based"));
                            }
                        }
                    }
                    break;
                #endregion
                #region Appointment
                case Appointment.EntityLogicalName:
                    var thisAppointment = (Appointment)thisRecord;
                    using (var srv = new Xrm(OrganizationService))
                    {
                        if (thisAppointment.cvt_serviceactivityid == null)
                            return false;
                        if (thisAppointment.cvt_Site == null)
                            return false;

                        var site = thisAppointment.cvt_Site;

                        if (site != null && site.Id != Guid.Empty)
                        {
                            findTeam = (from t in srv.TeamSet
                                        join f in srv.mcs_facilitySet on t.cvt_Facility.Id equals f.mcs_facilityId.Value
                                        join s in srv.mcs_siteSet on f.mcs_facilityId.Value equals s.mcs_FacilityId.Id
                                        where s.mcs_siteId.Value == site.Id && t.cvt_Type.Value == Teamcvt_Type.Scheduler.GetHashCode()
                                        select t).FirstOrDefault();

                            if (findTeam == null || findTeam.Id == Guid.Empty)
                                Logger.WriteDebugMessage(string.Format("Scheduler Team associated to the site on the Appointment {0} with Id {1} is null or empty. Hence the Appointment is not assigned to the Patient Facility Scheduler Team.", site.Name, site.Id));
                            else
                                Logger.WriteDebugMessage("Initiating the assignment to the Patient Facility Scheduler Team: " + findTeam.Id + ((Team)findTeam).Name);
                        }
                    }
                    break;
                #endregion
                #region PPEReview
                case cvt_ppereview.EntityLogicalName:
                    var thisReview = (cvt_ppereview)thisRecord;
                    using (var srv = new Xrm(OrganizationService))
                    {
                        if (thisReview.cvt_telehealthprivileging == null)
                            return false;

                        var thisPriv = srv.cvt_tssprivilegingSet.FirstOrDefault(t => t.Id == thisReview.cvt_telehealthprivileging.Id);

                        if (thisPriv == null)
                            return false;

                        var Facility = thisPriv.cvt_PrivilegedAtId;

                        if (Facility != null && Facility.Id != Guid.Empty)
                        {
                            findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == thisPriv.cvt_PrivilegedAtId.Id
                                && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.ServiceChief
                                && t.cvt_ServiceType != null && t.cvt_ServiceType.Id == thisPriv.cvt_ServiceTypeId.Id);

                            if (findTeam == null || findTeam.Id == Guid.Empty)
                            {
                                Logger.WriteDebugMessage("Can't find Service Chief team for Facility: " + thisPriv.cvt_PrivilegedAtId.Name + " with Specialty: " + thisPriv.cvt_ServiceTypeId.Name + ". Creating team.");
                                Team newTeam = new Team()
                                {
                                    cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, thisPriv.cvt_PrivilegedAtId.Id),
                                    cvt_Type = new OptionSetValue((int)Teamcvt_Type.ServiceChief),
                                    Name = thisPriv.cvt_ServiceTypeId.Name + " Service Chief Approval Group @ " + thisPriv.cvt_PrivilegedAtId.Name,
                                    TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                                    cvt_ServiceType = thisPriv.cvt_ServiceTypeId,
                                    BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisPriv.OwningBusinessUnit.Id)
                                };
                                findTeam = new Team();
                                findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                            }
                            else
                            {
                                if (thisReview.OwnerId.Id == findTeam.Id)
                                    return false;
                            }
                            Logger.WriteDebugMessage("Initiating the assignment to the Home Facility SC Team: " + findTeam.Id + ((Team)findTeam).Name);
                        }
                    }
                    break;
                    #endregion
            }
            #region Common Code for the Assign
            if (findTeam == null || findTeam.Id == Guid.Empty)
                return false;
            assignOwner.Id = findTeam.Id;

            AssignRequest assignRequest = new AssignRequest()
            {
                Assignee = assignOwner,
                Target = new EntityReference(thisRecord.LogicalName, thisRecord.Id)
            };

            OrganizationService.Execute(assignRequest);
            Logger.WriteDebugMessage(thisRecord.LogicalName + " assigned to a Team. Ending AssignOwner.");
            return true;
            #endregion
        }

        /// <summary>
        /// Create Team in the assign (add the security roles)
        /// </summary>
        /// <param name="newTeam"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns></returns>
        internal static Guid CreateTeamForAssign(Team newTeam, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            var newTeamId = OrganizationService.Create(newTeam);
            Logger.WriteDebugMessage("Team created.");

            //Rush the adding of security roles
            if (TeamCreatePostStageRunner.SetTeamRoles(newTeamId, Team.EntityLogicalName, OrganizationService, Logger))
            {
                Logger.WriteDebugMessage("Allowing time to process security changes.");
                System.Threading.Thread.Sleep(15000);
                Logger.WriteDebugMessage("Resuming assign.");
            }
            return newTeamId;
        }

        /// <summary>
        /// Runs assignowner for entire dataset passed in
        /// </summary>
        /// <param name="entitySet"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns></returns>
        internal static string FixOwnershipForEntity(IQueryable<Entity> entitySet, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.setMethod = "FixOwnershipForEntity";
            Logger.WriteDebugMessage("About to update Entity record ownership.");

            var count = 0;
            var entityName = "";
            foreach (var thisRecord in entitySet)
            {
                try
                {
                    bool update = CvtHelper.AssignOwner(thisRecord, Logger, OrganizationService);
                    entityName = thisRecord.LogicalName;
                    if (update)
                        count++;
                }
                catch (Exception ex)
                {
                    Logger.WriteDebugMessage("error message: " + ex.Message);
                }
            }
            var message = string.Format("Assigned ownership for {0}: {1}/{2}. ", entityName, count, entitySet.ToList().Count.ToString());
            Logger.WriteDebugMessage(message);
            return message;
        }
        #endregion

        #region Record Names
        /// <summary>
        /// Returns the record name if it needs to be changed.
        /// </summary>
        /// <param name="EntityObj"></param>
        /// <param name="RecordId"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns>Returns "" if no name change, or the name that should be changed to, if derived</returns>
        internal static string ReturnRecordNameIfChanged(Entity Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            string NameIfChanged = "";
            Logger.WriteDebugMessage("Deriving name for " + Record.LogicalName);
            switch (Record.LogicalName)
            {
                case Team.EntityLogicalName:
                    NameIfChanged = DeriveName((Team)Record, OnCreate, Logger, OrganizationService);
                    break;
                case cvt_tssprivileging.EntityLogicalName:
                    NameIfChanged = DeriveName((cvt_tssprivileging)Record, OnCreate, Logger, OrganizationService);
                    break;
                case mcs_resource.EntityLogicalName:
                    NameIfChanged = DeriveName((mcs_resource)Record, OnCreate, Logger, OrganizationService);
                    break;
                case mcs_resourcegroup.EntityLogicalName:
                    NameIfChanged = DeriveName((mcs_resourcegroup)Record, OnCreate, Logger, OrganizationService);
                    break;
                case cvt_mastertsa.EntityLogicalName:
                    NameIfChanged = DeriveName((cvt_mastertsa)Record, OnCreate, Logger, OrganizationService);
                    break;
                case mcs_services.EntityLogicalName:
                    NameIfChanged = DeriveName((mcs_services)Record, OnCreate, Logger, OrganizationService);
                    break;
                default:
                    Logger.WriteDebugMessage("Entity Name not found. No name generated.");
                    break;
            }

            return NameIfChanged;
        }

        internal static string DeriveName(Team Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Team thisRecord = new Team();
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<Team>();
            else //find record
            {
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.TeamSet.First(t => t.Id == Record.Id);
            }
            string derivedResultField = "";
            var teamServiceLine = new mcs_servicetype();
            if (thisRecord.cvt_ServiceType != null && thisRecord.cvt_ServiceType.Id != null)
            {
                using (var srv = new Xrm(OrganizationService))
                    teamServiceLine = srv.mcs_servicetypeSet.FirstOrDefault(st => st.Id == thisRecord.cvt_ServiceType.Id);
            }

            string serviceLineName = (teamServiceLine != null && teamServiceLine.mcs_name != null) ? teamServiceLine.mcs_name + " " : "";
            var role = (thisRecord.cvt_Type != null) ? (thisRecord.cvt_Type.Value) : -1;

            var teamFacility = new mcs_facility();
            if (thisRecord.cvt_Facility != null && thisRecord.cvt_Facility.Id != null)
            {
                using (var srv = new Xrm(OrganizationService))
                    teamFacility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == thisRecord.cvt_Facility.Id);
            }

            string facilityName = (teamFacility != null && teamFacility.mcs_name != null) ? " @ " + teamFacility.mcs_name : "";
            string roleName = "";

            var teamSite = new mcs_site();
            if (thisRecord.cvt_TMPSite != null && thisRecord.cvt_TMPSite.Id != null)
            {
                using (var srv = new Xrm(OrganizationService))
                    teamSite = srv.mcs_siteSet.FirstOrDefault(f => f.Id == thisRecord.cvt_TMPSite.Id);
            }
            string siteName = (teamSite != null && teamSite.mcs_name != null) ? " @ " + teamSite.mcs_name : "";

            switch (role)
            {
                case 917290000:
                    roleName = "FTC Approval Group";
                    break;
                case 917290001:
                    roleName = "Service Chief Approval Group";
                    break;
                case 917290002:
                    roleName = "Chiefs of Staff Approval Group";
                    break;
                case 917290003:
                    roleName = "Credentialing and Privileging Officer Approval Group";
                    break;
                case 917290004:
                    roleName = "TSA Notification Group";
                    break;
                case 917290005:
                    roleName = "Scheduler Group";
                    break;
                case 917290006:
                    roleName = "Data Administrators";
                    break;
                case 917290007:
                    roleName = "TCTs" + siteName;
                    break;
            }
            derivedResultField = String.Format("{0}{1}{2}", serviceLineName, roleName, facilityName);

            if (string.IsNullOrEmpty(serviceLineName + roleName) || thisRecord.Name == derivedResultField.Trim())
                derivedResultField = "";
            return derivedResultField;
        }

        internal static string DeriveName(cvt_tssprivileging Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            cvt_tssprivileging thisRecord = new cvt_tssprivileging();
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<cvt_tssprivileging>();
            else //find record
            {
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.cvt_tssprivilegingSet.First(p => p.Id == Record.Id);
            }
            string derivedResultField = "";

            var providerName = (thisRecord.cvt_ProviderId != null) ? thisRecord.cvt_ProviderId.Name : "";
            var serviceType = (thisRecord.cvt_ServiceTypeId != null) ? " - " + thisRecord.cvt_ServiceTypeId.Name : "";
            var facilityName = (thisRecord.cvt_PrivilegedAtId != null) ? " @ " + thisRecord.cvt_PrivilegedAtId.Name : "";
            var typeOfPrivileging = (thisRecord.cvt_TypeofPrivileging != null) ? thisRecord.cvt_TypeofPrivileging.Value : 0;
            var privText = "";
            switch (typeOfPrivileging)
            {
                case 917290000:
                    privText = " (Home/Parent)";
                    break;

                case 917290001:
                    privText = " (Proxy/Secondary)";
                    break;
            }

            derivedResultField = String.Format("{0}{1}{2}", providerName, serviceType, facilityName, privText);

            if (thisRecord.cvt_name == derivedResultField)
                derivedResultField = "";
            return derivedResultField;
        }

        internal static string DeriveName(mcs_resource Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            mcs_resource thisRecord;
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<mcs_resource>();
            else //find record
            {
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == Record.Id);

                if (thisRecord == null)
                    throw new InvalidPluginExecutionException("TMP Resource Record wasn't found for ID:" + Record.Id);

                if (Record.cvt_CartTypeId != null) thisRecord.cvt_CartTypeId = Record.cvt_CartTypeId;
                if (Record.mcs_UserNameInput != null) thisRecord.mcs_UserNameInput = Record.mcs_UserNameInput;
                if (Record.cvt_building != null) thisRecord.cvt_building = Record.cvt_building;
                if (Record.mcs_room != null) thisRecord.mcs_room = Record.mcs_room;
                if (Record.cvt_systemtype != null) thisRecord.cvt_systemtype = Record.cvt_systemtype;
                if (Record.cvt_room != null) thisRecord.cvt_room = Record.cvt_room;
                if (Record.cvt_uniqueid != null) thisRecord.cvt_uniqueid = Record.cvt_uniqueid;
            }

            string derivedResultField = "";

            //TMP Resource Name
            var mcs_usernameinput = thisRecord.mcs_UserNameInput;

            switch (thisRecord.mcs_Type.Value)
            {
                case (int)mcs_resourcetype.Room:
                    var building = thisRecord.cvt_building;
                    var room = thisRecord.mcs_room;

                    derivedResultField = (building != null) ? "Bldg. " + building : "";
                    derivedResultField += (derivedResultField != "" && room != null) ? ", " : "";
                    derivedResultField += (room != null) ? "Room " + room : "";

                    if ((derivedResultField == "") && (mcs_usernameinput != null))
                        derivedResultField = mcs_usernameinput;
                    break;
                case (int)mcs_resourcetype.Technology:
                    if (thisRecord.cvt_CartTypeId != null)
                    {
                        var cartType = OrganizationService.Retrieve(cvt_carttype.EntityLogicalName,
                            thisRecord.cvt_CartTypeId.Id, new ColumnSet("cvt_abbreviation"));
                        derivedResultField += cartType.GetAttributeValue<string>("cvt_abbreviation") + " ";
                        if (thisRecord.cvt_systemtype.Value ==
                            (int)cvt_carttypecvt_ResourceSystemType.TelehealthPatientCartSystem)
                            derivedResultField += "Cart ";
                    }
                    else if (thisRecord.FormattedValues.Contains("cvt_systemtype"))
                        derivedResultField += thisRecord.FormattedValues["cvt_systemtype"]; //SystemType

                    derivedResultField += thisRecord.cvt_room + " "; //Room Number
                    derivedResultField += (thisRecord.cvt_uniqueid != null) ? ": " + thisRecord.cvt_uniqueid : "";
                    break;
                default:
                    derivedResultField += mcs_usernameinput ?? "";
                    break;
            }

            if (thisRecord.mcs_RelatedSiteId != null)
            {
                using (var srv = new Xrm(OrganizationService))
                {
                    var site = srv.mcs_siteSet.FirstOrDefault(s => s.Id == thisRecord.mcs_RelatedSiteId.Id);
                    if (site != null) derivedResultField += " @ " + site.mcs_StationNumber;
                }
            }

            if (thisRecord.mcs_name == derivedResultField)
                derivedResultField = "";
            return derivedResultField;
        }

        internal static string DeriveName(mcs_resourcegroup Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            #region Getting/Validating the record
            mcs_resourcegroup thisRecord = new mcs_resourcegroup();
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<mcs_resourcegroup>();
            else //find record
            {
                Logger.WriteDebugMessage("Naming: Not on create, querying for record.");
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.mcs_resourcegroupSet.FirstOrDefault(r => r.Id == Record.Id);
            }

            if (thisRecord == null)
                throw new InvalidPluginExecutionException("TSS Resource Group Record wasn't found for GUID:" + thisRecord.Id);
            #endregion

            //New fields
            var locationText = "";
            var typeOfAppointmentText = "";
            var specialtyText = "";
            var specialtySubTypeText = "";
            var siteText = "";

            var resourceGroupTypeText = "";
            var allRequiredTech = "";
            var allRequiredRoom = "";

            #region Setting Resource Type
            var resourceType = -1;
            if (Record.Contains("mcs_type") && Record.mcs_Type != null)
            {
                resourceType = Record.mcs_Type.Value;
                Logger.WriteDebugMessage(String.Format("Updated mcs_Type from the context with {0}", Record.mcs_Type));
            }
            else if (thisRecord.Contains("mcs_type") && thisRecord.mcs_Type != null)
            {
                resourceType = thisRecord.mcs_Type.Value;
            }

            #endregion

            #region Location PRO or PAT

            var LocationOption = -1;
            if (Record.Contains("cvt_location") && Record.cvt_location != null)
            {
                LocationOption = Record.cvt_location.Value;
            }
            else if (thisRecord.Contains("cvt_location") && thisRecord.cvt_location != null)
            {
                LocationOption = thisRecord.cvt_location.Value;
            }

            switch (LocationOption)
            {
                case 917290000:
                    locationText = "PRO";
                    break;
                case 917290001:
                    locationText = "PAT";
                    break;

            }
            #endregion

            #region Type of Appointment IND or GRP
            var typeOfAppointmentOption = -1;
            if (Record.Contains("cvt_typeofappointment") && Record.cvt_typeofappointment != null)
            {
                typeOfAppointmentOption = Record.cvt_typeofappointment.Value;
            }
            else if (thisRecord.Contains("cvt_typeofappointment") && thisRecord.cvt_typeofappointment != null)
            {
                typeOfAppointmentOption = thisRecord.cvt_typeofappointment.Value;
            }

            switch (typeOfAppointmentOption)
            {
                case 917290000:
                    typeOfAppointmentText = "IND";
                    break;
                case 917290001:
                    typeOfAppointmentText = "GRP";
                    break;
            }
            #endregion

            #region Specialty and/or Specialty Sub-Type Name or Abbreviation
            //Get Specialty/Sub if PRO or PAT&GRP&AllRequired
            if ((locationText == "PRO") || (locationText == "PAT" && typeOfAppointmentText == "GRP" && resourceType == (int)mcs_resourcetype.AllRequired))
            {
                Guid specialtyGUID = Guid.Empty;
                //Use Spec from Context first if exists
                if (Record.Contains("cvt_specialty") && Record.cvt_specialty != null)
                {
                    specialtyGUID = Record.cvt_specialty.Id;
                    Logger.WriteDebugMessage(String.Format("Updated cvt_specialty from the context with {0}", Record.cvt_specialty.Name));
                }
                else if (thisRecord.Contains("cvt_specialty") && thisRecord.cvt_specialty != null)
                {
                    specialtyGUID = thisRecord.cvt_specialty.Id;
                }

                //Need to get the abbreviation off of the record
                if (specialtyGUID != Guid.Empty)
                {
                    var specRecord = (mcs_servicetype)OrganizationService.Retrieve(mcs_servicetype.EntityLogicalName, specialtyGUID, new ColumnSet(true));
                    if (specRecord != null)
                    {
                        specialtyText = !String.IsNullOrEmpty(specRecord.cvt_abbreviation) ? specRecord.cvt_abbreviation : specRecord.mcs_name;
                    }
                }

                Guid specialtySubTypeGUID = Guid.Empty;
                if (Record.Contains("cvt_specialtysubtype") && Record.cvt_specialtysubtype != null)
                {
                    specialtySubTypeGUID = Record.cvt_specialtysubtype.Id;
                    Logger.WriteDebugMessage(String.Format("Updated cvt_specialtysubtype from the context with {0}", Record.cvt_specialtysubtype.Name));
                }
                else if (thisRecord.Contains("cvt_specialtysubtype") && thisRecord.cvt_specialtysubtype != null)
                {
                    specialtySubTypeGUID = thisRecord.cvt_specialtysubtype.Id;
                }

                //Need to get the abbreviation off of the record
                if (specialtySubTypeGUID != Guid.Empty)
                {
                    var specSubTypeRecord = (mcs_servicesubtype)OrganizationService.Retrieve(mcs_servicesubtype.EntityLogicalName, specialtySubTypeGUID, new ColumnSet(true));
                    if (specSubTypeRecord != null)
                    {
                        specialtySubTypeText = !String.IsNullOrEmpty(specSubTypeRecord.cvt_abbreviation) ? specSubTypeRecord.cvt_abbreviation : specSubTypeRecord.mcs_name;
                    }
                }
            }
            else
                Logger.WriteDebugMessage("Not using the Specialty or Sub-Type for TSS Resource Group Naming.");
            #endregion

            #region TSS Site Name or Abbreviation
            Guid siteGUID = Guid.Empty;
            if (Record.Contains("mcs_relatedsiteid") && Record.mcs_relatedSiteId != null)
            {
                siteGUID = Record.mcs_relatedSiteId.Id;
                Logger.WriteDebugMessage(String.Format("Updated mcs_relatedSiteId from the context with {0}", Record.mcs_relatedSiteId.Name));
            }
            else if (thisRecord.Contains("mcs_relatedsiteid") && thisRecord.mcs_relatedSiteId != null)
            {
                siteGUID = thisRecord.mcs_relatedSiteId.Id;
            }

            //Need to get the abbreviation off of the record
            if (siteGUID != Guid.Empty)
            {
                var siteRecord = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, siteGUID, new ColumnSet(true));
                if (siteRecord != null)
                {
                    siteText = siteRecord.mcs_StationNumber != null ? siteRecord.mcs_StationNumber : siteRecord.mcs_name;
                    siteText = " @ " + siteText;
                }
            }
            else
                Logger.WriteToFile("This TSS Resource Group should have a TSS Site and does not: " + thisRecord.mcs_name + ". GUID: " + thisRecord.Id);
            #endregion

            #region UserName
            var userUniqueID = "";
            if (Record.Contains("mcs_usernameinput") && Record.mcs_UserNameInput != null)
            {
                userUniqueID = (!String.IsNullOrEmpty(Record.mcs_UserNameInput)) ? Record.mcs_UserNameInput : "";
                Logger.WriteDebugMessage(String.Format("Updated mcs_UserNameInput from the context with {0}", Record.mcs_UserNameInput));
            }
            else if (thisRecord.Contains("mcs_usernameinput") && thisRecord.mcs_UserNameInput != null)
            {
                userUniqueID = (!String.IsNullOrEmpty(thisRecord.mcs_UserNameInput)) ? thisRecord.mcs_UserNameInput : "";
            }
            #endregion

            #region Resource Type, All Required Tech/Room
            Logger.WriteDebugMessage("Checking for Resource Type.");
            //Check Resource Type
            switch (resourceType)
            {
                //If All Required
                case (int)mcs_resourcetype.AllRequired:
                case (int)mcs_resourcetype.Technology:
                    Logger.WriteDebugMessage("Resource Type is 'All Required' or 'Technology.'");
                    resourceGroupTypeText = thisRecord.FormattedValues["mcs_type"].ToString();
                    using (var srv = new Xrm(OrganizationService))
                    {
                        var groupResources = srv.mcs_groupresourceSet.Where(gr => gr.mcs_relatedResourceGroupId.Id == thisRecord.Id);

                        foreach (mcs_groupresource gr in groupResources)
                        {
                            if (gr.mcs_RelatedResourceId != null)
                            {
                                var resource = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == gr.mcs_RelatedResourceId.Id);
                                Logger.WriteDebugMessage("Found the TSS Resource.");
                                if (allRequiredTech == "" && resource.mcs_Type.Value == (int)mcs_resourcetype.Technology)
                                {
                                    allRequiredTech = resource.FormattedValues["cvt_systemtype"].ToString();
                                    Logger.WriteDebugMessage("resource.cvt_systemtype.ToString() = " + resource.FormattedValues["cvt_systemtype"].ToString());
                                }
                                if (resourceType != (int)mcs_resourcetype.Technology && allRequiredRoom == "" && resource.mcs_Type.Value == (int)mcs_resourcetype.Room && resource.mcs_room != null)
                                    allRequiredRoom = "Room " + resource.mcs_room;
                            }
                        }
                    }
                    break;
                case -1:
                    resourceGroupTypeText = "";
                    break;
                default:
                    resourceGroupTypeText = thisRecord.FormattedValues["mcs_type"].ToString() + "s";
                    break;
            }
            #endregion

            #region Constructing the Name
            //Specialty & Specialty Sub-Type - if PRO or PAT&GRP&AllRequired
            var derivedResultField = !String.IsNullOrEmpty(specialtyText) ? specialtyText + "." : "";
            derivedResultField += !String.IsNullOrEmpty(specialtySubTypeText) ? specialtySubTypeText + "." : "";

            //Tech Type - Pat&AR&Ind or Pat&Tech
            if ((locationText == "PAT" && resourceType == (int)mcs_resourcetype.AllRequired && typeOfAppointmentText == "IND") || (locationText == "PAT" && resourceType == (int)mcs_resourcetype.Technology))
            {
                derivedResultField += !String.IsNullOrEmpty(allRequiredTech) ? allRequiredTech + "." : "";
                Logger.WriteDebugMessage("Included Tech in the name");
            }
            else
                Logger.WriteDebugMessage("Did not include Tech in the name");

            //Room Number
            if (locationText == "PAT" && resourceType == (int)mcs_resourcetype.AllRequired)
                derivedResultField += !String.IsNullOrEmpty(allRequiredRoom) ? allRequiredRoom + "." : "";

            //Unique Id
            derivedResultField += !String.IsNullOrEmpty(userUniqueID) ? userUniqueID : "";

            //Type of Appt
            if (locationText == "PAT" && resourceType == (int)mcs_resourcetype.AllRequired)
            {
                //Trim leading or trailing "." from derivedResultField
                derivedResultField = derivedResultField.Trim('.');
                derivedResultField += !String.IsNullOrEmpty(typeOfAppointmentText) ? "." + typeOfAppointmentText : "";
            }
            //Trim leading or trailing "." from derivedResultField
            derivedResultField = derivedResultField.Trim('.');

            //: Resource Group Type
            derivedResultField += " : " + resourceGroupTypeText + siteText;

            //Location
            derivedResultField += !String.IsNullOrEmpty(locationText) ? "." + locationText : "";

            //In common:
            //{unique id} : {Resource Group Type} @ {TSS site}.{PAT/PRO}
            #endregion

            Logger.WriteDebugMessage("Automatically constructed name: " + derivedResultField);
            Logger.WriteDebugMessage("Name from the record: " + thisRecord.mcs_name.Trim());

            if (String.IsNullOrEmpty(derivedResultField) || thisRecord.mcs_name.Trim() == derivedResultField.Trim())
                derivedResultField = "";
            return derivedResultField;
        }

        internal static string DeriveName(cvt_mastertsa Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            cvt_mastertsa thisRecord = new cvt_mastertsa();
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<cvt_mastertsa>();
            else //find record
            {
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.cvt_mastertsaSet.FirstOrDefault(m => m.Id == Record.Id);
            }

            if (thisRecord == null)
                throw new InvalidPluginExecutionException("MTSA Record wasn't found for ID:" + thisRecord.Id);

            var servicetype = (thisRecord.cvt_servicetype != null) ? thisRecord.cvt_servicetype.Name : "";
            var servicesubtype = (thisRecord.cvt_servicesubtype != null) ? " : " + thisRecord.cvt_servicesubtype.Name : "";
            var relatedsiteid = (thisRecord.cvt_RelatedSiteid != null) ? thisRecord.cvt_RelatedSiteid.Name : "";
            var cvt_type = thisRecord.cvt_Type.Value;
            var cvt_groupappointment = thisRecord.cvt_GroupAppointment.Value;

            string derivedResultField = servicetype + servicesubtype + " @ " + relatedsiteid;

            if (!cvt_groupappointment && cvt_type)
                derivedResultField += " to Home/Mobile";

            if (thisRecord.cvt_name == derivedResultField)
                derivedResultField = "";
            return derivedResultField;
        }

        internal static string DeriveName(mcs_services Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            mcs_services thisRecord = new mcs_services();
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<mcs_services>();
            else //find record
            {
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.mcs_servicesSet.FirstOrDefault(s => s.Id == Record.Id);
            }

            if (thisRecord == null)
                throw new InvalidPluginExecutionException("TSA Record wasn't found for ID:" + thisRecord.Id);

            var servicetype = (thisRecord.cvt_servicetype != null) ? thisRecord.cvt_servicetype.Name : "";
            var servicesubtype = (thisRecord.cvt_servicesubtype != null) ? " : " + thisRecord.cvt_servicesubtype.Name : "";
            var relatedprovidersiteid = (thisRecord.cvt_relatedprovidersiteid != null) ? thisRecord.cvt_relatedprovidersiteid.Name : "";
            var cvt_type = thisRecord.cvt_Type.Value;
            var cvt_groupappointment = thisRecord.cvt_groupappointment.Value;

            string derivedResultField = servicetype + servicesubtype + " From " + relatedprovidersiteid;

            if (!cvt_groupappointment && cvt_type)
                derivedResultField += " to Home/Mobile";

            if (thisRecord.mcs_name == derivedResultField)
                derivedResultField = "";
            return derivedResultField;
        }

        #endregion
    }
}