﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.Shared;
using VA.TMP.OptionSets;
using VIMT.VideoVisit.Messages;

namespace VA.TMP.Integration.VIMT.VideoVisit.Mappers
{
    /// <summary>
    /// Default Mapping Resolvers class to hold all Resolver functions. 
    /// </summary>
    internal static class MappingResolvers
    {
        /// <summary>
        /// Person Identifier Resolver Function
        /// </summary>
        /// <param name="organizationService"></param>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpPersonIdentifier PersonIdentifierResolver(IOrganizationService organizationService, Contact source)
        {
            using (var srv = new Xrm(organizationService))
            {
                var identifiers = srv.mcs_personidentifiersSet.Where(x => x.mcs_patient.Id == source.Id).ToList();
                var icn = identifiers.FirstOrDefault(x => x.mcs_identifiertype.Value == (int)mcs_personidentifiersmcs_identifiertype.NationalIdentifier_NI && x.mcs_assigningauthority == "USVHA");
                if (icn == null) throw new Exception("Patient has no ICN: " + source.FullName);
                if (string.IsNullOrEmpty(icn.mcs_identifier)) throw new Exception("ICN is empty");
                var icnString = icn.mcs_identifier;
                return new VIMTTmpPersonIdentifier { AssigningAuthority = "ICN", UniqueId = icnString };
            }
        }
            
        /// <summary>
        /// Person Name Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpPersonName PersonNameResolver(Contact source)
        {
            return new VIMTTmpPersonName { FirstName = source.FirstName, LastName = source.LastName };         
        }

        /// <summary>
        /// Patient Contact Info Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpContactInformation PatientContactInformationResolver(Contact source)
        {
            var contactInformation = new VIMTTmpContactInformation
            {
                Mobile = source.Telephone1,
                PreferredEmail = source.EMailAddress1
            };

            if (source.cvt_TimeZone == null) return contactInformation;

            contactInformation.TimeZoneSpecified = true;
            contactInformation.TimeZone = source.cvt_TimeZone.Value;

            return contactInformation;
        }

        /// <summary>
        /// Patient Virutal Meeting Room Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpVirtualMeetingRoom PatientVirtualMeetingRoomResolver(ServiceAppointment source)
        {
            return new VIMTTmpVirtualMeetingRoom
            {
                Conference = source.mcs_meetingroomname,
                Pin = source.mcs_patientpin,
                Url = source.mcs_PatientUrl
            };
        }

        /// <summary>
        /// Provider Name Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpPersonName ProviderNameResolver(SystemUser source)
        {
            return new VIMTTmpPersonName { FirstName = source.FirstName, LastName = source.LastName };
        }

        /// <summary>
        /// Provider Contact Info Resolver Function 
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpContactInformation ProviderContactInformationResolver(SystemUser source)
        {
            var contactInformation = new VIMTTmpContactInformation
            {
                Mobile = source.MobilePhone,
                PreferredEmail = source.InternalEMailAddress
            };

            if (source.cvt_TimeZone == null) return contactInformation;

            contactInformation.TimeZoneSpecified = true;
            contactInformation.TimeZone = source.cvt_TimeZone.Value;

            return contactInformation;
        }

        /// <summary>
        /// Provider Location Resolver Function 
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        /// <param name="source">System User.</param>
        /// <param name="vimtTmpAppointmentKind">Appointment Kind.</param>
        /// <returns>Location</returns>
        internal static VIMTTmpLocation ProviderLocationResolver(IOrganizationService organizationService, ServiceAppointment serviceAppointment, SystemUser source, VIMTTmpAppointmentKind vimtTmpAppointmentKind)
        {
            using (var srv = new Xrm(organizationService))
            {
                var tsa = srv.mcs_servicesSet.FirstOrDefault(t => t.Id == serviceAppointment.mcs_relatedtsa.Id);
                var facility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == tsa.cvt_ProviderFacility.Id);

                if (facility == null) throw new Exception("Unable to retrieve Provider Facility");
                var siteCode = facility.mcs_StationNumber;
                var facilityName = facility.mcs_name;
                var timeZone = facility.mcs_Timezone ?? 0;

                var clinic = ClinicResolver(serviceAppointment, serviceAppointment.mcs_relatedprovidersite.Id, organizationService, Guid.Empty);
                
                if (string.IsNullOrEmpty(clinic.Ien) || string.IsNullOrEmpty(clinic.Name)) throw new Exception("No VistA Clinic is getting booked on the provider side (or Vista Clinic IEN is empty).  Unable to send Appointment to VistA.");
                
                return new VIMTTmpLocation
                {
                    Type = VIMTTmpLocationType.VA,
                    Facility = new VIMTTmpFacility { SiteCode = siteCode, Name = facilityName, TimeZone = timeZone },
                    Clinic = new VIMTTmpClinic { Name = clinic.Name, Ien = clinic.Ien}
                };
            }
        }

        /// <summary>
        /// Provider Virutal Meeting Room Resolver Function
        /// </summary>
        /// <param name="source">Service Appointment.</param>
        /// <returns>Virtual Meeting Room.</returns>
        internal static VIMTTmpVirtualMeetingRoom ProviderVirtualMeetingRoomResolver(ServiceAppointment source)
        {
            return new VIMTTmpVirtualMeetingRoom
            {
                Conference = source.mcs_meetingroomname,
                Pin = source.mcs_providerpin,
                Url = source.mcs_providerurl
            };
        }

        /// <summary>
        /// Reads List of Users and maps to List of VIMTTmpProviders
        /// </summary>
        /// <param name="users">List of users.</param>
        /// <param name="orgService">Organization Service for retrieving other data.</param>
        /// <param name="serviceAppointment">Service Appointment for getting location and VMR info.</param>
        /// <param name="vimtTmpAppointmentKind">Appointment Kind.</param>
        /// <returns>List of Providers.</returns>
        internal static List<VIMTTmpProviders> MapProviders(List<SystemUser> users, IOrganizationService orgService, ServiceAppointment serviceAppointment, VIMTTmpAppointmentKind vimtTmpAppointmentKind)
        {
            if (vimtTmpAppointmentKind == VIMTTmpAppointmentKind.STORE_FORWARD) return new List<VIMTTmpProviders>();

            var providerList = new List<VIMTTmpProviders>();

            foreach(var user in users)
            {
                var provider = new VIMTTmpProviders
                {
                    Name = ProviderNameResolver(user),
                    ContactInformation = ProviderContactInformationResolver(user),
                    Location = ProviderLocationResolver(orgService, serviceAppointment, user, vimtTmpAppointmentKind)
                };
                if (vimtTmpAppointmentKind == VIMTTmpAppointmentKind.MOBILE_ANY) provider.VirtualMeetingRoom = ProviderVirtualMeetingRoomResolver(serviceAppointment);
                providerList.Add(provider);
            }
            return providerList;
        }

        /// <summary>
        /// Maps Contacts to Patients.
        /// </summary>
        /// <param name="contacts">List of Contacts.</param>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        /// <param name="vimtTmpAppointmentKind">Appointment Kind.</param>
        /// <param name="isGroup">Is Group.</param>
        /// <param name="apptId">Appointment Id.</param>
        /// <returns>List of Patients.</returns>
        internal static List<VIMTTmpPatients> ResolveContacts(List<Guid> contacts, IOrganizationService organizationService, ServiceAppointment serviceAppointment, VIMTTmpAppointmentKind vimtTmpAppointmentKind, bool isGroup, Guid apptId)
        {
            var patients = new List<VIMTTmpPatients>();

            foreach (var contactId in contacts)
            {
                var contact = (Contact)organizationService.Retrieve(Contact.EntityLogicalName, contactId, new ColumnSet(true));
                var pat = new VIMTTmpPatients
                {
                    Id = PersonIdentifierResolver(organizationService, contact),
                    Name = PersonNameResolver(contact),
                    ContactInformation = PatientContactInformationResolver(contact),
                };
                if (!string.IsNullOrEmpty(serviceAppointment.mcs_meetingroomname)) pat.VirtualMeetingRoom = PatientVirtualMeetingRoomResolver(serviceAppointment);
                pat.Location = PatientLocationResolver(vimtTmpAppointmentKind, serviceAppointment, isGroup, apptId, organizationService);
                
                // Add in Facility and Clinic as optional
                patients.Add(pat);
            }
            return patients;
        }

        /// <summary>
        /// Maps Patient facility.
        /// </summary>
        /// <param name="vimtTmpAppointmentKind">Appointment Kind.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        /// <param name="isGroup">Is Group.</param>
        /// <param name="apptId">Appointment Id.</param>
        /// <param name="organizationService">Organization Service.</param>
        /// <returns>Location.</returns>
        internal static VIMTTmpLocation PatientLocationResolver(VIMTTmpAppointmentKind vimtTmpAppointmentKind, ServiceAppointment serviceAppointment, bool isGroup, Guid apptId, IOrganizationService organizationService)
        {
            if (vimtTmpAppointmentKind == VIMTTmpAppointmentKind.MOBILE_GFE || vimtTmpAppointmentKind == VIMTTmpAppointmentKind.MOBILE_ANY)
                return new VIMTTmpLocation { Type = VIMTTmpLocationType.NonVA };
            Guid siteId;// = Guid.Empty;
            using (var srv = new Xrm(organizationService))
            {
                mcs_facility patFacility;

                if (isGroup)
                {
                    var appointment = organizationService.Retrieve(Appointment.EntityLogicalName, apptId, new ColumnSet(true)).ToEntity<Appointment>();
                    siteId = appointment.cvt_Site?.Id ?? Guid.Empty;
                    var site = srv.mcs_siteSet.FirstOrDefault(s => s.Id == siteId);
                    patFacility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == site.mcs_FacilityId.Id);
                }
                else
                {
                    var tsa = srv.mcs_servicesSet.FirstOrDefault(t => t.Id == serviceAppointment.mcs_relatedtsa.Id);
                    patFacility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == tsa.cvt_PatientFacility.Id);
                    siteId = serviceAppointment.mcs_relatedsite?.Id ?? Guid.Empty;
                }

                if (patFacility == null) throw new Exception("Patient Facility cannot be null");

                var location = new VIMTTmpLocation
                {
                    Type = VIMTTmpLocationType.VA,
                    Clinic = ClinicResolver(serviceAppointment, siteId, organizationService, apptId),
                    Facility = new VIMTTmpFacility
                    {
                        Name = patFacility.mcs_name,
                        SiteCode = patFacility.mcs_StationNumber,
                        TimeZone = patFacility.mcs_Timezone ?? 0
                    }
                };
                return location;
            }
        }

        /// <summary>
        /// Maps Clinic.
        /// </summary>
        /// <param name="serviceAppointment">Service Appointment.</param>
        /// <param name="siteId">Site Id.</param>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="apptId">Appointment Id.</param>
        /// <returns>Clinic.</returns>
        internal static VIMTTmpClinic ClinicResolver (ServiceAppointment serviceAppointment, Guid siteId, IOrganizationService organizationService, Guid apptId)
        {
            using (var srv = new Xrm(organizationService))
            {
                // Look up the booked resources and find the Vista Clinic 
                var bookedResources = serviceAppointment.Resources.Where(r => r.PartyId.LogicalName == "equipment").ToList();
                if (apptId != Guid.Empty)
                {
                    var appt = organizationService.Retrieve(Appointment.EntityLogicalName, apptId, new ColumnSet("requiredattendees")).ToEntity<Appointment>();
                    if (appt.RequiredAttendees != null)
                        bookedResources.AddRange(appt.RequiredAttendees.Where(r => r.PartyId.LogicalName == "equipment").ToList());
                }
                var clinicName = string.Empty;
                var clinicIen = string.Empty;
                foreach (var equipmentParty in bookedResources)
                {
                    var resource = srv.mcs_resourceSet.FirstOrDefault(r => r.mcs_relatedResourceId != null && r.mcs_relatedResourceId.Id == equipmentParty.PartyId.Id);
                    if (resource?.mcs_Type != null && resource.mcs_Type.Value == (int)mcs_resourcetype.VistaClinic)
                    {
                        //Ensure that the VistA Clinic retrieved is the one for the appropriate site, otherwise skip to the next one
                        if (siteId == resource.mcs_RelatedSiteId.Id)
                        {
                            clinicName = resource.mcs_UserNameInput;
                            clinicIen = resource.cvt_ien;
                            break;
                        }
                        
                    }
                }

                return new VIMTTmpClinic { Name = clinicName, Ien = clinicIen };
            }
        }

        /// <summary>
        /// Gets Appointment Kind.
        /// </summary>
        /// <param name="serviceAppointment">Service Appointment.</param>
        /// <param name="organizationService">Organization Service.</param>
        /// <returns>Appointment Kind.</returns>
        internal static VIMTTmpAppointmentKind GetAppointmentKind(ServiceAppointment serviceAppointment, IOrganizationService organizationService)
        {
            if (serviceAppointment.cvt_TelehealthModality != null && serviceAppointment.cvt_TelehealthModality.Value) return VIMTTmpAppointmentKind.STORE_FORWARD;
            if (serviceAppointment.cvt_Type != null && !serviceAppointment.cvt_Type.Value) return VIMTTmpAppointmentKind.CLINIC_BASED;
            
            return PipelineUtilities.IsGfeServiceActivity(serviceAppointment, organizationService) ? VIMTTmpAppointmentKind.MOBILE_GFE : VIMTTmpAppointmentKind.MOBILE_ANY;
        }
    }
}