﻿using System;
using System.ServiceModel;
using System.Linq;
using Microsoft.Xrm.Sdk;
using VA.TMP.CRM;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;
using Microsoft.Xrm.Sdk.Query;
using VRMRest;
using VA.TMP.Integration.Plugins.Messages;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle creating a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentCreatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentCreatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        /// <summary>
        /// Gets or sets the ProxyAddFakeResponseType.
        /// </summary>
        private string ProxyAddFakeResponseType { get; set; }

        /// <summary>
        /// Gets or sets ProcessingCode.
        /// </summary>
        private string ProcessingCode { get; set; }

        /// <summary>
        /// Gets or sets ReturnMviMessagesInResponse.
        /// </summary>
        private bool ReturnMviMessagesInResponse { get; set; }

        /// <summary>
        /// Gets or sets PatientVeteran.
        /// </summary>
        private bool PatientVeteran { get; set; }

        /// <summary>
        /// Gets or sets PatientServiceConnected.
        /// </summary>
        private bool PatientServiceConnected { get; set; }

        /// <summary>
        /// Gets or sets PatientType.
        /// </summary>
        private int PatientType { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            try
            {
                var sa = PrimaryEntity.ToEntity<DataModel.ServiceAppointment>();

                if (sa.mcs_groupappointment == null || !sa.mcs_groupappointment.Value)
                {
                    if (sa.StatusCode.Value == (int)serviceappointment_statuscode.Pending || sa.StatusCode.Value == (int)serviceappointment_statuscode.ReservedScheduled)
                    {
                        var orgName = PluginExecutionContext.OrganizationName;
                        var user = OrganizationService.Retrieve(SystemUser.EntityLogicalName, PluginExecutionContext.UserId, new ColumnSet(true)).ToEntity<SystemUser>();

                        using (var context = new Xrm(OrganizationService))
                        {
                            Logger.WriteDebugMessage("Starting GetAndSetIntegrationSettings");
                            GetAndSetIntegrationSettings(context);
                            Logger.WriteDebugMessage("Finished GetAndSetIntegrationSettings");

                            Logger.WriteDebugMessage("Sending Proxy Add request to VIMT");
                            var response = CreateAndSendProxyAddToVista(orgName, user);
                            Logger.WriteDebugMessage("Finished sending Proxy Add request to VIMT");

                            if (response == null)
                            {
                                Logger.WriteDebugMessage("The Proxy Add response is null");
                                return;
                            }

                            ProcessProxyAddToVistaResponse(response);
                        }
                    }
                    else
                    {
                        Logger.WriteDebugMessage(string.Format("ServiceAppointment is not scheduled (status is {1}), Proxy Add Canceled: {0}", PrimaryEntity.Id, sa.StatusCode.Value));
                    }
                }
                else
                {
                    Logger.WriteDebugMessage("Proxy Add is executed on the Appointment, not the Service Activity for Group Appointments.");
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(string.Format("ERROR in ServiceAppointmentCreatePostStageRunner: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)));
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        /// <summary>
        /// Gets and sets the Integration Settings needed for Proxy Add.
        /// </summary>
        /// <param name="context">CRM context.</param>
        private void GetAndSetIntegrationSettings(Xrm context)
        {
            var vimtUrlSetting = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
            if (vimtUrlSetting != null) VimtUrl = vimtUrlSetting.mcs_value;
            else throw new InvalidPluginExecutionException("No VIMT Url listed in Integration Settings.  Please contact the Help Desk to add VIMT URL.  Proxy Add Canceled.");

            var proxyAddFakeResponseType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "MVI ProxyAdd Fake Response Type");
            if (proxyAddFakeResponseType != null) ProxyAddFakeResponseType = proxyAddFakeResponseType.mcs_value;
            else throw new InvalidPluginExecutionException("No ProxyAddFakeResponseType listed in Integration Settings.  Please contact the Help Desk to add ProxyAddFakeResponseType.  Proxy Add Canceled.");
            
            var processingCode = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "ProcessingCode");
            if (processingCode != null) ProcessingCode = processingCode.mcs_value;
            else throw new InvalidPluginExecutionException("No ProcessingCode listed in Integration Settings.  Please contact the Help Desk to add ProcessingCode.  Proxy Add Canceled.");

            var returnMviMessagesInResponse = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "ReturnMviMessagesInResponse");
            if (returnMviMessagesInResponse != null) ReturnMviMessagesInResponse = Convert.ToBoolean(returnMviMessagesInResponse.mcs_value);
            else throw new InvalidPluginExecutionException("No ReturnMviMessagesInResponse listed in Integration Settings.  Please contact the Help Desk to add ReturnMviMessagesInResponse.  Proxy Add Canceled.");

            var patientVeteran = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientVeteran");
            if (patientVeteran != null) PatientVeteran = Convert.ToBoolean(patientVeteran.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientVeteran listed in Integration Settings.  Please contact the Help Desk to add PatientVeteran.  Proxy Add Canceled.");

            var patientServiceConnected = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientServiceConnected");
            if (patientServiceConnected != null) PatientServiceConnected = Convert.ToBoolean(patientServiceConnected.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientServiceConnected listed in Integration Settings.  Please contact the Help Desk to add PatientServiceConnected.  Proxy Add Canceled.");

            var patientType = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "PatientType");
            if (patientType != null) PatientType = Convert.ToInt32(patientType.mcs_value);
            else throw new InvalidPluginExecutionException("No PatientType listed in Integration Settings.  Please contact the Help Desk to add PatientType.  Proxy Add Canceled.");
        }

        /// <summary>
        /// Sends the Proxy Add Request to VIMT.
        /// </summary>
        /// <param name="orgName">Organization Service.</param>
        /// <param name="user">CRM User.</param>
        /// <returns>ProxyAddResponseMessage.</returns>
        private ProxyAddResponseMessage CreateAndSendProxyAddToVista(string orgName, SystemUser user)
        {
            var request = new ProxyAddRequestMessage
            {
                LogRequest = true,
                ServiceAppointmentId = PrimaryEntity.Id,
                OrganizationName = orgName,
                FakeResponseType = ProxyAddFakeResponseType,
                UserFirstName = user.FirstName,
                UserLastName = user.LastName,
                UserId = user.Id,
                ProcessingCode = ProcessingCode,
                ReturnMviMessagesInResponse = ReturnMviMessagesInResponse,
                PatientVeteran = PatientVeteran,
                PatientServiceConnected = PatientServiceConnected,
                PatientType = PatientType
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

            Logger.WriteDebugMessage(string.Format("Sending Proxy Add Message to VIMT: {0}.", vimtRequest));

            try
            {
                var response = Utility.SendReceive<ProxyAddResponseMessage>(new Uri(VimtUrl), MessageRegistry.ProxyAddRequestMessage, request, null);
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Create Proxy Add to Vista", errorMessage, vimtRequest, typeof(ProxyAddRequestMessage).FullName,
                    typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage, PrimaryEntity.Id, OrganizationService);
                Logger.WriteToFile(errorMessage);

                return null;
            }
        }

        /// <summary>
        /// Process Proxy Add to VistA response.
        /// </summary>
        /// <param name="response">Proxy Add Response.</param>
        private void ProcessProxyAddToVistaResponse(ProxyAddResponseMessage response)
        {
            if (response == null) return;

            var errorMessage = response.ExceptionOccured ? response.ExceptionMessage : string.Empty;
            IntegrationPluginHelpers.CreateIntegrationResult("Proxy Add to VistA", response.ExceptionOccured, errorMessage,
                response.VimtRequest, response.SerializedInstance, response.VimtResponse,
                typeof(ProxyAddRequestMessage).FullName, typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage,
                PrimaryEntity.Id, OrganizationService);

            if (response.ExceptionOccured) return;

            Logger.WriteDebugMessage("Updating ProxyAddCompleted flag");

            var updateServiceAppointment = new DataModel.ServiceAppointment
            {
                Id = PrimaryEntity.Id,
                cvt_ProxyAddCompleted = true
            };

            try
            {
                OrganizationService.Update(updateServiceAppointment);
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException(string.Format("Failed to Update Proxy Add Completed Flag to initiate VVC/VVS plugin: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)), ex);
            }
        }
    }
}