﻿using MCSUtilities2011;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Shared
{
    public static class VistaPluginHelpers
    {
        private static List<Guid> GetPatients(VA.TMP.DataModel.ServiceAppointment serviceAppointment)
        {
            var patients = new List<Guid>();
            var patientAp = serviceAppointment.Customers.FirstOrDefault();
            if (patientAp != null && patientAp.PartyId != null)
                patients.Add(patientAp.PartyId.Id);
            return patients;
        }

        private static List<Guid> GetPatients(DataModel.Appointment appointment)
        {
            var patients = new List<Guid>();
            var patientAPs = appointment.OptionalAttendees.ToList();
            foreach (var ap in patientAPs)
            {
                patients.Add(ap.PartyId.Id);
            }
            return patients;
        }

        private static List<Guid> GetPatients(Entity record)
        {
            if (record.LogicalName == DataModel.Appointment.EntityLogicalName)
                return GetPatients(record.ToEntity<DataModel.Appointment>());
            if (record.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
                return GetPatients(record.ToEntity<DataModel.ServiceAppointment>());
            return new List<Guid>();
        }

        /// <summary>
        /// Create an instance of the Video Vist Service request and send to VIMT.
        /// </summary>
        /// <returns>VideoVisitCreateResponseMessage.</returns>
        internal static VideoVisitCreateResponseMessage CreateAndSendVideoVisitService(Entity entity, Guid userId, string organizationName, string vimtUrl, IOrganizationService organizationService, MCSLogger logger)
        {
            var request = new VideoVisitCreateRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                AppointmentId = entity.Id,
                AddedPatients = GetPatients(entity)
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

            try
            {
                logger.WriteDebugMessage("Sending Create VVS to VIMT");
                var response = Utility.SendReceive<VideoVisitCreateResponseMessage>(new Uri(vimtUrl), MessageRegistry.VideoVisitCreateRequestMessage, request, null);
                logger.WriteDebugMessage("VVS Create Successfully sent to VIMT");
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                if (entity.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
                    IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Create Video Visit", errorMessage, vimtRequest, typeof(VideoVisitCreateRequestMessage).FullName,
                        typeof(VideoVisitCreateResponseMessage).FullName, MessageRegistry.VideoVisitCreateRequestMessage, entity.Id, organizationService);
                else
                    IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Create Video Visit", errorMessage, vimtRequest, typeof(VideoVisitCreateRequestMessage).FullName, typeof(VideoVisitCreateResponseMessage).FullName, MessageRegistry.VideoVisitCreateRequestMessage, entity.Id, organizationService);
                logger.WriteToFile(errorMessage);

                return null;
            }
        }

        internal static VideoVisitUpdateResponseMessage UpdateAndSendVideoVisitService(Entity entity, Guid userId, string organizationName, string vimtUrl, IOrganizationService organizationService, MCSLogger logger)
        {
            var request = new VideoVisitUpdateRequestMessage
            {
                LogRequest = true,
                AppointmentId = entity.Id,
                OrganizationName = organizationName,
                UserId = userId,
                Contacts = GetPatients(entity)
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

            try
            {
                logger.WriteDebugMessage("Sending Update Retry to VIMT");
                var response = Utility.SendReceive<VideoVisitUpdateResponseMessage>(new Uri(vimtUrl), MessageRegistry.VideoVisitUpdateRequestMessage, request, null);
                logger.WriteDebugMessage("Retry Successfully sent to VIMT");
                return response;
            }
            catch(Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                if (entity.LogicalName == DataModel.ServiceAppointment.EntityLogicalName)
                    IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Update Video Visit", errorMessage, vimtRequest, typeof(VideoVisitUpdateRequestMessage).FullName, typeof(VideoVisitUpdateResponseMessage).FullName, MessageRegistry.VideoVisitUpdateRequestMessage, entity.Id, organizationService);
                else
                    IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Update Video Visit", errorMessage, vimtRequest, typeof(VideoVisitUpdateRequestMessage).FullName, typeof(VideoVisitUpdateResponseMessage).FullName, MessageRegistry.VideoVisitUpdateRequestMessage, entity.Id, organizationService);

                logger.WriteToFile(errorMessage);
                return null;
            }
        }

        /// <summary>
        /// Create an instance of the Video Vist Service request and send to VIMT.
        /// </summary>
        /// <returns>VideoVisitDeleteResponseMessage.</returns>
        internal static VideoVisitDeleteResponseMessage CancelAndSendVideoVisitServiceSa(DataModel.ServiceAppointment serviceAppointment, Guid userId, string organizationName, string vimtUrl, IOrganizationService organizationService, MCSLogger logger)
        {
            var cancels = IntegrationPluginHelpers.GetPatientsFromActivityPartyList(serviceAppointment.Customers.ToList());

            var request = new VideoVisitDeleteRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                AppointmentId = serviceAppointment.Id,
                CanceledPatients = cancels,
                WholeAppointmentCanceled = true
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

            try
            {
                logger.WriteDebugMessage("Sending Cancel to VIMT");
                var response = Utility.SendReceive<VideoVisitDeleteResponseMessage>(new Uri(vimtUrl), MessageRegistry.VideoVisitDeleteRequestMessage, request, null);
                logger.WriteDebugMessage("Cancel successfully sent to VIMT");
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Cancel Video Visit", errorMessage, vimtRequest, typeof(VideoVisitDeleteRequestMessage).FullName,
                    typeof(VideoVisitDeleteResponseMessage).FullName, MessageRegistry.VideoVisitDeleteRequestMessage, serviceAppointment.Id, organizationService, true);
                logger.WriteToFile(errorMessage);

                return null;
            }
        }

        internal static VideoVisitDeleteResponseMessage CancelAndSendVideoVisitServiceAppt(DataModel.Appointment appointment, string organizationName, Guid userId, MCSLogger logger, string vimtUrl, IOrganizationService organizationService, Guid integrationResultId)
        {
            var deleteRequest = new VideoVisitDeleteRequestMessage
            {
                AppointmentId = appointment.Id,
                LogRequest = true,
                OrganizationName = organizationName,
                UserId = userId,
                WholeAppointmentCanceled = GetWholeAppointmentCanceled(appointment),
                CanceledPatients = GetCancels(appointment, organizationService)
            };
            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(deleteRequest);

            try
            {
                logger.WriteDebugMessage("Sending Cancel Retry to VIMT");
                var response = Utility.SendReceive<VideoVisitDeleteResponseMessage>(new Uri(vimtUrl), MessageRegistry.VideoVisitDeleteRequestMessage, deleteRequest, null);
                logger.WriteDebugMessage("Retry successfully sent to VIMT");
                return response;
            }
            catch(Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateAppointmentIntegrationResultOnVimtFailure("Cancel Video Visit", errorMessage, vimtRequest, typeof(VideoVisitDeleteRequestMessage).FullName,
                    typeof(VideoVisitDeleteResponseMessage).FullName, MessageRegistry.VideoVisitDeleteRequestMessage, appointment.Id, organizationService);
                logger.WriteToFile(errorMessage);

                return null;
            }
        }

        internal static bool GetWholeAppointmentCanceled(DataModel.Appointment appointment)
        {
            return true;
        }
        
        internal static List<Guid> GetCancels(VA.TMP.DataModel.Appointment appointment, IOrganizationService organizationService)
        {
            var currentPatients = GetPatientsFromActivityPartyList(appointment.OptionalAttendees.ToList());

            //This means that specific patients were added or removed, but the overall appointment is still booked.  Need to find delta in booked vs current to get list of cancels or adds
            var previouslyBookedPatients = new List<Guid>();
            using (var srv = new Xrm(organizationService))
            {
                //Anthony review: is the Vista Integration Result set a reliable list to check to see who has already been booked?  
                var pastBookings = srv.cvt_vistaintegrationresultSet.Where(vir => vir.cvt_Appointment.Id == appointment.Id && vir.cvt_VistAStatus != VistaStatus.CANCELLED.ToString());
                foreach (var book in pastBookings)
                {
                    var contactId = IntegrationPluginHelpers.GetPatIdFromIcn(book.cvt_PersonId, organizationService);
                    //Only want distinct patients since WriteResults (aka vista integration results) will return 2 copies for a clinic based (1 for pat side and 1 for pro side with the same patient)
                    if (!previouslyBookedPatients.Contains(contactId))
                        previouslyBookedPatients.Add(contactId);
                }
            }
            var cancels = previouslyBookedPatients.Except(currentPatients).ToList();

            if (cancels.Count > 0)
                return cancels;
            else
                return new List<Guid>();

        }

        internal static List<Guid> GetPatientsFromActivityPartyList(List<ActivityParty> currentPatientsApList)
        {
            var patients = new List<Guid>();
            foreach (var ap in currentPatientsApList)
            {
                patients.Add(ap.PartyId.Id);
            }
            return patients;
        }

        public static bool RunVVS(DataModel.ServiceAppointment sa, Xrm context, MCSLogger Logger)
        {
            var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
            if (settings == null) throw new InvalidPluginExecutionException("Active Settings Cannot be Null");
            if (settings.cvt_UseVVS != null && settings.cvt_UseVVS.Value)
            {
                if (sa.cvt_Type.Value)
                {
                    if (settings.cvt_UseVVSHomeMobile != null && settings.cvt_UseVVSHomeMobile.Value)
                    {
                        Logger.WriteDebugMessage("Home/Mobile Encounter Vista Booking is turned on.");
                        return true;
                    }
                    else
                    {
                        Logger.WriteDebugMessage("Home/Mobile Encounter Vista Booking is turned off, skipping VVS Integration.");
                        return false;
                    }
                }
                else if (sa.cvt_TelehealthModality.HasValue && sa.cvt_TelehealthModality.Value)
                {
                    if (settings.cvt_UseVVSSingleEncounterNonHomeMobile != null && settings.cvt_UseVVSSingleEncounterNonHomeMobile.Value)
                    {
                        Logger.WriteDebugMessage("Non Home/Mobile Single Encounter Vista Booking is turned on.");
                        return true;
                    }
                    else
                    {
                        Logger.WriteDebugMessage("Non Home/Mobile Single Encounter Vista Booking is turned off, skipping VVS Integration.");
                        return false;
                    }
                }
                else
                {
                    //var specialtySwitch = true;
                    var specialtySwitch = CheckSpecialtySwitch(sa, context, Logger);
                    if (specialtySwitch)
                    {
                        var tsa = context.mcs_servicesSet.FirstOrDefault(s => s.Id == sa.mcs_relatedtsa.Id);

                        if (tsa.cvt_ServiceScope != null && tsa.cvt_ServiceScope.Value == (int)mcs_servicescvt_ServiceScope.InterFacility)
                        {

                            if (settings.cvt_UseVVSInterfacility != null && settings.cvt_UseVVSInterfacility.Value)
                            {
                                Logger.WriteDebugMessage("Interfacility Vista Booking is turned on.");
                                return true;
                            }
                            else
                            {
                                Logger.WriteDebugMessage("Interfacility Vista Booking is turned off, skipping VVS Integration.");
                                return false;
                            }
                        }
                        else
                        {
                            //Arbitrarily choosing Pro or Pat facility because they should be the same
                            var facility = context.mcs_facilitySet.FirstOrDefault(f => f.Id == tsa.cvt_ProviderFacility.Id);
                            if (facility.cvt_UseVistaIntrafacility.HasValue && facility.cvt_UseVistaIntrafacility.Value)
                            {
                                Logger.WriteDebugMessage("Intrafacility Vista Booking is turned on.");
                                return true;
                            }
                            else
                            {
                                Logger.WriteDebugMessage("Intrafacility Vista Booking is turned off, skipping VVS integration.");
                                return false;
                            }
                        }
                    }
                    else
                    {
                        Logger.WriteDebugMessage("Specialty Switch is turned off.");
                        return false;
                    }
                }
            }
            else
            {
                Logger.WriteDebugMessage("VVS is turned off system wide, skipping VVS integration");
                return false;
            }
        }

        private static bool CheckSpecialtySwitch(DataModel.ServiceAppointment sa, Xrm context, MCSLogger Logger)
        {
            var subSpecialty = sa.mcs_servicesubtype != null ? context.mcs_servicesubtypeSet.FirstOrDefault(s => s.Id == sa.mcs_servicesubtype.Id) : null;
            if (subSpecialty == null)
            {
                Logger.WriteDebugMessage("Sub-Specialty not specified on Service Activity, checking Specialty switch.");
            }
            else if (subSpecialty.cvt_UseVVS.HasValue && subSpecialty.cvt_UseVVS.Value)
            {
                Logger.WriteDebugMessage(string.Format("Sub-Specialty Vista Booking Switch is turned on for {0}, checking Service Activity Type.", subSpecialty.mcs_name));
                return true;
            }
            else if (!subSpecialty.cvt_UseVVS.HasValue || !subSpecialty.cvt_UseVVS.Value)
            {
                Logger.WriteDebugMessage(string.Format("Sub-Specialty Vista Booking Switch is turned off for {0}, skipping VVS Integration.", subSpecialty.mcs_name));
                return false;
            }
            var specialty = context.mcs_servicetypeSet.FirstOrDefault(s => s.Id == sa.mcs_servicetype.Id);
            if (specialty.cvt_UseVVS.HasValue && specialty.cvt_UseVVS.Value)
            {
                Logger.WriteDebugMessage(string.Format("Specialty switch turned on for {0}, checking Service Activity Type.", specialty.mcs_name));
                return true;
            }
            else
            {
                Logger.WriteDebugMessage(string.Format("Specialty switch turned off for {0}, skipping VVS Integration", specialty.mcs_name));
                return false;
            }
        }
    }
}