using System;
using System.Linq;
using VA.TMP.Integration.VIMT.Mvi.StateObject;
using VRM.Integration.Mvi.PersonSearch.Messages;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.Shared
{
    /// <summary>
    /// Class for Pearson Search Utilities.
    /// </summary>
    public static class PersonSearchUtilities
    {
        /// <summary>
        /// Send request to EC.
        /// </summary>
        /// <param name="state">State.</param>
        /// <returns></returns>
        public static RetrieveOrSearchPersonResponse SendPersonSearchToEc(PersonSearchStateObject state)
        {
            RetrieveOrSearchPersonResponse response = null;

            try
            {
                switch (state.PersonSearchFakeResponseType)
                {
                    case "0":
                        response = CreateFakeSuccess0Result();
                        break;
                    case "1":
                        response = CreateFakeSuccess1Result(state);
                        break;
                    case "2":
                        response = CreateFakeSuccessManyResults(state);
                        break;
                    case "3":
                        response = CreateFakeEdipiResult(state);
                        break;
                    default:
                        response = state.AttendedSearchRequest.SendReceive<RetrieveOrSearchPersonResponse>(MessageProcessType.Local);
                        break;
                }

                if (response == null) Logger.Instance.Error("Response returned from Attended Person Search is null");
                else Logger.Instance.Debug("Response returned from Attended Person Search is " + PipelineUtilities.SerializeInstance(response));

                state.RetrieveOrSearchPersonResponse = response;
            }
            catch(Exception ex)
            {
                Logger.Instance.Error(string.Format("Unable to serialize the output of the Attended Person Search response. Error was {0}", ex));
            }

            if (response == null)
            {
                ErrorOccurred(state, "No Response was receieved from Person Search");
            }
            else if (response.Acknowledgement == null)
            {
                ErrorOccurred(state, "No Acknowledgement was receieved in the Person Search Response");
            }
            else if (response.Acknowledgement.TypeCode == null)
            {
                ErrorOccurred(state, "Acknowledgment Type Code is null");
            }
            else
            {
                switch (response.Acknowledgement.TypeCode)
                {
                    case "AA":
                        if (response.QueryAcknowledgement.QueryResponseCode == "NF" || response.QueryAcknowledgement.QueryResponseCode == "QE") state.ExceptionMessage = response.QueryAcknowledgement.QueryResponseCode;
                        break;
                    case "AR":
                    case "AE":
                        if (response.Acknowledgement.AcknowledgementDetails == null || response.Acknowledgement.AcknowledgementDetails.Length == 0)
                        {
                            ErrorOccurred(state, "No Acknowledgment Details Listed in Person Search Fail Response");
                        }
                        else
                        {
                            var ack2 = response.Acknowledgement.AcknowledgementDetails.FirstOrDefault();
                            if (ack2 != null && ack2.Code != null)
                            {
                                if (!string.IsNullOrEmpty(ack2.Code.Code))
                                {
                                    var exceptionMessage = string.Format("Person Search failed with the following: {0} - {1} - {2}", ack2.Code.Code, ack2.Code.DisplayName, ack2.Text);
                                    ErrorOccurred(state, exceptionMessage);
                                }
                                else
                                {
                                    ErrorOccurred(state, "No Acknowledgment Code Listed in Person Search Fail Response");
                                }
                            }
                            else
                            {
                                ErrorOccurred(state, "Acknowledgment or Acknowledgment Code Fail is null");
                            }
                        }
                        break;
                }
            }

            return response;
        }

        /// <summary>
        /// Logs and set state on an exception.
        /// </summary>
        /// <param name="state">State.</param>
        /// <param name="errorMessage">Error Message.</param>
        private static void ErrorOccurred(PersonSearchStateObject state, string errorMessage)
        {
            state.ExceptionOccured = true;
            state.ExceptionMessage = errorMessage;
            Logger.Instance.Error(errorMessage);
        }

        /// <summary>
        /// Create fake EDIPI result.
        /// </summary>
        /// <param name="state">State.</param>
        /// <returns>RetrieveOrSearchPersonResponse.</returns>
        public static RetrieveOrSearchPersonResponse CreateFakeEdipiResult(PersonSearchStateObject state)
        {
            // TODO - parameterize the fake result based on user entry
            return new RetrieveOrSearchPersonResponse
            {
                MessageId = "88047127-5f3d-4090-8e93-1104a0384fdf",
                Person = new[] 
                { 
                    new PatientPerson()
                    {
                        SocialSecurityNumber = "898765678",
                        PhoneNumber = "(603)555-1111",
                        GenderCode = "M",
                        BirthDate = "19450820",
                        EdiPi = state.Edipi ?? "1606683207",
                        Address = new PatientAddress()
                        {
                            Use = AddressUse.Unspecified,
                            Country = "USA"
                        },
                        NameList = new[]
                        {
                            new Name
                            {
                                GivenName = "Jimmy",
                                FamilyName = "Chesney",
                                MiddleName = "David",
                                NameSuffix = "III",
                                Use = NameUse.Unspecified,
                                NameType = "Legal"
                            },
                            new Name
                            {
                                GivenName = "NAME",
                                FamilyName = "ALIAS",
                                MiddleName = "ONE",
                                Use = NameUse.Unspecified,
                                NameType = "Alias"
                            },
                            new Name
                            {
                                GivenName = "NAME",
                                FamilyName = "ALIAS",
                                MiddleName = "TWO",
                                Use = NameUse.Unspecified,
                                NameType = "Alias"
                            },                        
                            new Name
                            {
                                FamilyName = "NONAME",
                                Use = NameUse.Unspecified,
                                NameType = "Maiden"
                            }
                        },
                        CorrespondingIdList = new[]
                        { 
                            new UnattendedSearchRequest
                            {
                                MessageId = "0f1a0ebc-ccaa-4d5b-b9a2-4ffd04fb439f",
                                UseRawMviValue = false,
                                PatientIdentifier = "1008520256V373999",
                                IdentifierType = "NI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200M",
                                RawValueFromMvi = "1008520256V373999^NI^200M^USVHA^P",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "57d1aab1-122c-48e3-9562-05ad063b0c75",
                                UseRawMviValue = false,
                                PatientIdentifier = "898765678",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVBA",
                                AssigningFacility = "200BRLS",
                                RawValueFromMvi = "898765678^PI^200BRLS^USVBA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "0f2cd19a-462b-47c2-a2b6-bc2f7ed47bf0",
                                UseRawMviValue = false,
                                PatientIdentifier = "9100787222",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200M",
                                RawValueFromMvi = "9100787222^PI^200CORP^USVBA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "9592c1a5-2757-4cb3-b4ff-3e839173d905",
                                UseRawMviValue = false,
                                PatientIdentifier = "1606683207",
                                IdentifierType = "NI",
                                AssigningAuthority = "USDOD",
                                AssigningFacility = "200DOD",
                                RawValueFromMvi = "1606683207^NI^200DOD^USDOD^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "d9b95863-3fbb-4d26-9d4b-1f84432ce82f",
                                UseRawMviValue = false,
                                PatientIdentifier = "100000025",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "500",
                                RawValueFromMvi = "100000025^PI^500^USVHA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "eeea6d83-ee64-4baa-bacd-2e7f20da3d6e",
                                UseRawMviValue = false,
                                PatientIdentifier = "0000001008520256V373999000000",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200ESR",
                                RawValueFromMvi = "0000001008520256V373999000000^PI^200ESR^USVHA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "b1f93f22-8683-48e4-8914-835549d59d50",
                                UseRawMviValue = false,
                                PatientIdentifier = "600088657",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "500",
                                RawValueFromMvi = "600088657^PI^500^USVHA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "88ac5525-2949-45e8-8517-bfb517be4bbe",
                                UseRawMviValue = false,
                                PatientIdentifier = "600088657",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200CORP",
                                RawValueFromMvi = "600088657^PI^200CORP^USVHA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "d1ea6af7-72c1-423a-a541-24fbe8fe2b2b",
                                UseRawMviValue = false,
                                PatientIdentifier = "7172146",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "608",
                                RawValueFromMvi = "7172146^PI^608^USVHA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "ff3a0c8d-e2db-4de3-81ac-486a8f4195df",
                                UseRawMviValue = false,
                                PatientIdentifier = "100002049",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "553",
                                RawValueFromMvi = "100002049^PI^553^USVHA^A",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            },
                            new UnattendedSearchRequest
                            {
                                MessageId = "28867a22-c9d6-49c7-860e-5488a8dd8cb5",
                                UseRawMviValue = false,
                                PatientIdentifier = "0000001008591266V400774000000",
                                IdentifierType = "PI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200ESR",
                                RawValueFromMvi = "0000001008591266V400774000000^PI^200ESR^USVHA^PCE",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            }
                        },
                        RecordSource = "MVI"
                    }
                },
                ExceptionOccured = false,
                Message = "Your search in MVI found 1 matching record(s).",
                Acknowledgement = new Acknowledgement()
                {
                    TypeCode = "AA",
                    TargetMessage = "200CCVT-VRM.Integration.Servicebus.MVI.Services.TS-e600ef0e-0661-462f-b0ea-0a0f86874e26",
                },
                QueryAcknowledgement = new QueryAcknowledgement()
                {
                    QueryResponseCode = "OK",
                    ResultCurrentQuantity = "1"
                },
                FetchMessageProcessType = MessageProcessType.Local
            };
        }

        /// <summary>
        /// Create Fake Success with one result.
        /// </summary>
        /// <param name="state">State.</param>
        /// <returns>RetrieveOrSearchPersonResponse.</returns>
        public static RetrieveOrSearchPersonResponse CreateFakeSuccess1Result(PersonSearchStateObject state)
        {
            // TODO - parameterize the fake result based on user entry
            return new RetrieveOrSearchPersonResponse
            {
                MessageId = "88047127-5f3d-4090-8e93-1104a0384fdf",
                Person = new[] 
                { 
                    new PatientPerson
                    {
                        SocialSecurityNumber = state.Ss,
                        PhoneNumber = "(603)555-1111",
                        GenderCode = "M",
                        BirthDate = state.BirthDate,
                        Address = new PatientAddress
                        {
                            Use = AddressUse.Unspecified,
                            Country = "USA"
                        },
                        NameList = new[]
                        {
                            new Name
                            {
                                GivenName = state.FirstName,
                                FamilyName = state.FamilyName,
                                MiddleName = "David",
                                NameSuffix = "III",
                                Use = NameUse.Unspecified,
                                NameType = "Legal"
                            },
                            new Name
                            {
                                GivenName = "NAME",
                                FamilyName = "ALIAS",
                                MiddleName = "ONE",
                                Use = NameUse.Unspecified,
                                NameType = "Alias"
                            },
                            new Name
                            {
                                GivenName = "NAME",
                                FamilyName = "ALIAS",
                                MiddleName = "TWO",
                                Use = NameUse.Unspecified,
                                NameType = "Alias"
                            },                        
                            new Name
                            {
                                FamilyName = "NONAME",
                                Use = NameUse.Unspecified,
                                NameType = "Maiden"
                            },
                        },
                        CorrespondingIdList = new[]
                        { 
                            new UnattendedSearchRequest
                            {
                                MessageId = "bc748648-e7a9-49c4-97f6-5ac8cc433792",
                                UseRawMviValue = false,
                                PatientIdentifier = "1008520256V373999",
                                IdentifierType = "NI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200M",
                                RawValueFromMvi = "1008520256V373999^NI^200M^USVHA^P",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            }
                        },
                        RecordSource = "MVI"
                    }
                },
                ExceptionOccured = false,
                Message = "Your search in MVI found 1 matching record(s).",
                Acknowledgement = new Acknowledgement
                {
                    TypeCode = "AA",
                    TargetMessage = "200CCVT-VRM.Integration.Servicebus.MVI.Services.TS-ab0d95ef-4166-4286-828c-c595071a256f",
                    AcknowledgementDetails = new[]
                    {
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                Code = "132",
                                CodeSystemName = "MVI",
                                DisplayName = "IMT"
                            },
                            Text = "Identity Match Threshold"
                        },
                        new AcknowledgementDetail(){
                            Code = new AcknowledgementDetailCode
                            {
                                Code = "120",
                                CodeSystemName = "MVI",
                                DisplayName = "PDT"
                            },
                            Text = "Potential Duplicate Threshold"
                        }
                    }
                },
                QueryAcknowledgement = new QueryAcknowledgement
                {
                    QueryResponseCode = "OK",
                    ResultCurrentQuantity = "1"
                },
                FetchMessageProcessType = MessageProcessType.Local
            };
        }

        /// <summary>
        /// Create Fake success with many results.
        /// </summary>
        /// <param name="state">State.</param>
        /// <returns>RetrieveOrSearchPersonResponse</returns>
        public static RetrieveOrSearchPersonResponse CreateFakeSuccessManyResults(PersonSearchStateObject state)
        {
            // TODO - parameterize the fake result based on user entry
            return new RetrieveOrSearchPersonResponse
            {
                MessageId = "88047127-5f3d-4090-8e93-1104a0384fdf",
                Person = new[] 
                { 
                    new PatientPerson
                    {
                        SocialSecurityNumber = state.Ss ?? "239948835",
                        PhoneNumber = "(603)555-1111",
                        GenderCode = "M",
                        BirthDate = state.BirthDate ?? "19491109",
                        Address = new PatientAddress
                        {
                            Use = AddressUse.Unspecified,
                            Country = "USA"
                        },
                        NameList = new[]
                        {
                            new Name
                            {
                                GivenName = state.FirstName ?? "TYLER",
                                FamilyName = state.FamilyName ?? "NOHEC",
                                MiddleName = state.MiddleName ?? "David",
                                NameSuffix = "",
                                Use = NameUse.Unspecified,
                                NameType = "Legal"
                            },
                            new Name
                            {
                                FamilyName = "White",
                                Use = NameUse.Unspecified,
                                NameType = "Maiden"
                            },
                        },
                        CorrespondingIdList = new[]
                        { 
                            new UnattendedSearchRequest
                            {
                                MessageId = "adfef043-1caf-4bcc-beb8-6fe8655f767d",
                                UseRawMviValue = false,
                                PatientIdentifier = "1008593449V179009",
                                IdentifierType = "NI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200M",
                                RawValueFromMvi = "1008593449V179009^NI^200M^USVHA^P",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            }
                        },
                        RecordSource = "MVI"
                    },
                    new PatientPerson
                    {
                        SocialSecurityNumber = state.Ss ?? "239948835",
                        GenderCode = "M",
                        BirthDate = state.BirthDate ?? "19491109",
                        Address = new PatientAddress
                        {
                            Use = AddressUse.Unspecified,
                            Country = "USA"
                        },
                        NameList = new[]
                        {
                            new Name
                            {
                                GivenName = state.FirstName ?? "TYLER",
                                FamilyName = state.FamilyName ?? "NOHEC",
                                MiddleName = state.MiddleName ?? "David",
                                NameSuffix = "",
                                Use = NameUse.Unspecified,
                                NameType = "Legal"
                            },
                            new Name
                            {
                                FamilyName = "White",
                                Use = NameUse.Unspecified,
                                NameType = "Maiden"
                            },
                        },
                        CorrespondingIdList = new[]
                        { 
                            new UnattendedSearchRequest
                            {
                                MessageId = "adfef043-1caf-4bcc-beb8-6fe8655f767d",
                                UseRawMviValue = false,
                                PatientIdentifier = "1008696589V053728",
                                IdentifierType = "NI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200M",
                                RawValueFromMvi = "1008696589V053728^NI^200M^USVHA^P",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            }
                        },
                        RecordSource = "MVI"
                    },
                    new PatientPerson
                    {
                        SocialSecurityNumber = state.Ss ?? "239948835",
                        GenderCode = "M",
                        BirthDate = state.BirthDate ?? "19491109",
                        Address = new PatientAddress
                        {
                            Use = AddressUse.Unspecified,
                            Country = "USA"
                        },
                        NameList = new[]
                        {
                            new Name
                            {
                                GivenName = "TYLER",
                                FamilyName = "NOHEC",
                                MiddleName = "David",
                                NameSuffix = "",
                                Use = NameUse.Unspecified,
                                NameType = "Legal"
                            },
                            new Name
                            {
                                FamilyName = "White",
                                Use = NameUse.Unspecified,
                                NameType = "Maiden"
                            },
                        },
                        CorrespondingIdList = new[]
                        { 
                            new UnattendedSearchRequest
                            {
                                MessageId = "27ff9a53-3ea1-4661-b5a6-ea7f6e415702",
                                UseRawMviValue = false,
                                PatientIdentifier = "1008593447V264311",
                                IdentifierType = "NI",
                                AssigningAuthority = "USVHA",
                                AssigningFacility = "200M",
                                RawValueFromMvi = "1008593447V264311^NI^200M^USVHA^P",
                                UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                                FetchMessageProcessType = MessageProcessType.Local
                            }
                        },
                        RecordSource = "MVI"
                    }
                },
                ExceptionOccured = false,
                Message = "Your search in MVI found 3 matching record(s).",
                Acknowledgement = new Acknowledgement
                {
                    TypeCode = "AA",
                    TargetMessage = "200CCVT-VRM.Integration.Servicebus.MVI.Services.TS-ab0d95ef-4166-4286-828c-c595071a256f",
                    AcknowledgementDetails = new[]
                    {
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                Code = "132",
                                CodeSystemName = "MVI",
                                DisplayName = "IMT"
                            },
                            Text = "Identity Match Threshold"
                        },
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                Code = "120",
                                CodeSystemName = "MVI",
                                DisplayName = "PDT"
                            },
                            Text = "Potential Duplicate Threshold"
                        }
                    }
                },
                QueryAcknowledgement = new QueryAcknowledgement
                {
                    QueryResponseCode = "OK",
                    ResultCurrentQuantity = "1"
                },
                FetchMessageProcessType = MessageProcessType.Local
            };
        }

        /// <summary>
        /// Create Fake Success with no results.
        /// </summary>
        /// <returns>RetrieveOrSearchPersonResponse</returns>
        public static RetrieveOrSearchPersonResponse CreateFakeSuccess0Result()
        {
            // TODO - finish building this response object from the fake response
            return new RetrieveOrSearchPersonResponse
            {
                Acknowledgement = new Acknowledgement
                {
                    TypeCode = "AA",
                    TargetMessage = "200CCVT-VRM.Integration.Servicebus.MVI.Services.TS-814704af-1646-4130-b403-39f6a95d5fc4",
                    AcknowledgementDetails = new[]
                    {
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                Code = "132",
                                CodeSystemName = "MVI",
                                DisplayName = "IMT"
                            },
                            Text = "Identity Match Threshold"
                        },
                        new AcknowledgementDetail
                        {
                            Code = new AcknowledgementDetailCode
                            {
                                Code = "120",
                                CodeSystemName = "MVI",
                                DisplayName = "PDT"
                            },
                            Text = "Potential Duplicate Threshold"
                        }
                    }
                    
                },
                QueryAcknowledgement = new QueryAcknowledgement
                {
                    QueryResponseCode = "NF",
                    ResultCurrentQuantity = "0"
                },
            FetchMessageProcessType = MessageProcessType.Local
            };
        }

        /// <summary>
        /// Create Fake Selected Person Response.
        /// </summary>
        /// <param name="state">State.</param>
        /// <returns>CorrespondingIdsResponse.</returns>
        public static CorrespondingIdsResponse FakeSelectedPersonResponse(GetPersonIdentifiersStateObject state)
        {
            return new CorrespondingIdsResponse
            {
                CorrespondingIdList = new[]
                {
                    new UnattendedSearchRequest
                    {
                        MessageId = "5758c79d-7932-4e99-90b4-81dbd09bf73b",
                        UseRawMviValue = false,
                        PatientIdentifier = "1001179424V492193",
                        IdentifierType = "NI",
                        AssigningFacility = "200M",
                        AssigningAuthority = "USVHA",
                        RawValueFromMvi = "1001179424V492193^NI^200M^USVHA^P",
                        UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                        FetchMessageProcessType = MessageProcessType.Local
                    },
                    new UnattendedSearchRequest
                    {
                        MessageId = "2a05c76f-e32d-41cc-92b0-9903a437151a",
                        UseRawMviValue = false,
                        PatientIdentifier = "123456",
                        IdentifierType = "NI",
                        AssigningFacility = "200BOSS",
                        AssigningAuthority = "USNCA",
                        RawValueFromMvi = "123456^NI^200BOSS^USNCA^A",
                        UserId = new Guid("00000000-0000-0000-0000-000000000000"),
                        FetchMessageProcessType = MessageProcessType.Local
                    }
                },
                ExceptionOccured = false,
                OrganizationName = state.OrganizationName,
                FetchMessageProcessType = MessageProcessType.Local,
                UserId = new Guid("01c13ce7-8fa8-e411-acdb-00155d5575e0"),
                Acknowledgement = new Acknowledgement
                {
                    TypeCode = "AA",
                    TargetMessage = "200CCVT-VRM.Integration.Servicebus.MVI.Services.TS-c8909264-e536-45c4-bb81-60eb013f1e65"
                },
                QueryAcknowledgement = new QueryAcknowledgement
                {
                    QueryResponseCode = "OK"
                }
            };
        }

        /// <summary>
        /// Get Address Object.
        /// </summary>
        /// <param name="address">Address.</param>
        /// <returns>PatientAddress.</returns>
        internal static PatientAddress GetAddressObject(string address)
        {
            if (string.IsNullOrEmpty(address)) return null;
            
            // Address is of format (AL|CY|ST|ZP|CN)
            var names = address.Split('|');
            return new PatientAddress
            {
                StreetAddressLine = (names.Length >= 1) ? names[0] : string.Empty,
                City = (names.Length >= 2) ? names[1] : string.Empty,
                State = (names.Length >= 3) ? names[2] : string.Empty,
                PostalCode = (names.Length >= 4) ? names[3] : string.Empty,
                Country = (names.Length >= 5) ? names[4] : string.Empty
            };
        }

        /// <summary>
        /// Get Name Object.
        /// </summary>
        /// <param name="fullName">Full Name.</param>
        /// <returns>Name.</returns>
        internal static Name GetNameObject(string fullName)
        {
            if (string.IsNullOrEmpty(fullName)) return null;
            
            // Full name is of format (LN|FN|MN|SF|PF)
            var names = fullName.Split('|');
            return new Name
            {
                FamilyName = (names.Length >= 1) ? names[0] : string.Empty,
                GivenName = (names.Length >= 2) ? names[1] : string.Empty,
                MiddleName = (names.Length >= 3) ? names[2] : string.Empty,
                NameSuffix = (names.Length >= 4) ? names[3] : string.Empty,
                NamePrefix = (names.Length >= 5) ? names[4] : string.Empty
            };
        }

        /// <summary>
        /// Get Date.
        /// </summary>
        /// <param name="dateString">Date string.</param>
        /// <returns>DateTime.</returns>
        internal static DateTime? GetDate(string dateString)
        {
            if (string.IsNullOrEmpty(dateString)) return null;

            if (dateString.Length < 8) return null;

            var sYear = dateString.Substring(0, 4);
            var sMonth = dateString.Substring(4, 2);
            var sDay = dateString.Substring(6, 2);
            var sDate = string.Format("{0}/{1}/{2}", sMonth, sDay, sYear);

            DateTime date;
            if (DateTime.TryParse(sDate, out date)) return date;

            return null;
        }
    }
}