using MCSShared;
using MCSUtilities2011;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Metadata;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Shared
{
    // TODO (Joseph) - Lot of code. Consider refactor if possible

    /// <summary>
    /// Static class of reusable methods to facilitate integration plugin development.
    /// </summary>
    public static class IntegrationPluginHelpers
    {
        public const string VimtServerDown = "Error - VIMT is likely unavailable. Details of the error are as follows: {0}";

        /// <summary>
        ///  Update Service Appointment's Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="serviceAppointmentId">Service Appointment Id.</param>
        /// <param name="statusCode">Service Appointment Status.</param>
        public static void UpdateServiceAppointmentStatus(IOrganizationService organizationService, Guid serviceAppointmentId, serviceappointment_statuscode statusCode)
        {
            var sa = organizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointmentId, new ColumnSet(true)).ToEntity<VA.TMP.DataModel.ServiceAppointment>();
            ChangeEntityStatus(organizationService, sa, (int)statusCode, true);
        }

        /// <summary>
        ///  Update Service Appointment's Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="vodId">Service Appointment Id.</param>
        /// <param name="statusCode">Service Appointment Status.</param>
        public static void UpdateVodStatus(IOrganizationService organizationService, Guid vodId, cvt_vod_statuscode statusCode)
        {
            var vod = organizationService.Retrieve(DataModel.cvt_vod.EntityLogicalName, vodId, new ColumnSet(true)).ToEntity<VA.TMP.DataModel.cvt_vod>();
            ChangeEntityStatus(organizationService, vod, (int)statusCode, true);
        }

        public static void UpdateAppointment(IOrganizationService organizationService, Guid appointmentId, Appointmentcvt_IntegrationBookingStatus status)
        {
            var appointmentToUpdate = new DataModel.Appointment
            {
                Id = appointmentId,
                cvt_IntegrationBookingStatus = new OptionSetValue((int)status)
            };
            organizationService.Update(appointmentToUpdate);
        }

        /// <summary>
        /// Update Service Appointment to Scheduled status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="serviceAppointmentId">Service Appointment Id.</param>
        public static void UpdateServiceAppointmentToScheduledStatus(IOrganizationService organizationService, Guid serviceAppointmentId)
        {
            var serviceAppointment = new DataModel.ServiceAppointment
            {
                Id = serviceAppointmentId,
                StatusCode = new OptionSetValue((int)serviceappointment_statuscode.ReservedScheduled)
            };
            organizationService.Update(serviceAppointment);
        }

        /// <summary>
        /// Update Integration Result to ErrorRetrySuccess Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="integrationResultId">Integration Result Id.</param>
        public static void UpdateIntegrationResultToErrorRetrySuccessStatus(IOrganizationService organizationService, Guid integrationResultId)
        {
            var updateIntegrationResult = new mcs_integrationresult
            {
                Id = integrationResultId,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.ErrorRetrySuccess),
                mcs_retry = false
            };
            organizationService.Update(updateIntegrationResult);
        }

        /// <summary>
        /// Serializes an instance of a class to a string.
        /// </summary>
        /// <typeparam name="T">The Type of class.</typeparam>
        /// <param name="classInstance">The instance of the class.</param>
        /// <returns>Serialized instance of class.</returns>
        public static string SerializeInstance<T>(T classInstance)
        {
            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(T));
                serializer.Serialize(stream, classInstance);

                return Encoding.ASCII.GetString(stream.ToArray());
            }
        }

        /// <summary>
        /// Invoke SendReceive method using .NET reflection to send the request to VIMT.
        /// </summary>
        /// <param name="requestType">Request Type.</param>
        /// <param name="responseType">Response Type.</param>
        /// <param name="serializedRequest">Serialized Request.</param>
        /// <param name="registryMessage">Message Registry Name.</param>
        /// <param name="vimtBaseUri">VIMT base URI.</param>
        /// <returns></returns>
        public static object InvokeVimtDynamically(Type requestType, Type responseType, string serializedRequest, string registryMessage, string vimtBaseUri)
        {
            var serializer = new XmlSerializer(requestType);

            using (TextReader reader = new StringReader(serializedRequest))
            {
                var result = serializer.Deserialize(reader);
                var method = typeof(Utility).GetMethod("SendReceive");
                var generic = method.MakeGenericMethod(responseType);
                var response = generic.Invoke(null, new[] { new Uri(vimtBaseUri), registryMessage, result, null });

                var responseInstance = response;
                if (responseInstance == null) throw new Exception("The response value was not of the proper type");

                return responseInstance;
            }
        }

        /// <summary>
        /// Create Integration Result on successful completion of VIMT call.
        /// </summary>
        /// <param name="integrationResultName">>Name of the Integration Result.</param>
        /// <param name="exceptionOccured">Whether an exception occured.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="serviceAppointId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        public static Guid CreateIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid serviceAppointId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_integrationrequest = integrationRequest,
                mcs_vimtresponse = vimtResponse,
                mcs_status = exceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointId)
            };
            var id = organizationService.Create(integrationResult);

            if (exceptionOccured)
            {
                UpdateServiceAppointmentStatus(organizationService, serviceAppointId, serviceappointment_statuscode.InterfaceVIMTFailure);
            }

            return id;
        }

        /// <summary>
        /// Create Integration Result on successful completion of VIMT call.
        /// </summary>
        /// <param name="integrationResultName">>Name of the Integration Result.</param>
        /// <param name="exceptionOccured">Whether an exception occured.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="vodId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        public static Guid CreateVodIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid vodId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_integrationrequest = integrationRequest,
                mcs_vimtresponse = vimtResponse,
                mcs_status = exceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                cvt_vod = new EntityReference(DataModel.cvt_vod.EntityLogicalName, vodId)
            };
            var id = organizationService.Create(integrationResult);

            if (exceptionOccured)
            {
                UpdateVodStatus(organizationService, vodId, cvt_vod_statuscode.Failure);
            }

            return id;
        }


        /// <summary>
        /// Creates Integration Result.
        /// </summary>
        /// <param name="integrationResultName">Integration Result Name.</param>
        /// <param name="exceptionOccured">Whether an exception occurred.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Request Message Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Response Message Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="appointmentId">Appointment Id</param>
        /// <param name="organizationService">Organization Service Proxy.</param>
        /// <param name="updateAppointment">If true, updates appointment status (success or failure), if false, only updates status on failure.</param>
        public static Guid CreateAppointmentIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid appointmentId,
            IOrganizationService organizationService,
            bool updateAppointment = true)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_integrationrequest = integrationRequest,
                mcs_vimtresponse = vimtResponse,
                mcs_status = exceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_appointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, appointmentId)
            };
            var id = organizationService.Create(integrationResult);

            if (updateAppointment)
            {
                UpdateAppointment(organizationService, appointmentId, exceptionOccured
                    ? Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure
                    : Appointmentcvt_IntegrationBookingStatus.ReservedScheduled);
            }
            else
            {
                if (exceptionOccured)
                    UpdateAppointment(organizationService, appointmentId, Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure);
            }
            return id;
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Name of the Integration Result.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="serviceAppointId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        /// <param name="isCancelRequest">Whether call is for a Cancel Request.</param>
        public static void CreateIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string vimtRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid serviceAppointId,
            IOrganizationService organizationService,
            bool isCancelRequest = false)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.Error),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointId)
            };
            organizationService.Create(integrationResult);

            if (isCancelRequest) return;

            UpdateServiceAppointmentStatus(organizationService, serviceAppointId, serviceappointment_statuscode.InterfaceVIMTFailure);
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Name of the Integration Result.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="vodId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        /// <param name="isCancelRequest">Whether call is for a Cancel Request.</param>
        public static void CreateVodIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string vimtRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid vodId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.Error),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                cvt_vod = new EntityReference(DataModel.cvt_vod.EntityLogicalName, vodId)
            };
            organizationService.Create(integrationResult);


            UpdateVodStatus(organizationService, vodId, cvt_vod_statuscode.Failure);
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Integration Result Name.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Request Message Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Response Message Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="appointmentId">Appointment Id.</param>
        /// <param name="organizationService">Organization Service Proxy.</param>
        public static void CreateAppointmentIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string vimtRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid appointmentId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.Error),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_appointmentid = new EntityReference(DataModel.Appointment.EntityLogicalName, appointmentId)
            };
            organizationService.Create(integrationResult);

            UpdateAppointment(organizationService, appointmentId, Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure);
        }

        public static void CreateAppointmentIntegrationResult(string integrationResultName,
            string errorMessage,
            string vimtRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid appointmentId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.Error),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_appointmentid = new EntityReference(DataModel.Appointment.EntityLogicalName, appointmentId)
            };

            organizationService.Create(integrationResult);

        }

        public static int WriteVistaResults(WriteResults results, Guid parentRecordId, Guid apptId, Guid saId, string message, IOrganizationService organizationService, MCSLogger logger, int status = 0)
        {
            var failures = 0;
            foreach (var result in results.WriteResult)
            {
                CreateOrUpdateWriteResult(result, apptId, saId, parentRecordId, organizationService, logger);

                if (result.VistaStatus == VistaStatus.FAILED_TO_BOOK || result.VistaStatus == VistaStatus.FAILED_TO_CANCEL)
                {
                    failures++;
                    logger.WriteDebugMessage(string.Format("Failure in Cancel or Booking for {1}, {0} at {2}", result.Name.FirstName, result.Name.LastName, result.FacilityName));
                }
            }
            if (message.ToLower() == "book")
                return ReturnBookStatusFromWriteResults(failures, results.WriteResult.Length);
            if (message.ToLower() == "cancel")
                return ReturnCancelStatusFromWriteResults(failures, results.WriteResult.Length, status);
            throw new InvalidPluginExecutionException("unknown operation for WriteVistaResults");
        }

        private static void CreateOrUpdateWriteResult(WriteResult result, Guid apptId, Guid saId, Guid IntegrationResultId, IOrganizationService organizationService, MCSLogger logger)
        {
            logger.WriteDebugMessage("Creating/Updating Write Result");
            cvt_vistaintegrationresult crmResult = null;
            var reason = result.Reason;
            if (!String.IsNullOrEmpty(reason) && reason.Length > 4000)
                reason = result.Reason.Substring(0, 3999) + "*";

            using (var srv = new Xrm(organizationService))
            {
                //Figure out how to determine which Vista Booking 
                if (saId != Guid.Empty)
                    crmResult = srv.cvt_vistaintegrationresultSet.FirstOrDefault(wr =>
                        wr.cvt_ServiceActivity.Id == saId
                        && wr.cvt_PersonId == result.PersonId
                        && wr.cvt_FacilityCode == result.FacilityCode
                        && wr.cvt_ClinicIEN == result.ClinicIen);
                else if (apptId != Guid.Empty)
                    crmResult = srv.cvt_vistaintegrationresultSet.FirstOrDefault(wr =>
                        wr.cvt_Appointment.Id == apptId
                        && wr.cvt_PersonId == result.PersonId
                        && wr.cvt_FacilityCode == result.FacilityCode
                        && wr.cvt_ClinicIEN == result.ClinicIen);
            }
            if (crmResult == null)
            {
                logger.WriteDebugMessage("No Write Result found, creating new result");
                var record = new cvt_vistaintegrationresult
                {
                    cvt_ClinicIEN = result.ClinicIen,
                    cvt_ClinicName = result.ClinicName,
                    cvt_DateTime = result.DateTime,
                    cvt_FacilityCode = result.FacilityCode,
                    cvt_FacilityName = result.FacilityName,
                    cvt_FirstName = result.Name.FirstName,
                    cvt_LastName = result.Name.LastName,
                    cvt_PersonId = result.PersonId,
                    cvt_Reason = reason,
                    cvt_VistAStatus = result.VistaStatus.ToString(),
                    cvt_ParentResult = new EntityReference { Id = IntegrationResultId, LogicalName = mcs_integrationresult.EntityLogicalName },
                    cvt_name = string.Format("{0} {1} - at {2} ({3})", result.Name.FirstName, result.Name.LastName, result.FacilityName, result.FacilityCode)
                };
                if (saId != Guid.Empty) 
                    record.cvt_ServiceActivity = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, saId);

                if (apptId != Guid.Empty)
                    record.cvt_Appointment = new EntityReference(DataModel.Appointment.EntityLogicalName, apptId);
                try
                {
                    organizationService.Create(record);
                }
                catch(Exception ex)
                {
                    logger.WriteToFile("Unable to create Vista Integration Result.  Error: " + CvtHelper.BuildExceptionMessage(ex));
                }
            }
            else
            {
                logger.WriteDebugMessage("Found Write Result, updating Vista Integration record with id: " + crmResult.Id);
                if (crmResult.cvt_VistAStatus != result.VistaStatus.ToString())
                {
                    if ((crmResult.cvt_VistAStatus == VistaStatus.BOOKED.ToString() && result.VistaStatus.ToString() == VistaStatus.FAILED_TO_BOOK.ToString()) || 
                        (crmResult.cvt_VistAStatus == VistaStatus.CANCELLED.ToString() && result.VistaStatus.ToString() == VistaStatus.FAILED_TO_CANCEL.ToString()) )
                    {
                        logger.WriteToFile(string.Format("Write Result for {2} not updated, old status: {0}, new status: {1}", crmResult.cvt_VistAStatus, result.VistaStatus, crmResult.cvt_name));
                    }
                    else
                    {
                        var status = result.VistaStatus.ToString();
                        var updateResult = new cvt_vistaintegrationresult
                        {
                            Id = crmResult.Id,
                            cvt_VistAStatus = status,
                            cvt_Reason = reason
                        };
                        try
                        {
                            organizationService.Update(updateResult);
                        }
                        catch (Exception ex)
                        {
                            logger.WriteToFile(string.Format("Unable to update Vista Integration Result for {0}.  Error: {1}", crmResult.cvt_name, CvtHelper.BuildExceptionMessage(ex)));
                        }
                    }
                }
                else
                {
                    logger.WriteDebugMessage(string.Format("Status of Vista Integration Result did not change, skipping update of {0}", crmResult.cvt_name));
                }
            }
        }

        public static int ReturnBookStatusFromWriteResults(int failures, int writeResultCount)
        {
            if (failures == 0)
                return (int)serviceappointment_statuscode.ReservedScheduled;
            if (failures > 0 && failures < writeResultCount)
                return (int)serviceappointment_statuscode.PartialVistaFailure;
            if (failures > 0 && failures == writeResultCount)
                return (int)serviceappointment_statuscode.VistaFailure;
            throw new InvalidPluginExecutionException(string.Format("Unexpected number of Vista Books/Cancels.  {0} Failures, {1} Write Results(attempts)", failures, writeResultCount));
        }

        public static int ReturnCancelStatusFromWriteResults(int failures, int writeResultCount, int saStatus)
        {
            var status = 0;
            if (failures == 0)
                status = saStatus;
            else if (failures > 0 && failures < writeResultCount)
                status = (int)serviceappointment_statuscode.CancelFailure;
            else if (failures > 0 && failures == writeResultCount)
                status = (int)serviceappointment_statuscode.CancelFailure;
            else
                throw new InvalidPluginExecutionException(string.Format("Unexpected number of Vista Cancels.  {0} Failures, {1} Write Results(attempts)", failures, writeResultCount));
            return status;
        }

        internal static Guid GetPatIdFromIcn(string icn, IOrganizationService organizationService)
        {
            Guid id;
            using (var srv = new Xrm(organizationService))
            {
                var personIdentifier = srv.mcs_personidentifiersSet.FirstOrDefault(i => i.mcs_identifier == icn);
                if (personIdentifier != null && personIdentifier.mcs_patient != null)
                {
                    var pat = srv.ContactSet.FirstOrDefault(c => c.Id == personIdentifier.mcs_patient.Id);

                    if (pat == null)
                        throw new InvalidPluginExecutionException(string.Format("Person Identifier {0} is not linked to a patient.  ", icn));
                    id = pat.Id;
                }
                else
                {
                    throw new InvalidPluginExecutionException("No patient could be found with ICN = " + icn);
                }
            }
            return id;
        }

        public static List<Guid> GetPatientsFromActivityPartyList(List<ActivityParty> currentPatientsApList)
        {
            var patients = new List<Guid>();
            foreach (var ap in currentPatientsApList)
            {
                patients.Add(ap.PartyId.Id);
            }
            return patients;
        }

        /// <summary>
        /// Builds an error message from an exception recursively.
        /// </summary>
        /// <param name="ex">Exception.</param>
        /// <returns>Exception message.</returns>
        public static string BuildErrorMessage(Exception ex)
        {
            var errorMessage = string.Empty;

            if (ex.InnerException == null) return errorMessage;

            errorMessage += string.Format("\n\n{0}\n", ex.InnerException.Message);
            errorMessage += BuildErrorMessage(ex.InnerException);

            return errorMessage;
        }

        public static void ChangeEntityStatus(IOrganizationService service, Entity entity, int statusValue, bool useUpdateNotSsr = false)
        {
            var attributeRequest = new RetrieveAttributeRequest
            {
                EntityLogicalName = entity.LogicalName,
                LogicalName = "statuscode",
                RetrieveAsIfPublished = true
            };
            var attributeResponse = (RetrieveAttributeResponse)service.Execute(attributeRequest);
            var statusMetadata = (StatusAttributeMetadata)attributeResponse.AttributeMetadata;
            var status = (StatusOptionMetadata)statusMetadata.OptionSet.Options.FirstOrDefault(option => option.Value == statusValue);
            if (status == null)
                throw new InvalidPluginExecutionException(string.Format("{0} is an invalid status for Changing Entity State of {1} with id {2}", statusValue, entity.LogicalName, entity.Id));
            if (status.State == ((OptionSetValue)(entity.Attributes["statecode"])).Value && useUpdateNotSsr)
            {
                var updateEntity = new Entity { Id = entity.Id, LogicalName = entity.LogicalName };
                if (status.Value == null)
                    throw new InvalidPluginExecutionException(string.Format("Invalid Status Reason List: unable to Change Status for record: {0}; id={1}", entity.LogicalName, entity.Id));
                updateEntity.Attributes.Add(new KeyValuePair<string, object>("statuscode", new OptionSetValue(status.Value.Value)));
                try
                {
                    service.Update(updateEntity);
                }
                catch (Exception ex)
                {
                    throw new InvalidPluginExecutionException(string.Format("Failed to Update status of {0} with id {1}.  Error: {2}", entity.LogicalName, entity.Id, ex.Message));
                }
            }
            else
            {
                var stateRequest = new SetStateRequest
                {
                    State = new OptionSetValue((int)status.State),
                    Status = new OptionSetValue((int)status.Value),
                    EntityMoniker = new EntityReference
                    {
                        LogicalName = entity.LogicalName,
                        Id = entity.Id
                    }
                };
                try
                {
                    service.Execute(stateRequest);
                }
                catch (Exception ex)
                {
                    throw new InvalidPluginExecutionException(string.Format("Set State Request failed for {0} with id {1}.  Error: {2}", entity.LogicalName, entity.Id, ex.Message));
                }
            }
        }
    }
}