﻿using MCSShared;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace VA.TMP.CRM
{
    public class EmailAutomationCreatePostStageRunner : PluginRunner
    {        
        public EmailAutomationCreatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider) { }
        //Declare global variables
        string customMessage;

        #region Implementation
        /// <summary>
        /// Called by PluginRunner - Decide which email to send out (aka which branch of the plugin to run)
        /// </summary>
        public override void Execute()
        {

            var Name = PrimaryEntity.Attributes["cvt_name"].ToString();

            if (Name.Contains("System Automated"))
            {
                Logger.WriteDebugMessage("Starting PPE Review automatic emails");
                PPEReviewSummary();
                //Should we try to send all PPEFeedbacks through this process as well?
            }
        }

        #endregion

        #region Commonly Used Functions
        internal List<ActivityParty> RetrieveFacilityTeamMembers(Email email, Guid TeamId, out string unapprovedUsers)
        {
            Logger.WriteDebugMessage("Starting RetrieveFacilityTeamMembers function");
            unapprovedUsers = "";

            using (var srv = new Xrm(OrganizationService))
            {
                var teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == TeamId).ToList());
                var recipientList = new List<ActivityParty>();

                foreach (var member in teamMembers)
                {
                    var user = srv.SystemUserSet.FirstOrDefault(u => u.Id == member.SystemUserId);

                    if (user == null)
                    {
                        Logger.WriteDebugMessage("Team member is not a user.");
                        break;

                    }
                    else
                    {
                        Logger.WriteDebugMessage("Checking for the User's Email Address.");
                        if ((!String.IsNullOrEmpty(user.InternalEMailAddress)))
                        {
                            var party = new ActivityParty()
                            {
                                ActivityId = new EntityReference(email.LogicalName, email.Id),
                                PartyId = new EntityReference(SystemUser.EntityLogicalName, user.Id)
                            };
                            recipientList.Add(party);
                        }
                        else
                        {
                            if (unapprovedUsers != "")
                                unapprovedUsers += "; ";

                            unapprovedUsers += user.FullName;
                        }
                    }
                }
                return recipientList;
            }
        }
        #endregion

        #region PPE Review/Feedback
        internal void PPEReviewSummary()
        {
            Logger.setMethod = "FindOpenPPEReviews";
            Logger.WriteDebugMessage("Starting");
            string error = "";
            int successCount = 0;

            using (var srv = new Xrm(OrganizationService))
            {
                var facilitiesWithOpenPPEReview = (from f in srv.mcs_facilitySet
                                                   join r in srv.cvt_ppereviewSet on f.Id equals r.cvt_facility.Id
                                                   where r.cvt_outstandingppefeedbacks.Value > 0
                                                   select new
                                                   {
                                                       r.cvt_facility,
                                                       r.cvt_specialty
                                                   }).Distinct().ToList();
                Logger.WriteDebugMessage("Checking for Facilities with Open PPEReviews.");
                if (facilitiesWithOpenPPEReview == null)
                {
                    UpdateEmailAutomation(PrimaryEntity.Id, "No Open PPE Reviews. No emails sent.", "");
                    return;
                }
                Logger.WriteDebugMessage("Found Facility/Specialty Combinations: " + facilitiesWithOpenPPEReview.Count);
                foreach (var item in facilitiesWithOpenPPEReview)
                {
                    var team = srv.TeamSet.FirstOrDefault(t => t.cvt_Type.Value == (int)Teamcvt_Type.ServiceChief && t.cvt_ServiceType.Id == item.cvt_specialty.Id && t.cvt_Facility.Id == item.cvt_facility.Id);

                    if (team == null)
                    {
                        error += String.Format("Could Not Find {0} Service Chief Team for {1}.", item.cvt_specialty.Name, item.cvt_facility.Name);
                        break;
                    }
                    Logger.WriteDebugMessage("Found SC Team at the Facility.");
                    
                    //Create the emails
                    Email newEmail = new Email()
                    {
                        RegardingObjectId = new EntityReference(cvt_emailautomation.EntityLogicalName, PrimaryEntity.Id),
                        From = CvtHelper.GetWorkflowOwner("Privileging: PPE Submitted", OrganizationService),
                        Subject = "Daily PPE feedback tracking summary"
                    };
                    Guid newEmailID = OrganizationService.Create(newEmail);
                    Logger.WriteDebugMessage("Created the email object.");

                    Email newPPEReviewSummary = new Email()
                    {
                        Id = newEmailID,
                        To = null
                    };
                    var unapprovedUsers = "";
                    newPPEReviewSummary.To = RetrieveFacilityTeamMembers(newPPEReviewSummary, team.Id, out unapprovedUsers);

                    
                    if (unapprovedUsers != "")
                        error += String.Format("No Email addresses for the following users {0} on the team: {1}", unapprovedUsers, team.Name);
                    else
                        Logger.WriteDebugMessage("All Users listed on the team have approved emails.");

                    Logger.WriteDebugMessage("Count for " + item.cvt_facility.Name + ", TO Count: " + newPPEReviewSummary.To.Count().ToString());
                    if (newPPEReviewSummary.To.Count() == 0)
                    {
                        error += String.Format("No Team members for {0}", team.Name);
                        break;
                    }
                   
                    //Edit the E-mail body with the summary grid
                    var etc = CvtHelper.GetEntityTypeCode(OrganizationService, cvt_ppereview.EntityLogicalName);
                    string servernameAndOrgname = CvtHelper.getServerURL(OrganizationService);
                    string url = servernameAndOrgname + "/userDefined/edit.aspx?etc=" + etc + "&id=";

                    Logger.WriteDebugMessage("Building the email body.");
                    customMessage = "<table style = 'width:100%'><tr>";
                    //Table Headings
                    customMessage += "<th>Provider Name</th>";
                    customMessage += "<th># of Received Feedback</th>";
                    customMessage += "<th># of Requested Feedback</th>";
                    customMessage += "<th>Initiation Date</th>";
                    customMessage += "<th>Due Date</th>";
                    customMessage += "<th>Requests Escalated</th>";
                    customMessage += "<th>PPE Review record</th>";
                    customMessage += "</tr >";

                    var openReviews = srv.cvt_ppereviewSet.Where(r => r.cvt_facility.Id == item.cvt_facility.Id && r.cvt_specialty.Id == item.cvt_specialty.Id && r.cvt_outstandingppefeedbacks != 0 && r.cvt_outstandingppefeedbacks != null && r.cvt_submittedppefeedbacks != null);
                    var count = 0;
                    foreach (cvt_ppereview open in openReviews)
                    {
                        var escalated = (open.cvt_escalated.Value == true) ? "Yes" : "No";
                        customMessage += "<tr>";
                        customMessage += "<th>" + open.cvt_provider.Name + "</th>";
                        customMessage += "<th>" + open.cvt_submittedppefeedbacks + "</th>";
                        customMessage += "<th>" + open.cvt_requestedppefeedbacks + "</th>";
                        customMessage += "<th>" + ((DateTime)open.cvt_initiateddate).ToString("MM/dd/yyyy") + "</th>";
                        customMessage += "<th>" + ((DateTime)open.cvt_duedate).ToString("MM/dd/yyyy") + "</th>";
                        customMessage += "<th>" + escalated + "</th>";
                        customMessage += "<th>" + "<a href=\"" + url + open.Id + "\">View Record</a>" + "</th>"; //URL
                        customMessage += "</tr >";
                        count++;
                    }

                    customMessage += "</table >";
                    customMessage += "<br/><br/>This is an automated notification from the Telehealth Management Platform.";

                    newPPEReviewSummary.Description = customMessage;

                    if (count > 0)
                    {
                        CvtHelper.UpdateSendEmail(newPPEReviewSummary, OrganizationService);
                        successCount++;
                    }
                }

                string summary = "Successfully sent out " + successCount + " PPE Review summaries.";
                //Update the Email Automation record
                UpdateEmailAutomation(PrimaryEntity.Id, summary, error);

            }
        }

        internal void UpdateEmailAutomation(Guid recordId, string summary, string error)
        {
            Logger.WriteDebugMessage("About to update the Email Automation record.");

            var now = DateTime.Today;

            cvt_emailautomation updateRecord = new cvt_emailautomation()
            {
                Id = recordId,
                cvt_summary = summary,
                cvt_errors = error,
                cvt_name = "System Automated Emails " + now.ToString("yyyy/MM/dd")
            };
            OrganizationService.Update(updateRecord);
            Logger.WriteDebugMessage("Updated Email Automation record.");
        }

        #endregion

        #region Implementing additional interface methods
        public override string McsSettingsDebugField
        {
            get { return "cvt_ppereview"; }
        }
        #endregion
    }
}
