﻿using Microsoft.Xrm.Sdk;
using VA.TMP.DataModel;
using VIMT.VideoVisit.Messages;

namespace VA.TMP.Integration.VIMT.VideoVisit.Mappers
{
    /// <summary>
    /// Class to map a Service Appointment to VVS Appointment.
    /// </summary>
    internal class VideoVisitMapper
    {
        private readonly IOrganizationService _organizationService;
        private readonly ServiceAppointment _serviceAppointment;
        private readonly Contact _contact;
        private readonly SystemUser _systemUser;

        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="organizationService">CRM Organization Service.</param>
        /// <param name="serviceAppointment">CRM Service Appointment</param>
        /// <param name="contact">CRM Contact</param>
        /// <param name="systemUser">CRM Sysem User</param>
        public VideoVisitMapper(IOrganizationService organizationService, ServiceAppointment serviceAppointment, Contact contact, SystemUser systemUser)
        {
            _organizationService = organizationService;
            _serviceAppointment = serviceAppointment;
            _contact = contact;
            _systemUser = systemUser;
        }

        /// <summary>
        /// Map a schema Apppointment Initiation data to TMP entities.
        /// </summary>
        /// <returns>Appointment.</returns>
        internal VIMTTmpCreateAppointmentRequestData Map()
        {
            var createAppointmentRequest = new VIMTTmpCreateAppointmentRequestData
            {
                Version = "1.0",
                Id = _serviceAppointment.Id.ToString(),
                Duration = _serviceAppointment.ScheduledDurationMinutes.Value,
                DateTime = _serviceAppointment.ScheduledStart.ToString(),
                SchedulingRequestType = VIMTTmpSchedulingRequestType.OTHER_THAN_NEXT_AVAILABLE_PATIENT_REQUESTED,
                SchedulingRequestTypeSpecified = true,
                Type = VIMTTmpAppointmentType.REGULAR,
                TypeSpecified = true,             
                DesiredDate = _serviceAppointment.ScheduledStart.ToString(),
                DesiredDateSpecified = true
            };

            if (HasBookingNotes(_serviceAppointment))
                createAppointmentRequest.BookingNotes = _serviceAppointment.cvt_SchedulingInstructions;

            var contact = new VIMTTmpPatients
            {
                Id = MappingResolvers.PersonIdentifierResolver(_organizationService, _contact),
                Name = MappingResolvers.PersonNameResolver(_contact),
                ContactInformation =  MappingResolvers.PatientContactInformationResolver(_contact),
                Location = new VIMTTmpLocation { Type = VIMTTmpLocationType.NonVA },
                VirtualMeetingRoom = MappingResolvers.PatientVirtualMeetingRoomResolver(_serviceAppointment)
            };

            var systemUser = new VIMTTmpProviders
            {
                Name = MappingResolvers.ProviderNameResolver(_systemUser),
                ContactInformation = MappingResolvers.ProviderContactInformationResolver(_systemUser),
                Location = MappingResolvers.ProviderLocationResolver(_organizationService, _serviceAppointment, _systemUser),
                VirtualMeetingRoom = MappingResolvers.ProviderVirtualMeetingRoomResolver(_serviceAppointment)
            };

            createAppointmentRequest.Patients = new[] {contact};
            createAppointmentRequest.Providers = new[] {systemUser};
            return createAppointmentRequest;
        }

        /// <summary>
        /// Determine whether the Service Appointment has booking notes.
        /// </summary>
        /// <param name="serviceAppointment">Service Appointment.</param>
        /// <returns>Whether the Service Appointment has booking notes.</returns>
        private static bool HasBookingNotes(ServiceAppointment serviceAppointment)
        {
            return !string.IsNullOrEmpty(serviceAppointment.cvt_SchedulingInstructions);
        }
    }
}