﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System.Linq;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;
using VIMT.VideoVisit.Messages;

namespace VA.TMP.Integration.VIMT.VideoVisit.Mappers
{
    /// <summary>
    /// Default Mapping Resolvers class to hold all Resolver functions. 
    /// </summary>
    internal static class MappingResolvers
    {
        /// <summary>
        /// Person Identifier Resolver Function
        /// </summary>
        /// <param name="organizationService"></param>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpPersonIdentifier PersonIdentifierResolver(IOrganizationService organizationService, Contact source)
        {
            using (var srv = new Xrm(organizationService))
            {
                var identifiers = srv.mcs_personidentifiersSet.Where(x => x.mcs_patient.Id == source.Id).ToList();
                var icn = identifiers.First(x => x.mcs_identifiertype.Value == (int)mcs_personidentifiersmcs_identifiertype.NationalIdentifier_NI && x.mcs_assigningauthority == "USVHA");

                return new VIMTTmpPersonIdentifier { AssigningAuthority = "ICN", UniqueId = icn.mcs_identifier };
            }
        }

        /// <summary>
        /// Person Name Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpPersonName PersonNameResolver(Contact source)
        {
                var firstName = source.FirstName;
                var lastName = source.LastName;
                return new VIMTTmpPersonName { FirstName = firstName, LastName = lastName };         
        }

        /// <summary>
        /// Patient Contact Info Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpContactInformation PatientContactInformationResolver(Contact source)
        {
            var contactInformation = new VIMTTmpContactInformation
            {
                Mobile = source.Telephone1,
                PreferredEmail = source.EMailAddress1
            };

            if (source.cvt_TimeZone == null) return contactInformation;

            contactInformation.TimeZoneSpecified = true;
            contactInformation.TimeZone = source.cvt_TimeZone.Value;

            return contactInformation;
        }

        /// <summary>
        /// Patient Virutal Meeting Room Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpVirtualMeetingRoom PatientVirtualMeetingRoomResolver(ServiceAppointment source)
        {
            return new VIMTTmpVirtualMeetingRoom
            {
                Conference = source.mcs_meetingroomname,
                Pin = source.mcs_patientpin,
                Url = source.mcs_PatientUrl
            };
        }

        /// <summary>
        /// Provider Name Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpPersonName ProviderNameResolver(SystemUser source)
        {
            var firstName = source.FirstName;
            var lastName = source.LastName;
            return new VIMTTmpPersonName { FirstName = firstName, LastName = lastName };
        }

        /// <summary>
        /// Provider Contact Info Resolver Function 
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpContactInformation ProviderContactInformationResolver(SystemUser source)
        {
            var contactInformation = new VIMTTmpContactInformation
            {
                Mobile = source.MobilePhone,
                PreferredEmail = source.InternalEMailAddress
            };

            if (source.cvt_TimeZone == null) return contactInformation;

            contactInformation.TimeZoneSpecified = true;
            contactInformation.TimeZone = source.cvt_TimeZone.Value;

            return contactInformation;
        }

        /// <summary>
        /// Provider Location Resolver Function 
        /// </summary>
        /// <param name="organizationService"></param>
        /// <param name="serviceAppointment"></param>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpLocation ProviderLocationResolver(IOrganizationService organizationService, ServiceAppointment serviceAppointment, SystemUser source)
        {
            using (var srv = new Xrm(organizationService))
            {
                // Look up the Facility Information
                var facility = srv.mcs_facilitySet.First(i => i.Id == source.cvt_facility.Id);
                var siteCode = facility.mcs_StationNumber;
                var facilityName = facility.mcs_name;
                var timeZone = facility.mcs_Timezone ?? 0;

                // Look up the booked resources and find the Vista Clinic               
                var bookedResources = serviceAppointment.Resources.Where(r => r.PartyId.LogicalName == "equipment").ToList();
                var clinicName = "";

                if (bookedResources.Count <= 0) return new VIMTTmpLocation
                {
                    Type = VIMTTmpLocationType.VA,
                    Facility = new VIMTTmpFacility { SiteCode = siteCode, Name = facilityName, TimeZone = timeZone },
                    Clinic = new VIMTTmpClinic { Name = clinicName }
                };

                foreach (var t in bookedResources)
                {
                    var entRef = t.PartyId;
                    var cols = new ColumnSet("name", "cvt_type");
                    var equip = (Equipment)organizationService.Retrieve("equipment", entRef.Id, cols);

                    // Check if this is an equipment record of type Vista Clinic
                    if (equip.cvt_type.Value != (int)mcs_resourcetype.VistaClinic) continue;

                    // Once we find the equipment record with Clinic we will assign value and break from loop.
                    clinicName = equip.Name;
                    break;
                }

                return new VIMTTmpLocation
                {
                    Type = VIMTTmpLocationType.VA,
                    Facility = new VIMTTmpFacility { SiteCode = siteCode, Name = facilityName, TimeZone = timeZone },
                    Clinic = new VIMTTmpClinic { Name = clinicName }
                };
            }
        }
        /// <summary>
        /// Provider Virutal Meeting Room Resolver Function
        /// </summary>
        /// <param name="source"></param>
        /// <returns></returns>
        internal static VIMTTmpVirtualMeetingRoom ProviderVirtualMeetingRoomResolver(ServiceAppointment source)
        {
            return new VIMTTmpVirtualMeetingRoom
            {
                Conference = source.mcs_meetingroomname,
                Pin = source.mcs_providerpin,
                Url = source.mcs_providerurl
            };
        }
    }
}