﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle updating a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentVvsUpdatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentVvsUpdatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            var sa = OrganizationService.Retrieve(DataModel.ServiceAppointment.EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true)).ToEntity<DataModel.ServiceAppointment>();
            if (sa == null) throw new InvalidPluginExecutionException(string.Format("Unable to Find Service Appointment with id {0}", PrimaryEntity.Id));

            using (var context = new Xrm(OrganizationService))
            {
                var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
                if (settings == null) throw new InvalidPluginExecutionException("Active Settings Cannot be Null");
                if (settings.cvt_UseVVS != null && settings.cvt_UseVVS.Value)
                {
                    if (sa.cvt_VMRCompleted == null || !sa.cvt_VMRCompleted.Value) return;

                    var vimtUrlObj = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                    if (vimtUrlObj == null) throw new InvalidPluginExecutionException("VIMT Url cannot be null, please ensure there is a setting called \"VIMT URL\"");

                    VimtUrl = vimtUrlObj.mcs_value;
                    // Ensure this is a Home/Mobile Service Appointment of the correct status.
                    if (sa.cvt_Type == null || !sa.cvt_Type.Value) return;
                    if (sa.StateCode == null || sa.StateCode != ServiceAppointmentState.Scheduled) throw new InvalidPluginExecutionException("Service Appointment is not in Scheduled State.");
                    if (sa.StatusCode == null || sa.StatusCode.Value != (int)serviceappointment_statuscode.Pending) throw new InvalidPluginExecutionException("Service Appointment is not in Pending Status.");

                    // Call Video Visit Service
                    var videoVisitCreateResponseMessage = CreateAndSendVideoVisitService(sa.Id, PluginExecutionContext.InitiatingUserId, PluginExecutionContext.OrganizationName);
                    if (videoVisitCreateResponseMessage == null) return;
                    ProcessVideoVisitCreateResponseMessage(videoVisitCreateResponseMessage);
                }
                else
                {
                    Logger.WriteDebugMessage("Accenture Vyopta Integration turned off, Updating Service Activity Status");
                    IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.ReservedScheduled);
                }
            }
        }

        /// <summary>
        /// Create an instance of the Video Vist Service request and send to VIMT.
        /// </summary>
        /// <returns>VideoVisitCreateResponseMessage.</returns>
        private VideoVisitCreateResponseMessage CreateAndSendVideoVisitService(Guid appointmentId, Guid userId, string organizationName)
        {
            var request = new VideoVisitCreateRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                AppointmentId = appointmentId
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

            try
            {
                var response = Utility.SendReceive<VideoVisitCreateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VideoVisitCreateRequestMessage, request, null);
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Create Video Visit", errorMessage, vimtRequest, typeof(VideoVisitCreateRequestMessage).FullName,
                    typeof(VideoVisitCreateResponseMessage).FullName, MessageRegistry.VideoVisitCreateRequestMessage, PrimaryEntity.Id, OrganizationService);
                Logger.WriteToFile(errorMessage);

                return null;
            }
        }

        /// <summary>
        /// Create an Integration Result.
        /// </summary>
        /// <param name="videoVisitCreateResponseMessage">Video Visit Create Response Message.</param>
        private void ProcessVideoVisitCreateResponseMessage(VideoVisitCreateResponseMessage videoVisitCreateResponseMessage)
        {
            var errorMessage = videoVisitCreateResponseMessage.ExceptionOccured ? videoVisitCreateResponseMessage.ExceptionMessage : null;
            IntegrationPluginHelpers.CreateIntegrationResult("Create Video Visit", videoVisitCreateResponseMessage.ExceptionOccured, errorMessage,
                videoVisitCreateResponseMessage.VimtRequest, videoVisitCreateResponseMessage.SerializedInstance, videoVisitCreateResponseMessage.VimtResponse,
                typeof(VideoVisitCreateRequestMessage).FullName, typeof(VideoVisitCreateResponseMessage).FullName, MessageRegistry.VideoVisitCreateRequestMessage,
                PrimaryEntity.Id, OrganizationService);

            if (videoVisitCreateResponseMessage.ExceptionOccured) return;

            IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, PrimaryEntity.Id, serviceappointment_statuscode.ReservedScheduled);
        }
    }
}