﻿using MCSShared;
using Microsoft.Xrm.Sdk;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.ServiceAppointment
{
    /// <summary>
    ///  CRM Plugin Runner class to handle updating a ServiceAppointment.
    /// </summary>
    public class ServiceAppointmentVmrUpdatePostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public ServiceAppointmentVmrUpdatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            try
            {
                var serviceAppointmentEntity = PrimaryEntity.ToEntity<DataModel.ServiceAppointment>();

                // Only run VVC/VVS if Proxy Add has completed
                if (serviceAppointmentEntity.cvt_ProxyAddCompleted == null || !serviceAppointmentEntity.cvt_ProxyAddCompleted.Value) return;

                Logger.WriteDebugMessage("Proxy Add Completed, beginning VMR Generation");
                using (var context = new Xrm(OrganizationService))
                {
                    // Retrieve the Service Appointment.
                    var serviceAppointment = context.ServiceAppointmentSet.FirstOrDefault(x => x.Id == serviceAppointmentEntity.Id);
                    if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null.");

                    // Check the settings record to make sure we want to Call the Accenture & Vyopta Services
                    var settings = context.mcs_settingSet.FirstOrDefault(x => x.mcs_name == "Active Settings");
                    if (settings == null) throw new InvalidPluginExecutionException("Active Settings cannot be null.");
                    
                    // TODO check for the tablet with a static VMR, if exists, then skip the whole integration
                    if (settings.cvt_accenturevyopta != null && settings.cvt_accenturevyopta.Value && 
                        !CvtHelper.isGfeServiceActivity(serviceAppointment, OrganizationService, Logger))
                    {
                        var vimtUrl = context.mcs_integrationsettingSet.FirstOrDefault(x => x.mcs_name == "VIMT URL");
                        if (vimtUrl == null) throw new InvalidPluginExecutionException("No VIMT URL Integration Setting was found");
                        VimtUrl = vimtUrl.mcs_value;

                        // Ensure this is a Home/Mobile Service Appointment.
                        if (serviceAppointment.cvt_Type == null || !serviceAppointment.cvt_Type.Value) return;
                        if (serviceAppointment.StateCode == null || serviceAppointment.StateCode != ServiceAppointmentState.Scheduled) throw new InvalidPluginExecutionException("Service Appointment is not in Scheduled State.");
                        if (serviceAppointment.StatusCode == null || serviceAppointment.StatusCode.GetHashCode() != (int) serviceappointment_statuscode.Pending) throw new InvalidPluginExecutionException("Service Appointment is not in Pending Status.");                        

                        // Call Service to create the Virtual Meeting Room
                        Logger.WriteDebugMessage("Beginning CreateAndSendVirtualMeetingRoom");
                        var virtualMeetingRoomCreateResponseMessage = CreateAndSendVirtualMeetingRoom(serviceAppointment, PluginExecutionContext.InitiatingUserId, PluginExecutionContext.OrganizationName);
                        Logger.WriteDebugMessage("Finished CreateAndSendVirtualMeetingRoom");
                        if (virtualMeetingRoomCreateResponseMessage == null) return;

                        Logger.WriteDebugMessage("Beginning ProcessVirtualMeetingRoomCreateResponseMessage");
                        ProcessVirtualMeetingRoomCreateResponseMessage(virtualMeetingRoomCreateResponseMessage);
                        Logger.WriteDebugMessage("Finished ProcessVirtualMeetingRoomCreateResponseMessage");
                    }
                    else
                    {
                        // Just Set the Status to Scheduled
                        Logger.WriteDebugMessage("Accenture Vyopta Integration turned off, skipping logic");
                        var updateSa = new DataModel.ServiceAppointment()
                        {

                            Id = serviceAppointmentEntity.Id,
                            cvt_VMRCompleted = true
                        };
                        OrganizationService.Update(updateSa);
                        Logger.WriteDebugMessage("Successfully Triggered VVS");
                    }
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(string.Format("ERROR in ServiceAppointmentCreatePostStageRunner: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)));
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        /// <summary>
        /// Create an instance of the Virtual Meeting Room request and send to VIMT.
        /// </summary>
        /// <returns>VirtualMeetingRoomCreateResponseMessage.</returns>
        private VirtualMeetingRoomCreateResponseMessage CreateAndSendVirtualMeetingRoom(DataModel.ServiceAppointment serviceAppointment, Guid userId, string organizationName)
        {
            // Get the PatientId and ProviderId. Guard verbosely here against null references/objects.
            var bookedPatient = serviceAppointment.Customers.FirstOrDefault(p => p.PartyId.LogicalName == "contact");
            if (bookedPatient == null) throw new InvalidPluginExecutionException("Patient cannot be null.");

            if (bookedPatient.PartyId == null) throw new InvalidPluginExecutionException("The Patient PartyId cannot be null.");
            var patientId = bookedPatient.PartyId.Id;

            var bookedSysUser = serviceAppointment.Resources.FirstOrDefault(r => r.PartyId.LogicalName == "systemuser");
            if (bookedSysUser == null) throw new InvalidPluginExecutionException("Provider cannot be null.");

            if (bookedSysUser.PartyId == null) throw new InvalidPluginExecutionException("The Provider PartyId cannot be null.");
            var providerId = bookedSysUser.PartyId.Id;

            var request = new VirtualMeetingRoomCreateRequestMessage
            {
                LogRequest = true,
                UserId = userId,
                OrganizationName = organizationName,
                AppointmentId = serviceAppointment.Id,
                PatientId = patientId,
                ProviderId = providerId,
            };

            var vimtRequest = IntegrationPluginHelpers.SerializeInstance(request);

            try
            {
                var response = Utility.SendReceive<VirtualMeetingRoomCreateResponseMessage>(new Uri(VimtUrl), MessageRegistry.VirtualMeetingRoomCreateRequestMessage, request, null);
                return response;
            }
            catch (Exception ex)
            {
                var errorMessage = string.Format(IntegrationPluginHelpers.VimtServerDown, ex);
                IntegrationPluginHelpers.CreateIntegrationResultOnVimtFailure("Create Virtual Meeting Room", errorMessage, vimtRequest, typeof(VirtualMeetingRoomCreateRequestMessage).FullName,
                    typeof(VirtualMeetingRoomCreateResponseMessage).FullName, MessageRegistry.VirtualMeetingRoomCreateRequestMessage, PrimaryEntity.Id, OrganizationService);
                Logger.WriteToFile(errorMessage);

                return null;
            }
        }

        /// <summary>
        /// Update the Service Activity and create an Integration Result.
        /// </summary>
        /// <param name="virtualMeetingRoomCreateResponseMessage">Virtual Meeting Room Create Response Message.</param>
        private void ProcessVirtualMeetingRoomCreateResponseMessage(VirtualMeetingRoomCreateResponseMessage virtualMeetingRoomCreateResponseMessage)
        {
            var errorMessage = virtualMeetingRoomCreateResponseMessage.ExceptionOccured ? virtualMeetingRoomCreateResponseMessage.ExceptionMessage : null;
            IntegrationPluginHelpers.CreateIntegrationResult("Create Virtual Meeting Room", virtualMeetingRoomCreateResponseMessage.ExceptionOccured, errorMessage,
                virtualMeetingRoomCreateResponseMessage.VimtRequest, virtualMeetingRoomCreateResponseMessage.SerializedInstance, virtualMeetingRoomCreateResponseMessage.VimtResponse,
                typeof(VirtualMeetingRoomCreateRequestMessage).FullName, typeof(VirtualMeetingRoomCreateResponseMessage).FullName, MessageRegistry.VirtualMeetingRoomCreateRequestMessage,
                PrimaryEntity.Id, OrganizationService);
            
            if (virtualMeetingRoomCreateResponseMessage.ExceptionOccured) return;

            var serviceAppointment = new DataModel.ServiceAppointment
            {
                Id = PrimaryEntity.Id,
                mcs_meetingroomname = virtualMeetingRoomCreateResponseMessage.MeetingRoomName,
                mcs_PatientUrl = virtualMeetingRoomCreateResponseMessage.PatientUrl,
                mcs_providerurl = virtualMeetingRoomCreateResponseMessage.ProviderUrl,
                mcs_patientpin = virtualMeetingRoomCreateResponseMessage.PatientPin,
                mcs_providerpin = virtualMeetingRoomCreateResponseMessage.ProviderPin,
                mcs_dialingalias = virtualMeetingRoomCreateResponseMessage.DialingAlias,
                mcs_miscdata = virtualMeetingRoomCreateResponseMessage.MiscData,
                cvt_VMRCompleted = true
            };
            OrganizationService.Update(serviceAppointment);
        }
    }
}