﻿using Microsoft.Xrm.Sdk;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.Integration.Plugins.Messages;
using VA.TMP.Integration.Plugins.Shared;
using VA.TMP.OptionSets;

namespace VA.TMP.Integration.Plugins.IntegrationResult
{
    /// <summary>
    ///  CRM Plugin Runner class to handle creating a ServiceAppointment.
    /// </summary>
    public class IntegrationResultUpdateRetryPostStageRunner : PluginRunner
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="serviceProvider">Service Provider.</param>
        public IntegrationResultUpdateRetryPostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider)
        {
        }

        /// <summary>
        /// Gets the MCS Debug field.
        /// </summary>
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }

        /// <summary>
        /// Gets or sets the VIMT URL.
        /// </summary>
        private string VimtUrl { get; set; }

        /// <summary>
        /// Executes the plugin runner.
        /// </summary>
        public override void Execute()
        {
            DataModel.ServiceAppointment serviceAppointment = null;
            DataModel.Appointment appointment = null;
            try
            {
                var integrationResultEntity = PrimaryEntity.ToEntity<mcs_integrationresult>();

                using (var context = new Xrm(OrganizationService))
                {
                    // Retrieve the Integration Result and ensure it is in the correct state before proceeding.
                    var integrationResult = context.mcs_integrationresultSet.FirstOrDefault(x => x.Id == integrationResultEntity.Id);
                    if (integrationResult == null) throw new InvalidPluginExecutionException("Integration Result cannot be null.");
                    if (integrationResult.mcs_appointmentid == null && integrationResult.mcs_serviceappointmentid == null) throw new InvalidPluginExecutionException("Both AppointmentId and ServiceAppointmentId for Integration Result cannot be null.");

                    // Ensure that the Int Result Retry status = yes
                    if (integrationResult.mcs_retry == null || integrationResult.mcs_retry != true) return;

                    // Ensure that the integration result is in an error status. 
                    if (integrationResult.mcs_status == null || integrationResult.mcs_status.GetHashCode() != mcs_integrationresultmcs_status.Error.GetHashCode()) return;

                    // Look up the VIMT URL from Integration Settings. 
                    VimtUrl = context.mcs_integrationsettingSet.First(x => x.mcs_name == "VIMT URL").mcs_value;

                    // Use the integration result to dynamically invoke the call to VIMT. Look up the Int Result values. 
                    var vimtRegistryMessage = integrationResult.mcs_VimtMessageRegistryName;
                    var requestType = Type.GetType(integrationResult.mcs_VimtRequestMessageType);
                    var responseType = Type.GetType(integrationResult.mcs_VimtResponseMessageType);
                    var vimtRequest = integrationResult.mcs_vimtrequest;

                    // Retrieve the Service Appointment and ensure it is in the correct state before proceeding.
                    if (integrationResult.mcs_serviceappointmentid != null)
                    {
                        serviceAppointment = context.ServiceAppointmentSet.FirstOrDefault(x => x.Id == integrationResult.mcs_serviceappointmentid.Id);
                        if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null.");

                        // Ensure this is a Home/Mobile Service Appointment.
                        if ((serviceAppointment.cvt_Type == null || !serviceAppointment.cvt_Type.Value) && vimtRegistryMessage != MessageRegistry.ProxyAddRequestMessage) return;
                        if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                        if (serviceAppointment.StateCode.Value == ServiceAppointmentState.Open || serviceAppointment.StateCode.Value == ServiceAppointmentState.Closed) return;
                    }
                    else if (integrationResult.mcs_appointmentid != null)
                    {
                        appointment = context.AppointmentSet.FirstOrDefault(x => x.Id == integrationResult.mcs_appointmentid.Id);
                        if (appointment == null) throw new InvalidPluginExecutionException("Appointment cannot be null.");
                        if (appointment.StateCode == null) throw new InvalidPluginExecutionException("Appointment State Code cannot be null");
                        if (appointment.StateCode.Value == AppointmentState.Completed || appointment.StateCode.Value == AppointmentState.Open) return;
                    }

                    // Create the Response Object
                    var responseObject = IntegrationPluginHelpers.InvokeVimtDynamically(requestType, responseType, vimtRequest, vimtRegistryMessage, VimtUrl);
                    
                    // Cast to the appropriate Response type                  
                    switch (vimtRegistryMessage)
                    {
                        case MessageRegistry.ProxyAddRequestMessage:
                            var proxyAddResponse = responseObject as ProxyAddResponseMessage;
                            if (proxyAddResponse == null) throw new InvalidPluginExecutionException(string.Format("Exception casting VIMT response to type {0}", typeof(ProxyAddResponseMessage)));
                            if (integrationResult.mcs_serviceappointmentid != null) ProcessProxyAddResponse(proxyAddResponse, serviceAppointment);
                            else if (integrationResult.mcs_appointmentid != null) ProcessGroupProxyAddResponse(proxyAddResponse, appointment);
                            else throw new InvalidPluginExecutionException("Retry Failed: null appointment and serviceappointment for Proxy Add To Vista");
                            break;

                        case MessageRegistry.VideoVisitCreateRequestMessage:
                            if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null");
                            if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                            if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Scheduled) return;
                            var vvCreateResponse = responseObject as VideoVisitCreateResponseMessage;
                            if (vvCreateResponse == null) throw new InvalidPluginExecutionException(string.Format("Exception casting VIMT response to type {0}", typeof(VideoVisitCreateResponseMessage)));
                            ProcessVideoVisitCreateResponseMessage(vvCreateResponse, serviceAppointment);
                            break;

                        case MessageRegistry.VideoVisitDeleteRequestMessage:
                            if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null");
                            if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                            if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Canceled) return;
                            var vvDeleteResponse = responseObject as VideoVisitDeleteResponseMessage;
                            if (vvDeleteResponse == null) throw new InvalidPluginExecutionException(string.Format("Exception casting VIMT response to type {0}", typeof(VideoVisitDeleteResponseMessage)));
                            ProcessVideoVisitDeleteResponseMessage(vvDeleteResponse, serviceAppointment);
                            break;

                        case MessageRegistry.VirtualMeetingRoomCreateRequestMessage:
                            // If doing a retry for the VMR Create Request Message, we know the VVC Create was never reached, so we need to Dynamically invoke 
                            // the VVC create request message as well.
                            if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null");
                            if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                            if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Scheduled) return;
                            var vmrCreateResponse = responseObject as VirtualMeetingRoomCreateResponseMessage;
                            if (vmrCreateResponse == null) throw new InvalidPluginExecutionException(string.Format("Exception casting VIMT response to type {0}", typeof(VirtualMeetingRoomCreateResponseMessage)));
                            ProcessVirtualMeetingRoomCreateResponseMessage(vmrCreateResponse, serviceAppointment);
                            break;

                        case MessageRegistry.VirtualMeetingRoomDeleteRequestMessage:
                            // If doing a retry for the VMR Delete Request Message, we know the VVC Delete was never reached, so we need to Dynamically invoke 
                            // the VVC delete request message as well.
                            if (serviceAppointment == null) throw new InvalidPluginExecutionException("Service Appointment cannot be null");
                            if (serviceAppointment.StateCode == null) throw new InvalidPluginExecutionException("Service Appointment State Code cannot be null");
                            if (serviceAppointment.StateCode.Value != ServiceAppointmentState.Canceled) return;
                            var vmrDeleteResponse = responseObject as VirtualMeetingRoomDeleteResponseMessage;
                            if (vmrDeleteResponse == null) throw new InvalidPluginExecutionException(string.Format("Exception casting VIMT response to type {0}", typeof(VirtualMeetingRoomDeleteResponseMessage)));
                            ProcessVirtualMeetingRoomDeleteResponseMessage(vmrDeleteResponse, serviceAppointment);

                            // Need to manually construct VVC vimtrequest - serialized instance in the helper class
                            var vvDeleteRequestType = typeof(VideoVisitDeleteRequestMessage);
                            var vvDeleteResponseType = typeof(VideoVisitDeleteResponseMessage);
                            const string vvDeleteVimtRegistryMessage = MessageRegistry.VideoVisitDeleteRequestMessage;
                            
                            var vvDeleteRequest = new VideoVisitDeleteRequestMessage
                            {
                                LogRequest = true,
                                UserId = PluginExecutionContext.InitiatingUserId,
                                OrganizationName = PluginExecutionContext.OrganizationName,
                                AppointmentId = serviceAppointment.Id
                            };
                            var vvDeleteVimtRequest = IntegrationPluginHelpers.SerializeInstance(vvDeleteRequest);
                            var vvDeleteResponseObject = IntegrationPluginHelpers.InvokeVimtDynamically(vvDeleteRequestType, vvDeleteResponseType, vvDeleteVimtRequest, vvDeleteVimtRegistryMessage, VimtUrl);
                            var vvDeleteObjResponse = vvDeleteResponseObject as VideoVisitDeleteResponseMessage;
                            if (vvDeleteObjResponse == null) throw new InvalidPluginExecutionException(string.Format("Exception casting VIMT response to type {0}", typeof(VideoVisitDeleteResponseMessage)));
                            ProcessVideoVisitDeleteResponseMessage(vvDeleteObjResponse, serviceAppointment);
                            break;
                    }
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(string.Format("ERROR in IntegrationResultUpdateRetryPostStageRunner: {0}", IntegrationPluginHelpers.BuildErrorMessage(ex)));
            }
            catch (InvalidPluginExecutionException ex)
            {
                Logger.WriteDebugMessage(ex.Message);
                throw;
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw;
            }
        }

        /// <summary>
        /// Update the Service Activity and create an Integration Result.
        /// </summary>
        /// <param name="virtualMeetingRoomCreateResponseMessage">Virtual Meeting Room Create Response Message.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVirtualMeetingRoomCreateResponseMessage(VirtualMeetingRoomCreateResponseMessage virtualMeetingRoomCreateResponseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            // Create a new Int Result. Set to Complete if there was no error. 
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Create Virtual Meeting Room",
                mcs_error = virtualMeetingRoomCreateResponseMessage.ExceptionOccured ? virtualMeetingRoomCreateResponseMessage.ExceptionMessage : null,
                mcs_vimtrequest = virtualMeetingRoomCreateResponseMessage.VimtRequest,
                mcs_integrationrequest = virtualMeetingRoomCreateResponseMessage.SerializedInstance,
                mcs_vimtresponse = virtualMeetingRoomCreateResponseMessage.VimtResponse,
                mcs_status = virtualMeetingRoomCreateResponseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(VirtualMeetingRoomCreateRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(VirtualMeetingRoomCreateResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.VirtualMeetingRoomCreateRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);

            if (virtualMeetingRoomCreateResponseMessage.ExceptionOccured) return;

            var updateServiceAppointment = new DataModel.ServiceAppointment
            {
                Id = serviceAppointment.Id,
                mcs_meetingroomname = virtualMeetingRoomCreateResponseMessage.MeetingRoomName,
                mcs_PatientUrl = virtualMeetingRoomCreateResponseMessage.PatientUrl,
                mcs_providerurl = virtualMeetingRoomCreateResponseMessage.ProviderUrl,
                mcs_patientpin = virtualMeetingRoomCreateResponseMessage.PatientPin,
                mcs_providerpin = virtualMeetingRoomCreateResponseMessage.ProviderPin,
                mcs_dialingalias = virtualMeetingRoomCreateResponseMessage.DialingAlias,
                mcs_miscdata = virtualMeetingRoomCreateResponseMessage.MiscData,
                cvt_VMRCompleted = true
            };
            OrganizationService.Update(updateServiceAppointment);

            // Update the exisiting Int Result. Set to Error: Retry Success if there was no error. 
            IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
        }

        /// <summary>
        /// Create an Integration Result.
        /// </summary>
        /// <param name="videoVisitCreateResponseMessage">Video Visit Create Response Message.</param>
        /// <param name="serviceAppointment">Service Appointment</param>
        private void ProcessVideoVisitCreateResponseMessage(VideoVisitCreateResponseMessage videoVisitCreateResponseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Create Video Visit",
                mcs_error = videoVisitCreateResponseMessage.ExceptionOccured ? videoVisitCreateResponseMessage.ExceptionMessage : null,
                mcs_vimtrequest = videoVisitCreateResponseMessage.VimtRequest,
                mcs_integrationrequest = videoVisitCreateResponseMessage.SerializedInstance,
                mcs_vimtresponse = videoVisitCreateResponseMessage.VimtResponse,
                mcs_status = videoVisitCreateResponseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(VideoVisitCreateRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(VideoVisitCreateResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.VideoVisitCreateRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);
            
            // If Retry was unsuccessful, we will not update the existing Int Result. 
            if (videoVisitCreateResponseMessage.ExceptionOccured) return;
            
            // Update the exisiting Int Result. Set to Error: Retry Success if there was no error. 
            IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);

            // Check for successful In Results / Update Service Appt to Scheduled:Reserved
            // TODO - What if there are 2 error int results? 
            IntegrationPluginHelpers.UpdateServiceAppointmentToScheduledStatus(OrganizationService, serviceAppointment.Id);
        }

        /// <summary>
        /// Process Virtual Meeting Room Delete Response Message.
        /// </summary>
        /// <param name="virtualMeetingRoomDeleteResponseMessage">VirtualMeetingRoomDeleteResponseMessage.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVirtualMeetingRoomDeleteResponseMessage(VirtualMeetingRoomDeleteResponseMessage virtualMeetingRoomDeleteResponseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Cancel Virtual Meeting Room",
                mcs_error = virtualMeetingRoomDeleteResponseMessage.ExceptionOccured ? virtualMeetingRoomDeleteResponseMessage.ExceptionMessage : null,
                mcs_vimtrequest = virtualMeetingRoomDeleteResponseMessage.VimtRequest,
                mcs_integrationrequest = virtualMeetingRoomDeleteResponseMessage.SerializedInstance,
                mcs_vimtresponse = virtualMeetingRoomDeleteResponseMessage.VimtResponse,
                mcs_status = virtualMeetingRoomDeleteResponseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(VirtualMeetingRoomDeleteRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(VirtualMeetingRoomDeleteResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.VirtualMeetingRoomDeleteRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);

            // If Retry was unsuccessful, we will not update the existing Int Result. 
            if (virtualMeetingRoomDeleteResponseMessage.ExceptionOccured) return;
            
            // Update the exisiting Int Result. Set to Error: Retry Success if there was no error. 
            IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
        }

        /// <summary>
        /// Create an Integration Result.
        /// </summary>
        /// <param name="videoVisitDeleteResponseMessage">Video Visit Delete Response Message.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessVideoVisitDeleteResponseMessage(VideoVisitDeleteResponseMessage videoVisitDeleteResponseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Cancel Video Visit",
                mcs_error = videoVisitDeleteResponseMessage.ExceptionOccured ? videoVisitDeleteResponseMessage.ExceptionMessage : null,
                mcs_vimtrequest = videoVisitDeleteResponseMessage.VimtRequest,
                mcs_integrationrequest = videoVisitDeleteResponseMessage.SerializedInstance,
                mcs_vimtresponse = videoVisitDeleteResponseMessage.VimtResponse,
                mcs_status = videoVisitDeleteResponseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(VideoVisitDeleteRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(VideoVisitDeleteResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.VideoVisitDeleteRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);

            // If Retry was unsuccessful, we will not update the existing Int Result. 
            if (videoVisitDeleteResponseMessage.ExceptionOccured) return;
            
            // Update the exisiting Int Result. Set to Error: Retry Success if there was no error. 
            IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
        }

        /// <summary>
        /// Process Proxy Add to Vista response.
        /// </summary>
        /// <param name="responseMessage">ProxyAddResponseMessage.</param>
        /// <param name="serviceAppointment">Service Appointment.</param>
        private void ProcessProxyAddResponse(ProxyAddResponseMessage responseMessage, DataModel.ServiceAppointment serviceAppointment)
        {
            var newIntegrationResult = new mcs_integrationresult
            {
                mcs_name = "Proxy Add to VistA",
                mcs_error = responseMessage.ExceptionOccured ? responseMessage.ExceptionMessage : null,
                mcs_vimtrequest = responseMessage.VimtRequest,
                mcs_integrationrequest = responseMessage.SerializedInstance,
                mcs_vimtresponse = responseMessage.VimtResponse,
                mcs_status = responseMessage.ExceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = typeof(ProxyAddRequestMessage).FullName,
                mcs_VimtResponseMessageType = typeof(ProxyAddResponseMessage).FullName,
                mcs_VimtMessageRegistryName = MessageRegistry.ProxyAddRequestMessage,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointment.Id)
            };
            OrganizationService.Create(newIntegrationResult);

            if (responseMessage.ExceptionOccured) return;

            var updateServiceAppointment = new DataModel.ServiceAppointment
            {
                Id = serviceAppointment.Id,
                cvt_ProxyAddCompleted = true
            };

            try
            {
                var saStatus = serviceAppointment.cvt_Type != null ? serviceAppointment.cvt_Type.Value ? serviceappointment_statuscode.Pending : serviceappointment_statuscode.ReservedScheduled : serviceappointment_statuscode.ReservedScheduled;
                IntegrationPluginHelpers.UpdateServiceAppointmentStatus(OrganizationService, serviceAppointment.Id, saStatus);
                OrganizationService.Update(updateServiceAppointment);
                IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException(string.Format("Failed to Update Proxy Add Completed Flag to initiate VVC/VVS plugin: {0}", ex.Message), ex);
            }
        }

        /// <summary>
        /// Process Group Proxy Add response.
        /// </summary>
        /// <param name="responseMessage">ProxyAddResponseMessage.</param>
        /// <param name="appointment">Appointment.</param>
        private void ProcessGroupProxyAddResponse(ProxyAddResponseMessage responseMessage, DataModel.Appointment appointment)
        {
            try
            {
                IntegrationPluginHelpers.CreateAppointmentIntegrationResult("Group Proxy Add To Vista", responseMessage.ExceptionOccured, responseMessage.ExceptionMessage,
                    responseMessage.VimtRequest, responseMessage.SerializedInstance, responseMessage.VimtResponse, typeof(ProxyAddRequestMessage).FullName,
                    typeof(ProxyAddResponseMessage).FullName, MessageRegistry.ProxyAddRequestMessage, appointment.Id, OrganizationService);

                if (!responseMessage.ExceptionOccured) IntegrationPluginHelpers.UpdateIntegrationResultToErrorRetrySuccessStatus(OrganizationService, PrimaryEntity.Id);
            }
            catch (Exception ex)
            {
                throw new InvalidPluginExecutionException(string.Format("Failed to Process Proxy Add Group Response: {0}", ex.Message), ex);
            }
        }
    }
}