using System;
using System.Collections.Generic;
using System.Diagnostics;
using Microsoft.Practices.Unity.InterceptionExtension;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.Shared
{
    /// <summary>
    /// Interception behaviors implemented for each invocation of the pipelines.
    /// </summary>
    public class Interceptor : IInterceptionBehavior
    {
        /// <summary>
        /// Used to log pre and post method invocation.
        /// </summary>
        /// <param name="input">Inputs to the current call to the target.</param>
        /// <param name="getNext">Delegate to execute to get the next delegate in the behavior chain.</param>
        /// <returns>Return value from the target.</returns>
        public IMethodReturn Invoke(IMethodInvocation input, GetNextInterceptionBehaviorDelegate getNext)
        {
            var methodName = input.MethodBase.DeclaringType != null ? input.MethodBase.DeclaringType.FullName : input.MethodBase.Name;
            methodName = methodName.Replace("VA.TMP.Integration.VIMT.", string.Empty);
            methodName = methodName.Replace("PipelineSteps.", string.Empty);

            var stopWatch = Stopwatch.StartNew();

            Logger.Instance.Debug(string.Format("Calling TMP.{0}.", methodName));

            var message = getNext()(input, getNext);

            Logger.Instance.Debug(string.Format("Exiting TMP.{0}. Took {1} ms", methodName, stopWatch.ElapsedMilliseconds));

            return message;
        }

        /// <summary>
        /// Returns the interfaces required by the behavior for the objects it intercepts.
        /// </summary>
        /// <returns>The required interfaces.</returns>
        public IEnumerable<Type> GetRequiredInterfaces()
        {
            return Type.EmptyTypes;
        }

        /// <summary>
        /// Returns a flag indicating if this behavior will actually do anything when invoked.
        /// </summary>
        /// <remarks>This is used to optimize interception. If the behaviors won't actually do anything (for example, PIAB where no policies match) then the interception mechanism can be skipped completely.</remarks>
        public bool WillExecute
        {
            get { return true; }
        }
    }
}