using Microsoft.Xrm.Sdk;
using System;
using System.IO;
using System.Text;
using System.Xml.Serialization;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;
using VRMRest;

namespace VA.TMP.Integration.Plugins.Shared
{
    /// <summary>
    /// Static class of reusable methods to facilitate integration plugin development.
    /// </summary>
    public static class IntegrationPluginHelpers
    {
        public const string VimtServerDown = "Error - VIMT is likely unavailable. Details of the error are as follows: {0}";

        /// <summary>
        ///  Update Service Appointment's Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="serviceAppointmentId">Service Appointment Id.</param>
        /// <param name="statusCode">Service Appointment Status.</param>
        public static void UpdateServiceAppointmentStatus(IOrganizationService organizationService, Guid serviceAppointmentId, serviceappointment_statuscode statusCode)
        {
            var serviceAppointment = new DataModel.ServiceAppointment
            {
                Id = serviceAppointmentId,
                StatusCode = new OptionSetValue((int)statusCode)
            };
            organizationService.Update(serviceAppointment);
        }

        public static void UpdateAppointment(IOrganizationService organizationService, Guid appointmentId, Appointmentcvt_IntegrationBookingStatus status)
        {
            var appointmentToUpdate = new DataModel.Appointment
            {
                Id = appointmentId,
                cvt_IntegrationBookingStatus = new OptionSetValue((int)status)
            };
            organizationService.Update(appointmentToUpdate);
        }

        /// <summary>
        /// Update Service Appointment to Scheduled status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="serviceAppointmentId">Service Appointment Id.</param>
        public static void UpdateServiceAppointmentToScheduledStatus(IOrganizationService organizationService, Guid serviceAppointmentId)
        {
            var serviceAppointment = new DataModel.ServiceAppointment
            {
                Id = serviceAppointmentId,
                StatusCode = new OptionSetValue((int)serviceappointment_statuscode.ReservedScheduled)
            };
            organizationService.Update(serviceAppointment);
        }

        /// <summary>
        /// Update Integration Result to ErrorRetrySuccess Status.
        /// </summary>
        /// <param name="organizationService">Organization Service.</param>
        /// <param name="integrationResultId">Integration Result Id.</param>
        public static void UpdateIntegrationResultToErrorRetrySuccessStatus(IOrganizationService organizationService, Guid integrationResultId)
        {
            var updateIntegrationResult = new mcs_integrationresult
            {
                Id = integrationResultId,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.ErrorRetrySuccess),
                mcs_retry = false
            };
            organizationService.Update(updateIntegrationResult);
        }

        /// <summary>
        /// Serializes an instance of a class to a string.
        /// </summary>
        /// <typeparam name="T">The Type of class.</typeparam>
        /// <param name="classInstance">The instance of the class.</param>
        /// <returns>Serialized instance of class.</returns>
        public static string SerializeInstance<T>(T classInstance)
        {
            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(T));
                serializer.Serialize(stream, classInstance);

                return Encoding.ASCII.GetString(stream.ToArray());
            }
        }

        /// <summary>
        /// Invoke SendReceive method using .NET reflection to send the request to VIMT.
        /// </summary>
        /// <param name="requestType">Request Type.</param>
        /// <param name="responseType">Response Type.</param>
        /// <param name="serializedRequest">Serialized Request.</param>
        /// <param name="registryMessage">Message Registry Name.</param>
        /// <param name="vimtBaseUri">VIMT base URI.</param>
        /// <returns></returns>
        public static object InvokeVimtDynamically(Type requestType, Type responseType, string serializedRequest, string registryMessage, string vimtBaseUri)
        {
            var serializer = new XmlSerializer(requestType);

            using (TextReader reader = new StringReader(serializedRequest))
            {
                var result = serializer.Deserialize(reader);
                var method = typeof(Utility).GetMethod("SendReceive");
                var generic = method.MakeGenericMethod(responseType);
                var response = generic.Invoke(null, new[] { new Uri(vimtBaseUri), registryMessage, result, null });

                var responseInstance = response;
                if (responseInstance == null) throw new Exception("The response value was not of the proper type");

                return responseInstance;
            }
        }

        /// <summary>
        /// Create Integration Result on successful completion of VIMT call.
        /// </summary>
        /// <param name="integrationResultName">>Name of the Integration Result.</param>
        /// <param name="exceptionOccured">Whether an exception occured.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="serviceAppointId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        public static void CreateIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid serviceAppointId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_integrationrequest = integrationRequest,
                mcs_vimtresponse = vimtResponse,
                mcs_status = exceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointId)
            };
            organizationService.Create(integrationResult);

            if (exceptionOccured)
            {
                UpdateServiceAppointmentStatus(organizationService, serviceAppointId, serviceappointment_statuscode.InterfaceVIMTFailure);
            }
        }

        /// <summary>
        /// Creates Integration Result.
        /// </summary>
        /// <param name="integrationResultName">Integration Result Name.</param>
        /// <param name="exceptionOccured">Whether an exception occurred.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="integrationRequest">Integration Request.</param>
        /// <param name="vimtResponse">VIMT Response.</param>
        /// <param name="vimtRequestMessageType">VIMT Request Message Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Response Message Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="appointmentId">Appointment Id</param>
        /// <param name="organizationService">Organization Service Proxy.</param>
        public static void CreateAppointmentIntegrationResult(
            string integrationResultName,
            bool exceptionOccured,
            string errorMessage,
            string vimtRequest,
            string integrationRequest,
            string vimtResponse,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid appointmentId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_integrationrequest = integrationRequest,
                mcs_vimtresponse = vimtResponse,
                mcs_status = exceptionOccured ? new OptionSetValue((int)mcs_integrationresultmcs_status.Error) : new OptionSetValue((int)mcs_integrationresultmcs_status.Complete),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_appointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, appointmentId)
            };
            organizationService.Create(integrationResult);

            UpdateAppointment(organizationService, appointmentId, exceptionOccured
                    ? Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure
                    : Appointmentcvt_IntegrationBookingStatus.ReservedScheduled);
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Name of the Integration Result.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Message Request Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Message Response Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="serviceAppointId">Service Appointment Id.</param>
        /// <param name="organizationService">CRM Organization Service.</param>
        /// <param name="isCancelRequest">Whether call is for a Cancel Request.</param>
        public static void CreateIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string vimtRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid serviceAppointId,
            IOrganizationService organizationService,
            bool isCancelRequest = false)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.Error),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_serviceappointmentid = new EntityReference(DataModel.ServiceAppointment.EntityLogicalName, serviceAppointId)
            };
            organizationService.Create(integrationResult);

            if (isCancelRequest) return;

            UpdateServiceAppointmentStatus(organizationService, serviceAppointId, serviceappointment_statuscode.InterfaceVIMTFailure);
        }

        /// <summary>
        /// Creates an Error Integration Result when calls to VIMT fail.
        /// </summary>
        /// <param name="integrationResultName">Integration Result Name.</param>
        /// <param name="errorMessage">Error Message.</param>
        /// <param name="vimtRequest">VIMT Request.</param>
        /// <param name="vimtRequestMessageType">VIMT Request Message Type.</param>
        /// <param name="vimtResponseMessageType">VIMT Response Message Type.</param>
        /// <param name="vimtMessageRegistryName">VIMT Message Registry Name.</param>
        /// <param name="appointmentId">Appointment Id.</param>
        /// <param name="organizationService">Organization Service Proxy.</param>
        public static void CreateAppointmentIntegrationResultOnVimtFailure(
            string integrationResultName,
            string errorMessage,
            string vimtRequest,
            string vimtRequestMessageType,
            string vimtResponseMessageType,
            string vimtMessageRegistryName,
            Guid appointmentId,
            IOrganizationService organizationService)
        {
            var integrationResult = new mcs_integrationresult
            {
                mcs_name = integrationResultName,
                mcs_error = errorMessage,
                mcs_vimtrequest = vimtRequest,
                mcs_status = new OptionSetValue((int)mcs_integrationresultmcs_status.Error),
                mcs_VimtRequestMessageType = vimtRequestMessageType,
                mcs_VimtResponseMessageType = vimtResponseMessageType,
                mcs_VimtMessageRegistryName = vimtMessageRegistryName,
                mcs_appointmentid = new EntityReference(DataModel.Appointment.EntityLogicalName, appointmentId)
            };
            organizationService.Create(integrationResult);

            UpdateAppointment(organizationService, appointmentId, Appointmentcvt_IntegrationBookingStatus.InterfaceVIMTFailure);
        }

        /// <summary>
        /// Builds an error message from an exception recursively.
        /// </summary>
        /// <param name="ex">Exception.</param>
        /// <returns>Exception message.</returns>
        public static string BuildErrorMessage(Exception ex)
        {
            var errorMessage = string.Empty;

            if (ex.InnerException == null) return errorMessage;

            errorMessage += string.Format("\n\n{0}\n", ex.InnerException.Message);
            errorMessage += BuildErrorMessage(ex.InnerException);

            return errorMessage;
        }
    }
}