﻿using MCSShared;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace VA.TMP.CRM
{
    /// <summary>
    /// Update the Group Resource, the Constraint Based Group and the TSS Resource Group
    /// </summary>
    public class McsGroupResourceCreatePostStageRunner : PluginRunner
    {
        #region Constructor
        public McsGroupResourceCreatePostStageRunner(IServiceProvider serviceProvider)
            : base(serviceProvider)
        {
        }
        #endregion

        public override void Execute()
        {
            var thisGroupResource = CvtHelper.ValidateReturnRecord(PrimaryEntity, mcs_groupresource.EntityLogicalName, Logger, OrganizationService);
            UpdateGroupResource((mcs_groupresource)thisGroupResource);
            UpdateTSSResourceGroup((mcs_groupresource)thisGroupResource);
        }

        #region Logic
        /// <summary>
        /// Update the Group Resource
        /// </summary>
        /// <param name="thisGroupResource"></param>
        /// TODO: Does the Vista Clinic capacity need to be set as another check?
        internal void UpdateGroupResource(mcs_groupresource thisGroupResource)
        {
            Logger.setMethod = "UpdateGroupResource";
            Logger.WriteDebugMessage("starting UpdateGroupResource");

            try
            {
                if (thisGroupResource == null || thisGroupResource.mcs_Type == null)
                    return;

                if (thisGroupResource.mcs_Type.Value == (int)mcs_resourcetype.Room)
                {
                    var relatedResource = OrganizationService.Retrieve(mcs_resource.EntityLogicalName, thisGroupResource.mcs_RelatedResourceId.Id, new ColumnSet(true)).ToEntity<mcs_resource>();
                    var updateGroupResource = new mcs_groupresource()
                    {
                        Id = thisGroupResource.Id,
                        cvt_capacity = relatedResource.cvt_capacity
                    };
                    Logger.WriteDebugMessage("Updating Group Resource.");
                    OrganizationService.Update(updateGroupResource);
                }
                Logger.WriteDebugMessage("ending UpdateGroupResource");
            }

            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(ex.Message);
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(ex.Message);
            }
        }

        /// <summary>
        /// Update the TSS Resource Group
        /// </summary>
        /// <param name="thisId"></param>
        internal void UpdateTSSResourceGroup(mcs_groupresource thisGroupResource)
        {
            Logger.setMethod = "UpdateTSSResourceGroup";
            Logger.WriteDebugMessage("starting UpdateTSSResourceGroup");

            try
            {
                if (thisGroupResource.mcs_relatedResourceGroupId == null)
                    return;

                var thisTSSResourceGroupId = thisGroupResource.mcs_relatedResourceGroupId.Id;
                var thisTSSResourceGroup = OrganizationService.Retrieve(mcs_resourcegroup.EntityLogicalName, thisTSSResourceGroupId, new ColumnSet(true)).ToEntity<mcs_resourcegroup>();

                if (thisTSSResourceGroup == null || thisTSSResourceGroup.mcs_Type == null)
                    return;

                var builderCBG = new StringBuilder("<Constraints><Constraint><Expression>");
                string resourceString = null;
                var count = 0;
                using (var srv = new Xrm(OrganizationService))
                {
                    var getEquipmentResources = from resGroups in srv.mcs_groupresourceSet
                                                join mcsResourcs in srv.mcs_resourceSet on resGroups.mcs_RelatedResourceId.Id equals mcsResourcs.mcs_resourceId.Value
                                                where resGroups.mcs_relatedResourceGroupId.Id == thisTSSResourceGroupId && resGroups.statecode == 0
                                                select new
                                                {
                                                    mcsResourcs.mcs_name,
                                                    mcsResourcs.mcs_relatedResourceId
                                                };

                    var getUserResources = from resGroups in srv.mcs_groupresourceSet
                                           join Users in srv.SystemUserSet on resGroups.mcs_RelatedUserId.Id equals Users.Id
                                           where resGroups.mcs_relatedResourceGroupId.Id == thisTSSResourceGroupId && resGroups.statecode == 0
                                           select new
                                           {
                                               resGroups.mcs_RelatedUserId,
                                               resGroups.mcs_name
                                           };

                    switch (thisTSSResourceGroup.mcs_Type.Value)
                    {
                        case (int)mcs_resourcetype.VistaClinic:
                        case (int)mcs_resourcetype.Room:
                        case (int)mcs_resourcetype.Technology:
                            Logger.WriteDebugMessage("TSS Resource Group is either VistA Clinic, Room, or Technology.");
                            foreach (var mcsResource in getEquipmentResources)
                            {
                                builderCBG = buildFunction(mcsResource.mcs_relatedResourceId, resourceString, builderCBG, count, out count, out resourceString);
                            }
                            break;
                        case (int)mcs_resourcetype.TelepresenterImager:
                        case (int)mcs_resourcetype.Provider:
                            Logger.WriteDebugMessage("TSS Resource Group is either Telepresenter or Provider.");
                            foreach (var systemUser in getUserResources)
                            {
                                builderCBG = buildFunction(systemUser.mcs_RelatedUserId, resourceString, builderCBG, count, out count, out resourceString);
                            }
                            break;
                        case (int)mcs_resourcetype.AllRequired:
                            Logger.WriteDebugMessage("TSS Resource Group is All Required.");
                            foreach (var mcsResource in getEquipmentResources)
                            {
                                builderCBG = buildFunction(mcsResource.mcs_relatedResourceId, resourceString, builderCBG, count, out count, out resourceString);
                            }
                            foreach (var systemUser in getUserResources)
                            {
                                builderCBG = buildFunction(systemUser.mcs_RelatedUserId, resourceString, builderCBG, count, out count, out resourceString);
                            }
                            break;
                    }
                }
                if (count == 0)
                    builderCBG.Append("<Body>false");
                builderCBG.Append("</Body><Parameters><Parameter name=\"resource\" /></Parameters></Expression></Constraint></Constraints>");

                Logger.WriteDebugMessage("Constraints XML Constructed");
                var updateCBG = new ConstraintBasedGroup
                {
                    Id = thisTSSResourceGroup.mcs_RelatedResourceGroupId.Id,
                    Constraints = builderCBG.ToString()
                };
                
                var updateRG = new mcs_resourcegroup
                {
                    Id = thisTSSResourceGroupId,
                    cvt_resources = resourceString
                };

                Logger.WriteDebugMessage("Updating CBG");
                OrganizationService.Update(updateCBG);
                Logger.WriteDebugMessage("Updated ConstraintBasedGroup.");
                OrganizationService.Update(updateRG);
                Logger.WriteDebugMessage("Updated Resource Group.");
                Logger.WriteDebugMessage("System Resource Updated with " + count + " resources. Ending UpdateTSSResourceGroup");
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(ex.Message);
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException(ex.Message);
            }
        }

        internal static StringBuilder buildFunction(EntityReference field, string resourceString, StringBuilder builderCBG, int count, out int outCount, out string outResourceString)
        {
            //Add the resource name to the text field (for the view)
            resourceString += field.Name.ToString() + " ; ";

            //Add the relatedresourceId to the cbg
            builderCBG.Append(count > 0 ? " || resource[\"Id\"] == " : "<Body>resource[\"Id\"] == ");
            builderCBG.Append(field.Id.ToString("B"));
            count++;
            outCount = count;
            outResourceString = resourceString;
            return builderCBG;
        }

        #endregion
        #region Additional Interface Methods
        public override string McsSettingsDebugField
        {
            get { return "mcs_groupresourceplugin"; }
        }
        public override Entity GetPrimaryEntity()
        {
            return (Entity)PluginExecutionContext.InputParameters["Target"];
        }
        public override Entity GetSecondaryEntity()
        {
            return (Entity)PluginExecutionContext.PostEntityImages["post"];
        }
        #endregion
    }
}