﻿using MCSShared;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace VA.TMP.CRM
{
    public class EMailCreatePostStageRunner : PluginRunner
    {        
        public EMailCreatePostStageRunner(IServiceProvider serviceProvider) : base(serviceProvider) { }
        //Declare global variables
        string customMessage;
        //Emergency Contact Info
        string SiteMainPhone, ProTCTName, ProTCTPhone, ProTCTEmail, PatTCTName, PatTCTPhone, PatTCTEmail, ProRoom, PatRoom, SiteLocal911Phone, TSAProvEmergency, TSAPatEmergency;
        
        string PatientVirtualMeetingSpace = string.Empty, ProviderVirtualMeetingSpace = string.Empty;
        Boolean isCVTTablet = false;
        cvt_component VirtualMeetingSpace = new cvt_component();
        string stethIP = "";

        #region Implementation
        /// <summary>
        /// Called by PluginRunner - Decide which email to send out (aka which branch of the plugin to run)
        /// </summary>
        public override void Execute()
        {

            Email email = (Email)OrganizationService.Retrieve(Email.EntityLogicalName.ToString(), PrimaryEntity.Id, new ColumnSet(true));
            if (email.Subject.StartsWith("FW:") || email.Subject.StartsWith("RE:"))
                return;
            if (email.mcs_RelatedServiceActivity != null)
            {
                ServiceAppointment relatedAppt = (ServiceAppointment)OrganizationService.Retrieve(
                    ServiceAppointment.EntityLogicalName.ToString(), email.mcs_RelatedServiceActivity.Id, new ColumnSet(true));
                
                if ((relatedAppt.StatusCode.Value == 9 || relatedAppt.StatusCode.Value == 4 || relatedAppt.StatusCode.Value == 917290000) && email.RegardingObjectId == null)
                {
                    if (email.Subject.StartsWith("TSS Scheduler Action:"))
                    {
                        Logger.WriteDebugMessage("Beginning Vista Reminder Email");
                        SendVistaReminder(email);
                        Logger.WriteDebugMessage("Completed Vista Reminder Email");
                    }
                    else if (email.Subject.Contains("Telehealth Appointment Notification for"))
                    {
                        Logger.WriteDebugMessage("Beginning Service Activity Notification Email");
                        NotifyParticipantsOfAppointment(email, relatedAppt);
                        Logger.WriteDebugMessage("Completed Service Activity Notification Email");
                    }
                }
                //Send Patient Email (this email is created at the end of the NotifyParticipantsOfAppointment, so this code will trigger a 2nd round of the plugin execution to get to this branch)
                else if (email.RegardingObjectId != null && email.RegardingObjectId.LogicalName == Contact.EntityLogicalName)
                {
                    Logger.WriteDebugMessage("Beginning Patient Email");
                    CreateCalendarAppointmentAttachment(email, relatedAppt, relatedAppt.StatusCode.Value, "");
                    CvtHelper.UpdateSendEmail(email, OrganizationService);
                    Logger.WriteDebugMessage("Completed Patient Email");
                }
                else
                    return;
            }

            if (email.RegardingObjectId != null)
            {
                Logger.WriteDebugMessage("Beginning Send Email");
                switch (email.RegardingObjectId.LogicalName)
                {
                    //Regarding Object: TSA
                    case mcs_services.EntityLogicalName:
                        SendTSAEmail(email, email.RegardingObjectId.Id);
                        Logger.WriteDebugMessage("Completed Send TSA Email");
                        break;
                    //Regarding Object: TSS Privileging
                    case cvt_tssprivileging.EntityLogicalName:
                        SendPrivilegingEmail(email, email.RegardingObjectId.Id, email.RegardingObjectId.LogicalName);
                        Logger.WriteDebugMessage("Completed Send Privileging Email");
                        break;
                    //Regarding Object: Quality Check
                    case cvt_qualitycheck.EntityLogicalName:
                        SendTriggerEmail(email, email.RegardingObjectId.Id, email.RegardingObjectId.LogicalName);
                        Logger.WriteDebugMessage("Completed FPPE/OPPE Email");
                        break;
                }
            }
        }

#endregion

        #region Commonly Used Functions

        /// <summary>
        /// Overload for basic generateEmailBody - displays the url as the message for "Click Here"
        /// </summary>
        /// <param name="record">ID of the email</param>
        /// <param name="entityStringName">string name of the entity - to retrieve object type code</param>
        /// <param name="customMessage">The message</param>
        /// <returns></returns>
        internal string generateEmailBody(Guid record, string entityStringName, string customMessage){
            return generateEmailBody(record, entityStringName, customMessage, null);
        }

        /// <summary>
        /// Standard "boilerplate" E-Mail body
        /// </summary>
        /// <param name="record">ID of the email record</param>
        /// <param name="entityStringName">The string name of the object type code</param>
        /// <param name="customMessage">The custom string that goes into the email body</param>
        /// <param name="clickHereMessage">The message that is used as the display for the hyperlink</param>
        /// <returns>the body of the email</returns>
        internal string generateEmailBody(Guid record, string entityStringName, string customMessage, string clickHereMessage)
        {
            string body;
            var etc = CvtHelper.GetEntityTypeCode(OrganizationService, entityStringName);
            string servernameAndOrgname = CvtHelper.getServerURL(OrganizationService);
            string url = servernameAndOrgname + "/userDefined/edit.aspx?etc=" + etc + "&id=" + record;
            clickHereMessage = (clickHereMessage == null || clickHereMessage == string.Empty) ? url : clickHereMessage;
            //Custom email text
            //body = CvtHelper.GetTSSLogo(OrganizationService);
            body = "<br/><a href=\"" + url + "\">" + clickHereMessage + "</a>";
            body += "<br/><br/>" + customMessage;
            //Standard email text
            body += "<br/><br/>This is an automated notification from the Telehealth Scheduling System.";
            //removed hyperlink to main system window - per Jeff's Request 6/2/2015
            //body += "<br/><a href=\"" + servernameAndOrgname + "\">" + servernameAndOrgname + "</a>";

            return body;
        }

        #endregion

        //TO-DO: fix patient notification email when tsa is created (to send to team instead of individual user)
        #region SendTSAEmails

        /// <summary>
        /// Returns a string value representing the body of the email for TSA approval notification
        /// </summary>
        /// <param name="email">the object representing the email which is being sent</param>
        /// <param name="record">the Guid of the TSA which is causing this notification to be sent</param>
        /// <param name="entityStringName">the entity logical name of the tsa (i.e. "mcs_services")</param>
        /// <returns></returns>
        internal string ApprovalEmailBody(Email email)
        {

            var approver = String.Empty;
            var nextTeam = String.Empty;
            var FTC = String.Empty;
            var patFacility = String.Empty;
            //Get the Previous approvers by querying most recent note
            using (var srv = new Xrm(OrganizationService))
            {
                var TSANote = srv.AnnotationSet.Where(n => n.ObjectId.Id == email.RegardingObjectId.Id).OrderByDescending(n => n.CreatedOn).First(n => n.NoteText.Contains("Approved by"));
                //most recent approver
                approver = TSANote.CreatedBy.Name;
                var tsa = srv.mcs_servicesSet.FirstOrDefault(t => t.Id == email.RegardingObjectId.Id);
                patFacility = tsa.cvt_PatientFacility == null ? String.Empty : " To " + tsa.cvt_PatientFacility.Name;
                if (tsa.cvt_ServiceScope.Value == 917290001)
                    patFacility = " (Intrafacility)";
                //Get the next approver up and get the FTC who created the TSA (assumed to be provider side) and the FTC who first approved the TSA (assumed to be patient side)
                switch (tsa.statuscode.Value)
                {
                    case 917290002: 
                        nextTeam = "Provider FTC Team";
                        goto case 0;
                    case 917290000: 
                        nextTeam = "Provider Service Chief Team";
                        goto case 0;
                    case 917290001: 
                        nextTeam = "Provider Credentialing and Privileging Team";
                        goto case 0;
                    case 917290008: 
                        nextTeam = "Provider Chief of Staff Team";
                        goto case 0;
                    case 917290004: 
                        nextTeam = "Patient Service Chief Team";
                        goto case -1;
                    case 917290005: 
                        nextTeam = "Patient Credentialing and Privileging Team";
                        goto case -1;
                    case 917290006: 
                        nextTeam = "Patient Chief of Staff Team";
                        goto case -1;
                    case 0: //if Provider side - get the user who created the TSA - assumed to be the Provider FTC
                        FTC = srv.SystemUserSet.FirstOrDefault(u => u.Id == tsa.CreatedBy.Id).FullName;
                        break;
                    case -1: //If patient side - get user who first approved the TSA - assumed to be the Patient FTC
                        var firstApprover = srv.AnnotationSet.Where(n => n.ObjectId.Id == email.RegardingObjectId.Id).OrderBy(n => n.CreatedOn).First(n => n.NoteText.Contains("Approved by"));
                        FTC = firstApprover.CreatedBy.Name;
                        break;
                }
            }

            //TODO: Add patient facility, change spacing
            //get the FTC for whichever side the TSA is awaiting approval
            string hyperlink = GetTSALink(email);
            string OpsManual = "http://DNS.DNS   /sites/telehealth/cvtntc/docs/user_tsa_appr.docx";
            string RollOut = "http://DNS.DNS   /sites/telehelpdesk/TSS%20Roll-Out/default.aspx";
            string emailBody = String.Format("A Telehealth Service Agreement (TSA), {0} is awaiting your approval. <br/><ul><li>Previous Approver: {1}</li>" +
                "<li>{2} is the next in line for the TSA Approval Process. </ul>The hyperlink below will take you to the Telehealth Service Agreement.  If you wish to make changes to the TSA prior to approval, please contact {3}.  If you choose to approve the TSA, please select the Green Button on the top left corner.  If you choose to decline approval, please select the Red Button on the top left corner.<br/><br/><b>Click here to take action on the TSA</b>: {4} <br/><br/>", email.RegardingObjectId.Name + patFacility, approver, nextTeam, FTC, hyperlink);
            string loginNotes = String.Format("Note: A password is not required to access TMP.  Your credentials are passed from Windows authentication used to log on to your computer.  Simply click the link above.  For first time access, or access after a long period of time, you may be prompted to choose \"VA Accounts\" on a pop-up form.  After that, clicking the link will take you directly to the TSA.  <br/><br/>To see a brief tutorial for approvers, click this link: {0} <br/><br/>To access all resources (training materials, operations manual, etc.) for TMP users, click this link: {1}", "<a href=\"" + OpsManual + "\">" + OpsManual + "</a>", "<a href=\"" + RollOut + "\">" + RollOut + "</a>");
        
            return emailBody + loginNotes;
        }

        //Send appropriate email based on subject line (denial, under revision, reminder to take action, waiting for approval)
        internal void SendTSAEmail(Email email, Guid tsaID)
        {
            ////Denial
            //if (email.Subject.ToLower().IndexOf("denied") != -1)  //Denial
            //    email.Description = generateEmailBody(tsaID, "mcs_services", "The following Telehealth Service Agreement has been Denied.  Please review the notes to see the Denial Reason and correct any mistakes if applicable.", "Click Here to view this TSA");
            //else if (email.Subject.ToLower().IndexOf("revision") != -1)  //Revision
            //    customMessage = "The following Telehealth Service Agreement is Under Revision.";
            ////Reminder
            //else if (email.Subject.ToLower().IndexOf("action") != -1)  //Reminder
            //    customMessage = "This is a reminder that the following Telehealth Service Agreement is waiting for you to take action.";
            ////Approval - Privileging
            //else if (email.Subject.ToLower().IndexOf("privileging") != -1)  //Approval - Privileging TODO unable to find this subject anywhere - validate it is still in use
            //    customMessage = "Click on the link above to view this Telehealth Service Agreement and confirm Privileging.";
            ////Approval
            //else if (email.Subject.ToLower().IndexOf("waiting") != -1)  //Approval
            //    email.Description = ApprovalEmailBody(email);
            //else if (email.Subject.ToLower().IndexOf("completed") != -1)  //Production Notification Team
            //    email.Description = TSANotificationText(email);
            //else if (email.Subject.IndexOf("A TSA to your Facility has been created") != -1) //Notify patient site that TSA was created
            //    email.Description = generateEmailBody(tsaID, "mcs_services", "Please coordinate with the Provider Site FTC to set up the following TSA.  Once all the details are finalized, it is the responsibility of the Patient Site FTC to begin the Signature collection process.", "Click Here to view this TSA");
            //else
            //    return;

            //
            switch (email.Subject)
            {
                case "A Telehealth Service Agreement has been denied": //Denial
                    email.Description = generateEmailBody(tsaID, "mcs_services", "The following Telehealth Service Agreement has been Denied.  Please review the notes to see the Denial Reason and correct any mistakes if applicable.", "Click Here to view this TSA");
                    break;
                case "TSA under revision": //Revision
                    customMessage = "The following Telehealth Service Agreement is Under Revision.";
                    break;
                case "Please Take Action on the following TSA": //Reminder
                    customMessage = "This is a reminder that the following Telehealth Service Agreement is waiting for you to take action.";
                    break;
                case "FYI: A Telehealth Service Agreement has been completed": //Production Notification
                    email.Description = TSANotificationText(email);
                    break;
                case "A TSA to your Facility has been created": //Notify patient site that TSA was created
                    email.Description = generateEmailBody(tsaID, "mcs_services", "Please coordinate with the Provider Site FTC to set up the following TSA.  Once all the details are finalized, it is the responsibility of the Patient Site FTC to begin the Signature collection process.", "Click Here to view this TSA");
                    break;
                default:
                    if (email.Subject.Contains("Telehealth Service Agreement is awaiting your approval")) //Approval
                        email.Description = ApprovalEmailBody(email);
                    else
                    {
                        Logger.WriteToFile("Unable to match email subject to valid TSA email type, exiting plugin");
                        return;
                    }
                    break;
            }
            //Get Team Members will query the Team Members table and return an Activity Party List of the people listed on the team specified
            //If cant find team members, log the error and continue attempting to populate the message description
            try
            {
                email.To = GetTeamMembers(email, tsaID);
                Logger.WriteDebugMessage("Populated Email Recipients");
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
            }
            if (email.Description == null)
                email.Description = generateEmailBody(tsaID, "mcs_services", customMessage, "Click Here to approve/deny this TSA");
            //Get the owner of the workflow for the From field
            if (email.From.Count() == 0)
                email.From = CvtHelper.GetWorkflowOwner("TSA Approval Step 1 - Awaiting Prov FTC", OrganizationService);
            Logger.WriteDebugMessage("Sending TSA Email");
            CvtHelper.UpdateSendEmail(email, OrganizationService);
            Logger.WriteDebugMessage("TSA Email Sent");
        }

        internal string GetTSALink(Email email)
        {
            var etc = CvtHelper.GetEntityTypeCode(OrganizationService, mcs_services.EntityLogicalName);
            string servernameAndOrgname = CvtHelper.getServerURL(OrganizationService);
            string url = servernameAndOrgname + "/userDefined/edit.aspx?etc=" + etc + "&id=" + email.RegardingObjectId.Id;
            return String.Format("<a href=\"{0}\">{1}</a>", url, url);
        }

        internal string TSANotificationText(Email email)
        {
            return String.Format("For your information, a Telehealth Service Agreement (TSA), {0} has been approved.  <br/>The hyperlink below will take you to the Telehealth Service Agreement. \n\nClick here to view the TSA: {1} <br /><br />Note: A password is not required to access TSS. Your credentials are passed from Windows authentication used to log on to your computer. Simply click the link above. For first time access, or access after a long period of time, you may be prompted to choose \"VA Accounts\" on a pop-up form.  After that, clicking the link will take you directly to the TSA.  <br /><br />To access all resources (training materials, operations manual, etc.) for TMP users, click this link: {2}", email.RegardingObjectId.Name, GetTSALink(email), String.Format("<a href=\"{0}\">{0}</a>","http://DNS.DNS   /sites/telehelpdesk/TSS%20Roll-Out/default.aspx"));
        }

        /// <summary>
        /// Query the team membership table to get the team appropriate for the status of the TSA.  return the list of activity parties corresponding to the system users that are members of the team.  
        /// </summary>
        /// <param name="email">This is the email record that is being built and eventually gets sent out</param>
        /// <param name="tsaID">This is the ID of the TSA that generated this email.  Based on the status of the TSA, a specific team will be selected</param>
        /// <returns>Activity Party List corresponding to System users that are members of the team</returns>
        internal List<ActivityParty> GetTeamMembers(Email email, Guid tsaID)
        {
            //Status Listing: 917290002==Approved by Pat FTC, 917290000==Prov FTC, 917290001==Prov SC, 917290004==Prov CoS, 917290005==Pat SC, 917290006==Pending Privileging, 251920000==PROD, 917290003==DENIED, 917290007==UNDER REVISION; 917290008==Approved by Prov C&P
            var members = new List<ActivityParty>();
            var teamMembers = new List<TeamMembership>();
            using (var srv = new Xrm(OrganizationService))
            {
                //Get both the Patient and Provider Facility to check for the teams on both sides
                var tsa = srv.mcs_servicesSet.First(t => t.Id == tsaID);
                var proFacilityId = tsa.cvt_ProviderFacility.Id;
                var patFacilityId = tsa.cvt_PatientFacility != null ? tsa.cvt_PatientFacility.Id : Guid.Empty;
                var team = new Team();
                switch (tsa.statuscode.Value)
                {
                    case 1: //For Draft TSAs, send notification that TSA has been created for their site. 
                        team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.FTC);
                        break;
                    case 917290002://Approved by Patient Site FTC (get Provider Site FTC Team) - Workflow Step 1
                        team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == proFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290000);
                        break;
                    case 917290000://Approved by Provider Site FTC (get Provider Service Chief Team) - Workflow Step 2
                        team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == proFacilityId &&
                            t.cvt_Type != null && t.cvt_Type.Value == 917290001 && t.cvt_ServiceType.Id == tsa.cvt_servicetype.Id);
                        break;
                    case 917290001://Approved by Provider Service Chief (get Prov C&P Team) - Workflow Step 3
                        team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == proFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290003);
                        break;
                    case 917290008://Approved by Provider C&P Team (get Prov Chief of Staff Team) - Workflow Step 4
                        team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == proFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290002);
                        break;
                    case 917290004://Approved by Provider Site Chief of Staff (Get Patient Site Service Chief) - Workflow Step 5
                        if (patFacilityId != Guid.Empty)
                            team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId &&
                                t.cvt_Type != null && t.cvt_Type.Value == 917290001 && t.cvt_ServiceType.Id == tsa.cvt_servicetype.Id);
                        break;
                    case 917290005://Approved by Patient Site Service Chief (Get Patient Site Chief of Staff) - Workflow Step 6
                        if (patFacilityId != Guid.Empty)
                            team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290003);
                        break;
                    case 917290006://Approved by C&P (Get Chief of Staff Team) - Workflow Step 7
                        if (patFacilityId != Guid.Empty)
                            team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290002);
                        break;
                    case 917290007://Get both side FTCs whether it is in Denied status or in Under Revision
                    case 917290003:
                        team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == proFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290000);
                        if (team != null)
                            teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == team.Id).ToList());

                        //repurpose team variable to get patient facility (prov facility team members have already been added above) and add team members from pat facility
                        if (patFacilityId != Guid.Empty)
                        {
                            team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290000);
                            if (team != null)
                            {
                                if (teamMembers.Count == 0)
                                    teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == team.Id).ToList());
                                else
                                    teamMembers.AddRange((List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == team.Id).ToList()));
                            }
                        }
                        break;
                    case 251920000: //PROD - Get Both sides notification team for TSA Notification email
                        team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == proFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290004);
                        if (team != null)
                            teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == team.Id).ToList());

                        //repurpose team variable to get patient facility (prov facility team members have already been added above) and add team members from pat facility (if not intrafacility)
                        if (patFacilityId != Guid.Empty && patFacilityId != proFacilityId)
                        {
                            team = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId && t.cvt_Type != null && t.cvt_Type.Value == 917290004);
                            if (team != null)
                            {
                                if (teamMembers.Count == 0)
                                    teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == team.Id).ToList());
                                else
                                    teamMembers.AddRange((List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == team.Id).ToList()));
                            }
                        }
                        break; 
                }
                if (team == null)
                    throw new InvalidPluginExecutionException("No Team was found to receive this email, please verify the team is set up");
                if (teamMembers.Count == 0) //if you havent already added the team members (everthing other than prod notification, under revision and denial) then add now
                    teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == team.Id).ToList());
                foreach (var member in teamMembers)
                {
                    var party = new ActivityParty(){
                        ActivityId = new EntityReference(email.LogicalName, email.Id),
                        PartyId = new EntityReference(SystemUser.EntityLogicalName, member.SystemUserId.Value)
                    };
                    members.Add(party);
                }
            }
            if (members.Count == 0)
                members.AddRange(email.To);
            return members;
        }

        #endregion

        #region SendserviceappointmentNotifications

        /// <summary>
        /// Primary function which generates the Service Activity notification (including ical)
        /// </summary>
        /// <param name="email">The email object corresponding to the email record created which triggered this plugin</param>
        /// <param name="relatedAppt">The service activity which generated the email and is the subject of the notification</param>
        internal void NotifyParticipantsOfAppointment(Email email, ServiceAppointment relatedAppt)
        {
            List<SystemUser> proTCTuser = new List<SystemUser>();
            List<SystemUser> patTCTuser = new List<SystemUser>();

            //Get the TSA so you can figure out whether the resource is provider or patient side
            mcs_services tsa = (mcs_services)OrganizationService.Retrieve(mcs_services.EntityLogicalName, relatedAppt.mcs_relatedtsa.Id, new ColumnSet(true));

            //Get the Pro/Pat Emergency Info from TSA
            TSAProvEmergency = tsa.cvt_ProviderStaffEmergencyResponsibilities;
            TSAPatEmergency = tsa.cvt_PatientStaffEmergencyResponsibilities;

            if (tsa.cvt_PatientSiteClinicalPOC != null)
            {
                Guid TCTonTSA = tsa.cvt_PatientSiteClinicalPOC.Id;
                //Get the TCTs Mobile or Office phone 
                SystemUser tct = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, TCTonTSA, new ColumnSet("mobilephone", "cvt_officephone", "firstname", "lastname", "internalemailaddress"));
                PatTCTPhone = tct.MobilePhone;  //Use the TCT number
                PatTCTName = tct.FirstName + " " + tct.LastName;
                PatTCTEmail = tct.InternalEMailAddress;
                patTCTuser.Add(tct);
                if (PatTCTPhone == null)
                    PatTCTPhone = tct.cvt_officephone;  //Use the TCT number
            }

            //Get the Pro TCT
            if (tsa.cvt_ProviderSiteClinicalPOC != null)
            {
                Guid TCTonTSA = tsa.cvt_ProviderSiteClinicalPOC.Id;
                //Get the TCTs Mobile or Office phone 
                SystemUser tct = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, TCTonTSA, new ColumnSet("mobilephone", "cvt_officephone", "firstname", "lastname", "internalemailaddress"));
                ProTCTPhone = tct.MobilePhone;  //Use the TCT number
                ProTCTName = tct.FirstName + " " + tct.LastName;
                ProTCTEmail = tct.InternalEMailAddress;
                proTCTuser.Add(tct);
                if (ProTCTPhone == null)
                    ProTCTPhone = tct.cvt_officephone;  //Use the TCT number
            }
            //Pro Lead TCT
            if (ProTCTPhone == null) //No Phone from TCT from TSA, look at Site specified one.
            {
                if (tsa.cvt_relatedprovidersiteid.Id != null) //Check for TSA Provider Site
                {
                    //Get Provider Site
                    //get the TSA's related patient site
                    mcs_site relatedprosite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, tsa.cvt_relatedprovidersiteid.Id, new ColumnSet(true));

                    if (relatedprosite.cvt_LeadTCT != null)
                    {
                        //Get the Lead TCTs Mobile or Office phone 
                        SystemUser tct = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, relatedprosite.cvt_LeadTCT.Id, new ColumnSet("mobilephone", "cvt_officephone", "firstname", "lastname", "internalemailaddress"));
                        ProTCTPhone = tct.MobilePhone;  //Use the TCT number
                        ProTCTName = tct.FirstName + " " + tct.LastName;
                        ProTCTEmail = tct.InternalEMailAddress;
                        proTCTuser.Clear();
                        proTCTuser.Add(tct);
                        if (ProTCTPhone == null)
                            ProTCTPhone = tct.cvt_officephone;  //Use the TCT number
                    }
                }
            }
            //If CVT to Home, there is no patient site
            if ((tsa.cvt_Type != true) && (tsa.cvt_groupappointment != true))
            {
                //get the TSA's related patient site
                mcs_site relatedpatsite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, tsa.cvt_relatedpatientsiteid.Id, new ColumnSet(true));
                //Get the Site's Emergency Contact Info
                if (relatedpatsite.cvt_Local911 != null)
                    SiteLocal911Phone = relatedpatsite.cvt_Local911; //Use the Local 911
                if (relatedpatsite.cvt_phone != null)
                    SiteMainPhone = relatedpatsite.cvt_phone; //Use the Site Phone

                //Pat Lead TCT
                if (PatTCTPhone == null) //No Phone from TCT from TSA, look at Pat Site specified one.
                {
                    if (relatedpatsite.cvt_LeadTCT != null)
                    {
                        //Get the Lead TCTs Mobile or Office phone 
                        SystemUser tct = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, relatedpatsite.cvt_LeadTCT.Id, new ColumnSet("mobilephone", "cvt_officephone", "firstname", "lastname", "internalemailaddress"));
                        PatTCTPhone = tct.MobilePhone;  //Use the TCT number
                        PatTCTName = tct.FirstName + " " + tct.LastName;
                        PatTCTEmail = tct.InternalEMailAddress;
                        patTCTuser.Clear();
                        patTCTuser.Add(tct);
                        if (PatTCTPhone == null)
                            PatTCTPhone = tct.cvt_officephone;  //Use the TCT number
                    }
                }
            }

            //Get the Pat TCT
            //Get the resources listed on the service activity
            var resources = relatedAppt.GetAttributeValue<EntityCollection>("resources");
            EntityCollection users = new EntityCollection();
            EntityCollection equipmentResources = new EntityCollection();
            resources.Entities.AddRange(GetApptResources(relatedAppt, string.Empty));

            //Get the users from the resource list (filter out equipment)
            foreach (var res in resources.Entities)
            {
                var party = res.ToEntity<ActivityParty>();
                if (party.PartyId.LogicalName == SystemUser.EntityLogicalName)
                {
                    ActivityParty p = new ActivityParty()
                    {
                        PartyId = new EntityReference(SystemUser.EntityLogicalName, party.PartyId.Id)
                    };
                    users.Entities.Add(p);
                }
                else
                {
                    Equipment e = (Equipment)OrganizationService.Retrieve(Equipment.EntityLogicalName, party.PartyId.Id, new ColumnSet("equipmentid", "mcs_relatedresource"));
                    mcs_resource equip = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName, e.mcs_relatedresource.Id, new ColumnSet(true));
                    equipmentResources.Entities.Add(equip);
                }
            }

            //Get the rooms and techs and segment them by patient/provider site
            var tsaProResources = getPRGs(tsa.Id, "provider");
            var tsaPatResources = getPRGs(tsa.Id, "patient");
            var providerTechs = ClassifyResources(equipmentResources, tsaProResources, (int)mcs_resourcetype.Technology);
            var patientTechs = ClassifyResources(equipmentResources, tsaPatResources, (int)mcs_resourcetype.Technology);
            var providerRooms = ClassifyResources(equipmentResources, tsaProResources, (int)mcs_resourcetype.Room);
            var patientRooms = ClassifyResources(equipmentResources, tsaPatResources, (int)mcs_resourcetype.Room);

            //select which users to send the email to (providers/patients): null means provider side, 1 means patient side (and 0 means both)
            var providerResources = GetRecipients(users, tsaProResources);
            var patientResources = GetRecipients(users, tsaPatResources);

            //Get providers so we can pass in string for list of clinicians to patient email
            var clinicians = providerResources;

            //format body of the email (telepresenters can be duplicated)
            var date = string.Empty;
            var patSite = tsa.cvt_relatedpatientsiteid;
            var patSiteString = patSite != null ? patSite.Name : string.Empty;
            patSiteString += tsa.cvt_groupappointment.Value == true ? tsa.cvt_PatientFacility.Name : "Home/Mobile";
            email.Description = formatNotificationEmailBody(providerTechs, patientTechs, providerRooms, patientRooms, patientResources, relatedAppt, tsa, providerResources, out date);
            email.Subject = email.Subject.Trim() + " " + patSiteString + " " + date;
            //Combine the lists and then add them as the email recipients
            providerResources.AddRange(patientResources);

            //Add the Pro and Pat TCT to the .To
            providerResources.AddRange(proTCTuser);
            providerResources.AddRange(patTCTuser);

            email.To = CvtHelper.SetPartyList(providerResources);

            //Get the owner of the workflow for the From field    
            if (email.From.Count() == 0)
                email.From = CvtHelper.GetWorkflowOwner("Service Activity Notification", OrganizationService);
            //OrganizationService.Update(email);

            //Send a Calendar Appointment if the appointment is scheduled (if canceled, send cancellation update)
            CreateCalendarAppointmentAttachment(email, relatedAppt, relatedAppt.StatusCode.Value, stethIP);
            CvtHelper.UpdateSendEmail(email, OrganizationService);

            if (tsa.cvt_Type.Value == true) // Only send patient email for Home/Mobile TSAs
            {
                //Create and Send Email to Patient/Veteran (copy sender from Provider Email)
                var providerEmailSender = new EntityCollection();
                providerEmailSender.Entities.AddRange(email.From);
                SendPatientEmail(relatedAppt, providerEmailSender, clinicians);
            }
        }

        internal string getPatientVirtualMeetingSpace(ServiceAppointment sa, out bool? patientSpace)
        {
            Logger.WriteDebugMessage("Getting Virtual Meeting Space");
            patientSpace = null;
            if (sa.mcs_PatientUrl != null && sa.mcs_providerurl != null)
            {
                PatientVirtualMeetingSpace = sa.mcs_PatientUrl;
                ProviderVirtualMeetingSpace = sa.mcs_providerurl;
                Logger.WriteDebugMessage("Virtual Meeting Space is from Service Activity Record: " + PatientVirtualMeetingSpace + ", " + ProviderVirtualMeetingSpace);
            }
            else
            {
                var patientAP = sa.Customers.FirstOrDefault();
                if (patientAP == null || patientAP.PartyId == null)
                    return string.Empty;
                Logger.WriteDebugMessage(sa.Customers.ToList().Count().ToString() + " patients " + patientAP.PartyId.Name.ToString());
                var patient = (Contact)OrganizationService.Retrieve(Contact.EntityLogicalName, patientAP.PartyId.Id, new ColumnSet(true));
                Logger.WriteDebugMessage("Contact: " + patient.FullName + " VMR: " + patient.cvt_PatientVirtualMeetingSpace + " and Tablet: " + patient.cvt_BLTablet);
                if (patient != null && patient.cvt_PatientVirtualMeetingSpace != null)
                {
                    patientSpace = true;
                    PatientVirtualMeetingSpace = patient.cvt_PatientVirtualMeetingSpace;
                    ProviderVirtualMeetingSpace = patient.cvt_ProviderVirtualMeetingSpace;
                }
                else if (patient != null && patient.cvt_BLTablet != null)
                {
                    patientSpace = false;
                    PatientVirtualMeetingSpace = patient.cvt_BLTablet;
                    isCVTTablet = true;
                }
                else if (VirtualMeetingSpace.Id != new Guid())
                    PatientVirtualMeetingSpace = VirtualMeetingSpace.cvt_webinterfaceurl;
                else
                    PatientVirtualMeetingSpace = "Please Contact Your TCT for Web Meeting Details";
                Logger.WriteDebugMessage(PatientVirtualMeetingSpace + ": Virtual Meeting Space is from Patient record = " + patientSpace.ToString());
            }
            return PatientVirtualMeetingSpace;
        }

        internal void SendPatientEmail(ServiceAppointment sa, EntityCollection From, List<SystemUser> provs)
        {
            if (VirtualMeetingSpace != null && VirtualMeetingSpace.Id != Guid.Empty || PatientVirtualMeetingSpace.IndexOf("Please Contact Your") == -1) 
                // last check is only relevant if we do not want to generate email at all given the scenario where no virtual meeting space is provided
            {
                //Get the Patient and their timezone
                var patientAP = sa.Customers.FirstOrDefault();
                if (patientAP == null)
                    Logger.WriteToFile("No Patient was found to receive the email for following Service Activity: " + sa.Id);
                else
                {
                    var recipient = new ActivityParty()
                    {
                        PartyId = new EntityReference(Contact.EntityLogicalName, patientAP.PartyId.Id)
                    };
                    Logger.WriteDebugMessage("Sending Patient Email to " + patientAP.PartyId.Name);
                    var patient = (Contact)OrganizationService.Retrieve(Contact.EntityLogicalName, recipient.PartyId.Id, new ColumnSet(true));

                    //Setup variables to get timeZone conversion properly
                    DateTime timeConversion = sa.ScheduledStart.Value;
                    string fullDate = string.Empty, timeZonesString = string.Empty;
                    bool convertSuccess = false, isCancelled = (sa.StatusCode.Value == 9 || sa.StatusCode.Value == 917290000) ? true : false; //Cancellation
                    int timeZone = patient.cvt_TimeZone != null ? patient.cvt_TimeZone.Value : 35; //default time zone to East Coast if its not listed on the patient record
                    timeConversion = ConvertTimeZone(sa.ScheduledStart.Value, timeZone, out timeZonesString, out convertSuccess);
                    Logger.WriteDebugMessage("Time converted to " + timeZonesString);
                    fullDate = convertSuccess ? timeConversion.ToString("ddd dd MMM yyyy HH:mm") + " " + timeZonesString : timeConversion.ToString("ddd dd MMM yyyy HH:mm") + " GMT";

                    //Creating the Subject text
                    var subject = "Your VA Video Visit has been ";
                    subject += (isCancelled) ? "canceled" : "scheduled";
                    subject += " for " + fullDate.Trim();
                    Logger.WriteDebugMessage("Local Time: " + fullDate);

                    //Getting the Subject Specialty, Specialty Sub Type
                    var tsa = (mcs_services)OrganizationService.Retrieve(mcs_services.EntityLogicalName, sa.mcs_relatedtsa.Id, new ColumnSet(true));
                    var serviceText = (tsa.cvt_servicetype != null) ? "<b>Specialty:</b> " + tsa.cvt_servicetype.Name + "<br />" : "";
                    serviceText += (tsa.cvt_servicesubtype != null) ? "<b>Specialty Sub Type:</b> " + tsa.cvt_servicesubtype.Name + "<br />" : "";


                    var clinicians = string.Empty;
                    foreach (SystemUser user in provs)
                    {
                        clinicians += user.FullName + "; ";
                    }
                    if (clinicians != string.Empty)
                        clinicians = "<b>Clinician:</b> " + clinicians.Remove(clinicians.Length - 2) + "<br />";

                    //if you can't find "Please Contact Your" that means a real url was entered, so use it as a hyperlink, otherwise, display the "Please Contact Your..." message as it comes across
                    var meetingSpace = PatientVirtualMeetingSpace.IndexOf("Please Contact Your") == -1 ?
                        CvtHelper.buildHTMLUrl(PatientVirtualMeetingSpace, "Click Here to Join the Virtual Meeting")
                        : "<b>Your virtual meeting room was not found, " + PatientVirtualMeetingSpace + "</b>";

                    var dynamicBody = isCVTTablet ? "Your provider will call your CVT tablet for the appointment." : "Please click the following link to access the virtual meeting.  This will take you into the virtual waiting room until your provider joins.<br />";

                    //Set up difference in Scheduled vs Cancelation text
                    var descrStatus = "reminder of your";
                    var attachmentText = "<br /><br />A calendar appointment is attached to this email, you can open the attachment and save it to your calendar.";
                    if (isCancelled) //Canceled
                    {
                        descrStatus = "cancelation notice for your previously scheduled";
                        attachmentText = "<br /><br />A calendar appointment cancelation is attached to this email, you can open the attachment and click \"Remove from Calendar\" to remove this event from your calendar.";
                        dynamicBody = "";
                        meetingSpace = "";
                    }

                    var description = String.Format("This is a {0} Video Visit with a VA clinician on <b>{1}</b>. {2}{3}<br /><br />{4}{5}<br />If you have any questions or concerns, please contact your clinic. <br />{6}",
                        descrStatus,
                        fullDate,
                        dynamicBody,
                        (!isCVTTablet ? meetingSpace : ""),
                        serviceText,
                        clinicians,
                        attachmentText);
                    Email patientEmail = new Email()
                    {
                        Subject = subject,
                        Description = description,
                        mcs_RelatedServiceActivity = new EntityReference(ServiceAppointment.EntityLogicalName, sa.Id),
                        RegardingObjectId = new EntityReference(Contact.EntityLogicalName, patientAP.PartyId.Id),
                        From = CvtHelper.GetWorkflowOwner("Service Activity Notification", OrganizationService)
                    };
                    patientEmail.To = CvtHelper.SetPartyList(recipient);

                    OrganizationService.Create(patientEmail);
                    Logger.WriteDebugMessage("Patient Email Created Successfully");
                }
            }
            else
                Logger.WriteToFile("No VMR information could be found");
        }

        public DateTime ConvertTimeZone(DateTime date, int CRMTimeZoneCode, out string timeZonesString, out bool success)
        {
            success = false;
            timeZonesString = string.Empty;
            try
            {
                Logger.WriteDebugMessage("Converting Time to Appropriate Time Zone");
                using (var srv = new Xrm(OrganizationService))
                {
                    var timeZonerecord = srv.TimeZoneDefinitionSet.FirstOrDefault(t => t.TimeZoneCode != null && t.TimeZoneCode.Value == CRMTimeZoneCode);
                    timeZonesString = timeZonerecord.StandardName;
                }
                var timeZoneCode = TimeZoneInfo.FindSystemTimeZoneById(timeZonesString);
                var localTime = TimeZoneInfo.ConvertTimeFromUtc(date, timeZoneCode);
                success = true;
                return localTime;
            }
            catch (TimeZoneNotFoundException ex)
            {
                Logger.WriteToFile("Could not find " + timeZonesString + " time zone with code " + CRMTimeZoneCode.ToString() + ": " + ex.Message + " ; using UTC instead");
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Time Zone conversion issue" + ex.Message + " ; using UTC instead");
            }
            return date;
        }

        /// <summary>
        /// This method creates the .ics attachment and appends it to the email
        /// </summary>
        /// <param name="email">This is the email that the attachment is attaching to</param>
        /// <param name="sa">The service appointment which </param>
        /// <param name="statusCode">The status of the email - which sets the status of the attachment as well as the subject of the email</param>
        internal void CreateCalendarAppointmentAttachment(Email email, ServiceAppointment sa, int statusCode, string stethIP)
        {
            bool group = false;
            if (sa.mcs_groupappointment != null)
            {
                group = sa.mcs_groupappointment.Value;
            }
            Logger.WriteTxnTimingMessage("Begin Creating Calendar Appointment");
            string schLocation = "See Description";
            string schSubject = group == true ? "Telehealth Visit-Group Appointment: Do Not Reply" : 
                "Telehealth Visit-Single Appointment: Do Not Reply";
            string schDescription = email.Description;
            System.DateTime schBeginDate = (System.DateTime)sa.ScheduledStart;
            System.DateTime schEndDate = (System.DateTime)sa.ScheduledEnd;
            string sequence = "";
            string status = "CONFIRMED";
            string method = "";
            //if the appointment is canceled, send a cancellation notice based on the UID of the previous entry sent
            if (statusCode == 9 || statusCode == 917290000)
            {
                method = "METHOD:CANCEL\n";
                sequence = "SEQUENCE:1\n";
                status = "CANCELLED";
                schSubject = "Canceled: Telehealth Visit: Do Not Reply";
            }

            //attach a ClearSteth CVL file if a steth is in the components
            string cvlAtttachment = string.IsNullOrEmpty(stethIP) ? "" :
                "ATTACH;ENCODING=BASE64;VALUE=BINARY;X-FILENAME=invitation.cvl:" + Convert.ToBase64String(new ASCIIEncoding().GetBytes("<?xml version=\"1.0\" encoding=\"UTF-8\"?><CVL><IP>" + stethIP + "</IP ><Port>9005</Port><ConferenceId>12345</ConferenceId></CVL>")) + "\n";
 
            string att =  "BEGIN:VCALENDAR\n"+
                              "PRODID:-//VA//Veterans Affairs//EN\n"+
                              method +
                              "BEGIN:VEVENT\n"+
                              cvlAtttachment +
                              "UID:" + sa.Id + "\n" + sequence + 
                              "DTSTART:" + schBeginDate.ToUniversalTime().ToString("yyyyMMdd\\THHmmss\\Z")+"\n"+
                              "DTEND:" + schEndDate.ToUniversalTime().ToString("yyyyMMdd\\THHmmss\\Z")+"\n"+
                              "LOCATION:" + schLocation +
                              //Use Description tag for email clients that cant handle x-alt-desc tag with HTML
                              "\nDESCRIPTION;ENCODING=QUOTED-PRINTABLE:" + schDescription.Replace("<br/>", "").Replace("<b>", "").Replace("</b>", "").Replace("<u>", "").Replace("</u>", "") +
                              "\nSUMMARY:" + schSubject + "\nPRIORITY:3\n" +
                              "STATUS:" + status + "\n" +
                              //Include alternate description if the calendar client can handle html x-alt-desc tag
                              "X-ALT-DESC;FMTTYPE=text/html:<html>"+ schDescription.Replace("\n","<br/>") +"</html>" + "\n" +
                              //"ATTACH:invitation.cvl\n" +
                          "END:VEVENT\n" + "END:VCALENDAR\n";

            ActivityMimeAttachment calendarAttachment = new ActivityMimeAttachment()
            {
                ObjectId = new EntityReference(Email.EntityLogicalName, email.Id),
                ObjectTypeCode = Email.EntityLogicalName,
                Subject = string.Format("Telehealth Visit"),
                Body = Convert.ToBase64String(
                        new ASCIIEncoding().GetBytes(att)),
                FileName = string.Format(CultureInfo.CurrentCulture, "Telehealth-Appointment.ics"),
                
            };
            OrganizationService.Create(calendarAttachment);
            Logger.WriteTxnTimingMessage("Finished Creating Calendar Appointment");
            return;
        }

        internal string formatNotificationEmailBody(List<mcs_resource> providerTechs, List<mcs_resource> patientTechs, List<mcs_resource> providerRooms,
            List<mcs_resource> patientRooms, List<SystemUser> telepresenters, ServiceAppointment sa, mcs_services tsa, List<SystemUser> providers, out string convertedDate)
        {
            Logger.WriteDebugMessage("Starting Formatting Email Body");
            convertedDate = string.Empty;
            string emailBody = "";
            string providerTechsString = "";
            string patientTechsString = null;
            string providerRoomsString = null;
            string patientRoomsString = null;
            string telepresentersString = null;
            string providersString = null;
            string DEALicensed = "";

            //DEA Licensed
            DEALicensed = (sa.cvt_Type.Value == true) ? "This is a Home/Mobile visit, consider Ryan Haight regulations prior to prescribing any controlled medications." : "";

            foreach (mcs_resource r in providerTechs)
            {
                providerTechsString += r.mcs_name;
                if (r.cvt_relateduser != null)
                {
                    SystemUser poc = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, r.cvt_relateduser.Id, new ColumnSet("fullname", "mobilephone", "cvt_officephone", "cvt_teleworkphone"));
                    providerTechsString += "; POC Name: " + poc.FullName + "; ";
                    var phone = poc.MobilePhone == null ? poc.cvt_officephone : poc.MobilePhone;

                    //If TSA is telework (true), then add that number here as well.
                    providerTechsString += ((tsa.cvt_ProviderLocationType != null) && (tsa.cvt_ProviderLocationType.Value == true) && (poc.cvt_TeleworkPhone != null)) ? "POC Telework Phone #: " + poc.cvt_TeleworkPhone + ";" : "";
                    providerTechsString += (phone != null) ? "POC Phone #: " + phone : "";               
                }
                providerTechsString += "<br/>";
                providerTechsString += getComponents(r, sa);
            }

            foreach (mcs_resource r in patientTechs)
            {
                patientTechsString += r.mcs_name;
                if (r.cvt_relateduser != null)
                {
                    SystemUser poc = (SystemUser)OrganizationService.Retrieve(
                        SystemUser.EntityLogicalName, r.cvt_relateduser.Id, new ColumnSet("fullname", "mobilephone", "cvt_officephone"));
                    patientTechsString += "; POC Name: " + poc.FullName + "; ";
                    var phone = poc.MobilePhone == null ? poc.cvt_officephone : poc.MobilePhone;
                    patientTechsString += "POC Phone #: " + phone;
                }
                patientTechsString += "<br/>";
                patientTechsString += getComponents(r, sa);
            }

            foreach (mcs_resource r in providerRooms)
            {
                providerRoomsString += "<b><u>Room:</u></b> " + r.mcs_name;
                if (r.cvt_phone != null)
                    ProRoom += (ProRoom == null) ? r.cvt_phone : ", " + r.cvt_phone;
                        
                providerRoomsString += "<br/>";
            }

            foreach (mcs_resource r in patientRooms)
            {
                patientRoomsString += "<b><u>Room:</u></b> " + r.mcs_name;
                if (r.cvt_phone != null)
                {
                    PatRoom += (PatRoom == null) ? r.cvt_phone : ", " + r.cvt_phone;
                }
                if (DEALicensed == "" && r.mcs_RelatedSiteId != null) {                    
                    var resourceSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, r.mcs_RelatedSiteId.Id, new ColumnSet(true));

                    if (resourceSite.cvt_DEALicensed != null && resourceSite.cvt_DEALicensed.Value == true)
                        patientRoomsString += ";  <u><b>Note: The patient care site is DEA registered.</u></b>";
                    else
                        patientRoomsString += ";  <u><b>Note: The patient care site is NOT DEA registered.</u></b>";
                }
                patientRoomsString += "<br/>";
            }

            foreach (SystemUser t in telepresenters)
            {
                var phone = t.cvt_officephone != null ? t.cvt_officephone : t.MobilePhone;
                telepresentersString += "<b><u>Telepresenter:</u></b> " + t.FullName + ": " + phone + "<br/>";
            }
            foreach (SystemUser t in providers)
            {
                var phone = t.cvt_officephone != null ? t.cvt_officephone : t.MobilePhone;
                providersString += "<b><u>Provider:</u></b> " + t.FullName;
                providersString += (phone != null) ? "; Phone: " + phone : "";
                
                //If TSA is telework (true), then add that number here as well.
                if ((tsa.cvt_ProviderLocationType != null) && (tsa.cvt_ProviderLocationType.Value == true))
                {
                    //Check user for telework number                  
                    providersString += (t.cvt_TeleworkPhone != null) ? "; Telework Phone: " + t.cvt_TeleworkPhone + ";" : "";
                    
                }
                providersString += "<br/>";
            }

           

            Logger.WriteDebugMessage("Getting Time Zone to convert String");
            var proSiteId = sa.mcs_relatedprovidersite;
            int proSiteTimeZone = 35;
            if (proSiteId != null)
            {
                using (var srv = new Xrm(OrganizationService))
                {
                    var site = srv.mcs_siteSet.FirstOrDefault(s => s.Id == proSiteId.Id);
                    if (site != null)
                        proSiteTimeZone = site.mcs_TimeZone != null ? site.mcs_TimeZone.Value : proSiteTimeZone;
                }
            }
            var conversionSuccess = false;
            var timeZoneString = string.Empty; //this will be retrieved in ConvertTimeZone
            var convertedTime = ConvertTimeZone(sa.ScheduledStart.Value, proSiteTimeZone, out timeZoneString, out conversionSuccess);
            var fullDate = conversionSuccess ? convertedTime.ToString("ddd dd MMM yyyy HH:mm") + " " + timeZoneString : convertedTime.ToString("ddd dd MMM yyyy HH:mm") + " GMT";
            convertedDate = fullDate;
            Logger.WriteDebugMessage(string.Format("Converted Time Zone to {0}", timeZoneString));

            if (sa.StatusCode.Value == 9 || sa.StatusCode.Value == 917290000)
            {
                emailBody += "This is an automated Message to notify you that a Telehealth Appointment previously scheduled for " + fullDate +
                "has been <font color='red'><u>Canceled</u></font>.  Please open the attachment and click \"Remove from Calendar\" to remove this event from your calendar.  " +
                "The Details are listed below: <br/><br/>";
            }
            else if (sa.StatusCode.Value == 4)
            {
                emailBody += "This is an automated Message to notify you that a Telehealth Appointment has been <font color='green'><u>Scheduled</u></font> " +
                    "for " + fullDate + ".  " + 
                    "Please open the attachment and click \"Save and Close\" to add this event to your calendar.  " +
                    "The Details are listed below: <br/><br/>";
            }

            //Provider Info
            emailBody += "<br/><font size='5' color='blue'>Provider Site Information:</font><br/>";
            emailBody += providerRoomsString;
            emailBody += (tsa.cvt_provsitevistaclinics != null) ? "<b><u>Vista Clinic:</u></b> " + tsa.cvt_provsitevistaclinics.ToString() : ""; //Needs to be specific Prov VC, not all
            emailBody += (!String.IsNullOrEmpty(providerTechsString)) ? "<br/><b><u>Technologies: </u></b><br/> " + providerTechsString + "<br/>" : "";
            emailBody += providersString;
            emailBody += (TSAProvEmergency != "") ? "<u><b>Provider Site Emergency Responsibilities:</u></b><br/> " + TSAProvEmergency + "<br/>" : "";

            if ((ProRoom != null) || (ProTCTPhone != null && ProTCTName != null))
            {
                emailBody += "<u><b>Telephone Contact Information:</u></b><br/> <ul>";
                emailBody += (ProRoom != null) ? "<li>To direct dial the room: " + ProRoom + "</li>" : "";
                emailBody += (ProTCTPhone != null && ProTCTName != null) ? "<li>To contact the TCT at the provider site, call " + ProTCTName + " at " + ProTCTPhone + ".</li><br/><br/>" : "";
                emailBody += "</ul>";
            }

            //Patient Info
            if (tsa.cvt_Type != true)
            {
                emailBody += "<br/><font size='5' color='blue'>Patient Site Information:</font><br/>";
                emailBody += patientRoomsString;
                emailBody += (tsa.cvt_patsitevistaclinics != null) ? "<b><u>Vista Clinic:</u></b> " + tsa.cvt_patsitevistaclinics.ToString() + "<br/>" : "";
                emailBody += (!String.IsNullOrEmpty(patientTechsString)) ? "<b><u>Technologies: </u></b><br/>" + patientTechsString + "<br/>" : "";
                emailBody += telepresentersString;
                emailBody += (TSAPatEmergency != "") ? "<b><u>Patient Site Emergency Responsibilities:</u></b><br/> " + TSAPatEmergency + "<br/>" : "";
                
                if ((PatRoom != null) || (SiteMainPhone != null) || (SiteLocal911Phone != null) || (PatTCTPhone != null && PatTCTName != null))
                {
                    emailBody += "<u><b>Telephone Contact Information:</u></b><br/> <ul>";
                    emailBody += (PatRoom != null) ? "<li>To direct dial the room: " + PatRoom + "</li>" : "";
                    emailBody += (SiteMainPhone != null) ? "<li>To reach the main phone number for the patient side clinic: " + SiteMainPhone + "</li>" : "";
                    emailBody += (SiteLocal911Phone != null) ? "<li>If you are out of area, this is the number to reach emergency services: " + SiteLocal911Phone + "</li>" : "";
                    emailBody += (PatTCTPhone != null && PatTCTName != null) ? "<li>To contact the TCT at the patient site, call " + PatTCTName + " at " + PatTCTPhone + ".</li>" : "";
                    emailBody += "</ul>";
                }
            }
            else  //Condition for CVT to Home
            {           
                bool? patient = null;
                var meetingSpace = getPatientVirtualMeetingSpace(sa, out patient);

                emailBody += "<br/><br/><font size='5' color='blue'>Home/Mobile Information:</font><br/>";
                emailBody += (DEALicensed != "") ? "<u><b>Site DEA Licensed:</u></b><br/> " + DEALicensed + "<br/>" : "";
                emailBody += (patient == false) ? "<br/> Patient CVT Tablet: <br/>" : "<br/> Virtual Meeting Space: <br/>";

                if (meetingSpace == string.Empty)
                    meetingSpace = "Please Contact Your Clinician for Web Meeting Details";

                //Change to read the ProviderVirtualMeetingSpace on the patient record.
                // https://pexipdemo.com/px/vatest/#/?name=ProviderName&join=1&media=&escalate=1&conference=vatest@pexipdemo.com&pin=1234  
                if (patient == true || ProviderVirtualMeetingSpace != string.Empty)
                {
                    emailBody += "From your Web browser: " + CvtHelper.buildHTMLUrl(ProviderVirtualMeetingSpace, ProviderVirtualMeetingSpace) + "<br/>";

                    var conf = getParamValue(ProviderVirtualMeetingSpace, "conference=");
                    var cid = getParamValue(ProviderVirtualMeetingSpace, "pin=");

                    if (!string.IsNullOrEmpty(conf) && !string.IsNullOrEmpty(cid) )
                        emailBody += String.Format("<br/>And if you wanted to dial from your VTC device:<br/><br/>From any VTC device: {0}<br/>Host CID: {1}", conf, cid);
                }
                else
                    emailBody += meetingSpace + "<br/>";
            }
            emailBody += "<br/><br/>Please Do Not Reply to this message.  It comes from an unmonitored mailbox.";

            Logger.WriteDebugMessage("Finishing Formatting Email Body");
            return emailBody;
        }

        internal string getParamValue(string url, string key)
        {
            var result = string.Empty;
            var parameter = url.Split('&').LastOrDefault(s => s.ToLower().Contains(key));
            var parameterKeyValue = parameter != null ? parameter.Split('=') : null;
            if (parameterKeyValue != null && parameterKeyValue.Count() == 2)
                result = parameterKeyValue[1];
            return result;
        }

        internal string getComponents(mcs_resource technology, ServiceAppointment SA)
        {
            //Get all the components and include the CEVN Alias and IP Addresses for each.  Return the formatted string with a line for each Component
            //virtualMeetingSpace = null;
            string components = null; 
            using (var context = new Xrm(OrganizationService))
            {
                var compList = context.cvt_componentSet.Where(c => c.cvt_relatedresourceid.Id == technology.Id);
                foreach (cvt_component c in compList)
                {
                    if (components == null)
                        components += "<ul>";
                    components += "<li>" + c.cvt_name;
                    switch (c.cvt_name)
                    {
                        case "Codec, Hardware":
                            if (c.cvt_cevnalias != null)
                                components += "; CEVN Alias: " + c.cvt_cevnalias;
                            break;
                        case "Telemedicine Encounter Management":
                        case "Telemed Encounter Management":
                        case "TEMS (Telemedicine Encounter Management Software)":
                        case "TEMS (Telemed Encounter Management Software)":
                            if (c.cvt_ipaddress != null)
                                components += "; IP Address: " + CvtHelper.buildHTMLUrl(c.cvt_ipaddress);
                            break;
                        case "CVT Patient Tablet":
                            if (c.cvt_serialnumber != null)
                                components += "; Serial Number: " + c.cvt_serialnumber;
                            break;
                        case "Virtual Meeting Space":
                            VirtualMeetingSpace = c;
                            break;
                        case "Digital Stethoscope Peripheral":
                            stethIP = c.cvt_ipaddress;
                            break;
                    }
                    //Send URL
                    var url = "";
                    var contact = getDummyContact();
                    var secondaryEntities = new List<Entity>();
                    secondaryEntities.Add(SA);
                    secondaryEntities.Add(contact);
                    if (UrlBuilder.TryGetUrl(OrganizationService, this.GetType().ToString(), c, secondaryEntities, out url))
                        components += "; <a href=" + url + ">" + url + "</a>";
                    components += "</li>";                    
                }
                if (components != null)
                    components += "</ul>";
            }
            return components;        
        }

        internal Contact getDummyContact()
        {
            Contact c = new Contact();
            using (var srv = new Xrm(OrganizationService)){
                c = srv.ContactSet.FirstOrDefault();
            }
            return c;
        }

        internal List<Entity> getPRGs(Guid tsaId, string location)
        {
            QueryByAttribute qa = new QueryByAttribute("cvt_"+location+"resourcegroup");
            qa.ColumnSet = new ColumnSet("cvt_tsaresourcetype", "cvt_relateduserid", "cvt_relatedresourcegroupid", "cvt_relatedresourceid");
            qa.AddAttributeValue("cvt_relatedtsaid", tsaId);
            var results = OrganizationService.RetrieveMultiple(qa);
            return results.Entities.ToList();
        }

        //if patientOrProvider == 1, then get provider resources, otherwise get patient resources
        internal EntityCollection getPRGs(mcs_services tsa, int? patientOrProvider)
        {
            EntityCollection PRGCollection = new EntityCollection();
            using (var context = new Xrm(OrganizationService)){
                if (patientOrProvider == 1)
                {
                    var ProvRGs = context.cvt_providerresourcegroupSet.Where(prg => prg.cvt_RelatedTSAid.Id == tsa.Id);
                    foreach (var res in ProvRGs)
                        PRGCollection.Entities.Add(res);
                }
                else
                {
                    var PatRGs = context.cvt_patientresourcegroupSet.Where(prg => prg.cvt_RelatedTSAid.Id == tsa.Id);
                    foreach (var res in PatRGs)
                        PRGCollection.Entities.Add(res);
                }
            }
            return PRGCollection;
        }

        /// <summary>
        /// looks to find a group resource record for the group and user listed
        /// </summary>
        /// <param name="user">user Id of the group resource</param>
        /// <param name="group">group id of the group resource</param>
        /// <returns>true if a record exists for the group passed in and the user passed it</returns>
        internal bool MatchResourceToGroup(Guid userId, Guid groupId)
        {
            using (var srv = new Xrm(OrganizationService))
                return srv.mcs_groupresourceSet.FirstOrDefault(gr => gr.mcs_relatedResourceGroupId.Id == groupId && gr.mcs_RelatedUserId.Id == userId) != null;
        }

        /// <summary>
        /// looks to find a group resource record for the group and resource listed
        /// </summary>
        /// <param name="resource">resource record of the group resource</param>
        /// <param name="group">resource record of the group resource</param>
        /// <returns>true if a record exists for the group passed in and the resource passed it</returns>

        internal bool MatchResourceToGroup(mcs_resource resource, mcs_resourcegroup group)
        {
            using (var srv = new Xrm(OrganizationService))
                return srv.mcs_groupresourceSet.FirstOrDefault(gr => gr.mcs_relatedResourceGroupId.Id == group.Id && gr.mcs_RelatedResourceId.Id == resource.Id) != null;
        }

        /// <summary>
        /// returns the list of users based on the location (provider or patient) and 
        /// </summary>
        /// <param name="users">list of user activity parties in the SA.Resources field</param>
        /// <param name="location">"patient" or "provider"</param>
        /// <param name="tsaId">id of the tsa associated with the service activity that generated this email</param>
        /// <returns></returns>
        internal List<SystemUser> GetRecipients(EntityCollection users, List<Entity> prgs)
        {
            List<SystemUser> recipients = new List<SystemUser>();
            var singles = prgs.Where(p => p.Attributes.Contains("cvt_tsaresourcetype") && ((OptionSetValue)p.Attributes["cvt_tsaresourcetype"]).Value == (int)cvt_tsaresourcetype.SingleProvider).ToList();
            var groups = prgs.Where(p => p.Attributes.Contains("cvt_tsaresourcetype") && ((OptionSetValue)p.Attributes["cvt_tsaresourcetype"]).Value == (int)cvt_tsaresourcetype.ResourceGroup).ToList();

            foreach (var singlePRG in singles)
            {
                if (singlePRG.Attributes.Contains("cvt_relateduserid") && singlePRG.Attributes["cvt_relateduserid"] != null)
                {
                    SystemUser singleUser = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName,
                        ((EntityReference)singlePRG.Attributes["cvt_relateduserid"]).Id, new ColumnSet(true));
                    if (singleUser != null && singleUser.Id != Guid.Empty)
                    {
                        foreach (ActivityParty u in users.Entities)
                        {
                            if (singleUser.Id == u.PartyId.Id)
                            {
                                recipients.Add(singleUser);
                                break;
                            }
                        }
                    }
                }
            }
            foreach (var groupPRG in groups)
            {
                if (groupPRG.Attributes.Contains("cvt_relatedresourcegroupid") && groupPRG.Attributes["cvt_relatedresourcegroupid"] != null)
                {
                    mcs_resourcegroup group = (mcs_resourcegroup)OrganizationService.Retrieve(
                            mcs_resourcegroup.EntityLogicalName, ((EntityReference)groupPRG.Attributes["cvt_relatedresourcegroupid"]).Id, new ColumnSet(true));
                    //if group type value is any of the "user-type" groups (provider or all required or telepresenter)
                    if ((group.mcs_Type.Value == (int)mcs_resourcetype.Provider) || (group.mcs_Type.Value == (int)mcs_resourcetype.AllRequired) || 
                        (group.mcs_Type.Value == (int)mcs_resourcetype.TelepresenterImager))
                    {
                        //if the user selected is in the resource group, return true and add the user to the entitycollection
                        foreach (ActivityParty u in users.Entities)
                        {
                            var user = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, u.PartyId.Id, new ColumnSet(true));
                            if (MatchResourceToGroup(user.Id, group.Id))
                                recipients.Add(user);
                        }
                    }
                }
            }
            return recipients;
        }

        //separates user types based on value passed into parameter - if user type = null, then only providers, if 0 then both, if 1 then patient
        internal EntityCollection GetRecipients(EntityCollection users, int? userType, mcs_services tsa)
        {
            EntityCollection providers = new EntityCollection();

            foreach (ActivityParty u in users.Entities)
            {
                SystemUser user = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, u.PartyId.Id, new ColumnSet(true));

                //search provider resource group records for a resource of type single provider who matches the user from the list of resources selected
                //otherwise look for a resource group that contains the user
                //if userType is null, return providers, if 0 - return both (patients added below)
                if (userType == null || userType == 0)
                {
                    //get providerResourceGroups for the given TSA
                    var queryResult = getPRGs(tsa, 1).Entities;

                    var singleProviders = queryResult.Where(p => ((cvt_providerresourcegroup)p).cvt_TSAResourceType.Value == 2);
                    var resourceGroups = queryResult.Where(p => ((cvt_providerresourcegroup)p).cvt_TSAResourceType.Value == 0);
                    if (singleProviders.Count() > 0)
                    {
                        foreach (cvt_providerresourcegroup p in singleProviders)
                        {
                            SystemUser singleProviderUser = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, p.cvt_RelatedUserId.Id, new ColumnSet(true));
                            if (singleProviderUser != null && singleProviderUser.Id == user.Id)
                               providers.Entities.Add(user);
                        }
                    }
                    if (resourceGroups.Count() > 0)
                    {
                        //for each resource group, check if the type is provider.  If so, then retrieve the resource group, retrieve get all the resources listed, and compare 
                        //the list of resources in the group to the resource on the Service Activity
                        foreach (cvt_providerresourcegroup rg in resourceGroups)
                        {
                            mcs_resourcegroup group = (mcs_resourcegroup)OrganizationService.Retrieve(
                                mcs_resourcegroup.EntityLogicalName, rg.cvt_RelatedResourceGroupid.Id, new ColumnSet(true));
                            //if group type value is provider or all required
                            if ((group.mcs_Type.Value == 99999999) || (group.mcs_Type.Value == 917290000))
                            {
                                //if the user selected is in the resource group, return true and add the user to the entitycollection
                                if (MatchResourceToGroup(user.Id, group.Id))
                                    providers.Entities.Add(user);
                            }
                        }
                    }               
                }
                //if userType is 1, return patients only, if 0 - return both (providers added above)
                if (userType == 0 || userType == 1)
                {
                    //get patientresourcegroup records for the TSA
                    var queryResult = getPRGs(tsa, 0).Entities;
                    var singlePatients = queryResult.Where(psr => ((cvt_patientresourcegroup)psr).cvt_TSAResourceType.Value == 3);
                    var resourceGroups = queryResult.Where(psr => ((cvt_patientresourcegroup)psr).cvt_TSAResourceType.Value == 0);
                    if (singlePatients.Count() > 0)
                    {
                        foreach (cvt_patientresourcegroup p in singlePatients)
                        {
                            SystemUser singlePatientUser = (SystemUser)OrganizationService.Retrieve(
                                SystemUser.EntityLogicalName, p.cvt_RelatedUserId.Id, new ColumnSet(true));
                            if (singlePatientUser != null && singlePatientUser.Id == user.Id)
                            providers.Entities.Add(user);
                        }
                    }
                    if (resourceGroups.Count() > 0)
                    {
                        foreach (cvt_patientresourcegroup rg in resourceGroups)
                        {
                            mcs_resourcegroup group = (mcs_resourcegroup)OrganizationService.Retrieve(
                                mcs_resourcegroup.EntityLogicalName, rg.cvt_RelatedResourceGroupid.Id, new ColumnSet(true));
                            if (group.mcs_Type.Value == 100000000)
                            {
                                if (MatchResourceToGroup(user.Id, group.Id))
                                    providers.Entities.Add(user);
                            }
                        }
                    }
                }
            }
            return providers;
        }

        /// <summary>
        /// filters down the list of all equipment on SA based on criteria provided
        /// </summary>
        /// <param name="equipment">collection of mcs_resources that correspond to the equipment in the resources field on the sa</param>
        /// <param name="tsa">tsa for the Service Activity</param>
        /// <param name="prgs">cvt_patientresourcegroup or cvt_providerresourcegroup for all resources on the tsa</param>
        /// <param name="equipType">type of mcs_resource (room, tech, vista clinic, etc.)</param>
        /// <returns>the list of mcs_resources based on the filters listed (pro or pat location and equipment type)</returns>
        internal List<mcs_resource> ClassifyResources(EntityCollection equipment, List<Entity> prgs, int? equipType)
        {
            List<mcs_resource> relevantResources = new List<mcs_resource>();

            var singles = prgs.Where(prg => ((OptionSetValue)(prg.Attributes["cvt_tsaresourcetype"])).Value == (int)cvt_tsaresourcetype.SingleResource).ToList();
            var groups = prgs.Where(prg => ((OptionSetValue)(prg.Attributes["cvt_tsaresourcetype"])).Value == (int)cvt_tsaresourcetype.ResourceGroup).ToList();

            foreach (Entity singlePRG in singles)
            {
                if (singlePRG.Attributes.Contains("cvt_relatedresourceid") && singlePRG.Attributes["cvt_relatedresourceid"] != null)
                {
                    foreach (mcs_resource r in equipment.Entities)
                    {
                        if (r.mcs_Type.Value != equipType && equipType != null)
                            continue;

                        mcs_resource resource = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName,
                                ((EntityReference)singlePRG.Attributes["cvt_relatedresourceid"]).Id, new ColumnSet(true));
                        if (resource != null && resource.Id == r.Id)
                            relevantResources.Add(r);
                    }
                }
            }
            foreach (Entity groupPRG in groups)
            {
                if (groupPRG.Attributes.Contains("cvt_relatedresourcegroupid") && groupPRG.Attributes["cvt_relatedresourcegroupid"] != null)
                {
                    mcs_resourcegroup group = (mcs_resourcegroup)OrganizationService.Retrieve(mcs_resourcegroup.EntityLogicalName, 
                                ((EntityReference)groupPRG.Attributes["cvt_relatedresourcegroupid"]).Id, new ColumnSet(true));
                    if (group.mcs_Type.Value == (int)mcs_resourcetype.Room || group.mcs_Type.Value == (int)mcs_resourcetype.Technology || group.mcs_Type.Value == (int)mcs_resourcetype.AllRequired)
                    {
                        foreach (mcs_resource r in equipment.Entities)
                        {
                            if (r.mcs_Type.Value != equipType && equipType != null)
                                continue;

                            if (MatchResourceToGroup(r, group))
                            {
                                relevantResources.Add(r);
                                break;
                            }
                        }
                    }
                }
            }
            
            return relevantResources;
        }
        #endregion

        #region Vista Reminder Email
        internal void SendVistaReminder(Email email)
        {
            Logger.WriteDebugMessage("Beginning Vista Reminder");
            var provTeamMembers = new List<TeamMembership>();
            var patTeamMembers = new List<TeamMembership>();
            ServiceAppointment sa = (ServiceAppointment)OrganizationService.Retrieve(ServiceAppointment.EntityLogicalName, email.mcs_RelatedServiceActivity.Id, new ColumnSet(true));
            var creator = sa.CreatedBy;
            //Either get the Prov/Pat Facility from TSA or from the SA.
            mcs_services tsa = (mcs_services)OrganizationService.Retrieve(mcs_services.EntityLogicalName, sa.mcs_relatedtsa.Id, new ColumnSet(true));

            Logger.WriteDebugMessage("Retrieved Service Activity and TSA associated with this Email");

            //if (tsa.cvt_Type != null && tsa.cvt_Type.Value)
            //{
            //    DeleteVistaReminder(email, "Home/Mobile email does not require a VistA Reminder, deleting email");
            //    return;
            //}

            var provFacilityId = tsa.cvt_ProviderFacility.Id;
            var patFacilityId = tsa.cvt_PatientFacility != null ? tsa.cvt_PatientFacility.Id : Guid.Empty;
            var intraFacility = (provFacilityId == patFacilityId) ? true : false;

            using (var srv = new Xrm(OrganizationService))
            {
                var provTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == provFacilityId && t.cvt_Type.Value == 917290005);
                if (provTeam != null)
                provTeamMembers = srv.TeamMembershipSet.Where(TM => TM.TeamId == provTeam.Id).ToList();
                else
                    Logger.WriteToFile("The provider side Scheduler Team was unable to be found for Service Activity: " + sa.Id);

                var patTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == patFacilityId && t.cvt_Type.Value == 917290005);
                if (patTeam != null)
                patTeamMembers = srv.TeamMembershipSet.Where(TM => TM.TeamId == patTeam.Id).ToList();
                else
                    Logger.WriteToFile("The patient side Scheduler Team was unable to be found for Service Activity: " + sa.Id);
            }

            bool provCheck = false;
            bool patCheck = false;
            EntityCollection provMembers = new EntityCollection();
            EntityCollection patMembers = new EntityCollection();
            if (provTeamMembers.Count == 0)
                Logger.WriteToFile("There are no members of the Scheduler team at " + tsa.cvt_ProviderFacility.Name + ".  Please contact the FTC and ensure this is corrected.");
            else
            {
            foreach (TeamMembership tm in provTeamMembers)
            {
                ActivityParty p = new ActivityParty()
                {
                    PartyId = new EntityReference(SystemUser.EntityLogicalName, tm.SystemUserId.Value)
                };
                provMembers.Entities.Add(p);
                if (creator.Id == tm.SystemUserId.Value)
                    provCheck = true;
                }
            }
            if (patTeamMembers.Count == 0 && !tsa.cvt_Type.Value)
                Logger.WriteToFile(string.Format("There are no members of the Scheduler team at {0}.  Please contact the FTC and ensure this is corrected.", tsa.cvt_PatientFacility != null ? tsa.cvt_PatientFacility.Name : "\"No Facility Listed\""));
            else
            {
            foreach (TeamMembership tm in patTeamMembers)
            {
                ActivityParty p = new ActivityParty()
                {
                    PartyId = new EntityReference(SystemUser.EntityLogicalName, tm.SystemUserId.Value)
                };
                patMembers.Entities.Add(p);
                if (creator.Id == tm.SystemUserId.Value)
                    patCheck = true;
                }
            }
            //If TSA is Store Forward and the scheduler is on the patient Scheduler team side OR if the scheduler is on both Scheduler Teams, then don't send email
            if ((tsa.cvt_AvailableTelehealthModalities != null && tsa.cvt_AvailableTelehealthModalities.Value == 917290001 && patCheck)
                || (patCheck && provCheck))
            {
                Logger.WriteDebugMessage("No need to send out email, Deleting Email.  TSA is SFT and Scheduler is on Pat Team OR Scheduler is on Both Pat and Prov Team");
                try
                {
                    OrganizationService.Delete(email.LogicalName, email.Id);
                    Logger.WriteDebugMessage("Email Deleted");
                }
                catch (Exception ex)
                {
                    Logger.WriteToFile("Unable to Delete Email " + ex.Message + ".  Leaving email as is.");
                }
                return;
            }
            else
            {
                SetupVistaReminderEmail(email, provMembers, patMembers, tsa, provCheck, patCheck, sa);
            }
        }

        internal void DeleteVistaReminder(Email email, string debugMessage)
        {
            Logger.WriteDebugMessage(debugMessage);
            try
            {
                OrganizationService.Delete(email.LogicalName, email.Id);
                Logger.WriteDebugMessage("Email Deleted");
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to Delete Email " + ex.Message + ".  Leaving email as is.");
            }
        }

        internal void SetupVistaReminderEmail(Email email, EntityCollection provMembers, EntityCollection patMembers, mcs_services tsa, bool provCheck, bool patCheck, ServiceAppointment sa)
        {
            Logger.WriteDebugMessage("Beginning SetupVistaReminderEmail");
            mcs_facility patFacility = null;
            if (tsa.cvt_PatientFacility != null)
                patFacility = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, tsa.cvt_PatientFacility.Id, new ColumnSet("mcs_stationnumber"));
            var patStation = patFacility == null ? string.Empty : " (" + patFacility.mcs_StationNumber + ")";
            mcs_facility proFacility = null;
            if (tsa.cvt_ProviderFacility != null)
                proFacility = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, tsa.cvt_ProviderFacility.Id, new ColumnSet("mcs_stationnumber"));
            var proStation = proFacility == null ? string.Empty : " (" + proFacility.mcs_StationNumber + ")";
            email.From = CvtHelper.SetPartyList(sa.CreatedBy);
            int timeZone = 0;
            List<ActivityParty> To = new List<ActivityParty>();
            Logger.WriteDebugMessage(string.Format("Retrieved pat {0} and pro {1} facilities and set the email sender {2}", patStation, proStation, email.From.ToString()));
            if (provCheck == false)
            {
                //Add Prov Scheduler Team Members to To Line
                foreach (ActivityParty ap in provMembers.Entities)
                {
                    To.Add(ap);
                }
                //To = provMembers.Entities.ToList<ActivityParty>();
                Entity proSite; //Either the site or facility of the provider
                if (tsa.cvt_relatedprovidersiteid != null)
                    proSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, tsa.cvt_relatedprovidersiteid.Id, new ColumnSet(true));
                else
                    proSite = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, tsa.cvt_ProviderFacility.Id, new ColumnSet(true));
                timeZone = (int)proSite.Attributes["mcs_timezone"];
            }
            if (patCheck == false)
            {
                //Add Pat Scheduler Team Members to To Line
                foreach (ActivityParty ap in patMembers.Entities)
                {
                    To.Add(ap);
                }
                To = To.GroupBy(A => A.PartyId.Id).Select(g => g.First()).ToList<ActivityParty>(); //Method to select distinct recipients based on recipient ID (since entire Activity Party may not be duplicate)
                Entity patSite;
                if (tsa.cvt_relatedprovidersiteid != null)
                    patSite = (mcs_site)OrganizationService.Retrieve(mcs_site.EntityLogicalName, tsa.cvt_relatedprovidersiteid.Id, new ColumnSet(true));
                else
                    patSite = (mcs_facility)OrganizationService.Retrieve(mcs_facility.EntityLogicalName, tsa.cvt_ProviderFacility.Id, new ColumnSet(true));
                timeZone = (int)patSite.Attributes["mcs_timezone"];
            }
            email.To = To;
            string timeZonesString = string.Empty;
            bool convertSuccess = false;
            var timeConversion = ConvertTimeZone(sa.ScheduledStart.Value, timeZone, out timeZonesString, out convertSuccess);
            Logger.WriteDebugMessage("Vista Reminder (for scheduler action) Time converted to " + timeZonesString);
            var equips = sa.Resources.Where(ap => ap.PartyId.LogicalName == Equipment.EntityLogicalName).ToList();

            //Added to ensure that resources from child appointments (for Group SAs) are also retrieved and included in the list of resources
            var childEquips = GetApptResources(sa, Equipment.EntityLogicalName); 
            equips.AddRange(childEquips);
            var vistaClinics = "Vista Clinic(s): ";
            foreach (var equipment in equips)
            {
                var e = (Equipment)OrganizationService.Retrieve(Equipment.EntityLogicalName, equipment.PartyId.Id, new ColumnSet("mcs_relatedresource"));
                var resource = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName, e.mcs_relatedresource.Id, new ColumnSet("mcs_name","mcs_type"));
                if (resource.mcs_Type.Value == 251920000)
                    vistaClinics += resource.mcs_name + "; ";
            }
            Logger.WriteDebugMessage("Added vista clinics to Scheduler Action email: " + vistaClinics);
            var displayTime = "Appointment Start Time: " + timeConversion + " " + timeZonesString + "; <br/>";
            var body = generateEmailBody(sa.Id, ServiceAppointment.EntityLogicalName, displayTime + vistaClinics, "Click Here to open the Service Activity in TSS");
            var serviceType = tsa.cvt_servicetype.Name;
            if (tsa.cvt_servicesubtype != null)
                serviceType += " : " + tsa.cvt_servicesubtype.Name;
            var status = sa.StatusCode.Value == 4 ? "scheduled" : "canceled";
            var proFacName = tsa.cvt_ProviderFacility == null ? string.Empty : tsa.cvt_ProviderFacility.Name + proStation;
            var patFacName = tsa.cvt_PatientFacility == null  ? string.Empty : tsa.cvt_PatientFacility.Name + patStation;

            if (tsa.cvt_Type != null && tsa.cvt_Type.Value)
                patFacName = "Home/Mobile";

            email.Description = string.Format("A {0} telehealth appointment has been {1} at a remote facility, please {2} this patient in VistA. The provider facility is: {3}. The patient facility is: {4}. {5}",
                serviceType,
                status,
                status == "scheduled" ? "schedule" : "cancel",
                proFacName, 
                patFacName, 
                body);

            if (tsa.cvt_relatedpatientsiteid == null)
                email.Subject += patFacName;

            CvtHelper.UpdateSendEmail(email, OrganizationService);
        }

        internal List<ActivityParty> GetApptResources(ServiceAppointment sa, string filter = "")
        {
            var childResources = new List<ActivityParty>();
            var childAppts = new List<Appointment>();
            using (var srv = new Xrm(OrganizationService))
            {
                childAppts = srv.AppointmentSet.Where(a => a.cvt_serviceactivityid.Id == sa.Id && a.ScheduledStart.Value == sa.ScheduledStart.Value).ToList();
            }
            foreach (var appt in childAppts)
            {
                //If there is no entityType filter listed, then just add all members of appointment requiredAttendees
                if (string.IsNullOrEmpty(filter))
                    childResources.AddRange(appt.RequiredAttendees);
                else
                {
                    foreach (var resource in appt.RequiredAttendees)
                    {
                        //PartyID should never be null, but added null check just in case.  
                        if (resource.PartyId != null && resource.PartyId.LogicalName == filter) 
                            childResources.Add(resource);
                    }
                }
            }
            Logger.WriteDebugMessage("Appointment Resources retrieved for Service Activity: " + sa.Id);
            return childResources;
        }
        #endregion

        #region TSS Privilege e-mails
        internal List<ActivityParty> RetrieveFacilityTeamMembers(Email email, Guid TeamId, IEnumerable<ActivityParty> originalParty)
        {
            using (var srv = new Xrm(OrganizationService))
            {
                var teamMembers = (List<TeamMembership>)(srv.TeamMembershipSet.Where(t => t.TeamId == TeamId).ToList());
                var recipientList = new List<ActivityParty>();

                if (originalParty != null)
                    recipientList.AddRange(originalParty);
                foreach (var member in teamMembers)
                {
                    var party = new ActivityParty()
                    {
                        ActivityId = new EntityReference(email.LogicalName, email.Id),
                        PartyId = new EntityReference(SystemUser.EntityLogicalName, member.SystemUserId.Value)
                    };
                    recipientList.Add(party);
                }
                return recipientList;
            }   
        }

        internal void SendPrivilegingEmail(Email email, Guid tssprivilegeId, string recordType)
        {
            using (var srv = new Xrm(OrganizationService))
            {              
                //Get the related TSS Privileging record
                cvt_tssprivileging tssprivileging = (cvt_tssprivileging)OrganizationService.Retrieve(cvt_tssprivileging.EntityLogicalName, tssprivilegeId, new ColumnSet(true));
                if (tssprivileging.cvt_PrivilegedAtId != null) //Always filled
                {
                    //Establish parameters to clean up queries
                    List<ActivityParty> recipient = new List<ActivityParty>();
                    Boolean isRegardingPrivHome = true;
                    cvt_tssprivileging homePrivRecord = tssprivileging;
                    List<Team> homeCPTeam = new List<Team>();
                    cvt_tssprivileging proxyPrivRecord = new cvt_tssprivileging();
                    List<Team> proxyCPTeam = new List<Team>();

                    //Regarding is Proxy, overwrite homeProvRecord and isRegardingPrivHome
                    if ((tssprivileging.cvt_TypeofPrivileging != null) && (tssprivileging.cvt_TypeofPrivileging.Value == 917290001) && (tssprivileging.cvt_ReferencedPrivilegeId != null))
                    {
                        isRegardingPrivHome = false;
                        homePrivRecord = (cvt_tssprivileging)OrganizationService.Retrieve(cvt_tssprivileging.EntityLogicalName, tssprivileging.cvt_ReferencedPrivilegeId.Id, new ColumnSet(true));
                        proxyPrivRecord = tssprivileging;
                        proxyCPTeam = srv.TeamSet.Where(p => p.cvt_Facility.Id == proxyPrivRecord.cvt_PrivilegedAtId.Id && p.cvt_Type.Value == 917290003).Distinct().ToList();
                    }

                    //Home CPTeam is always set
                    homeCPTeam = srv.TeamSet.Where(p => p.cvt_Facility.Id == homePrivRecord.cvt_PrivilegedAtId.Id && p.cvt_Type.Value == 917290003).Distinct().ToList();

                    //Get the owner of the workflow for the From field
                    email.From = CvtHelper.GetWorkflowOwner("Privileging: PPE Submitted", OrganizationService);

                    //Initial Privileging
                    if (email.Subject.IndexOf("Telehealth Notification: A Provider is now privileged") != -1)
                    {
                        if (isRegardingPrivHome) //Home
                        {
                            foreach (var cp in homeCPTeam)
                            {
                                email.To = RetrieveFacilityTeamMembers(email, cp.Id, email.To);
                            }
                            customMessage = "A Home Privilege has been granted at Facility: " + homePrivRecord.cvt_PrivilegedAtId.Name;
                        }
                        else //Proxy
                        {
                            foreach (var cp in proxyCPTeam)
                            {
                                email.To = RetrieveFacilityTeamMembers(email, cp.Id, email.To);
                            }
                            foreach (var cp in homeCPTeam)
                            {
                                email.Cc = RetrieveFacilityTeamMembers(email, cp.Id, email.To);
                            }
                            customMessage = "A Proxy Privilege has been granted at Facility: " + proxyPrivRecord.cvt_PrivilegedAtId.Name;
                            customMessage += "<br/>Home Privilege: The provider's HOME privileging is at Facility: " + homePrivRecord.cvt_PrivilegedAtId.Name;
                            customMessage += "<br/>Reminder: Please enter the provider into your local PPE process.";
                        }
                    }
                    //Check if E-mail subject = "Renewal"
                    else if (email.Subject.IndexOf("Telehealth Notification: Upcoming Renewal for a Provider") != -1)
                    {
                        if (isRegardingPrivHome == true)
                        {
                            //Update Home/Primary TSS Privilege //If Status Reason = Privileged; set to In Renewal
                            if (homePrivRecord.statuscode.Value == 917290001)
                            {
                                //Declare new object
                                cvt_tssprivileging homeRecord = new cvt_tssprivileging()
                                {
                                    Id = homePrivRecord.Id,
                                    statuscode = new OptionSetValue(917290002)
                                };
                                //homePrivRecord.statuscode.Value = 917290002;
                                OrganizationService.Update(homeRecord);
                            }
                            ////Get Proxy TSS Privileging records
                            //var proxyPrivs = srv.cvt_tssprivilegingSet.Where(p => p.cvt_ReferencedPrivilegeId.Id == homePrivRecord.Id).Distinct().ToList();
                            //foreach (var result in proxyPrivs)
                            //{
                            //    //If status reason == privileged, set to Renewal
                            //    if (result.statuscode.Value == 917290001)
                            //    {
                            //        //Declare new object
                            //        cvt_tssprivileging childRecord = new cvt_tssprivileging()
                            //        {
                            //            Id = result.Id,
                            //            statuscode = new OptionSetValue(917290002)
                            //        };
                            //        //result.statuscode.Value = 917290002;
                            //        OrganizationService.Update(childRecord);
                            //    }
                            //}
                            foreach (var cp in homeCPTeam)
                            {
                                email.To = RetrieveFacilityTeamMembers(email, cp.Id, email.To);
                            }
                            //Edit the E-mail body
                            customMessage = "Provider's Home Privilege is up for renewal at Facility: " + homePrivRecord.cvt_PrivilegedAtId.Name;
                            //customMessage += "<br/>Note: Home and any Proxy Privileges have been set to 'In Renewal' status.";
                            customMessage += "<br/>Note: Home Privilege has been set to 'In Renewal' status.";
                        }
                    }
                    //Else if Suspended
                    else if (email.Subject.IndexOf("Telehealth Notification: A Provider's Privileging has been Suspended") != -1)
                    {
                        if (isRegardingPrivHome == true)
                        {
                            foreach (var cp in homeCPTeam)
                            {
                                email.To = RetrieveFacilityTeamMembers(email, cp.Id, email.To);
                            }
                            //Update the provider's record
                            SystemUser provider = (SystemUser)OrganizationService.Retrieve(SystemUser.EntityLogicalName, tssprivileging.cvt_ProviderId.Id, new ColumnSet(true));
                            provider.cvt_disable = true;
                            OrganizationService.Update(provider);

                            //Edit the E-mail body
                            customMessage = "A provider's HOME privileging has been suspended at Facility: " + homePrivRecord.cvt_PrivilegedAtId.Name;
                            customMessage += "<br/>Note: THE PROVIDER'S USER RECORD HAS BEEN DISABLED.  This Provider can no longer be scheduled in the system.";
                            customMessage += "<br/>Suspension: The suspension occurred at Facility: " + homePrivRecord.cvt_PrivilegedAtId.Name;
                        }
                        else
                        {
                            foreach (var cp in homeCPTeam)
                            {
                                email.To = RetrieveFacilityTeamMembers(email, cp.Id, email.To);
                            }
                            foreach (var cp in proxyCPTeam)
                            {
                                email.Cc = RetrieveFacilityTeamMembers(email, cp.Id, email.Cc);
                            }

                            //Edit the E-mail body
                            customMessage = "A provider's PROXY privileging has been suspended at Facility: " + proxyPrivRecord.cvt_PrivilegedAtId.Name;
                            customMessage += "<br/>Note: This Provider is still schedulable in the system.";
                            customMessage += "<br/>Suspension: The suspension occurred at Facility:" + proxyPrivRecord.cvt_PrivilegedAtId.Name;
                            customMessage += "<br/>Home Privilege: The provider's HOME privileging is at Facility: " + homePrivRecord.cvt_PrivilegedAtId.Name;                                     
                        }                        
                    }
                    //Generate body and then send
                    customMessage += "<br/>Reminder: Notify all pertinent C&P Officers and Service Chiefs.";
                    email.Description = generateEmailBody(tssprivilegeId, "cvt_tssprivileging", customMessage, "Please click this link to view the Privileging record.");
                    if (email.To != null)
                        CvtHelper.UpdateSendEmail(email, OrganizationService);
                }
            }
        }
        #endregion 

        #region FPPE/OPPE Check e-mail
        //Add SC Team to TO/CC 7/24/15
        internal void SendTriggerEmail(Email email, Guid fppeID, string recordType)
        {
            Logger.setMethod = "SendTriggerEmail";
            Logger.WriteDebugMessage("Starting");
            using (var srv = new Xrm(OrganizationService))
            {
                //Check system generated e-mail
                if (email.Subject.IndexOf("Telehealth Notification: PPE Submitted") != -1)
                {
                    //Get the owner of the workflow for the From field
                    email.From = CvtHelper.GetWorkflowOwner("Privileging: PPE Submitted", OrganizationService);

                    Logger.WriteDebugMessage("Get the PPE related to the email");
                    cvt_qualitycheck fppe = (cvt_qualitycheck)OrganizationService.Retrieve(cvt_qualitycheck.EntityLogicalName, fppeID, new ColumnSet(true));

                    //Find the Privilege record associated and navigate to that record
                    if (fppe.cvt_TSSPrivilegingId != null)
                    {               
                        cvt_tssprivileging fppePriv = (cvt_tssprivileging)OrganizationService.Retrieve(cvt_tssprivileging.EntityLogicalName, fppe.cvt_TSSPrivilegingId.Id, new ColumnSet(true));
                        //Assuming Home
                        cvt_tssprivileging homePriv = fppePriv;
                        Boolean isRegardingRelatedPrivHome = true;
                        Guid homeServiceType = fppePriv.cvt_ServiceTypeId != null ? fppePriv.cvt_ServiceTypeId.Id : Guid.Empty;
                        List<Team> homeSCTeams = new List<Team>();
                        List<Team> proxySCTeams = new List<Team>();

                        if (homePriv.cvt_TypeofPrivileging.Value != 917290000) //Overwriting since Proxy
                        {
                            isRegardingRelatedPrivHome = false;
                            homePriv = (cvt_tssprivileging)OrganizationService.Retrieve(cvt_tssprivileging.EntityLogicalName, fppePriv.cvt_ReferencedPrivilegeId.Id, new ColumnSet(true));
                        }

                        if (homeServiceType == Guid.Empty)
                            homeServiceType = homePriv.cvt_ServiceTypeId != null ? homePriv.cvt_ServiceTypeId.Id : Guid.Empty;

                        //Add Service Chief Team - should only ever be one
                        homeSCTeams = srv.TeamSet.Where(t => t.cvt_Facility.Id == homePriv.cvt_PrivilegedAtId.Id && t.cvt_Type.Value == 917290001 && t.cvt_ServiceType.Id == homeServiceType).Distinct().ToList();

                        foreach (var result in homeSCTeams)
                        {
                            email.To = RetrieveFacilityTeamMembers(email, result.Id, email.To);
                        }
                        var flag = "Green";
                        if (fppe.cvt_Flag != null && fppe.cvt_Flag.Value != 917290000)
                            flag = "Red";

                        //Edit the E-mail body
                        customMessage = "A " + flag + " flagged FPPE/OPPE has been submitted.";

                        //If actually from Proxy, set those team members as Cc
                        if (isRegardingRelatedPrivHome == false)
                        {
                            if (fppePriv.cvt_PrivilegedAtId != null)
                                proxySCTeams = srv.TeamSet.Where(t => t.cvt_Facility.Id == fppePriv.cvt_PrivilegedAtId.Id && 
                                    t.cvt_Type.Value == 917290001 && t.cvt_ServiceType.Id == homeServiceType).Distinct().ToList();
                            foreach (var proxyTeam in proxySCTeams)
                            {
                                email.Cc = RetrieveFacilityTeamMembers(email, proxyTeam.Id, email.Cc);
                            }
                            customMessage += "<br/>This FPPE/OPPE was submitted regarding the Proxy Privilege.";
                            customMessage += "<br/>Proxy Privilege is at Facility: " + fppePriv.cvt_PrivilegedAtId.Name;
                            
                        }

                        customMessage += "<br/>Home Privilege is at Facility: " + homePriv.cvt_PrivilegedAtId.Name;
                        customMessage += "<br/>Specialty: " + homePriv.cvt_ServiceTypeId.Name;
                        //customMessage += "Date Range: " + fppe.cvt_EvaluationStartDate + " to " + fppe.cvt_EvaluationEndDate;
                        customMessage += "<br/>Reminder: Notify all pertinent C&P Officers and Service Chiefs.";
                        email.Description = generateEmailBody(fppeID, "cvt_qualitycheck", customMessage, "Please click this link to view the FPPE/OPPE record.");
                        if (email.To != null)
                            CvtHelper.UpdateSendEmail(email, OrganizationService);
                    }
                }
            }
        }
        #endregion

        #region Implementing additional interface methods
        public override string McsSettingsDebugField
        {
            get { return "cvt_serviceactivityplugin"; }
        }
        #endregion
    }
}
