﻿using MCSUtilities2011;
using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using VA.TMP.CRM;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace MCSShared
{
    public static partial class CvtHelper
    {
        #region Team
        /// <summary>
        /// Passing in either a SystemUser or a Team, associate security roles
        /// </summary>
        /// <param name="thisId">Accepts either Team or SystemUser Guids</param>
        /// <param name="EntityLogicalName"></param>
        /// <param name="OrganizationService"></param>
        /// <param name="Logger"></param>
        internal static void UpdateSecurityRoles(Guid thisId, string EntityLogicalName, IOrganizationService OrganizationService, MCSLogger Logger, out int addRoles, out int removeRoles)
        {
            using (var srv = new Xrm(OrganizationService))
            {
                //Vars to Track Counts
                bool addedRoles = false;
                bool removedRoles = false;
                addRoles = 0;
                removeRoles = 0;

                IQueryable currentRoleResults = null;
                string rolesToAdd = "";
                int totalRoles = 0;
                Guid businessUnitId = Guid.Empty;
                string thisRecordName = "";
                string displayMessage = "";
                EntityReferenceCollection RolesToAssociate = new EntityReferenceCollection();
                EntityReferenceCollection RolesToDisassociate = new EntityReferenceCollection();

                if (EntityLogicalName == Team.EntityLogicalName)
                {
                    var thisTeam = srv.TeamSet.First(t => t.Id == thisId);
                    if (thisTeam.BusinessUnitId == null)
                    {
                        Logger.WriteDebugMessage("No BU set on Team, so no roles were added.");
                        //Maybe check For Facility value and align to that BU.
                        return; //no BU, so exit                      
                    }
                    businessUnitId = thisTeam.BusinessUnitId.Id;
                    thisRecordName = thisTeam.Name;
                    rolesToAdd = "TMP User";
                    if (thisTeam.cvt_Type != null)
                    {
                        switch (thisTeam.cvt_Type.Value)
                        {
                            case (int)Teamcvt_Type.FTC:
                                //User + Approver + FTC
                                rolesToAdd += "|TMP TSA Approver|TMP TSA Developer";
                                break;
                            case (int)Teamcvt_Type.ServiceChief:
                                rolesToAdd += "|TMP TSA Approver|TMP Privileging Feedback";
                                break;
                            case (int)Teamcvt_Type.ChiefofStaff:
                                rolesToAdd += "|TMP TSA Approver";
                                break;
                            case (int)Teamcvt_Type.CredentialingandPrivileging:
                                rolesToAdd += "|TMP TSA Approver|TMP Privileging";
                                break;
                            //case "917290004": //TSA Notification
                            //    break;
                            case (int)Teamcvt_Type.Scheduler:
                                //User + Scheduler
                                rolesToAdd += "|TMP Scheduler";
                                break;
                            case (int)Teamcvt_Type.DataAdministrator:
                                rolesToAdd += "|TSS Data Administrator";
                                break;
                        }
                    }
                    else
                    {
                        if (thisTeam.Name.Contains("("))
                            rolesToAdd = "TMP Site Team";
                        else
                            rolesToAdd += "|TMP BU Team";
                    }

                    displayMessage = "Missing Default Role(s) systematically associated (based on Type) to Team: ";
                    currentRoleResults = srv.TeamRolesSet.Where(tr => tr.TeamId == thisId);
                }
                else if (EntityLogicalName == SystemUser.EntityLogicalName)
                {
                    var thisUser = srv.SystemUserSet.First(u => u.Id == thisId);
                    if (thisUser.cvt_SecurityRolesString == null)
                        return; //Future: Auto add Roles based on Type field on User Record (like Team)
                    rolesToAdd = thisUser.cvt_SecurityRolesString;
                    businessUnitId = thisUser.BusinessUnitId.Id;
                    thisRecordName = thisUser.FullName;
                    displayMessage = " Former Roles re-associated in new Business Unit for user: ";
                    currentRoleResults = srv.SystemUserRolesSet.Where(u => u.SystemUserId == thisId);
                }

                //Common Code
                if (rolesToAdd == "TMP User")
                {
                    //Check for Site Team, Facility Team, VISN Team.
                    //Could query for site or facility or visn for exact match of name and if so, add team depending on which result?
                    Logger.WriteDebugMessage(String.Format("No roles to assign for {0}: {1}", EntityLogicalName, thisRecordName));
                    return;
                }
                var SecRoles = rolesToAdd.Contains('|') ? rolesToAdd.Split('|') : new string[] { rolesToAdd }; //split or only one value
                totalRoles = SecRoles.Count();

                try
                {
                    //Get the Roles to Add
                    foreach (var defaultRole in SecRoles)
                    {
                        Boolean matchToIgnoreAdd = false;
                        //If no current roles, do not try to match
                        if (currentRoleResults != null)
                        {
                            foreach (Entity role in currentRoleResults)
                            {
                                if (matchToIgnoreAdd)
                                    break;

                                Guid roleGuid = Guid.Empty;
                                if (role != null && role.Attributes != null && role.Attributes["roleid"] != null)
                                    roleGuid = (Guid)(role.Attributes["roleid"]);
                                var roleRecord = srv.RoleSet.FirstOrDefault(r => r.Id == roleGuid);
                                if ((roleRecord != null) && (roleRecord.Name != null))
                                    matchToIgnoreAdd = (defaultRole == roleRecord.Name) ? true : false;
                            }
                        }
                        if (!matchToIgnoreAdd) //No Match, so add
                        {
                            var newRole = srv.RoleSet.FirstOrDefault(sr => sr.BusinessUnitId.Id == businessUnitId && sr.Name == defaultRole.ToString());
                            RolesToAssociate.Add(new EntityReference(Role.EntityLogicalName, newRole.Id));
                            addRoles++;
                            addedRoles = true;
                        }
                    }

                    //If no current roles, do not try to match
                    if (currentRoleResults != null)
                    {
                        //Get the Roles to Remove
                        foreach (Entity role in currentRoleResults)
                        {
                            Guid roleGuid = Guid.Empty;
                            if (role != null && role.Attributes != null && role.Attributes["roleid"] != null)
                                roleGuid = (Guid)(role.Attributes["roleid"]);
                            var roleRecord = srv.RoleSet.FirstOrDefault(r => r.Id == roleGuid);
                            Boolean matchToRemove = SecRoles.FirstOrDefault(r => r == roleRecord.Name) == null;
                            if (matchToRemove)
                            {
                                RolesToDisassociate.Add(new EntityReference(Role.EntityLogicalName, roleGuid));
                                removeRoles++;
                                removedRoles = true;
                            }
                        }
                    }

                    if (addedRoles)
                    {
                        OrganizationService.Associate(EntityLogicalName, thisId, new Relationship(EntityLogicalName.ToLower() + "roles_association"), RolesToAssociate);
                        Logger.WriteDebugMessage(String.Format("{0}/{1} {2}{3}", addRoles, totalRoles, displayMessage, thisRecordName));
                    }

                    if (removedRoles)
                    {
                        OrganizationService.Disassociate(EntityLogicalName, thisId, new Relationship(EntityLogicalName.ToLower() + "roles_association"), RolesToDisassociate);
                        Logger.WriteDebugMessage(String.Format("{0} role(s) removed from {1}", removeRoles, thisRecordName));
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Logger.WriteToFile(ex.Message);
                }
                catch (Exception ex)
                {
                    Logger.WriteToFile(ex.Message);
                }
            }
        }

        /// <summary>
        /// Update Site Team to the specified Business Unit and associate the TMP Site Team role from that new Business Unit
        /// </summary>
        /// <param name="teamId"></param>
        /// <param name="buId"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        internal static void UpdateSiteTeam(Guid teamId, Guid buId, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.setMethod = "CvtHelper.UpdateSiteTeam";
            try
            {
                var req = new SetParentTeamRequest()
                {
                    TeamId = teamId,
                    BusinessId = buId
                };

                OrganizationService.Execute(req);
                Logger.WriteDebugMessage("Site team record updated to new BU");

                //Need to reassign the TMP Site Team security role
                //Logger.WriteDebugMessage("About to query for appropriate security role.");
                var siteTeamrole = new Role();
                using (var srv = new Xrm(OrganizationService))
                {
                    siteTeamrole = srv.RoleSet.FirstOrDefault(r => r.Name == "TMP Site Team" && r.BusinessUnitId.Id == buId);
                }
                if (siteTeamrole != null)
                {
                    // Associate the user with the role.
                    OrganizationService.Associate(Team.EntityLogicalName, teamId, new Relationship("teamroles_association"),
                        new EntityReferenceCollection() { new EntityReference(Role.EntityLogicalName, siteTeamrole.Id) });
                    Logger.WriteDebugMessage("Associated TMP Site Team role to team.");
                }
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Exception with UpdateSiteTeam for " + teamId + ". Message: " + ex.Message);
            }
        }

        /// <summary>
        /// Update the Primary Team field to match the VISN on the user record.
        /// </summary>
        /// <param name="thisId"></param>
        internal static void UpdatePrimaryTeam(Guid thisId, IOrganizationService OrganizationService, MCSLogger Logger)
        {
            Logger.setMethod = "UpdatePrimaryTeam";
            Logger.WriteDebugMessage("Starting UpdatePrimaryTeam");
            using (var srv = new Xrm(OrganizationService))
            {
                var thisUser = srv.SystemUserSet.FirstOrDefault(su => su.Id == thisId);
                Logger.WriteDebugMessage("Retrieved User Set.");
                if (thisUser.cvt_PrimaryTeam == null || (thisUser.cvt_PrimaryTeam.Name != thisUser.BusinessUnitId.Name))
                {
                    Logger.WriteDebugMessage("Mismatched Primary Team and VISN name, update team.");
                    //Need to update the team to match the VISN
                    var VISNTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisUser.BusinessUnitId.Name);
                    if (VISNTeam != null)
                    {
                        Logger.WriteDebugMessage("Found Team of user's current VISN. About to Update.");
                        SystemUser userUpdate = new SystemUser()
                        {
                            Id = thisUser.Id,
                            cvt_PrimaryTeam = new EntityReference() { LogicalName = Team.EntityLogicalName, Id = VISNTeam.Id, Name = VISNTeam.Name }
                        };
                        OrganizationService.Update(userUpdate);
                        Logger.WriteDebugMessage("Update successful.");
                    }
                }
            }
        }

        #endregion

        #region AssignOwner
        /// <summary>
        /// Assigns Ownership of the record to the correct team. 
        /// </summary>
        /// <param name="thisRecord"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns>Assigned - true/false</returns>
        internal static bool AssignOwner(Entity thisRecord, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            #region Setup
            Logger.setMethod = "AssignOwner";
            Logger.WriteDebugMessage("Starting AssignOwner");

            EntityReference assignOwner = new EntityReference()
            {
                LogicalName = Team.EntityLogicalName,
            };
            Entity findTeam = new Entity();
            #endregion

            switch (thisRecord.LogicalName)
            {
                #region TSS Site
                case mcs_site.EntityLogicalName:
                    var thisSite = (mcs_site)thisRecord;
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisSite.mcs_name);
                    break;
                #endregion
                #region TSS Resource
                case mcs_resource.EntityLogicalName:

                    var thisResource = (mcs_resource)thisRecord;
                    //Check that the TSS Resource's owner is the same name as the site
                    if (thisResource.mcs_RelatedSiteId == null || thisResource.OwnerId.Name == thisResource.mcs_RelatedSiteId.Name)
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisResource.mcs_RelatedSiteId.Name);

                    break;
                #endregion
                #region Component
                case cvt_component.EntityLogicalName:

                    var thisComponent = (cvt_component)thisRecord;
                    if (thisComponent.cvt_relatedresourceid == null)
                        return false;

                    mcs_resource parentResource = (mcs_resource)OrganizationService.Retrieve(mcs_resource.EntityLogicalName, thisComponent.cvt_relatedresourceid.Id, new ColumnSet(true));
                    if (parentResource.mcs_RelatedSiteId == null)
                        return false;

                    //Check that the TSS Resource's owner is the same name as the site, if so, pass along or query for correct one.
                    if (parentResource.OwnerId.Name == parentResource.mcs_RelatedSiteId.Name)
                        findTeam.Id = parentResource.OwnerId.Id;
                    else
                    {
                        //Get the correct team
                        using (var srv = new Xrm(OrganizationService))
                            findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == parentResource.mcs_RelatedSiteId.Name);
                        if (findTeam == null)
                            return false;

                        //Also Reassign TSS Resource
                        AssignRequest assignTSSResource = new AssignRequest()
                        {
                            Assignee = findTeam.ToEntityReference(),
                            Target = new EntityReference(mcs_resource.EntityLogicalName, parentResource.Id)
                        };

                        OrganizationService.Execute(assignTSSResource);
                        Logger.WriteDebugMessage("TSS Resource reassigned to Site Team.");
                    }
                    //No need to reassign, corrrect owner
                    if (thisComponent.OwnerId.Id == findTeam.Id)
                        return false;
                    break;
                #endregion
                #region TSS Resource Group
                case mcs_resourcegroup.EntityLogicalName:

                    var thisResourceGroup = (mcs_resourcegroup)thisRecord;
                    if (thisResourceGroup.mcs_relatedSiteId == null || thisResourceGroup.OwnerId.Name == thisResourceGroup.mcs_relatedSiteId.Name)
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisResourceGroup.mcs_relatedSiteId.Name);
                    break;
                #endregion
                #region Group Resource
                case mcs_groupresource.EntityLogicalName:

                    var thisGroupResource = (mcs_groupresource)thisRecord;
                    if (thisGroupResource.mcs_relatedSiteId == null || thisGroupResource.OwnerId.Name == thisGroupResource.mcs_relatedSiteId.Name)
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name == thisGroupResource.mcs_relatedSiteId.Name);
                    break;
                #endregion
                #region MTSA
                case cvt_mastertsa.EntityLogicalName:

                    var thisMTSA = (cvt_mastertsa)thisRecord;
                    //Check the Provider Facility vs OwnerId
                    if ((thisMTSA.OwnerId.Name.Contains("FTC") && (thisMTSA.OwnerId.Name.Contains(thisMTSA.cvt_ProviderFacility.Name))))
                        return false;

                    //Get the correct team
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility == thisMTSA.cvt_ProviderFacility && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.FTC);

                    if (findTeam == null)
                    {
                        string name = (thisMTSA.cvt_ProviderFacility != null) ? thisMTSA.cvt_ProviderFacility.Name : "No Facility Listed";
                        Logger.WriteDebugMessage("FTC Team not found for Facility: " + name + " Creating team.");
                        Team newTeam = new Team()
                        {
                            cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, thisMTSA.cvt_ProviderFacility.Id),
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.FTC),
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisMTSA.OwningBusinessUnit.Id),
                            Name = "FTC Approval Group @ " + thisMTSA.cvt_ProviderFacility.Name
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }
                    else
                    {
                        //Check that the TSS Resource's owner is the same name as the site, if so, pass along or query for correct one.
                        if (thisMTSA.OwnerId != null && thisMTSA.OwnerId.Id == findTeam.Id)
                            return false;
                    }
                    break;
                #endregion
                #region TSA
                case mcs_services.EntityLogicalName:

                    var thisTSA = (mcs_services)thisRecord;

                    //Check the Provider Facility vs OwnerId
                    if ((thisTSA.OwnerId.Name.Contains("FTC") && (thisTSA.OwnerId.Name.Contains(thisTSA.cvt_ProviderFacility.Name))))
                        return false;

                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility == thisTSA.cvt_ProviderFacility && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.FTC);

                    if (findTeam == null)
                    {
                        string name = (thisTSA.cvt_ProviderFacility != null) ? thisTSA.cvt_ProviderFacility.Name : "No Facility Listed";
                        Logger.WriteDebugMessage("FTC Team not found for Facility: " + name + " Creating team.");
                        Team newTeam = new Team()
                        {
                            cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, thisTSA.cvt_ProviderFacility.Id),
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.FTC),
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisTSA.OwningBusinessUnit.Id),
                            Name = "FTC Approval Group @ " + thisTSA.cvt_ProviderFacility.Name
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }

                    if (thisTSA.OwnerId != null && thisTSA.OwnerId.Id == findTeam.Id)
                        return false;
                    break;
                #endregion
                #region Telehealth Privileging
                case cvt_tssprivileging.EntityLogicalName:

                    var thisPrivileging = (cvt_tssprivileging)thisRecord;

                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == thisPrivileging.cvt_PrivilegedAtId.Id && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.CredentialingandPrivileging);

                    //Can't find team or already corrrect owner
                    if (findTeam == null)
                    {
                        Logger.WriteDebugMessage("Can't find CP Team for Facility: " + thisPrivileging.cvt_PrivilegedAtId.Name + ". Creating team.");
                        Team newTeam = new Team()
                        {
                            cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, thisPrivileging.cvt_PrivilegedAtId.Id),
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.CredentialingandPrivileging),
                            Name = "Credentialing and Privileging Officer Approval Group @ " + thisPrivileging.cvt_PrivilegedAtId.Name,
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisPrivileging.OwningBusinessUnit.Id)
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }
                    else
                    {
                        if (thisPrivileging.OwnerId.Id == findTeam.Id)
                            return false;
                    }

                    break;
                #endregion
                #region FPPE/OPPE
                case cvt_qualitycheck.EntityLogicalName:

                    var thisPPE = (cvt_qualitycheck)thisRecord;
                    Logger.WriteDebugMessage("Querying for the parent priv.");
                    var privilegeParent = (cvt_tssprivileging)OrganizationService.Retrieve(cvt_tssprivileging.EntityLogicalName, thisPPE.cvt_TSSPrivilegingId.Id, new ColumnSet(true));
                    Logger.WriteDebugMessage("Querying for the Service Chief Team for the facility.");

                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.cvt_Facility.Id == privilegeParent.cvt_PrivilegedAtId.Id
                            && t.cvt_Type != null && t.cvt_Type.Value == (int)Teamcvt_Type.ServiceChief
                            && t.cvt_ServiceType != null && t.cvt_ServiceType.Id == privilegeParent.cvt_ServiceTypeId.Id);

                    //Can't find team or already correct owner
                    if (findTeam == null)
                    {
                        Logger.WriteDebugMessage("Can't find Service Chief team for Facility: " + privilegeParent.cvt_PrivilegedAtId.Name + " with Specialty: " + privilegeParent.cvt_ServiceTypeId.Name + ". Creating team.");
                        Team newTeam = new Team()
                        {
                            cvt_Facility = new EntityReference(mcs_facility.EntityLogicalName, privilegeParent.cvt_PrivilegedAtId.Id),
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.ServiceChief),
                            Name = privilegeParent.cvt_ServiceTypeId.Name + " Service Chief Approval Group @ " + privilegeParent.cvt_PrivilegedAtId.Name,
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            cvt_ServiceType = privilegeParent.cvt_ServiceTypeId,
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, privilegeParent.OwningBusinessUnit.Id)
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }
                    else
                    {
                        if (thisPPE.OwnerId.Id == findTeam.Id)
                            return false;
                    }
                    break;

                #endregion
                #region Prov/Pat Site Resources
                case cvt_providerresourcegroup.EntityLogicalName:
                case cvt_patientresourcegroup.EntityLogicalName:
                    var owner = (EntityReference)thisRecord["ownerid"];
                    var resourceSite = OrganizationService.Retrieve(mcs_site.EntityLogicalName, ((EntityReference)thisRecord["cvt_relatedsiteid"]).Id, new ColumnSet(true)).ToEntity<mcs_site>();
                    if (resourceSite.mcs_FacilityId == null)
                        throw new InvalidPluginExecutionException(string.Format("Site {0} does not have a facility", resourceSite.mcs_name));
                    if (owner.Name.Contains("FTC") && owner.Name.Contains(resourceSite.mcs_FacilityId.Name))
                        return false;
                    using (var srv = new Xrm(OrganizationService))
                        findTeam = srv.TeamSet.FirstOrDefault(t => t.Name.Contains("FTC") && t.Name.Contains(resourceSite.mcs_FacilityId.Name));
                    if (findTeam == null)
                    {
                        var facility = OrganizationService.Retrieve(mcs_facility.EntityLogicalName, resourceSite.mcs_FacilityId.Id, new ColumnSet(true)).ToEntity<mcs_facility>();
                        Team newTeam = new Team()
                        {
                            cvt_Facility = resourceSite.mcs_FacilityId,
                            cvt_Type = new OptionSetValue((int)Teamcvt_Type.FTC),
                            TeamType = new OptionSetValue((int)TeamTeamType.Owner),
                            BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, facility.OwningBusinessUnit.Id),
                            Name = "FTC Approval Group @ " + facility.mcs_name
                        };
                        findTeam = new Team();
                        findTeam.Id = CreateTeamForAssign(newTeam, Logger, OrganizationService);
                    }
                    break;
                #endregion
                #region Service Activity
                case ServiceAppointment.EntityLogicalName:
                    var thisServiceAppointment = (ServiceAppointment)thisRecord;
                    using (var srv = new Xrm(OrganizationService))
                    {
                        if (thisServiceAppointment.cvt_Type.HasValue)
                        {
                            var site = (thisServiceAppointment.cvt_Type.Value) ? thisServiceAppointment.mcs_relatedprovidersite : thisServiceAppointment.mcs_relatedsite;

                            if (site != null && site.Id != Guid.Empty)
                            {
                                findTeam = (from t in srv.TeamSet
                                            join f in srv.mcs_facilitySet on t.cvt_Facility.Id equals f.mcs_facilityId.Value
                                            join s in srv.mcs_siteSet on f.mcs_facilityId.Value equals s.mcs_FacilityId.Id
                                            where s.mcs_siteId.Value == site.Id && t.cvt_Type.Value == Teamcvt_Type.Scheduler.GetHashCode()
                                            select t).FirstOrDefault();

                                if (findTeam == null || findTeam.Id == Guid.Empty)
                                    Logger.WriteDebugMessage(string.Format("Scheduler Team associated to the site on the Service Activity {0} with Id {1} is null or empty. Hence the Service appointment is not assigned to the Patient site Scheduler Team.", site.Name, site.Id));
                                else
                                    Logger.WriteDebugMessage("Initiating the assignment to the Patient site Scheduler Team: " + findTeam.Id + ((Team)findTeam).Name);
                            }
                            else
                            {
                                Logger.WriteDebugMessage(string.Format("Associated site on the Service Activity is null or empty. Service Activity Type: {0}",(thisServiceAppointment.cvt_Type.Value) ? "Home/Mobile":"Clinic Based" ));
                            }
                        }
                    }
                    break;
                    #endregion
            }
            #region Common Code for the Assign
            if (findTeam == null || findTeam.Id == Guid.Empty)
                return false;
            assignOwner.Id = findTeam.Id;

            AssignRequest assignRequest = new AssignRequest()
            {
                Assignee = assignOwner,
                Target = new EntityReference(thisRecord.LogicalName, thisRecord.Id)
            };

            OrganizationService.Execute(assignRequest);
            Logger.WriteDebugMessage(thisRecord.LogicalName + " assigned to a Team. Ending AssignOwner.");
            return true;
            #endregion
        }

        /// <summary>
        /// Create Team in the assign (add the security roles)
        /// </summary>
        /// <param name="newTeam"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns></returns>
        internal static Guid CreateTeamForAssign(Team newTeam, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            var newTeamId = OrganizationService.Create(newTeam);
            Logger.WriteDebugMessage("Team created.");

            //Rush the adding of security roles
            if (TeamCreatePostStageRunner.SetTeamRoles(newTeamId, Team.EntityLogicalName, OrganizationService, Logger))
            {
                Logger.WriteDebugMessage("Allowing time to process security changes.");
                System.Threading.Thread.Sleep(15000);
                Logger.WriteDebugMessage("Resuming assign.");
            }
            return newTeamId;
        }

        /// <summary>
        /// Runs assignowner for entire dataset passed in
        /// </summary>
        /// <param name="entitySet"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns></returns>
        internal static string FixOwnershipForEntity(IQueryable<Entity> entitySet, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.setMethod = "FixOwnershipForEntity";
            Logger.WriteDebugMessage("About to update Entity record ownership.");

            var count = 0;
            var entityName = "";
            foreach (var thisRecord in entitySet)
            {
                try
                {
                    bool update = CvtHelper.AssignOwner(thisRecord, Logger, OrganizationService);
                    entityName = thisRecord.LogicalName;
                    if (update)
                        count++;
                }
                catch (Exception ex)
                {
                    Logger.WriteDebugMessage("error message: " + ex.Message);
                }
            }
            var message = string.Format("Assigned ownership for {0}: {1}/{2}. ", entityName, count, entitySet.ToList().Count.ToString());
            Logger.WriteDebugMessage(message);
            return message;
        }
        #endregion

        #region Record Names
        /// <summary>
        /// Returns the record name if it needs to be changed.
        /// </summary>
        /// <param name="EntityObj"></param>
        /// <param name="RecordId"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns>Returns "" if no name change, or the name that should be changed to, if derived</returns>
        internal static string ReturnRecordNameIfChanged(Entity Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            string NameIfChanged = "";
            switch (Record.LogicalName)
            {
                case Team.EntityLogicalName:
                    NameIfChanged = DeriveNameForTeam(Record, OnCreate, Logger, OrganizationService);
                    break;
                case cvt_tssprivileging.EntityLogicalName:
                    NameIfChanged = DeriveNameForTSSPrivileging(Record, OnCreate, Logger, OrganizationService);
                    break;
                default:
                    Logger.WriteDebugMessage("Entity Name not found. No name generated.");
                    break;
            }

            return NameIfChanged;
        }

        internal static string DeriveNameForTeam(Entity Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Team thisRecord = new Team();
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<Team>();
            else //find record
            {
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.TeamSet.First(t => t.Id == Record.Id);
            }
            string derivedResultField = "";
            var teamServiceLine = new mcs_servicetype();
            if (thisRecord.cvt_ServiceType != null && thisRecord.cvt_ServiceType.Id != null)
            {
                using (var srv = new Xrm(OrganizationService))
                    teamServiceLine = srv.mcs_servicetypeSet.FirstOrDefault(st => st.Id == thisRecord.cvt_ServiceType.Id);
            }

            string serviceLineName = (teamServiceLine != null && teamServiceLine.mcs_name != null) ? teamServiceLine.mcs_name + " " : "";
            var role = (thisRecord.cvt_Type != null) ? (thisRecord.cvt_Type.Value) : -1;

            var teamFacility = new mcs_facility();
            if (thisRecord.cvt_Facility != null && thisRecord.cvt_Facility.Id != null)
            {
                using (var srv = new Xrm(OrganizationService))
                    teamFacility = srv.mcs_facilitySet.FirstOrDefault(f => f.Id == thisRecord.cvt_Facility.Id);
            }

            string facilityName = (teamFacility != null && teamFacility.mcs_name != null) ? " @ " + teamFacility.mcs_name : "";
            string roleName = "";
            switch (role)
            {
                case 917290000:
                    roleName = "FTC Approval Group";
                    break;
                case 917290001:
                    roleName = "Service Chief Approval Group";
                    break;
                case 917290002:
                    roleName = "Chiefs of Staff Approval Group";
                    break;
                case 917290003:
                    roleName = "Credentialing and Privileging Officer Approval Group";
                    break;
                case 917290004:
                    roleName = "TSA Notification Group";
                    break;
                case 917290005:
                    roleName = "Scheduler Group";
                    break;
                case 917290006:
                    roleName = "Data Administrators";
                    break;
                case 917290007:
                    roleName = "TCTs";
                    break;
            }
            derivedResultField = String.Format("{0}{1}{2}", serviceLineName, roleName, facilityName);

            if (thisRecord.Name == derivedResultField.Trim())
                derivedResultField = "";
            return derivedResultField;
        }

        internal static string DeriveNameForTSSPrivileging(Entity Record, Boolean OnCreate, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            cvt_tssprivileging thisRecord = new cvt_tssprivileging();
            if (OnCreate) //use record
                thisRecord = Record.ToEntity<cvt_tssprivileging>();
            else //find record
            {
                using (var srv = new Xrm(OrganizationService))
                    thisRecord = srv.cvt_tssprivilegingSet.First(p => p.Id == Record.Id);
            }
            string derivedResultField = "";

            var providerName = (thisRecord.cvt_ProviderId != null) ? thisRecord.cvt_ProviderId.Name : "";
            var serviceType = (thisRecord.cvt_ServiceTypeId != null) ? " - " + thisRecord.cvt_ServiceTypeId.Name : "";
            var facilityName = (thisRecord.cvt_PrivilegedAtId != null) ? " @ " + thisRecord.cvt_PrivilegedAtId.Name : "";
            var typeOfPrivileging = (thisRecord.cvt_TypeofPrivileging != null) ? thisRecord.cvt_TypeofPrivileging.Value : 0;
            var privText = "";
            switch (typeOfPrivileging)
            {
                case 917290000:
                    privText = " (Home/Parent)";
                    break;

                case 917290001:
                    privText = " (Proxy/Secondary)";
                    break;
            }

            derivedResultField = String.Format("{0}{1}{2}", providerName, serviceType, facilityName, privText);

            if (thisRecord.cvt_name == derivedResultField)
                derivedResultField = "";
            return derivedResultField;
        }
        #endregion

    }
}