﻿using MCSUtilities2011;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using VA.TMP.DataModel;
using VA.TMP.OptionSets;

namespace MCSShared
{
    public static partial class CvtHelper
    {
        #region Helper Functions

        /// <summary>
        /// This accepts 1 or 2 strings and outputs them as a clickable link
        /// </summary>
        /// <param name="input">the string to be converted to a URL</param>
        /// <param name="clickDisplay">the display text that is clickable - defaults to be the same as the actual URL unless otherwise specified</param>
        /// <returns>formatted url</returns>
        public static string buildHTMLUrl(string input, string clickDisplay = "")
        {
            var http = (input.Length < 4 || input.Substring(0, 4).ToLower() != "http") ? "https://" + input : input;
            clickDisplay = (clickDisplay == string.Empty) ? http : clickDisplay;
            return "<a href=" + http + ">" + clickDisplay + "</a>";
        }
        /// <summary>
        /// Create a note related to this object
        /// </summary>
        /// <param name="Regarding"></param>
        /// <param name="Message"></param>
        /// <param name="Title"></param>
        /// <param name="OrgService"></param>
        public static void CreateNote(EntityReference Regarding, String Message, String Title, IOrganizationService OrgService)
        {
            var newNote = new Entity("annotation");
            newNote["subject"] = Title;
            newNote["notetext"] = Message;
            newNote["objectid"] = Regarding;
            OrgService.Create(newNote);
        }

        /// <summary>
        /// gets the URL for the environment from the settings table's cvt_URL field
        /// </summary>
        /// <param name="OrganizationService">OrganizationService for local environment</param>
        /// <returns>returns url</returns>
        public static string getServerURL(IOrganizationService OrganizationService)
        {
            var url = "";
            using (var srv = new Xrm(OrganizationService))
            {
                var setting = srv.mcs_settingSet.FirstOrDefault(s => s.mcs_name == "Active Settings");
                if (setting != null && !string.IsNullOrEmpty(setting.cvt_URL))
                {
                    url = setting.cvt_URL;
                    url = url.EndsWith("/") ? url : url + "/";
                    url = url.StartsWith("http") ? url : "https://" + url;
                }
                else
                    throw new InvalidPluginExecutionException("URL setting is not configured, please contact the Help Desk and request the URL be set up");
            }
            return url;
        }

        /// <summary>
        /// Retrieves the ETC
        /// </summary>
        /// <param name="service"></param>
        /// <param name="entity"></param>
        /// <returns></returns>
        public static int GetEntityTypeCode(IOrganizationService service, Entity entity)
        {
            return GetEntityTypeCode(service, entity.LogicalName);
        }

        /// <summary>
        /// Get Entity Type Code because it can change per environment
        /// </summary>
        /// <param name="service"></param>
        /// <param name="entityname"></param>
        /// <returns></returns>
        public static int GetEntityTypeCode(IOrganizationService service, String entityname)
        {
            RetrieveEntityRequest request = new RetrieveEntityRequest();
            request.LogicalName = entityname;
            RetrieveEntityResponse response = (RetrieveEntityResponse)service.Execute(request);
            int objectTypecode = response.EntityMetadata.ObjectTypeCode.Value;
            return objectTypecode;
        }

        /// <summary>
        /// This method accepts the resource (or user) id from the patient or provider site resource, and then returns the user or equipment record
        /// </summary>
        /// <param name="id"></param>
        /// <param name="context"></param>
        /// <returns></returns>
        public static Entity getPartyParticipantFromResource(Guid id, Xrm context)
        {
            var participant = new Entity();
            participant = context.EquipmentSet.FirstOrDefault(e => e.mcs_relatedresource.Id == id);
            if (participant == null)
                participant = context.SystemUserSet.FirstOrDefault(u => u.Id == id);
            return participant;
        }

        /// <summary>
        /// Get owner of workflow
        /// </summary>
        /// <param name="WorkflowName"></param>
        /// <param name="OrganizationService"></param>
        /// <returns></returns>
        public static List<ActivityParty> GetWorkflowOwner(string WorkflowName, IOrganizationService OrganizationService)
        {
            //Get the workflow definition's owner
            using (var context = new Xrm(OrganizationService))
            {
                var workflow = ((from w in context.WorkflowSet
                        where (w.Type.Value == 1 &&
                        w.Name == WorkflowName)
                        select new
                        {
                            w.OwnerId
                        }).FirstOrDefault());

                //var workflow = context.WorkflowSet.FirstOrDefault(c =>
                //    c.Type.Value == 1 && c.Name == WorkflowName);
                
                if (workflow != null && workflow.OwnerId != null)
                {
                    var owner = new ActivityParty()
                    {
                        PartyId = workflow.OwnerId
                    };
                    return CvtHelper.SetPartyList(owner);
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// Builds an error message from an exception recursively, excluding the first level message
        /// </summary>
        /// <param name="ex">Exception.</param>
        /// <returns>Exception message.</returns>
        public static string BuildErrorMessage(Exception ex)
        {
            var errorMessage = string.Empty;

            if (ex.InnerException == null) return errorMessage;

            errorMessage += string.Format("\n\n{0}\n", ex.InnerException.Message);
            errorMessage += BuildErrorMessage(ex.InnerException);

            return errorMessage;
        }

        /// <summary>
        /// Concatenates the error message including the first level message
        /// </summary>
        /// <param name="ex"></param>
        /// <param name="errorMessage"></param>
        /// <returns></returns>
        public static string BuildExceptionMessage(Exception ex, string errorMessage = "")
        {
            var message = string.Format("{0} \n{1}\n", errorMessage, ex.Message);
            if (ex.InnerException != null) 
                BuildExceptionMessage(ex.InnerException, message);
            return message;
        }

        /// <summary>
        /// Conditionally sets the field on the return Entity object if it has changed, otherwise returns the Entity object unchanged
        /// </summary>
        /// <param name="update">the object that is the result of the operation</param>
        /// <param name="existing">The source object to check if something has changed</param>
        /// <param name="newer">the proper target to match against source, and to be used if they aren't equal</param>
        /// <param name="patGroupFieldName">the name of the field on the patient resource group record</param>
        /// <param name="resourceFieldName">the name of the field on the tss resource record</param>
        /// <returns>the cvt_patientresourcegroup record either updated if it is out of date or the same as before if no update is needed</returns>
        /// <remarks>use this for updating entities so that only changes are recorded and persisted instead of brute forcing all changes</remarks>
        public static Entity UpdateField(Entity update, Entity existing, Entity newer, string targetFieldName, string sourceFieldName, bool overrideNulls = true)
        {
            if (!newer.Attributes.Contains(sourceFieldName))
            {
                if (overrideNulls && existing.Attributes.Contains(targetFieldName) && existing[targetFieldName] != null)
                    update[targetFieldName] = null;
                return update;
            }
            if (!existing.Attributes.Contains(targetFieldName))
            {
                update[targetFieldName] = newer[sourceFieldName];
                return update;
            }
            if ((existing[targetFieldName]).GetType().ToString() == "Microsoft.Xrm.Sdk.EntityReference")
            {
                if (newer[sourceFieldName].GetType().ToString() == "System.Guid")
                {
                    if (((EntityReference)(existing[targetFieldName])).Id != ((Guid)(newer[sourceFieldName])))
                        update[targetFieldName] = (new EntityReference(newer.LogicalName, (Guid)newer[sourceFieldName]));
                }
                else if (newer[sourceFieldName].GetType().ToString() == "Microsoft.Xrm.Sdk.EntityReference")
                {
                    if (((EntityReference)(existing[targetFieldName])).Id != ((EntityReference)(newer[sourceFieldName])).Id)
                        update[targetFieldName] = ((EntityReference)(newer[sourceFieldName]));
                }
            }
            else if ((existing[targetFieldName]).GetType().ToString() == "Microsoft.Xrm.Sdk.OptionSetValue")
            {
                if (((OptionSetValue)(existing[targetFieldName])).Value != ((OptionSetValue)(newer[sourceFieldName])).Value)
                    update[targetFieldName] = ((OptionSetValue)(newer[sourceFieldName]));
            }
            else
            {
                if (existing[targetFieldName].ToString().Trim() != newer[sourceFieldName].ToString().Trim())
                    update[targetFieldName] = newer[sourceFieldName];
            }
            return update;
        }

        /// <summary>
        /// retrieves the record and verifies that it is of the correct type
        /// </summary>
        /// <param name="PrimaryEntity"></param>
        /// <param name="EntityLogicalName"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <returns></returns>
        public static Entity ValidateReturnRecord(Entity PrimaryEntity, string EntityLogicalName, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.setMethod = "ValidateReturnRecord";
            //Logger.WriteDebugMessage("Starting ValidateReturnRecord");
            if (PrimaryEntity.Id == null)
                throw new InvalidPluginExecutionException("Entity has no ID.");

            if (PrimaryEntity.LogicalName != EntityLogicalName)
                throw new InvalidPluginExecutionException("Entity is not Type: " + EntityLogicalName);

            var ThisRecord = OrganizationService.Retrieve(EntityLogicalName, PrimaryEntity.Id, new ColumnSet(true));
            //Logger.WriteDebugMessage("Returning Record. Ending ValidateReturnRecord");
            return ThisRecord;
        }
        #endregion

        #region MTSA/TSA/Service Related Functions
        //TODO - finish out commenting Parameters
        //TODO - check all ifs
        /// <summary>
        /// Update TSAs provider and vista clinic string fields and build the service if it is in status of Production.
        /// </summary>
        /// <param name="TSAId">ID of the TSA to be updated</param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <param name="McsSettings"></param>
        /// <remarks>3-25-2015: Major Refactoring to improve readability and consolidate code.  Errors currently exist, so this should fix them as well</remarks>
        public static void CreateUpdateService(Guid TSAId, MCSLogger Logger, IOrganizationService OrganizationService, MCSSettings McsSettings)
        {
            Logger.setMethod = "CvtHelper.CreateUpdateService";
            Logger.WriteDebugMessage("Starting CreateUpdateService");

            using (var srv = new Xrm(OrganizationService))
            {
                try
                {
                    if (TSAId == Guid.Empty)
                    {
                        Logger.WriteDebugMessage("No TSA value, exiting.");
                        return;
                    }

                    //Get current TSA
                    var thisTSA = srv.mcs_servicesSet.FirstOrDefault(i => i.Id == TSAId);
                    if (thisTSA == null)
                        throw new InvalidPluginExecutionException("TSA to be updated could not be found.");

                    Logger.WriteDebugMessage("Starting to build the Service components");

                    //Declaration of Variables
                    var builder = new System.Text.StringBuilder("");
                    var builderProv = new System.Text.StringBuilder("");
                    var builderProvAllRequired = new System.Text.StringBuilder("");
                    var builderPat = new System.Text.StringBuilder("");
                    var builderPatAllRequired = new System.Text.StringBuilder("");
                    var builderGroupProvOnlybranch = new System.Text.StringBuilder("");
                    var builderGroupProvOnlyAllRequiredBranch = new System.Text.StringBuilder("");
                    string provSiteVCs = null;
                    string providers = null;
                    string patSiteVCs = null;
                    string patSites = null;
                    var tsaType = "Clinic";
                    Guid resourceSpecId;
                    String patActivityParty = String.Empty;

                    #region PROVIDER SITE RESOURCES
                    //Retrieve all the provider site resources
                    Logger.WriteDebugMessage("Start sorting through Provider Site Resources");
                    var getProvResources = from provGroups in srv.cvt_providerresourcegroupSet
                                           where provGroups.cvt_RelatedTSAid.Id == TSAId
                                           where provGroups.statecode == 0
                                           select new
                                           {
                                               provGroups.Id,
                                               provGroups.cvt_RelatedResourceId,
                                               provGroups.cvt_resourcespecguid,
                                               provGroups.cvt_TSAResourceType,
                                               provGroups.cvt_Type,
                                               provGroups.cvt_RelatedUserId,
                                               provGroups.cvt_RelatedResourceGroupid,
                                               provGroups.cvt_name
                                           };

                    //Loop through all of the Provider Site resources
                    foreach (var provGroups in getProvResources)
                    {
                        //Verify that the Resource is typed, not required, but should be filled in
                        if (provGroups.cvt_Type != null)
                        {
                            //StrBuilder
                            if (provGroups.cvt_resourcespecguid != null)
                            {
                                Guid sysResId = Guid.Empty;
                                switch (provGroups.cvt_Type.Value)
                                {
                                    case 917290000: //All Required
                                        builderProvAllRequired.Append(AddResourceToConstraintGroup(provGroups.cvt_resourcespecguid));
                                        var grResAR = srv.mcs_resourcegroupSet.FirstOrDefault(rg => rg.Id == provGroups.cvt_RelatedResourceGroupid.Id);
                                        if (grResAR == null)
                                            throw new InvalidPluginExecutionException(String.Format("Provider Site Resource Group is invalid, please remove and re-add the All Required Resource group to this TSA {0}.", provGroups.cvt_name));
                                        sysResId = grResAR.mcs_RelatedResourceGroupId.Id;
                                        var nestedARcbg = AddResourceToConstraintGroup(sysResId.ToString());
                                        var nestedARrs = BuildOut(thisTSA, new StringBuilder(nestedARcbg), OrganizationService, -1, 0, false);
                                        builderGroupProvOnlyAllRequiredBranch.Append(AddResourceToConstraintGroup(nestedARrs.ToString()));
                                        break;
                                    default: //Others
                                        builderProv.Append(AddResourceToConstraintGroup(provGroups.cvt_resourcespecguid));

                                        //for Group Prov Only Branch
                                        switch (provGroups.cvt_TSAResourceType.Value)
                                        {
                                            case 1: //Single Resource = 1
                                                var res = srv.mcs_resourceSet.FirstOrDefault(r => r.Id == provGroups.cvt_RelatedResourceId.Id);
                                                //TODO - throw exception if null
                                                sysResId = res.mcs_relatedResourceId.Id;
                                                break;
                                            case 2://Single Provider = 2
                                                sysResId = provGroups.cvt_RelatedUserId.Id;
                                                break;
                                            case 0: //Resource Group = 0
                                                var grRes = srv.mcs_resourcegroupSet.FirstOrDefault(rg => rg.Id == provGroups.cvt_RelatedResourceGroupid.Id);
                                                //TODO - throw exception if null
                                                sysResId = grRes.mcs_RelatedResourceGroupId.Id;
                                                var GroupResourceSpec = AddResourceToConstraintGroup(sysResId.ToString());
                                                sysResId = BuildOut(thisTSA, new StringBuilder(GroupResourceSpec), OrganizationService, 1, 0, false);
                                                break;
                                            default: //Unknown scenario
                                                Logger.WriteDebugMessage(String.Format("Provider Site Resource Record Resource Type is invalid for Resource {0}.", provGroups.cvt_name));
                                                //TODO - check if any negative impacts come out of TSA Resource not matching option set values
                                                break;
                                        }

                                        //Nest the Individual Group into a Choose 1
                                        builderGroupProvOnlybranch.Append(AddResourceToConstraintGroup(sysResId.ToString()));
                                        break;
                                }
                            }
                            else
                            {
                                Logger.WriteDebugMessage(String.Format("Resource Spec for resource - {0} - unable to be found, please recreate provider site resource.", provGroups.cvt_name));
                                //TODO - Consider throwing Exception
                            }
                            //Naming
                            switch (provGroups.cvt_Type.Value)
                            {
                                case 251920000: //Vista Clinic
                                    if (provGroups.cvt_TSAResourceType.Value == 0) //Group of Vista Clinics
                                    {
                                        //Query for child names
                                        var groupResourceRecords = srv.mcs_groupresourceSet.Where(g => g.mcs_relatedResourceGroupId.Id == provGroups.cvt_RelatedResourceGroupid.Id && g.statecode == 0);
                                        foreach (var child in groupResourceRecords)
                                        {
                                            if (child.mcs_RelatedResourceId != null)
                                                provSiteVCs += child.mcs_RelatedResourceId.Name.ToString() + " ; ";
                                        }
                                    }
                                    else
                                    {
                                        if (provGroups.cvt_RelatedResourceId != null)
                                            provSiteVCs += provGroups.cvt_RelatedResourceId.Name.ToString() + " ; ";
                                    }
                                    break;
                                case 100000000: //Telepresenter/Imager
                                case 99999999: //Provider
                                    if (provGroups.cvt_TSAResourceType.Value == 0) //Group of Providers
                                    {
                                        //Query for child names
                                        var groupResourceRecords = srv.mcs_groupresourceSet.Where(g => g.mcs_relatedResourceGroupId.Id == provGroups.cvt_RelatedResourceGroupid.Id && g.statecode == 0);
                                        foreach (var child in groupResourceRecords)
                                        {
                                            if (child.mcs_RelatedUserId != null)
                                                providers += child.mcs_RelatedUserId.Name.ToString() + " ; ";
                                        }
                                    }
                                    else
                                    {
                                        if (provGroups.cvt_RelatedUserId != null)
                                            providers += provGroups.cvt_RelatedUserId.Name.ToString() + " ; ";
                                    }
                                    break;
                                case 917290000: //All Required
                                    //Query for child names
                                    var childgroupResourceRecords = srv.mcs_groupresourceSet.Where(g => g.mcs_relatedResourceGroupId.Id == provGroups.cvt_RelatedResourceGroupid.Id && g.statecode == 0);
                                    foreach (var child in childgroupResourceRecords)
                                    {
                                        if (child.mcs_RelatedUserId != null)
                                            providers += child.mcs_RelatedUserId.Name.ToString() + " ; ";
                                        else
                                        { //Check the related Resource if it is a Vista Clinic
                                            var childR = srv.mcs_resourceSet.First(r => r.Id == child.mcs_RelatedResourceId.Id);
                                            if (childR.mcs_Type.Value == 251920000)
                                                provSiteVCs += childR.mcs_name + " ; ";
                                        }
                                    }
                                    break;
                                //default: No Default Required - Room or Technology (or unknown type), do nothing
                                //    break;
                            }
                        }
                        else //Probably Single Provider, but check.
                        {
                            //Provider or Telepresenter
                            if ((provGroups.cvt_TSAResourceType.Value == 2) || (provGroups.cvt_TSAResourceType.Value == 3))
                            {
                                if (provGroups.cvt_RelatedUserId != null)
                                {
                                    builderProv.Append(AddResourceToConstraintGroup(provGroups.cvt_resourcespecguid));
                                    providers += provGroups.cvt_RelatedUserId.Name.ToString() + " ; ";

                                    //Add to Group Prov Only Branch
                                    builderGroupProvOnlybranch.Append(AddResourceToConstraintGroup(provGroups.cvt_RelatedUserId.Id.ToString()));
                                }
                                else
                                    Logger.WriteDebugMessage(String.Format("User not listed in Provider Group {0}.", provGroups.cvt_name));
                            }
                            else
                                Logger.WriteDebugMessage(String.Format("Type is not null and provider site resource is not a user {0}", provGroups.cvt_name));
                        }
                    }
                    //Buildout Group Prov Only All Required here
                    if (builderGroupProvOnlyAllRequiredBranch.Length > 0)
                    {
                        builderGroupProvOnlybranch.Append(AddResourceToConstraintGroup(BuildOut(thisTSA, builderGroupProvOnlyAllRequiredBranch, OrganizationService, 1, 0, false).ToString()));
                    }
                    Logger.WriteDebugMessage("Finished Sorting through Provider Site Resources");
                    #endregion

                    #region PATIENT SITE RESOURCES
                    //Retrieve all the patient site resources
                    Logger.WriteDebugMessage("Start Sorting through Patient Site Resources");
                    var getPatResources = from patGroups in srv.cvt_patientresourcegroupSet
                                          where patGroups.cvt_RelatedTSAid.Id == TSAId
                                          where patGroups.statecode == 0
                                          select new
                                          {
                                              patGroups.Id,
                                              patGroups.cvt_RelatedResourceId,
                                              patGroups.cvt_resourcespecguid,
                                              patGroups.cvt_TSAResourceType,
                                              patGroups.cvt_type,
                                              patGroups.cvt_RelatedResourceGroupid
                                          };

                    //Loop through all of the Patient Site resources
                    foreach (var patGroups in getPatResources)
                    {
                        //Verify that the Resource is typed
                        if (patGroups.cvt_type != null)
                        {
                            //Check that ResourceSpecGuid is not null
                            if (patGroups.cvt_resourcespecguid != null)
                            {
                                switch (patGroups.cvt_type.Value)
                                {
                                    case 917290000: //All Required
                                        builderPatAllRequired.Append(AddResourceToConstraintGroup(patGroups.cvt_resourcespecguid));
                  
                                        

                                    //Query for child names, if Vista Clinics then write into the string
                                        var childgroupResourceRecords = srv.mcs_groupresourceSet.Where(g => g.mcs_relatedResourceGroupId.Id == patGroups.cvt_RelatedResourceGroupid.Id && g.statecode == 0);
                                        foreach (var child in childgroupResourceRecords)
                                        {
                                            if (child.mcs_RelatedUserId == null)
                                            {
                                                //Check the related Resource if it is a Vista Clinic
                                                var childR = srv.mcs_resourceSet.First(r => r.Id == child.mcs_RelatedResourceId.Id);
                                                if (childR.mcs_Type.Value == 251920000)
                                                    patSiteVCs += childR.mcs_name + " ; ";

                                                if (String.IsNullOrEmpty(patActivityParty))
                                                    patActivityParty = childR.mcs_relatedResourceId.Id + "|" + Equipment.EntityLogicalName + "|" + patGroups.cvt_resourcespecguid;
                                            }
                                            else //User for Group
                                            {
                                                if (String.IsNullOrEmpty(patActivityParty))
                                                    patActivityParty = child.mcs_RelatedUserId.Id + "|" + SystemUser.EntityLogicalName + "|" + patGroups.cvt_resourcespecguid;
                                            }
                                        }
                                        break;
                                    case 251920000: //Vista Clinic
                                        builderPat.Append(AddResourceToConstraintGroup(patGroups.cvt_resourcespecguid));
                                        if (patGroups.cvt_RelatedResourceId != null)
                                            patSiteVCs += patGroups.cvt_RelatedResourceId.Name.ToString() + " ; ";
                                        break;
                                    default: //Others                           
                                        builderPat.Append(AddResourceToConstraintGroup(patGroups.cvt_resourcespecguid));
                                        break;
                                }
                            }
                        }
                        //If not typed, could be single provider, shouldn't be on patient side, but if
                        else
                        {   //Provider or Telepresenter
                            if ((patGroups.cvt_TSAResourceType.Value == 2) || (patGroups.cvt_TSAResourceType.Value == 3))
                            {
                                if (patGroups.cvt_resourcespecguid != null)
                                {
                                    builderPat.Append(AddResourceToConstraintGroup(patGroups.cvt_resourcespecguid));
                                }
                            }
                        }
                    }

                    //Getting all the unique Patient Sites to add into a field for the view.                      
                    var getPatientSites = (from patGroupSites in srv.cvt_patientresourcegroupSet
                                           where patGroupSites.cvt_RelatedTSAid.Id == TSAId
                                           where patGroupSites.statecode == 0
                                           select new
                                           {
                                               patGroupSites.cvt_relatedsiteid,
                                           }).Distinct();

                    foreach (var patGroupSites in getPatientSites)
                    {
                        if (patGroupSites.cvt_relatedsiteid != null)
                            patSites += patGroupSites.cvt_relatedsiteid.Name.ToString() + " ; ";
                    }
                    Logger.WriteDebugMessage("Finished Sorting through Patient Site Resources");
                    #endregion

                    //Check for TSA status != Production
                    if (thisTSA.statuscode.Value != 251920000)
                    {
                        Logger.WriteDebugMessage("TSA is not in status Production, just update the string fields");
                        UpdateTSA(thisTSA, patSites, providers, patSiteVCs, provSiteVCs, Logger, OrganizationService, srv, Guid.Empty);
                        return;
                    }

                    //Else continue building
                    #region Logic - Constructing the builders etc
                    //Validation: No Resources, throw error
                    if (builderProv.Length == 0 && builderProvAllRequired.Length == 0 && builderPat.Length == 0 && builderPatAllRequired.Length == 0)
                        throw new InvalidPluginExecutionException("A TSA must have a resource listed in order to be put into production");
                    //Determine the Type of TSA, and check specifically
                    if (thisTSA.cvt_AvailableTelehealthModalities != null && thisTSA.cvt_AvailableTelehealthModalities.Value == 917290001)
                        tsaType = "SFT";
                    else if (thisTSA.cvt_groupappointment == true)
                        tsaType = "Group";
                    else if (thisTSA.cvt_Type == true)
                        tsaType = "Home";

                    switch (tsaType)
                    {
                        case "Home"://Home/Mobile - only Provider Site
                            if (builderProv.Length == 0 && builderProvAllRequired.Length == 0)
                                throw new InvalidPluginExecutionException("A Home/Mobile TSA must have provider resources Listed in order to be put into production");
                            else
                            {
                                if (builderProv.Length > 0)
                                    builder.Append(builderProv);

                                if (!string.IsNullOrEmpty(builderProvAllRequired.ToString()))
                                {
                                    Guid ProvAllRequiredSpecId = BuildOut(thisTSA, builderProvAllRequired, OrganizationService, 1, 0, true);
                                    builder.Append(AddResourceToConstraintGroup(ProvAllRequiredSpecId.ToString()));
                                }
                            }
                            break;
                        case "SFT"://SFT - only Patient Site
                            if (builderPat.Length == 0 && builderPatAllRequired.Length == 0)
                                throw new InvalidPluginExecutionException("A Store Forward TSA must have patient resources listed in order to be put into production");
                            else
                            {
                                if (builderPat.Length > 0)
                                    builder.Append(builderPat);

                                if (builderPatAllRequired.Length > 0)
                                {
                                    Guid PatAllRequiredSpecId = BuildOut(thisTSA, builderPatAllRequired, OrganizationService, 1, 0, true);
                                    builder.Append(AddResourceToConstraintGroup(PatAllRequiredSpecId.ToString()));
                                }
                            }
                            break;
                        default: // Clinic Based or Group (PatAllReq is all) - both Provider and Patient Site
                            if (builderProv.Length == 0 && builderProvAllRequired.Length == 0)
                                throw new InvalidPluginExecutionException("A TSA must have a provider resource listed in order to be put into production");
                            if (builderPat.Length == 0 && builderPatAllRequired.Length == 0)
                                throw new InvalidPluginExecutionException("A TSA must have a patient resource listed in order to be put into production");
                            if (tsaType == "Group" && builderPatAllRequired.Length == 0) // && patActivityParty.Length > 0)
                                throw new InvalidPluginExecutionException("A Group TSA must have at least one patient all required group listed in order to be put into production");
                            if (builderProv.Length > 0) //Prov Normal
                            {
                                Logger.WriteDebugMessage("Detected Prov resource");
                                builder.Append(builderProv);
                                //builderGroupProvOnlybranch.Append(builderProv);
                            }

                            if (builderProvAllRequired.Length > 0) //Prov AR
                            {
                                Logger.WriteDebugMessage("Detected Prov All Required groups");
                                Guid ProvAllRequiredSpecId = BuildOut(thisTSA, builderProvAllRequired, OrganizationService, 1, 0, true);
                                builder.Append(AddResourceToConstraintGroup(ProvAllRequiredSpecId.ToString()));

                                //Guid ProvGroupBranchOnlyId = BuildOut(thisTSA, builderProvAllRequired, OrganizationService, 1, 0, true);
                                //builderGroupProvOnlybranch.Append(AddResourceToConstraintGroup(ProvGroupBranchOnlyId.ToString()));  
                            }

                            if (builderPat.Length > 0  && tsaType != "Group")
                                builder.Append(builderPat);

                            if (builderPatAllRequired.Length > 0)
                            {
                                Guid PatAllRequiredSpecId;
                                if (tsaType == "Group")
                                    PatAllRequiredSpecId = BuildOut(thisTSA, builderPatAllRequired, OrganizationService, -1, 0, true);
                                else
                                    PatAllRequiredSpecId = BuildOut(thisTSA, builderPatAllRequired, OrganizationService, 1, 0, true);

                                builder.Append(AddResourceToConstraintGroup(PatAllRequiredSpecId.ToString()));
                            }
                            break;
                    }
                    #endregion

                    #region Logic - Constructing the service
                    Logger.WriteDebugMessage("Building out ResourceSpec; TSASpec " +tsaType);
                    
                    if (tsaType == "Group")
                    {//Need to default a resource to second branch for the search.  Preplugin will remove all pat resources.
                        Logger.WriteDebugMessage("Group - Building out both branches");
                        StringBuilder builderGroup = new System.Text.StringBuilder("");
                        //Stage branch 2 - prov only
                        Guid groupProvId = BuildOut(thisTSA, builderGroupProvOnlybranch, OrganizationService, -1, 0, false);
                        builderGroup.Append(AddResourceToConstraintGroup(groupProvId.ToString()));

                        //Stage branch 1 - all
                        Guid groupCombinedId = BuildOut(thisTSA, builder, OrganizationService, -1, 0, false);
                        builderGroup.Append(AddResourceToConstraintGroup(groupCombinedId.ToString()));
                        
                        Logger.WriteDebugMessage(builderGroup.ToString());

                        //build those out into resourceSpecId. Choose either prov or prov+pat
                        resourceSpecId = BuildOut(thisTSA, builderGroup, OrganizationService, 1, 2, false);
                    }
                    else
                    {
                        Logger.WriteDebugMessage("Individual - Building out normal");
                        resourceSpecId = BuildOut(thisTSA, builder, OrganizationService, -1, 2, false);
                    }
                    int initialstatus;
                    if (thisTSA.cvt_Type != null && thisTSA.cvt_Type.Value)
                        initialstatus = (int)service_initialstatuscode.Pending;
                    else
                        initialstatus = (int)service_initialstatuscode.Reserved;
                    Service newService = new Service
                    {
                        Name = thisTSA.mcs_name.ToString(),
                        AnchorOffset = 480,
                        Duration = thisTSA.cvt_duration.Value,
                        InitialStatusCode = new OptionSetValue(initialstatus),
                        Granularity = "FREQ=MINUTELY;INTERVAL=" + thisTSA.cvt_startevery.ToString() + ";",
                        ResourceSpecId = new EntityReference(ResourceSpec.EntityLogicalName, resourceSpecId)
                    };

                    //create the service
                    Logger.WriteDebugMessage("Creating new Service");
                    var newServiceId = OrganizationService.Create(newService);
                    UpdateTSA(thisTSA, patSites, providers, patSiteVCs, provSiteVCs, Logger, OrganizationService, srv, newServiceId, patActivityParty);
                    #endregion
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Logger.WriteToFile(ex.Message);
                    throw new InvalidPluginExecutionException(McsSettings.getUnexpectedErrorMessage + ":" + ex.Message);
                }
                catch (Exception ex)
                {
                    if (ex.Message.StartsWith("custom"))
                    {
                        Logger.WriteDebugMessage(ex.Message.Substring(6));

                        throw new InvalidPluginExecutionException(ex.Message.Substring(6));
                    }
                    else
                    {
                        Logger.setMethod = "Execute";
                        Logger.WriteToFile(ex.Message);
                        throw new InvalidPluginExecutionException(McsSettings.getUnexpectedErrorMessage + ":" + ex.Message);
                    }
                }
            }
        }

        /// <summary>
        /// Passing in the Resource Group GUID, return the StringBuilder of the associated child records.
        /// </summary>
        /// <param name="resourceGroupId"></param>
        /// <param name="srv"></param>
        /// <param name="resourceNames"></param>
        /// <param name="count"></param>
        /// <returns></returns>
        public static StringBuilder GetResources(Guid resourceGroupId, Xrm srv, out string resourceNames, out int count)
        {
            resourceNames = String.Empty;
            var resources = new StringBuilder();
            var grs = from resGroup in srv.mcs_groupresourceSet
                      join r in srv.mcs_resourceSet on resGroup.mcs_RelatedResourceId.Id equals r.mcs_resourceId.Value into resource
                      from res in resource.DefaultIfEmpty()
                      where resGroup.mcs_relatedResourceGroupId.Id == resourceGroupId && resGroup.statecode == 0
                      select new
                      {
                          name = resGroup.mcs_name,
                          user = resGroup.mcs_RelatedUserId,
                          resource = res.mcs_relatedResourceId
                      };
            count = grs.ToList().Count;

            foreach (var res in grs)
            {
                resourceNames += " " + res.name + " ;";
                if (res.user != null)
                    resources.Append(CvtHelper.AddResourceToConstraintGroup(res.user.Id.ToString()));
                else if (res.resource != null)
                    resources.Append(CvtHelper.AddResourceToConstraintGroup(res.resource.Id.ToString()));
            }
            return resources;
        }

        /// <summary>
        /// Add the Resources to the Constraint Group
        /// </summary>
        /// <param name="resourceId"></param>
        /// <returns></returns>
        public static string AddResourceToConstraintGroup(string resourceId)
        {
            var builder = "resource[\"Id\"] == ";
            builder += new Guid(resourceId).ToString("B");
            builder += " || ";
            return builder;
        }

        /// <summary>
        /// wraps the xml body string with the appropriate tags to make it a valid constraints field in a constraintbasedgroup record
        /// </summary>
        /// <param name="strBuilder">the string that becomes the body of the constraintbasedgroup</param>
        /// <returns>the full Constraints xml string</returns>
        public static StringBuilder BuildConstraintsXML(StringBuilder strBuilder)
        {
            var mainStrBuilder = new System.Text.StringBuilder("<Constraints><Constraint><Expression><Body>");
            if (strBuilder.Length == 0)
                strBuilder.Append("false || ");
            mainStrBuilder.Append(strBuilder);
            //Remove " || " from end and close the Constraint Group with the correct tags
            mainStrBuilder.Remove(mainStrBuilder.Length - 4, 4);
            mainStrBuilder.Append("</Body><Parameters><Parameter name=\"resource\" /></Parameters></Expression></Constraint></Constraints>");
            return mainStrBuilder;
        }

        /// <summary>
        /// Build Constraint Based Group
        /// </summary>
        /// <param name="thisTSA"></param>
        /// <param name="strBuilder"></param>
        /// <param name="OrgService"></param>
        /// <param name="ReqCount"></param>
        /// <param name="constraintBasedGroupTypeCode"></param>
        /// <param name="Site"></param>
        /// <returns></returns>
        public static Guid BuildOut(mcs_services thisTSA, StringBuilder strBuilder, IOrganizationService OrgService, int ReqCount, int constraintBasedGroupTypeCode, Boolean Site)
        {
            //constraintBasedGroupTypeCode
            //        Static = 0,
            //        Dynamic = 1,
            //        Implicit = 2

            //Correctly tag the XML
            var mainStrBuilder = BuildConstraintsXML(strBuilder);

            var constraintBasedGroupSetup = new ConstraintBasedGroup
            {
                BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisTSA.OwningBusinessUnit.Id),
                Constraints = mainStrBuilder.ToString(),
                Name = thisTSA.mcs_name,
                GroupTypeCode = new OptionSetValue(constraintBasedGroupTypeCode)
            };
            //Create the new Constraint Based Group
            var newConstraintGroup = OrgService.Create(constraintBasedGroupSetup);

            var newSpec = new ResourceSpec
            {
                BusinessUnitId = new EntityReference(BusinessUnit.EntityLogicalName, thisTSA.OwningBusinessUnit.Id),
                ObjectiveExpression = @"<Expression><Body>udf ""Random""(factory,resource,appointment,request,leftoffset,rightoffset)</Body><Parameters><Parameter name=""factory"" /><Parameter name=""resource"" /><Parameter name=""appointment"" /><Parameter name=""request"" /><Parameter name=""leftoffset"" /><Parameter name=""rightoffset"" /></Parameters><Properties EvaluationInterval=""P0D"" evaluationcost=""0"" /></Expression>",
                RequiredCount = ReqCount,
                Name = "Selection Rule",
                GroupObjectId = newConstraintGroup,
                SameSite = Site
            };
            var specId = OrgService.Create(newSpec);
            return specId;
        }

        /// <summary>
        /// Update TSA fields. Only update if values are different (clean up audit history)
        /// </summary>
        /// <param name="thisTSA"></param>
        /// <param name="patSites"></param>
        /// <param name="providers"></param>
        /// <param name="patSiteVCs"></param>
        /// <param name="provSiteVCs"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        /// <param name="srv"></param>
        /// <param name="newServiceId"></param>
        public static void UpdateTSA(mcs_services thisTSA, String patSites, String providers, String patSiteVCs, String provSiteVCs, MCSLogger Logger, IOrganizationService OrganizationService, Xrm srv, Guid newServiceId, String patActivityParty = "")
        {
            Logger.setMethod = "Update TSA";
            Logger.WriteDebugMessage("About to update TSA");
            var updateTSA = new Entity("mcs_services") { Id = thisTSA.Id };
            var updateCount = 0;
            Guid formerService = Guid.Empty;

            if (newServiceId == Guid.Empty) //Draft
            {
                if (thisTSA.mcs_RelatedServiceId != null)
                {
                    //Set to null
                    formerService = thisTSA.mcs_RelatedServiceId.Id;
                    updateTSA["mcs_relatedserviceid"] = null;
                    updateTSA["cvt_grouppatientbranch"] = null;
                    updateCount += 1;
                }
            }
            else //Production - always use the new id if present
            {
                //Check if there is a former value in service
                if (thisTSA.mcs_RelatedServiceId != null)
                    formerService = thisTSA.mcs_RelatedServiceId.Id;
                updateTSA["mcs_relatedserviceid"] = new EntityReference(Service.EntityLogicalName, newServiceId);
                Logger.WriteDebugMessage("Group: patActivityParty=" + patActivityParty);
                if (patActivityParty != "")
                {
                    updateTSA["cvt_grouppatientbranch"] = patActivityParty;
                    Logger.WriteDebugMessage("Attempting to update the Group Patient Branch on the TSA.");
                }
                updateCount += 1;
            }

            if (thisTSA.cvt_patientsites != ValidateLength(patSites, 2500))
            {
                updateTSA["cvt_patientsites"] = ValidateLength(patSites, 2500);
                updateCount += 1;
            }
            if (thisTSA.cvt_providers != ValidateLength(providers, 2500))
            {
                updateTSA["cvt_providers"] = ValidateLength(providers, 2500);
                updateCount += 1;
            }
            if (thisTSA.cvt_patsitevistaclinics != ValidateLength(patSiteVCs, 2500))
            {
                updateTSA["cvt_patsitevistaclinics"] = ValidateLength(patSiteVCs, 2500);
                updateCount += 1;
            }
            if (thisTSA.cvt_provsitevistaclinics != ValidateLength(provSiteVCs, 2500))
            {
                updateTSA["cvt_provsitevistaclinics"] = ValidateLength(provSiteVCs, 2500);
                updateCount += 1;
            }

            //Clear Bulk Edit Fields
            if (thisTSA.cvt_BulkRemoveResource != null)
            {
                updateTSA["cvt_bulkremoveresource"] = null;
                updateCount += 1;
            }
            if (thisTSA.cvt_BulkAddResource != null)
            {
                updateTSA["cvt_bulkaddresource"] = null;
                updateCount += 1;
            }
            if (thisTSA.cvt_BulkRemoveUser != null)
            {
                updateTSA["cvt_bulkremoveuser"] = null;
                updateCount += 1;
            }
            if (thisTSA.cvt_BulkAddUser != null)
            {
                updateTSA["cvt_bulkadduser"] = null;
                updateCount += 1;
            }
            if (thisTSA.cvt_BulkRemoveResourceGroup != null)
            {
                updateTSA["cvt_bulkremoveresourcegroup"] = null;
                updateCount += 1;
            }
            if (thisTSA.cvt_BulkAddResourceGroup != null)
            {
                updateTSA["cvt_bulkaddresourcegroup"] = null;
                updateCount += 1;
            }

            if (updateCount > 0)
            {
                OrganizationService.Update(updateTSA);
                Logger.WriteDebugMessage("TSA Updated");
                if (formerService != Guid.Empty)
                {
                    //Search on the prior service and attempt to delete it.
                    var ServiceActivity = srv.ServiceAppointmentSet.FirstOrDefault(i => i.ServiceId.Id == formerService);
                    //If an associated SA doesn't exist, delete
                    if (ServiceActivity == null)
                    {
                        OrganizationService.Delete(Service.EntityLogicalName, formerService);
                        Logger.WriteDebugMessage("Prior Service successfully deleted.");
                    }
                    else
                        Logger.WriteDebugMessage("Prior Service could not be deleted, associated Service Activities exist.");
                }
            }
            else
                Logger.WriteDebugMessage("No systematically built values to update on TSA");
        }

        /// <summary>
        /// Update MTSA fields
        /// </summary>
        /// <param name="mtsaId"></param>
        /// <param name="deleteId"></param>
        /// <param name="Logger"></param>
        /// <param name="OrganizationService"></param>
        public static void UpdateMTSA(Guid mtsaId, Guid deleteId, MCSLogger Logger, IOrganizationService OrganizationService)
        {
            Logger.setMethod = "UpdateMTSA";
            Logger.WriteDebugMessage("starting UpdateMTSA");
            using (var srv = new Xrm(OrganizationService))
            {
                if (mtsaId == Guid.Empty)
                {
                    Logger.WriteDebugMessage("No MTSA value, exiting.");
                    return;
                }

                var getProvResources = from provGroups in srv.cvt_providerresourcegroupSet
                                        where provGroups.cvt_RelatedMasterTSAId.Id == mtsaId
                                        where provGroups.statecode == 0
                                        select new
                                        {
                                            provGroups.Id,
                                            provGroups.cvt_RelatedResourceId,
                                            provGroups.cvt_resourcespecguid,
                                            provGroups.cvt_TSAResourceType,
                                            provGroups.cvt_Type,
                                            provGroups.cvt_RelatedUserId,
                                            provGroups.cvt_RelatedResourceGroupid
                                        };

                //grab all the provider groups
                string provSiteVCs = null;
                string providers = null;
                foreach (var provGroups in getProvResources)
                {
                    if (provGroups.Id != deleteId)
                    {
                        //Verify that the Resource is typed, not required, but should be filled in
                        if (provGroups.cvt_Type != null)
                        {
                            //Naming
                            switch (provGroups.cvt_Type.Value)
                            {
                                case 251920000: //Vista Clinic
                                    if (provGroups.cvt_TSAResourceType.Value == 0) //Group of Vista Clinics
                                    {
                                        //Query for child names
                                        var groupResourceRecords = srv.mcs_groupresourceSet.Where(g => g.mcs_relatedResourceGroupId.Id == provGroups.cvt_RelatedResourceGroupid.Id);
                                        foreach (var child in groupResourceRecords)
                                        {
                                            if (child.mcs_RelatedResourceId != null)
                                                provSiteVCs += child.mcs_RelatedResourceId.Name.ToString() + " ; ";
                                        }
                                    }
                                    else
                                    {
                                        if (provGroups.cvt_RelatedResourceId != null)
                                            provSiteVCs += provGroups.cvt_RelatedResourceId.Name.ToString() + " ; ";
                                    }
                                    break;
                                case 100000000: //Telepresenter/Imager
                                case 99999999: //Provider
                                    if (provGroups.cvt_TSAResourceType.Value == 0) //Group of Providers
                                    {
                                        //Query for child names
                                        var groupResourceRecords = srv.mcs_groupresourceSet.Where(g => g.mcs_relatedResourceGroupId.Id == provGroups.cvt_RelatedResourceGroupid.Id);
                                        foreach (var child in groupResourceRecords)
                                        {
                                            if (child.mcs_RelatedUserId != null)
                                                providers += child.mcs_RelatedUserId.Name.ToString() + " ; ";
                                        }
                                    }
                                    else
                                    {
                                        if (provGroups.cvt_RelatedUserId != null)
                                            providers += provGroups.cvt_RelatedUserId.Name.ToString() + " ; ";
                                    }
                                    break;
                                case 917290000: //All Required
                                    //Query for child names
                                    var childgroupResourceRecords = srv.mcs_groupresourceSet.Where(g => g.mcs_relatedResourceGroupId.Id == provGroups.cvt_RelatedResourceGroupid.Id);
                                    foreach (var child in childgroupResourceRecords)
                                    {
                                        if (child.mcs_RelatedUserId != null)
                                            providers += child.mcs_RelatedUserId.Name.ToString() + " ; ";
                                        else
                                        { //Check the related Resource if it is a Vista Clinic
                                            var childR = srv.mcs_resourceSet.First(r => r.Id == child.mcs_RelatedResourceId.Id);
                                            if (childR.mcs_Type.Value == 251920000)
                                                provSiteVCs += childR.mcs_name + " ; ";
                                        }
                                    }
                                    break;
                            }
                        }
                        else //Probably Single Provider, but check.
                        {
                            //Provider or Telepresenter
                            if ((provGroups.cvt_TSAResourceType.Value == 2) || (provGroups.cvt_TSAResourceType.Value == 3))
                            {
                                if (provGroups.cvt_RelatedUserId != null)
                                {
                                    providers += provGroups.cvt_RelatedUserId.Name.ToString() + " ; ";
                                }
                            }
                        }                   
                    }
                }
                
                var mtsa = srv.cvt_mastertsaSet.FirstOrDefault(i => i.Id == mtsaId);
                var updateMTSA = new Entity("cvt_mastertsa") { Id = mtsaId };
                var updateCount = 0;

                if (mtsa.cvt_providers != ValidateLength(providers, 2500))
                {
                    updateMTSA["cvt_providers"] = ValidateLength(providers, 2500);
                    updateCount += 1;
                }
                if (mtsa.cvt_providersitevistaclinics != ValidateLength(provSiteVCs, 2500))
                {
                    updateMTSA["cvt_providersitevistaclinics"] = ValidateLength(provSiteVCs, 2500);
                    updateCount += 1;
                }
                if (updateCount > 0)
                {
                    OrganizationService.Update(updateMTSA);
                    Logger.WriteDebugMessage("MTSA Updated");
                }
                else
                    Logger.WriteDebugMessage("No systematically built values to update on MTSA");
            }
        }

        /// <summary>
        /// Check length of a string and either reduce or return the original value
        /// </summary>
        /// <param name="value"></param>
        /// <param name="length"></param>
        /// <returns></returns>
        public static String ValidateLength(String value, int length)
        {
            if (value != null)
            {
                if (value.Length > length)
                    return value.Substring(0, length);
            }
            return value;
        }
        #endregion

    }
}