unit fOptionsSurrogate;
{
  Surrogate Management Functionality within CPRS Graphical User Interface (GUI)
  (Request #20071216)
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ORCtrls, ORDtTmRng, ORFn, ExtCtrls, fBase508Form,
  VA508AccessibilityManager, Vcl.ComCtrls, Vcl.Menus;

type
  TIDItem = class(TCollectionItem)
  private
  public
    IDString: String;
    Caption, PublicName: String;
    Comments: String;
    dateFrom, dateUntil: TDateTime;
    function strFrom: String;
    function strUntil: String;

    function toRawString: String;
    procedure setListItem(anItem: TListItem);
    function getInfo: String;
    function getInfoDebug: String;
    function IsOpen: Boolean;
  end;

  TfrmOptionsSurrogate = class(TfrmBase508Form)
    clvSurrogates: TCaptionListView;
    pnlParams: TPanel;
    pnlSurrogateTools: TPanel;
    pnlUpdateIndicator: TPanel;
    stxtChanged: TStaticText;
    pnlInfo: TPanel;
    pnlToolBar: TPanel;
    btnSurrEdit: TButton;
    btnRemove: TButton;
    VA508StaticText1: TVA508StaticText;
    cbUseDefaultDates: TCheckBox;
    edDefaultPeriod: TEdit;
    Label1: TLabel;
    pnlDebug: TPanel;
    procedure btnRemoveClick(Sender: TObject);
    procedure clvSurrogatesCompare(Sender: TObject; Item1, Item2: TListItem;
      Data: Integer; var Compare: Integer);
    procedure clvSurrogatesCustomDrawItem(Sender: TCustomListView;
      Item: TListItem; State: TCustomDrawState; var DefaultDraw: Boolean);
    procedure clvSurrogatesChange(Sender: TObject; Item: TListItem;
      Change: TItemChange);
    procedure btnSurrEditClick(Sender: TObject);
    procedure clvSurrogatesDblClick(Sender: TObject);
    procedure clvSurrogatesKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure clvSurrogatesMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure cbUseDefaultDatesClick(Sender: TObject);
    procedure edDefaultPeriodKeyPress(Sender: TObject; var Key: Char);
    procedure edDefaultPeriodExit(Sender: TObject);
  private
    fIDCollection: TCollection;
    fRawParams, fRawServerData: String;
    fIgnore, fSurrogateUpdated: Boolean;
    procedure LoadServerData;
    procedure LoadListViewByStringList(aList: TStringList);
    procedure setRangeInfo;
    procedure clearRanges;
    procedure mergeItems;
    procedure reNumItems;
    procedure ParseServerRecord(aValue: String;
      var sData, sName, sFrom, sUntil: String);
    function IsOpen(anItem: Integer): Boolean;
    function ListViewToRaw: String;
    procedure setSurrogateUpdated(aValue: Boolean);
    procedure setByCurrentItem;
    function ServerDeleteAll: String;
    function ServerSaveAll: String;
    function getItemID(anItem: Integer): String;
    procedure NotifyParent;
  public
    property SurrogateUpdated: Boolean read fSurrogateUpdated
      write setSurrogateUpdated;
    function ApplyChanges: Boolean;
    procedure UpdateWithServerData;
    procedure RefreshList;
    procedure RefreshParams;
  end;

var
  frmOptionsSurrogate: TfrmOptionsSurrogate;

function StrDateToDate(aDate: String): TDateTime;

const
  fmtListDateTime = 'mm/dd/yyyy@hh:nn'; // date/time format used in the table
  fmtListDateTimeSec = 'mm/dd/yyyy@hh:nn:ss';

implementation

uses rOptions, uOptions, rCore, fSurrogateEdit, System.UITypes, fOptions,
  uConst, ORDtTm;

{$R *.DFM}

const
  csNeverLoaded = '......';

  ciOpen = '...'; // indicator of the period available for assignment
  ciUnknownName = 'Dbl-click here to add surrogate';
  // SDS V32 Defect 101 4/4/2017
  ciActive = 'Active'; // indicator of currently active surrogate

  maxSurrPeriod = 365.25 * 4; // max assignment period is four years
  dtGap = 0.0; // required gap between surrogate assignments
  msgSurrogateRemove = 'Remove Surrogate?';
  msgSurrogateChangesSinceLastLoad =
    'Surrogate settings were changed since the last request' + CRLF +
    'Please review the current settings prior to saving the updates';

  MAX_PERIOD = 30;
  DEFAULT_PERIOD = 7;
  DEFAULT_INPUT = '1';

  /// /////////////////////////////////////////////////////////////////////////////

function StrDateToDate(aDate: String): TDateTime;
var
  dtDate, dtTime: Real;
  sDate, sTime: String;
begin
  // expected format: 'mm/dd/yyyy@hh:nn';  Sample user edit: 04/17/2017@2100
  sDate := piece(aDate, '@', 1);
  sTime := piece(aDate, '@', 2);

  dtDate := strToDate(sDate);
  dtTime := strToTime(sTime);

  Result := dtDate + dtTime;
end;

function StrDateToFMDate(aDate: String): TDateTime;
begin
  try
    Result := DateTimeToFMDateTime(StrDateToDate(aDate));
  except
    on E: Exception do
    begin
      Result := 0;
{$IFDEF DEBUG}
      MessageDlg('Error converting string to FM date:' + CRLF + CRLF +
        E.Message, mtError, [mbOK], 0);
{$ENDIF}
    end;
  end;
end;

function StrDateToFMDateStr(aDate: String): String;
begin
  Result := FloatToStr(StrDateToFMDate(aDate));
end;

/// /////////////////////////////////////////////////////////////////////////////

function newTIDItem(aCollection: TCollection;
  anID, aCaption, aName, aComment: String; aFrom, aUntil: TDateTime): TIDItem;
begin
  Result := TIDItem.Create(aCollection);
  Result.IDString := anID;
  Result.Caption := aCaption;
  Result.PublicName := aName;
  Result.Comments := aComment;
  Result.dateFrom := aFrom;
  Result.dateUntil := aUntil;
end;

/// /////////////////////////////////////////////////////////////////////////////
function TIDItem.toRawString: String;
begin
  Result := IDString + U + FloatToStr(DateTimeToFMDateTime(dateFrom)) + U +
    FloatToStr(DateTimeToFMDateTime(dateUntil));
end;

function TIDItem.getInfo: String;
begin
  Result := PublicName + '  ' + FormatDateTime(fmtListDateTime, dateFrom) +
    ' .. ' + FormatDateTime(fmtListDateTime, dateUntil);
end;

function TIDItem.getInfoDebug: String;
begin
  Result := IDString + '  ' + Caption + '  ' + PublicName + '  ' +
    FormatDateTime(fmtListDateTimeSec, dateFrom) + ' .. ' +
    FormatDateTime(fmtListDateTimeSec, dateUntil) + '  ' + Comments;
end;

function TIDItem.strFrom: String;
begin
  Result := FormatDateTime(fmtListDateTime, dateFrom);
end;

function TIDItem.strUntil: String;
begin
  Result := FormatDateTime(fmtListDateTime, dateUntil);
end;

function TIDItem.IsOpen: Boolean;
begin
  Result := PublicName = ciUnknownName;
end;

procedure TIDItem.setListItem(anItem: TListItem);
begin
  if assigned(anItem) then
  begin
    while anItem.SubItems.Count < 4 do
      anItem.SubItems.Add('');
    anItem.Caption := Caption;
    anItem.SubItems[0] := PublicName;
    anItem.SubItems[1] := FormatDateTime(fmtListDateTime, dateFrom);
    anItem.SubItems[2] := FormatDateTime(fmtListDateTime, dateUntil);
    anItem.SubItems[3] := Comments;
    anItem.Data := self;
  end;
end;

/// /////////////////////////////////////////////////////////////////////////////

procedure TfrmOptionsSurrogate.FormCreate(Sender: TObject);
begin
  inherited;
  fIDCollection := TCollection.Create(TIDItem);
{$IFDEF DEBUG_AA}
  pnlDebug.Visible := True;
{$ENDIF}
  fRawServerData := csNeverLoaded;
  fRawParams := csNeverLoaded;
end;

procedure TfrmOptionsSurrogate.FormDestroy(Sender: TObject);
begin
  fIDCollection.Free;
  inherited;
end;

procedure TfrmOptionsSurrogate.RefreshList;
begin
  clearRanges;
  setRangeInfo;
  mergeItems;
  reNumItems;
  if clvSurrogates.Items.Count = 1 then
    clvSurrogates.ItemIndex := 0;
  setByCurrentItem;
end;

procedure TfrmOptionsSurrogate.RefreshParams;
var
  i: Integer;
begin
  cbUseDefaultDates.Checked := piece(fRawParams, '^', 1) = DEFAULT_INPUT;
  edDefaultPeriod.Text := piece(fRawParams, '^', 2);
  i := StrToIntDef(edDefaultPeriod.Text, -1);
  if i < 0 then
    edDefaultPeriod.Text := intToStr(DEFAULT_PERIOD);
end;

procedure TfrmOptionsSurrogate.UpdateWithServerData;
begin
  LoadServerData;
  RefreshList;

  rpcGetSurrogateParams(fRawParams);
  RefreshParams;
end;

function TfrmOptionsSurrogate.IsOpen(anItem: Integer): Boolean;
begin
  if assigned(clvSurrogates.Items[anItem].Data) then
    Result := TIDItem(clvSurrogates.Items[anItem].Data)
      .PublicName = ciUnknownName
  else
    Result := True;
end;

procedure TfrmOptionsSurrogate.btnSurrEditClick(Sender: TObject);
var
  srvNow: TDateTime;
  IDNext, IDPrev, IDItem: TIDItem;
  i, iPeriod: LongInt;
  bUseDefaultDates, bLastRecord: Boolean;
  sDataOld, sData, sSurrogate, sStart, sStop, sMin, sMax: String;
begin
  inherited;
  iPeriod := StrToIntDef(edDefaultPeriod.Text, -1);
  if iPeriod < 0 then
  begin
    InfoBox('Incorrect value for default Period: ' + CRLF + CRLF + Char(VK_Tab)
      + '"' + edDefaultPeriod.Text + '"' + CRLF + CRLF +
      'Enter integer value greater than 7 days and try again ', 'Error',
      MB_ICONERROR or MB_OK);
    exit;
  end
  else
    defaultSurrPeriod := iPeriod;

  if clvSurrogates.Items.Count < 1 then
{$IFDEF DEBUG}
    ShowMessage('Surrogates list should not be blank... :( ')
{$ENDIF}
  else
  begin
    i := clvSurrogates.ItemIndex;

    bLastRecord := i = clvSurrogates.Items.Count - 1;

    if i < 0 then
      i := 0; // assuming the surrogate list is not empty

    IDItem := TIDItem(clvSurrogates.Items[i].Data);
    if not assigned(IDItem) then
    begin
      ShowMessage('No item to process...');
      exit;
    end;
    if i > 0 then
      IDPrev := TIDItem(clvSurrogates.Items[i - 1].Data)
    else
      IDPrev := nil;

    if i < clvSurrogates.Items.Count - 1 then
      IDNext := TIDItem(clvSurrogates.Items[i + 1].Data)
    else
      IDNext := nil;

    sStart := IDItem.strFrom;
    sStop := IDItem.strUntil;
    sSurrogate := IDItem.PublicName;

    srvNow := FMDateTimeToDateTime(ServerFMNow);

    if (i > 0) and assigned(IDPrev) and IDPrev.IsOpen then
      sMin := IDPrev.strFrom
    else
      sMin := IDItem.strFrom;

    if StrDateToDate(sMin) < srvNow then
      sMin := FormatDateTime(fmtListDateTime, srvNow);

    // Unresolved question SDS!  -> sent email to Ty and Andrey, asking what we do with a surrogate completely in the past
    // do we delete it? do we allow editing?
    if (StrDateToDate(sMin) > StrDateToDate(sStart)) and
      (StrDateToDate(sMin) < StrDateToDate(sStop)) then
      sStart := sMin;
    // V32 Defect 101 SDS: 4/11/17 Enforce now as the earliest start date, even for existing items

    if (i < clvSurrogates.Items.Count - 1) and assigned(IDNext) and IDNext.IsOpen
    then
      sMax := IDNext.strUntil
    else
      sMax := IDItem.strUntil;

    sData := getItemID(i);
    sDataOld := sData;

    bUseDefaultDates := cbUseDefaultDates.Checked;
    if editSurrogate(sSurrogate, sStart, sStop, sMin, sMax, sData, bLastRecord,
      bUseDefaultDates) = mrOK then
    begin
{$IFDEF DEBUG}
      ShowMessage('Edit Surrogate Results: ' + CRLF + CRLF + 'Name: ' +
        sSurrogate + '(' + sData + ')' + CRLF + 'Start: ' + sStart + CRLF +
        'Stop: ' + sStop + CRLF);
{$ENDIF}
      if sDataOld <> sData then
      begin
        IDItem := newTIDItem(fIDCollection, sData, '', sSurrogate, '',
          StrDateToDate(sStart), StrDateToDate(sStop));
      end
      else
      begin
        IDItem.IDString := sData;
        IDItem.PublicName := sSurrogate;
        IDItem.dateFrom := StrDateToDate(sStart);
        IDItem.dateUntil := StrDateToDate(sStop);
      end;

      IDItem.setListItem(clvSurrogates.Items[i]);

      RefreshList;
      SurrogateUpdated := fRawServerData <> ListViewToRaw;
    end;
  end;
end;

procedure TfrmOptionsSurrogate.btnRemoveClick(Sender: TObject);
var
  i: Integer;
  sMsg: String;
begin
  i := clvSurrogates.ItemIndex;
  if i < 0 then
    exit;
  if not assigned(clvSurrogates.Items[clvSurrogates.ItemIndex].Data) then
    ShowMessage('Item should have object assigned!')
  else
  begin
    sMsg := TIDItem(clvSurrogates.Items[clvSurrogates.ItemIndex].Data).getInfo;

    if MessageDlg(msgSurrogateRemove + CRLF + CRLF + sMsg, mtConfirmation,
      [mbOK, mbCancel], 0) <> mrOK then
      exit;

    clvSurrogates.Items.BeginUpdate;
    clvSurrogates.Items.Delete(i);
    clearRanges;
    setRangeInfo;
    reNumItems;
    clvSurrogates.Items.EndUpdate;
    SurrogateUpdated := fRawServerData <> ListViewToRaw;
    setByCurrentItem;
  end;
end;

procedure TfrmOptionsSurrogate.cbUseDefaultDatesClick(Sender: TObject);
begin
  inherited;
  edDefaultPeriod.Enabled := cbUseDefaultDates.Checked;
  if cbUseDefaultDates.Checked then
    fRawParams := '1^' + piece(fRawParams,'^',2)
  else
    fRawParams := '0^' + piece(fRawParams,'^',2);
end;

procedure TfrmOptionsSurrogate.clearRanges;
var
  i: Integer;
begin
  i := 0;
  while i < clvSurrogates.Items.Count do
  begin
    if IsOpen(i) then
      clvSurrogates.Items.Delete(i)
    else
      inc(i);
  end;
end;

procedure TfrmOptionsSurrogate.reNumItems;
var
  n, i: Integer;
begin
  i := 0;
  n := 1;
  while i < clvSurrogates.Items.Count do
  begin
    if IsOpen(i) then
      clvSurrogates.Items[i].Caption := ciOpen
    else
    begin
      clvSurrogates.Items[i].Caption := intToStr(n);
      if (n = 1) and (i = 0) then
        clvSurrogates.Items[i].SubItems[3] := ciActive;
      inc(n);
    end;
    inc(i);
  end;
end;

procedure TfrmOptionsSurrogate.mergeItems;
var
  i: Integer;
  IDItem, IDPrev: TIDItem;
begin
  i := 1;
  IDPrev := TIDItem(clvSurrogates.Items[0].Data);
  while i < clvSurrogates.Items.Count do
  begin
    IDItem := TIDItem(clvSurrogates.Items[i].Data);
    if IDItem.IDString = IDPrev.IDString then
    begin
      IDPrev.dateUntil := IDItem.dateUntil;
      IDPrev.setListItem(clvSurrogates.Items[i - 1]);
      clvSurrogates.Items.Delete(i);
    end
    else
    begin
      IDPrev := IDItem;
      inc(i);
    end;
  end;
end;

procedure TfrmOptionsSurrogate.clvSurrogatesChange(Sender: TObject;
  Item: TListItem; Change: TItemChange);
begin
  if fIgnore then
    exit;
  inherited;
  if (Sender as TListView).ItemIndex = -1 then
    btnSurrEdit.Enabled := (Sender as TListView).Items.Count > 0
  else
  begin
    fIgnore := True;
    btnSurrEdit.Enabled := True;
    setByCurrentItem;
    fIgnore := False;
  end;
end;

procedure TfrmOptionsSurrogate.clvSurrogatesCompare(Sender: TObject;
  Item1, Item2: TListItem; Data: Integer; var Compare: Integer);
var
  dt1, dt2: TDateTime;
begin
  inherited;
  dt1 := 0.0;
  dt2 := 0.0;
  if assigned(Item1.Data) then
    dt1 := TIDItem(Item1.Data).dateFrom;
  if assigned(Item2.Data) then
    dt2 := TIDItem(Item2.Data).dateFrom;

  Compare := 0;
  if dt1 < dt2 then
    Compare := -1
  else if dt2 < dt1 then
    Compare := 1;
end;

procedure TfrmOptionsSurrogate.clvSurrogatesCustomDrawItem
  (Sender: TCustomListView; Item: TListItem; State: TCustomDrawState;
  var DefaultDraw: Boolean);
begin
  inherited;
  if Item.Caption = ciOpen then
    Sender.Canvas.Font.Color := cl3dDkShadow // clHighlight
  else
    Sender.Canvas.Font.Color := clWindowText; // clDkGray
end;

procedure TfrmOptionsSurrogate.clvSurrogatesDblClick(Sender: TObject);
begin
  inherited;
  btnSurrEditClick(nil);
end;

procedure TfrmOptionsSurrogate.clvSurrogatesKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  if Key = VK_INSERT then
  begin
    if ssCtrl in Shift then
    begin
      btnSurrEditClick(nil);
      Key := 0;
    end;
  end
  else if Key = VK_DELETE then
  begin
    if ssCtrl in Shift then
    begin
      btnRemoveClick(nil);
      Key := 0;
    end;
  end;

  inherited;
end;

procedure TfrmOptionsSurrogate.clvSurrogatesMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited;
  if ssCtrl in Shift then
    btnRemoveClick(nil);
end;

procedure TfrmOptionsSurrogate.edDefaultPeriodExit(Sender: TObject);

  function validPeriod: Boolean;
  var
    i: Integer;
  begin
    i := StrToIntDef(edDefaultPeriod.Text, -1);
    Result := (1 <= i) and (i <= MAX_PERIOD);
  end;

begin
  inherited;
  if not ValidPeriod then
    begin
      beep;
      edDefaultPeriod.Text := intToStr(Default_Period);
      edDefaultPeriod.SetFocus;
    end;
  fRawParams := piece(fRawParams,'^',1) + '^'+ edDefaultPeriod.Text;
end;

procedure TfrmOptionsSurrogate.edDefaultPeriodKeyPress(Sender: TObject;
  var Key: Char);
begin
  inherited;
  if Key = Char(VK_RETURN) then
    Key := #0; // Don't want the deault to be ignited
  if not CharInSet(Key,[#8, '0' .. '9']) then // #8 - VK_BACK
  begin
    Key := #0;
    beep;
  end;
end;

procedure TfrmOptionsSurrogate.setRangeInfo;
var
  IDItem: TIDItem;
  li: TListItem;
  i: Integer;
  dtLimit, dtNow, dtFrom, dtUntil, dtMax: TDateTime;
begin
  inherited;
  dtNow := FMDateTimeToDateTime(ServerFMNow);
  dtLimit := dtNow + maxSurrPeriod;
  dtMax := dtNow;
  dtUntil := dtNow;
//  dtFrom := 0.0;

  fIgnore := True;
  i := 0;
  while i < clvSurrogates.Items.Count do
  begin
    if assigned(clvSurrogates.Items[i].Data) then
    begin
      IDItem := TIDItem(clvSurrogates.Items[i].Data);
      dtFrom := IDItem.dateFrom;
      dtUntil := IDItem.dateUntil;
    end
    else
    begin
{$IFDEF DEBUG}
      MessageDlg('DEBUG: No Object assigned to the record <' + intToStr(i) + '>!',
        mtError, [mbOK], 0);
{$ENDIF}
      continue;
    end;

    if dtUntil < dtNow then
    begin
      inc(i);
      continue; // ignore ranges ended in the past
    end;

    if dtFrom < dtNow then
    begin
      dtMax := dtUntil;
      inc(i);
      continue;
    end;

    if IsOpen(i) then
    // no open records are expected while setting open ranges
    begin
{$IFDEF DEBUG}
      MessageDlg('DEBUG: No Open records expected at this time! (record: ' +
        intToStr(i) + ')', mtError, [mbOK], 0);
{$ENDIF}
      inc(i);
      continue;
    end;

    if (dtFrom - dtMax > 2 * dtGap) then
    begin
      li := clvSurrogates.Items.Insert(i);
      IDItem := newTIDItem(fIDCollection, '', ciOpen, ciUnknownName, '',
        dtMax + dtGap, dtFrom - dtGap);
      IDItem.setListItem(li);
      dtMax := dtUntil;
      inc(i, 2);
    end
    else
    begin
      dtMax := dtUntil;
      inc(i);
    end;
  end;

  if dtLimit - dtUntil >= dtGap then
  begin
    li := clvSurrogates.Items.Insert(i);
    IDItem := newTIDItem(fIDCollection, '', ciOpen, ciUnknownName, '',
      dtUntil + dtGap, dtLimit);
    IDItem.setListItem(li);
  end;
  fIgnore := False;
end;

procedure TfrmOptionsSurrogate.LoadListViewByStringList(aList: TStringList);
var
  IDItem: TIDItem;
  i: Integer;
  sData, sName, sFrom, sUntil: String;
  li: TListItem;
begin
  if not assigned(aList) then
    exit;
  for i := 0 to aList.Count - 1 do
  begin
    ParseServerRecord(aList[i], sData, sName, sFrom, sUntil);

    if IsFMDateTime(sFrom) and IsFMDateTime(sUntil) then
    // v32 Issue Tracker #57?
    begin
      li := clvSurrogates.Items.Add;
      IDItem := newTIDItem(fIDCollection, sData, '', sName, '',
        FMDateTimeToDateTime(StrToFloat(sFrom)),
        FMDateTimeToDateTime(StrToFloat(sUntil)));
      IDItem.setListItem(li);
    end
    else
      MessageDlg('Incorrect or missing date/time for surrogate ' + sName + CRLF
        + CRLF + 'From date: ' + sFrom + CRLF + 'Until date: ' + sUntil + CRLF +
        CRLF + 'Record won''t be included in the Surrogates list', mtError,
        [mbOK], 0);
  end;
end;

procedure TfrmOptionsSurrogate.LoadServerData;
var
  ts: TStringList;
begin
  inherited;
  fIDCollection.Clear;
  clvSurrogates.Clear;
  ts := TStringList.Create;
  rpcGetSurrogateInfoList(ts);
  try
    if ts.Count > 1 then
    begin
      ts.Delete(0);
      fRawServerData := ts.Text;
      LoadListViewByStringList(ts);
    end;
  finally
    ts.Free;
  end;
  fRawServerData := ListViewToRaw;
  SurrogateUpdated := False;
end;

procedure TfrmOptionsSurrogate.NotifyParent;
var
  obj: TObject;
begin
  obj := Parent;
  while (obj <> nil) and not(obj is TForm) do
    obj := TWinControl(obj).Parent;
  if obj <> nil then
    SendMessage(TWinControl(obj).Handle, UM_CHECKAPPLY, 0, 0);
end;

procedure TfrmOptionsSurrogate.setSurrogateUpdated(aValue: Boolean);
begin
  fSurrogateUpdated := aValue;
  stxtChanged.Visible := fSurrogateUpdated;
  NotifyParent;
end;

procedure TfrmOptionsSurrogate.setByCurrentItem;
var
  ind: Integer;
  IDItem: TIDItem;
begin
  ind := clvSurrogates.ItemIndex;
  if ind < 0 then
  begin
    btnRemove.Enabled := False;
    btnSurrEdit.Enabled := False;
  end
  else
  begin
    btnSurrEdit.Enabled := True;
    IDItem := TIDItem(clvSurrogates.Items[clvSurrogates.ItemIndex].Data);
    if assigned(IDItem) then
    begin
      btnRemove.Enabled := not IDItem.IsOpen;
      if btnRemove.Enabled then
        btnSurrEdit.Caption := '&Edit Surrogate'
      else
        btnSurrEdit.Caption := '&Add Surrogate';

      if (ind = 0) then
        if IDItem.IsOpen then
          IDItem.Comments := ''
        else
          IDItem.Comments := ciActive;
    end
    else
      ShowMessage('Item should be assigned!...');
  end;
end;

function TfrmOptionsSurrogate.ListViewToRaw: String;
var
  s: String;
  IDItem: TIDItem;
  SL: TStringList;
  i: Integer;
begin
  SL := TStringList.Create;
  for i := 0 to clvSurrogates.Items.Count - 1 do
  begin
    if assigned(TIDItem(clvSurrogates.Items[i].Data)) then
    begin
      IDItem := TIDItem(clvSurrogates.Items[i].Data);
      if IDItem.IsOpen then
        continue;
      s := IDItem.toRawString;
      SL.Add(s);
    end
{$IFDEF DEBUG}
    else
      ShowMessage('DEBUG: All Records should have an object assigned!');
{$ENDIF}
  end;
  Result := SL.Text;
  SL.Free;
end;

function TfrmOptionsSurrogate.ServerDeleteAll: String;
var
  s, sMsg: String;
  bOK: Boolean;
  SL: TStringList;
begin
  SL := TStringList.Create;
  SL.Text := fRawServerData;
  while SL.Count > 0 do
  begin
    s := SL[0];
    try
      s := pieces(s, U, 1, 2);
      rpcSetSurrogateInfo(s, bOK, sMsg);
    except
      On E: Exception do
      begin
        sMsg := E.Message;
        bOK := False;
      end;
    end;
    if not bOK then
    begin
      MessageDlg('Error deleting surrogate(s):' + CRLF + CRLF + sMsg, mtError,
        [mbOK], 0);
      Result := Result + sMsg + CRLF;
    end;
    SL.Delete(0);
  end;
  SL.Free;
end;

function TfrmOptionsSurrogate.ServerSaveAll: String;
var
  sMsg: String;
  bOK: Boolean;
  SL: TStringList;
begin
  Result := '';
  SL := TStringList.Create;
  SL.Text := ListViewToRaw;
  while SL.Count > 0 do
  begin
    rpcSetSurrogateInfo(SL[0], bOK, sMsg);
    if not bOK then
    begin
      // MessageDlg('Error saving surrogate(s):' + CRLF + CRLF + sMsg, mtError,
      // [mbOK], 0);
      Result := Result + sMsg + CRLF;
    end;
    SL.Delete(0);
  end;
  SL.Free;

end;

function TfrmOptionsSurrogate.ApplyChanges: Boolean;
var
  sMsg, sNewList, sRaw: String;

  function getRawList(aString: String): String;
  var
    i: Integer;
    lst: TStringList;
  begin
    Result := '';
    lst := TStringList.Create;
    rpcGetSurrogateInfoList(lst);
    try
      if lst.Count > 1 then
      begin
        for i := 1 to lst.Count - 1 do
          Result := Result + piece(lst[i], U, 1) + U + piece(lst[i], U, 3) + U +
            piece(lst[i], U, 4) + CRLF;
      end;
    finally
      lst.Free;
    end;
  end;

begin
  // check if params were changed since the last load
  rpcGetSurrogateParams(sRaw);
  if sRaw <> fRawParams then
    rpcSetSurrogateParams(fRawParams);

  if fRawServerData = csNeverLoaded then // fixing defect 331720
  begin
    Result := True;
    exit;
  end;
  // check if the surrogates settings were updated since the last load
  sRaw := getRawList(sNewList);
  if fRawServerData <> sRaw then
  begin
    MessageDlg(msgSurrogateChangesSinceLastLoad
{$IFDEF DEBUG}
      + CRLF + CRLF + 'DEBUG. Server before: ' + CRLF + fRawServerData + CRLF + CRLF +
      'Server Now   : ' + CRLF + sRaw
{$ENDIF}
      , mtWarning, [mbOK], 0);
    Result := False;
  end
  else
  begin
    Result := not SurrogateUpdated;
    if Result then // no need to save changes (or update the table)
      exit;
    sMsg := ServerDeleteAll;
    if sMsg = '' then
      sMsg := ServerSaveAll;

    Result := sMsg = '';

    if Result then
      SurrogateUpdated := False
    else
      MessageDlg('Error applying saving surrogates changes' + CRLF + CRLF +
        sMsg, mtError, [mbOK], 0);
  end;

  UpdateWithServerData;
end;

procedure TfrmOptionsSurrogate.ParseServerRecord(aValue: String;
  var sData, sName, sFrom, sUntil: String);
begin
  sData := piece(aValue, U, 1);
  sName := piece(aValue, U, 2);
  sFrom := piece(aValue, U, 3);
  sUntil := piece(aValue, U, 4);
end;

function TfrmOptionsSurrogate.getItemID(anItem: Integer): String;
begin
  if assigned(clvSurrogates.Items[anItem].Data) then
    Result := TIDItem(clvSurrogates.Items[anItem].Data).IDString
  else
    Result := '';
end;

end.
