unit rODAnatPath;

// Developer: Theodore Fontana

interface

uses
  Winapi.Windows, Winapi.Messages, System.SysUtils, System.Classes, Vcl.Forms,
  Vcl.Controls, Vcl.StdCtrls, Vcl.ComCtrls, Vcl.Graphics;

type
  // A value for an element can have an associated TDateTimePicker to TEdit and
  // trigger another Builder Element to be required.
  TBuilderClusterItem = class(TObject)
    Associated: TObject;              // TDateTimePicker or TEdit
    Trigger: string;                  // IEN of Element to make Required
    OrderID: string;                  // Code of Order Prompt
    OrderValue: string;               // New value of Order Prompt (from OrderID)
  end;

  // An element has many values with the potential for an associated object and
  // an object to make required for each value.
  TBuilderClusterElement = class(TObject)
  private
    FList: TStringList;               // 'value', TBuilderClusterItem
  public
    constructor Create;
    destructor Destroy; override;
    function GetItem(Value: string): TBuilderClusterItem;
  end;

  TBuilderLocation = (blNone,blSpecimen,blText);

  TbeValidation = procedure(Sender: TObject; var beValid: Boolean; nVal,oVal: string) of object;

  TBuilderElement = class(TCustomControl)
  private
    FClusterElement: TBuilderClusterElement;
    FContainer: TGroupBox;
    FbtnReset: TButton;
    FValue: string;
    FValueEx: string;
    FIEN: string;
    FValidation: TbeValidation;
    procedure Initalize;
    procedure ResetClick(Sender: TObject);
    procedure SetRequired(const Value: Boolean);
    procedure SetValue(const Value: string);
    function GetValueEx: string;
    function GetControl: TControl;
    function GetRequired: Boolean;
  protected
    procedure SetCaption(const Value: string);
    property Control: TControl read GetControl;
  public
    constructor Create(aIEN: string; AOwner: TFrame); overload;
    destructor Destroy; override;
    procedure Add(vVals,vDefault,vHide: string);
    procedure Revert;
    function Valid: Boolean;          // This compares Required to Valued
    function OwnedBy: TBuilderLocation;
    property Caption;
    property onValidate: TbeValidation read FValidation write FValidation;
    property IEN: string read FIEN write FIEN;
    property Required: Boolean read GetRequired write SetRequired Default False;
    property Value: string read FValue write SetValue;
    property ValueEx: string read GetValueEx;
  end;

  TBuilderCluster = class(TObject)
  private
    FList: TStringList;               // 'IEN', TBuilderElement
  public
    constructor Create;
    destructor Destroy; override;
    function GetItem(Value: string): TBuilderElement;
  end;

  TBuilderCombo = class(TComboBox)
  private
    FElement: TBuilderElement;
    FPreviousIndex: Integer;
  protected
    procedure Change; override;
    procedure DropDown; override;
  public
    constructor Create(AContainer: TBuilderElement); overload;
    procedure Initalize;
    procedure Revert;
  end;

  TBuilderEdit = class(TEdit)
  private
    FElement: TBuilderElement;
    FPreviousText: string;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
  public
    constructor Create(AContainer: TBuilderElement); overload;
    procedure Initalize;
    procedure Revert;
  end;

  TBuilderRadioGroup = class;

  TBuilderRadio = class(TRadioButton)
  private
    FContainer: TBuilderRadioGroup;
  protected
    procedure SetChecked(Value: Boolean); override;
  public
    constructor Create(AContainer: TBuilderRadioGroup); overload;
  end;

  TBuilderRadioGroup = class(TCustomControl)
  private
    FElement: TBuilderElement;
    FRadioCollection: TList;
    FPreviousChecked: TBuilderRadio;
    FValue: string;
    procedure SetValue(const Value: string);
  protected
    procedure RecordChecked(Value: TBuilderRadio);
  public
    constructor Create(AContainer: TBuilderElement); overload;
    destructor Destroy; override;
    procedure Add(vVals: TStringList; vDefault: string);
    procedure Initalize;
    procedure Revert;
    property Value: string read FValue write SetValue;
  end;

var
  BuilderCluster: TBuilderCluster;

implementation

uses
  ORfn, fODAnatPath, fODAnatPathBuilder, fODAnatpathSpecimen;

{$REGION 'TBuilderCluster'}

constructor TBuilderCluster.Create;
begin
  FList := TStringList.Create;
end;

destructor TBuilderCluster.Destroy;
begin
  FList.Free;
end;

function TBuilderCluster.GetItem(Value: string): TBuilderElement;
  var
  I: Integer;
begin
  Result := nil;

  // FList = ('IEN', TBuilderElement)
  I := FList.IndexOf(Value);
  if I <> -1 then
    Result := TBuilderElement(FList.Objects[I]);
end;

{$ENDREGION}

{$REGION 'TBuilderClusterElement'}

constructor TBuilderClusterElement.Create;
begin
  FList := TStringList.Create;
end;

destructor TBuilderClusterElement.Destroy;
var
  I: Integer;
begin
  for I := FList.Count - 1 downto 0 do
    FList.Objects[I].Free;
  FList.Free;
end;

function TBuilderClusterElement.GetItem(Value: string): TBuilderClusterItem;
  var
  I: Integer;
begin
  Result := nil;

  // FList = ('value', TBuilderClusterItem)
  I := FList.IndexOf(Value);
  if I <> -1 then
    Result := TBuilderClusterItem(FList.Objects[I]);
end;

{$ENDREGION}

{$REGION 'TBuilderCombo'}

// Protected -------------------------------------------------------------------

procedure TBuilderCombo.Change;
begin
  FPreviousIndex := ItemIndex;

  inherited;

  if ((FElement <> nil) and (FElement is TBuilderElement)) then
    FElement.Value := Items[ItemIndex];
end;

procedure TBuilderCombo.DropDown;
var
  cbLength,I: Integer;
begin
  inherited;

  cbLength := Width;
  for I := 0 to Items.Count - 1 do
    if Canvas.TextWidth(Items[I]) > cbLength then
      cbLength := Canvas.TextWidth(Items[I]) + GetSystemMetrics(SM_CXVSCROLL);

  SendMessage(Handle, CB_SETDROPPEDWIDTH, (cbLength + 7), 0);
end;

// Public ----------------------------------------------------------------------

constructor TBuilderCombo.Create(AContainer: TBuilderElement);
begin
  inherited Create(AContainer);

  Anchors := [akLeft,akRight];
  Style := csDropDownList;
//  OnDropDown := cbDropDown;    // Dropdown width check?
  FElement := AContainer;
  FPreviousIndex := -1;
end;

procedure TBuilderCombo.Initalize;
begin
  ItemIndex := -1;
  FPreviousIndex := -1;
end;

procedure TBuilderCombo.Revert;
begin
  ItemIndex := FPreviousIndex;
end;

{$ENDREGION}

{$REGION 'TBuilderEdit'}

// Private ---------------------------------------------------------------------

procedure TBuilderEdit.CMExit(var Message: TCMExit);
begin
  FPreviousText := Text;

  if ((FElement <> nil) and (FElement is TBuilderElement)) then
    FElement.Value := Text;
end;

// Public ----------------------------------------------------------------------

constructor TBuilderEdit.Create(AContainer: TBuilderElement);
begin
  inherited Create(AContainer);

  Anchors := [akLeft,akRight];
  FElement := AContainer;
  FPreviousText := '';
end;

procedure TBuilderEdit.Initalize;
begin
  Clear;
  FPreviousText := '';
end;

procedure TBuilderEdit.Revert;
begin
  Text := FPreviousText;
end;

{$ENDREGION}

{$REGION 'TBuilderRadio'}

// Protected -------------------------------------------------------------------

procedure TBuilderRadio.SetChecked(Value: Boolean);
begin
  inherited;

  if ((Value) and (FContainer <> nil)) then
  begin
    FContainer.RecordChecked(Self);
    FContainer.Value := Caption;
  end;
end;

// Public ----------------------------------------------------------------------

constructor TBuilderRadio.Create(AContainer: TBuilderRadioGroup);
begin
  inherited Create(AContainer);

  WordWrap := True;
  FContainer := AContainer;
end;

{$ENDREGION}

{$REGION 'TBuilderRadioGroup'}

// Private ---------------------------------------------------------------------

procedure TBuilderRadioGroup.SetValue(const Value: string);
begin
  FValue := Value;
  if FElement <> nil then
    FElement.Value := Value;
end;

// Protected -------------------------------------------------------------------

procedure TBuilderRadioGroup.RecordChecked(Value: TBuilderRadio);
begin
  FPreviousChecked := Value;
end;

// Public ----------------------------------------------------------------------

constructor TBuilderRadioGroup.Create(AContainer: TBuilderElement);
begin
  inherited Create(AContainer);

  FRadioCollection := TList.Create;
  FElement := AContainer;
  FPreviousChecked := nil;
end;

destructor TBuilderRadioGroup.Destroy;
var
  I: Integer;
begin
  for I := FRadioCollection.Count - 1 downto 0 do
    TBuilderRadio(FRadioCollection.Items[I]).Free;
  FRadioCollection.Free;

  inherited;
end;

procedure TBuilderRadioGroup.Add(vVals: TStringList; vDefault: string);
var
  iColumns,iLeft,iTop,I,rText,rRows: Integer;
  fRadio: TBuilderRadio;
  cBit: TBitmap;
  vTmp,sTmp: string;
  vItem: TBuilderClusterItem;
  faDate: TDateTimePicker;
  faEdit: TEdit;
begin
  if FElement = nil then
    Exit;

  cBit := TBitmap.Create;
  try
    iColumns := 1;
    if FElement.OwnedBy = blSpecimen then
      iColumns := vVals.Count;

    iLeft := 10;
    iTop := (FElement.FContainer.Height div 2) - 3;

    for I := 0 to vVals.Count - 1 do
    begin
      fRadio := TBuilderRadio.Create(Self);
      fRadio.Parent := Self;
      fRadio.Caption := Piece(vVals[I],';',1);
      fRadio.Left := iLeft;
      fRadio.Top := iTop;

      FRadioCollection.Add(@fRadio);

      if iColumns > 1 then
      begin
        fRadio.Width := cBit.Canvas.TextWidth(fRadio.Caption) + 25;
        iLeft := iLeft + fRadio.Width;
      end else
      begin
        iTop := iTop + fRadio.Height;
        fRadio.Width := FElement.FContainer.Width - iLeft - 75;

        rText := cBit.Canvas.TextWidth(fRadio.Caption);
        if (rText + 25) div fRadio.Width > 0 then
        begin
          rRows := (rText + 25) div fradio.Width;
          if ((rText + 25) mod fRadio.Width > 0) then
            rRows := rRows + 1;
          fRadio.Height := fRadio.Height + ((rRows - 1) * Abs(Font.Height));

          if iColumns = 1 then
            iTop := iTop + ((rRows - 1) * Abs(Font.Height));
        end;
      end;

      // value ; code (D,E,"",OrderPromptID) ; # OR value
      vItem := TBuilderClusterItem.Create;
      vItem.Trigger := Piece(vVals[I],';',3);

      sTmp := Piece(vVals[I],';',2);
      if sTmp = 'D' then
      begin
        faDate := TDateTimePicker.Create(Self);
        faDate.Parent := Self;
        faDate.Format := ' ';
        faDate.Enabled := False;
        faDate.Width := 100;
        faDate.Left := fRadio.Left + 17;
        faDate.Top := iTop;

        if iColumns = 1 then
          iTop := iTop + faDate.Height + 5;

        vItem.Associated := faDate;
      end
      else if sTmp = 'E' then
      begin
        faEdit := TEdit.Create(Self);
        faEdit.Parent := Self;
        faEdit.Enabled := False;
        faEdit.Width := 100;
        faEdit.Left := fradio.Left + 17;
        faEdit.Top := iTop;

        if iColumns = 1 then
          iTop := iTop + faEdit.Height + 5;

        vItem.Associated := faEdit;
      end
      else if sTmp <> '' then
      begin
        // sTmp = OrderPromptID and sl[I],';',3 = the value to change it to
        vItem.OrderID := sTmp;
        vItem.OrderValue := Piece(vVals[I],';',3);
      end;
      FElement.FClusterElement.FList.AddObject(vTmp, vItem);

      if fRadio.Caption = vDefault then
        fRadio.Checked := True;
    end;
  finally
    cBit.Free;
  end;
end;

procedure TBuilderRadioGroup.Initalize;
var
  I: Integer;
begin
  for I := 0 to FRadioCollection.Count - 1 do
    if Assigned(FRadioCollection.Items[I]) then
      TBuilderRadio(FRadioCollection.Items[I]).Checked := False;

  FPreviousChecked := nil;
  Value := '';
end;

procedure TBuilderRadioGroup.Revert;
begin
  if FPreviousChecked <> nil then
    FPreviousChecked.Checked := True;
end;

{$ENDREGION}

{$REGION 'TBuilderElement'}

// Private ---------------------------------------------------------------------

procedure TBuilderElement.Initalize;
begin
  if Control <> nil then
  begin
    if Control is TBuilderCombo then
      TBuilderCombo(Control).Initalize
    else if Control is TBuilderEdit then
      TBuilderEdit(Control).Initalize
    else if Control is TBuilderRadioGroup then
      TBuilderRadioGroup(Control).Initalize;
  end;
end;

procedure TBuilderElement.ResetClick(Sender: TObject);
begin
  Initalize;
end;

procedure TBuilderElement.SetRequired(const Value: Boolean);
var
  sCaption: string;
begin
  sCaption := FContainer.Caption;

  if sCaption <> '' then
  begin
    if Value then
    begin
      if sCaption[1] <> '*' then
        FContainer.Caption := '*' + sCaption;
    end else
    begin
      if sCaption[1] = '*' then
      begin
        Delete(sCaption, 1, 1);
        FContainer.Caption := sCaption;
      end;
    end;
  end;
end;

procedure TBuilderElement.SetValue(const Value: string);
var
  vItem,tItem: TBuilderClusterItem;
  iValid: Boolean;
  tElement: TBuilderElement;
  I: Integer;
begin
  iValid := True;
  // Execute validation addon first
  if Assigned(onValidate) then
    onValidate(Control, iValid, Value, FValue);
  if not iValid then
  begin
    Revert;
    Exit;
  end;

  FValue := Value;

  vItem := FClusterElement.GetItem(FValue);
  if vItem = nil then
    Exit;

  // Enable/Disable an associated Control
  if vItem.Associated is TDateTimePicker then
  begin
    if not Valid then
    begin
      TDateTimePicker(vItem.Associated).Enabled := True;
      TDateTimePicker(vItem.Associated).Format := 'MMM dd, yyyy';
    end else
    begin
      TDateTimePicker(vItem.Associated).Enabled := False;
      TDateTimePicker(vItem.Associated).Format := ' ';
    end;
  end
  else if vItem.Associated is TEdit then
  begin
    if not Valid then
    begin
      TEdit(vItem.Associated).Enabled := True;
    end else
    begin
      TEdit(vItem.Associated).Enabled := False;
      TEdit(vItem.Associated).Clear;
    end;
  end;

  // Make other Builder Elements Required/Unrequired
  if vItem.Trigger <> '' then
  begin
    for I := 0 to FClusterElement.FList.Count - 1 do
    begin
      tItem := TBuilderClusterItem(FClusterElement.FList.Objects[I]);
      tElement := BuilderCluster.GetItem(tItem.Trigger);
      if tElement <> nil then
        tElement.Required := False;
    end;

    tElement := BuilderCluster.GetItem(vItem.Trigger);
    if tElement <> nil then
      tElement.Required := True;
  end;

  // Change the value of an Order Prompt (Legacy)
  // This is a one-way change, there is no change back
  if ((vItem.OrderID <> '') and (vItem.OrderValue <> '')) then
    frmODAnatPath.ChangeOrderPromptValue(vItem.OrderID, vItem.OrderValue);
end;

function TBuilderElement.GetValueEx: string;
var
  vItem: TBuilderClusterItem;
begin
  Result := '';

  vItem := FClusterElement.GetItem(FValue);
  if vItem <> nil then
  begin
    if vItem.Associated is TDateTimePicker then
      Result := DateToStr(TDateTimePicker(vItem.Associated).Date)
    else if vItem.Associated is TEdit then
      Result := TEdit(vItem.Associated).Text;
  end;
end;

function TBuilderElement.GetControl: TControl;
begin
  Result := nil;
  if FContainer.ControlCount > 0 then
    Result := FContainer.Controls[0];
end;

function TBuilderElement.GetRequired: Boolean;
var
  sCaption: string;
begin
  Result := False;
  sCaption := FContainer.Caption;

  if sCaption <> '' then
    if sCaption[1] = '*' then
      Result := True;
end;

// Protected -------------------------------------------------------------------

procedure TBuilderElement.SetCaption(const Value: string);
begin
  if FContainer <> nil then
    FContainer.Caption := Value;
end;

// Public ----------------------------------------------------------------------

constructor TBuilderElement.Create(aIEN: string; AOwner: TFrame);
begin
  inherited Create(AOwner);

  FClusterElement := TBuilderClusterElement.Create;
  FIEN := aIEN;
  BuilderCluster.FList.AddObject(aIEN, Self);

  FContainer := TGroupBox.Create(Self);
  FContainer.Parent := Self;
  FContainer.Width := Width - 19;
  FContainer.align := alLeft;

  FbtnReset := TButton.Create(Self);
  FbtnReset.Parent := Self;
  FbtnReset.onClick := ResetClick;
  FbtnReset.Caption := 'X';
  FbtnReset.Width := 15;
  FbtnReset.Height := 15;
  FbtnReset.Left := FContainer.Width + 1;
  FbtnReset.Top := (Height div 2) - 7;
  FbtnReset.Align := alCustom;
  FbtnReset.Anchors := [akRight];

  FValue := '';
  FValueEx := '';
end;

destructor TBuilderElement.Destroy;
begin
  FClusterElement.Free;

  inherited;
end;

procedure TBuilderElement.Add(vVals,vDefault,vHide: string);
var
  sl: TStringList;
  I: Integer;
  fEdit: TBuilderEdit;
  fCombo: TBuilderCombo;
  fRadio: TBuilderRadioGroup;
  vTmp,sTmp: string;
  vItem: TBuilderClusterItem;
  faDate: TDateTimePicker;
  faEdit: TEdit;

  function IsTooLong(vList: TStringList): Boolean;
  var
    I,eTotal: Integer;
    cBit: TBitmap;
  begin
    Result := False;
    cBit := TBitmap.Create;
    try
      eTotal := 0;
      for I := 0 to vList.Count - 1 do
        eTotal := eTotal + cBit.Canvas.TextWidth(Piece(vList[I], ';', 1)) + 35;

      if eTotal > FContainer.Width then
        Result := True;
    finally
      cBit.Free;
    end;
  end;

begin
  sl := TStringList.Create;
  try
    sl.Delimiter := '|';
    sl.StrictDelimiter := True;
    sl.DelimitedText := vVals;

    // EditBox
    if sl.Count = 0 then
    begin
      fEdit := TBuilderEdit.Create(Self);
      fEdit.Parent := FContainer;
      fEdit.Width := FContainer.Width - 20;
      fEdit.Left := 10;
      fEdit.Top := (FContainer.Height div 2) - (fEdit.Height div 2) + 6;

      // Defaulting
      fEdit.Text := vDefault;
      fEdit.OnExit(fEdit);
    end
    // ComboBox
    else if (sl.Count > 3) or ((IsTooLong(sl))) then
    begin
      fCombo := TBuilderCombo.Create(Self);
      fCombo.Parent := FContainer;
      fCombo.Width := FContainer.Width - 20;
      fCombo.Left := 10;
      fCombo.Top := (FContainer.Height div 2) - (FContainer.Height div 2) + 6;

      // Adding Values
      for I := 0 to sl.Count - 1 do
      begin
        vTmp := Piece(sl[I],';',1);
        fCombo.Items.Add(vTmp);

        // value ; code (D,E,"",OrderPromptID) ; # OR value
        vItem := TBuilderClusterItem.Create;
        vItem.Trigger := Piece(sl[I],';',3);

        sTmp := Piece(sl[I],';',2);
        if sTmp = 'D' then
        begin
          faDate := TDateTimePicker.Create(Self);
          faDate.Parent := Self;
          faDate.Format := ' ';
          faDate.Enabled := False;

          fCombo.Width := fCombo.Width - 105;

          faDate.Width := 100;
          faDate.Left := fCombo.Left + fCombo.Width + 5;
          faDate.Top := fCombo.Top;

          vItem.Associated := faDate;
        end
        else if sTmp = 'E' then
        begin
          faEdit := TEdit.Create(Self);
          faEdit.Parent := Self;
          faEdit.Enabled := False;

          fCombo.Width := fCombo.Width - 105;

          faEdit.Width := 100;
          faEdit.Left := fCombo.Left + fCombo.Width + 5;
          faEdit.Top := fCombo.Top;

          vItem.Associated := faEdit;
        end
        else if sTmp <> '' then
        begin
          // sTmp = OrderPromptID and sl[I],';',3 = the value to change it to
          vItem.OrderID := sTmp;
          vItem.OrderValue := Piece(sl[I],';',3);
        end;
        FClusterElement.FList.AddObject(vTmp, vItem);
      end;

      // Defaulting
      if vDefault <> '' then
      begin
        fCombo.ItemIndex := fCombo.Items.IndexOf(vDefault);
        fCombo.Change;
      end;
    end
    // RadioGroup
    else
    begin
      fRadio := TBuilderRadioGroup.Create(Self);
      fRadio.Parent := FContainer;

      // Adding Values + Defaulting
      fRadio.Add(sl, vDefault);
    end;
  finally
    sl.Free;

    // Hiding
    if vHide = '1' then
      Visible := False;
  end;
end;

procedure TBuilderElement.Revert;
begin
  if Control <> nil then
  begin
    if Control is TBuilderCombo then
      TBuilderCombo(Control).Revert
    else if Control is TBuilderEdit then
      TBuilderEdit(Control).Revert
    else if Control is TBuilderRadioGroup then
      TBuilderRadioGroup(Control).Revert;
  end;
end;

function TBuilderElement.Valid;
begin
  Result := True;

  if not Required then
    Exit;

  if FValue = '' then
    Result := False;
end;

function TBuilderElement.OwnedBy: TBuilderLocation;
begin
  if Owner <> nil then
  begin
    if Owner is TfAnatPathSpecimen then
      Result := blSpecimen
    else if Owner is TfAnatPathBuilder then
      Result := blText
    else Result := blNone;
  end else Result := blNone;
end;

{$ENDREGION}

initialization
  BuilderCluster := TBuilderCluster.Create;

finalization
  BuilderCluster.Free;

end.
