unit fODAnatPathSpecimen;

// Developer: Theodore Fontana

interface

uses
  Winapi.Windows, Winapi.Messages, System.StrUtils, System.SysUtils,
  System.Variants, System.Classes, Vcl.Forms, Vcl.Controls, Vcl.StdCtrls,
  Vcl.ComCtrls, Vcl.ExtCtrls, Vcl.Graphics, Vcl.Menus, Vcl.CheckLst,
  Vcl.Buttons, ORCtrls, fODBase;

type
  TCollSamp = class(TObject)
  public
    CollSampID: Integer;                  { IEN of CollSamp }
    CollSampName: string;                 { Name of CollSamp }
    SpecimenID: Integer;                  { IEN of selected specimen }
    SpecimenName: string;                 { Name of the specimen }
    SpecimenDescription: string;
    TubeColor: string;                    { TubeColor (text) }
    MinInterval: Integer;                 { Minimum days between orders }
    MaxPerDay: Integer;                   { Maximum orders per day }
    SampReqComment: string;               { Name of required comment }
    WardComment: TStringList;             { CollSamp specific comment }
    constructor Create;
    destructor Destroy; override;
  end;

  TBuildReturn = class(TObject)
    IEN: string;
    SPB: string;
    SPV: string;
  end;

  TfAnatPathSpecimen = class(TFrame)
    pnlHeader: TPanel;
    lbCharLimit: TLabel;
    btnDelete: TBitBtn;
    gpBody: TGridPanel;
    cboCollSamp: TORComboBox;
    lbCollSamp: TLabel;
    edSpecimenDesc: TCaptionEdit;
    lbDescription: TLabel;
    edSpecimen: TCaptionEdit;
    lbSpecimen: TLabel;
    procedure SpecimenDescChange(Sender: TObject; var iValid: Boolean; nVal,oVal: string);
    procedure btnDeleteClick(Sender: TObject);
    procedure cboCollSampChange(Sender: TObject);
    procedure cboCollSampAction(Sender: TObject);
  private
    FCollectionSample: TCollSamp;           // The Collection Sample for this Specimen
    CollSampList: TList;                    // The list of possible Collection Samples
    FElements: TStringList;                 // TBuilderElement sorted by Position + Specimen (optional)
    FHideSpec: Boolean;
    FSpecPosition: Integer;
    FCSHide: Boolean;
    FCSDefaultIEN: Integer;
    procedure LoadCollSamp;
    procedure LoadAllSamples;
    procedure FillCollSampList(LoadData: TStringList);
    procedure GetAllCollSamples;
    function IndexOfCollSamp(CollSampIEN: Integer): Integer;
    function PageReference: string;
    function PageNumber: Integer;
  protected
    procedure ShowControl(AControl: TControl); override;
  public
    constructor Create(AOwner: TComponent; sName: string; sIEN: Integer); overload;
    destructor Destroy; override;
    procedure UpdateResponses;
    property CollectionSample: TCollSamp read FCollectionSample write FCollectionSample;
  end;

implementation

{$R *.dfm}

uses
  ORNet, ORfn, VAUtils, rODLab, fODLabOthCollSamp, rODAnatPath, fODAnatPath;

{$REGION 'TCollSamp'}

constructor TCollSamp.Create;
begin
  WardComment := TStringList.Create;
end;

destructor TCollSamp.Destroy;
begin
  WardComment.Free;
end;

{$ENDREGION}

{$REGION 'TfAnatPathSpecimen'}

procedure TfAnatPathSpecimen.SpecimenDescChange(Sender: TObject; var iValid: Boolean; nVal,oVal: string);
var
  vVal: string;

  function BuildSpecDescription: string;
  var
    tmp: array of string;
    I: Integer;
    Build: TBuilderElement;
    nVal: string;
    sl: TStringList;

    procedure AddNext(iIndex: Integer; Value: string);
    var
      I: Integer;
      vAdded: Boolean;
    begin
      vAdded := False;

      for I := 0 to Length(tmp) - 1 do
        if tmp[I] = '' then
        begin
          tmp[I] := Value;
          vAdded := True;
          Break;
        end;

      if not vAdded then
      begin
        SetLength(tmp, Length(tmp) + 1);
        tmp[Length(tmp) - 1] := Value;
      end;
    end;

  begin
    if (edSpecimenDesc.Text = '') or (edSpecimenDesc.Text = edSpecimen.Text) then
    begin    
      SetLength(tmp, FElements.Count + 1);
      try
        // Specimen
        if not FHideSpec then
          tmp[FSpecPosition] := edSpecimen.Text;

        // Build Components
        FElements.Sort;
        for I := 0 to FElements.Count - 1 do
        begin
          Build := TBuilderElement(FElements.Objects[I]);
          AddNext(I, Build.Value + ' ' + Build.ValueEx);
        end;
      finally
        for I := 0 to Length(tmp) - 1 do
          if tmp[I] <> '' then
            if Result <> '' then
              Result := Result + ',' + tmp[I]
            else
              Result := tmp[I];

        SetLength(tmp, 0);
      end;
    end else
    begin
      sl := TStringList.Create;
      try
        sl.Delimiter := ',';
        sl.StrictDelimiter := True;
        sl.DelimitedText := edSpecimenDesc.Text;

        for I := 0 to sl.Count - 1 do
          if AnsiContainsText(sl[I], oVal) then
          begin
            sl[I] := nVal;
            Break;
          end;
      finally
        for I := 0 to sl.Count - 1 do
        begin
          if Result <> '' then
            Result := Result + ',' + sl[I]
          else Result := sl[I];
        end;
        sl.Free;
      end;
    end;
  end;

begin
  // Quick Check - check the difference between the new and old values before
  // bothering with a complete build of the specimen description.
  if Length(edSpecimenDesc.Text) + (Length(nVal) - Length(oVal)) > 75 then
  begin
    ShowMsg('Specimen Description at max length.'); //, smiWarning, smrOK);
    iValid := False;
    Exit;
  end;

  vVal := BuildSpecDescription;
  if Length(vVal) > 75 then
  begin
    ShowMsg('Specimen Description at max length.'); //, smiWarning, smrOK);
    iValid := False;
    Exit;
  end;

  edSpecimenDesc.Text :=  vVal;
end;

procedure TfAnatPathSpecimen.btnDeleteClick(Sender: TObject);
begin
  if ShowMsg('Are you sure you want to delete this entry - doing so means you will have to re-enter data.',
             smiWarning, smbYesNo) = smrYes then
  begin
    if ((Owner <> nil) and (Owner is TTabSheet)) then
      frmODAnatPath.DeletePage(TTabSheet(Owner));
  end;
end;

procedure TfAnatPathSpecimen.cboCollSampChange(Sender: TObject);
var
  I: Integer;
  ACollSamp: TCollSamp;
begin
  if cboCollSamp.Text = 'Other...' then
    if (cboCollSamp.ItemIndex >= 0) and (cboCollSamp.ItemIEN = 0) then
    begin
      GetAllCollSamples;
      Exit;
    end;

  if cboCollSamp.ItemIEN < 1 then
    Exit;

  I := IndexOfCollSamp(cboCollSamp.ItemIEN);
  if I <> -1 then
    ACollSamp := TCollSamp(CollSampList.Items[I])
  else
    Exit;

  FCollectionSample.CollSampID := ACollSamp.CollSampID;
  FCollectionSample.CollSampName := ACollSamp.CollSampName;

  if not frmODAnatPath.Changing then
    if ACollSamp.WardComment.Count > 0 then
    begin
      frmODAnatPath.pnlMessage.TabOrder := cboCollSamp.TabOrder + 1;       // Huh?
      frmODAnatPath.OrderMessage(ACollSamp.WardComment.Text);
    end;
end;

procedure TfAnatPathSpecimen.cboCollSampAction(Sender: TObject);
begin
  inherited;

  if cboCollSamp.Text = 'Other...' then
    if (cboCollSamp.ItemIndex > -1) and (cboCollSamp.ItemIEN = 0) then
      GetAllCollSamples;
end;

procedure TfAnatPathSpecimen.GetAllCollSamples;
var
  OtherSamp: string;
begin
  cboCollSamp.DroppedDown := False;

  if ((CollSampList.Count + 1) <= cboCollSamp.Items.Count) then
    LoadAllSamples;

  OtherSamp := SelectOtherCollSample(Font.Size, 0, CollSampList);

  if OtherSamp = '-1' then
    Exit;

  if cboCollSamp.SelectByID(Piece(OtherSamp,U,1)) = -1 then
    cboCollSamp.Items.Insert(0, OtherSamp);

  cboCollSamp.SelectByID(Piece(OtherSamp, U, 1));
  cboCollSamp.OnChange(cboCollSamp);
end;

procedure TfAnatPathSpecimen.LoadAllSamples;
var
  LoadList: TStringList;
begin
  LoadList := TStringList.Create;
  try
    LoadSamples(LoadList);
    FillCollSampList(LoadList);
  finally
    LoadList.Free;
  end;
end;

procedure TfAnatPathSpecimen.FillCollSampList(LoadData: TStringList);
var
  I: Integer;
  ACollSamp: TCollSamp;
begin
  // 1  2        3         4       5         6          7         8          9                     10
  // n^IEN^CollSampName^SpecIEN^TubeTop^MinInterval^MaxPerDay^LabCollect^SampReqCommentIEN;name^SpecName
  ACollSamp := nil;
  for I := 0 to LoadData.Count - 1 do
  begin
    if LoadData[I] = '~CollSamp' then
      Break;

    if LoadData[I] <> '' then
    begin
      if LoadData[I][1] = 'i' then
      begin
        ACollSamp := TCollSamp.Create;
        ACollSamp.CollSampID     := StrToInt(Piece(LoadData[I],'^',2));
        ACollSamp.CollSampName   := Piece(LoadData[I],'^',3);
        ACollSamp.SpecimenID     := StrToIntDef(Piece(LoadData[I],'^',4), 0);
        ACollSamp.SpecimenName   := Piece(LoadData[I],'^',10);
        ACollSamp.TubeColor      := Piece(LoadData[I],'^',5);
        ACollSamp.MinInterval    := StrToIntDef(Piece(LoadData[I],'^',6), 0);
        ACollSamp.MaxPerDay      := StrToIntDef(Piece(LoadData[I],'^',7), 0);
        ACollSamp.SampReqComment := Piece(LoadData[I],'^',9);
      end else if LoadData[I][1] = 't' then
        if ACollSamp <> nil then
          ACollSamp.WardComment.Add(Copy(LoadData[I],2,255));
    end;
  end;
end;

// Private ---------------------------------------------------------------------

procedure TfAnatPathSpecimen.LoadCollSamp;
var
  I: Integer;
  ACollSamp: TCollSamp;
  tmp: string;
begin
  cboCollSamp.Clear;

  FillCollSampList(ALabTest.FLoadedTestData);

  if CollSampList.Count > 0 then
  begin
    for I := 0 to CollSampList.Count - 1 do
    begin
      ACollSamp := TCollSamp(CollSampList.Items[I]);

      tmp := IntToStr(ACollSamp.CollSampID) + '^' + ACollSamp.CollSampName;
      if ACollSamp.TubeColor <> '' then
        tmp := tmp + ' (' + ACollSamp.TubeColor + ')';
      cboCollSamp.Items.Add(tmp);

      if ACollSamp.CollSampID = FCSDefaultIEN then
        cboCollSamp.ItemIndex := I;
    end;
  end else
  begin
    cboCollSamp.Items.Add('0^Other...');
    cboCollSamp.ItemIndex := 0;
  end;

  cboCollSampChange(cboCollSamp);
end;

function TfAnatPathSpecimen.IndexOfCollSamp(CollSampIEN: Integer): Integer;
var
  I: Integer;
  ACollSamp: TCollSamp;
begin
  Result := -1;

  for I := 0 to CollSampList.Count - 1 do
  begin
    ACollSamp := TCollSamp(COllSampList.Items[I]);

    if ACollSamp.CollSampID = CollSampIEN then
    begin
      Result := I;
      Break;
    end;
  end;
end;

function TfAnatPathSpecimen.PageReference: string;
begin
  Result := 'SP;' + IntToStr(FCollectionSample.SpecimenID);
end;

function TfAnatPathSpecimen.PageNumber: Integer;
begin
  Result := 0;

  if ((Owner = nil) or not (Owner is TTabSheet)) then
    Exit;

  Result := TTabSheet(Owner).PageIndex + 1;
end;

// Protected -------------------------------------------------------------------

procedure TfAnatPathSpecimen.ShowControl(AControl: TControl);
begin
  inherited;

  edSpecimenDesc.SetFocus;
end;

// Public ----------------------------------------------------------------------

constructor TfAnatPathSpecimen.Create(AOwner: TComponent; sName: string; sIEN: Integer);
var
  sl,BuildList: TStringList;
  I,J: Integer;
  Build: TBuildReturn;

  procedure AddControlItem(vID,vTitle,vHide,vReq,vDefault,vPosition,vVals: string);
  var
    nControlItem: TControlItem;
    BuilderElement: TBuilderElement;
    vCol,vRow: Integer;

    function NextCell(var vCol: Integer; var vRow: Integer): Boolean;
    var
      Ir,Ic: Integer;
    begin
      Result := False;

      // If there is no ControlItem or that ControlItem has no control or that
      // control is not visible then we can use that cell.
      for Ir := 0 to gpBody.RowCollection.Count - 1 do
        for Ic := 0 to gpBody.ColumnCollection.Count - 1 do
          if (gpBody.ControlCollection.ControlItems[Ic,Ir] = nil) or
             ((gpBody.ControlCollection.ControlItems[Ic,Ir].Control <> nil) and
              (not gpBody.ControlCollection.ControlItems[Ic,Ir].Control.Visible)) then
          begin
            Result := True;
            vCol := Ic;
            vRow := Ir;
            Exit;
          end;
    end;

  begin
    if not NextCell(vCol,vRow) then
      Exit;

    nControlItem := gpBody.ControlCollection.Add;

    BuilderElement := TBuilderElement.Create(vID, Self);
    BuilderElement.Parent := gpBody;
    BuilderElement.Caption := vTitle;
    BuilderElement.Add(vVals, vDefault, vHide);
    BuilderElement.Align := alClient;
    BuilderElement.Required := StrToBool(vReq);
    BuilderElement.onValidate := SpecimenDescChange;

    nControlItem.Control := BuilderElement;
    nControlItem.SetLocation(vCol, vRow);

    FElements.AddObject(vPosition, BuilderElement);
  end;

begin
  inherited Create(AOwner);

  CollSampList := TList.Create;

  FCollectionSample := TCollSamp.Create;
  FCollectionSample.SpecimenID := sIEN;
  FCollectionSample.SpecimenName := sName;

  FElements := TStringList.Create(True);

  edSpecimen.Text := sName;

  gpBody.ControlCollection.BeginUpdate;

  sl := TStringList.Create;
  try
    try
      tCallV(sl, 'ORWLRAP1 CONFIG', [PageReference, ALabTest.TestID]);

      if (sl.Count < 1) or (sl[0] = '0') then
        Exit;

      // SPH^SP^HIDE_FROM_DESCRIPTION^POSITION^COLLECTION_SAMPLE_HIDE(1,0)^COLLECTION_SAMPLE_DEFAULT
      // SPB^SP^ID^TITLE^HIDE^REQUIRED^DEFAULT_VALUE^POSITION
      // SPV^SP^ID^VAL|VAL
      BuildList := TStringList.Create(True);
      try
        for I := 0 to sl.Count - 1 do
        begin
          // SPH applies to the whole form and only has to be processed once
          if Piece(sl[I],U,1) = 'SPH' then
          begin
            // Hide Specimen from the Specimen Description
            if Piece(sl[I],U,3) = '1' then
              FHideSpec := True
            else FHideSpec := False;

            // If Specimen is part of the Specimen Description this will be its position
            FSpecPosition := StrToIntDef(Piece(sl[I],U,4), 0);

            // Hide the Collection Sample
            if Piece(sl[I],U,5) = '1' then
              FCSHide := True
            else FCSHide := False;

            // Default the Collection Sample to a based on IEN
            if TryStrToInt(Piece(sl[I],U,6),J) then
              FCSDefaultIEN := J
            else FCSDefaultIEN := 0;
          end else
          begin
            // SPB applies to a Builder Element
            // SPV applies to the values of a Builder Element
            J := BuildList.IndexOf(Piece(sl[I],U,3));
            if J <> -1 then
              Build := TBuildReturn(BuildList.Objects[J])
            else
            begin
              Build := TBuildReturn.Create;
              BuildList.AddObject(Piece(sl[I],U,3), Build);
            end;

            if Piece(sl[I],U,1) = 'SPB' then
              Build.SPB := sl[I];

            if Piece(sl[I],U,1) = 'SPV' then
              Build.SPV := sl[I];
          end;
        end;

        for I := 0 to BuildList.Count - 1 do
        begin
          Build := TBuildReturn(BuildList.Objects[I]);
          AddControlItem(Piece(Build.SPB,U,3), Piece(Build.SPB,U,4), Piece(Build.SPB,U,5),
                         Piece(Build.SPB,U,6), Piece(Build.SPB,U,7), Piece(Build.SPB,U,8),
                         Piece(Build.SPV,U,4));
        end;
      finally
        BuildList.Free;
      end;

      FElements.Sort;

      if FCSDefaultIEN < 1 then
      begin
        if uDfltCollSamp > 0 then
          FCSDefaultIEN := uDfltCollSamp
        else
          FCSDefaultIEN := StrToIntDef(LRFSAMP, 0);
      end;

      // Collection Sample ----------------------

      LoadCollSamp;

      if FCSHide then
      begin
        lbCollSamp.Visible := False;
        cboCollSamp.Visible := False;
      end;
    except
      on E: Exception do
      ShowMsg(E.Message, smiError, smbOK);
    end;
  finally
    sl.Free;
    gpBody.ControlCollection.EndUpdate;
  end;
end;

destructor TfAnatPathSpecimen.Destroy;
var
  I: Integer;
begin
  I := PageNumber;
  frmODAnatPath.Responses.Remove('SPECIMEN', I);
  frmODAnatPath.Responses.Remove('SPECDESC', I);
  frmODAnatPath.Responses.Remove('SAMPLE', I);
  frmODAnatPath.UpdateOrderText;

  for I := CollSampList.Count - 1 downto 0 do
    TCollSamp(CollSampList.Items[I]).Free;
  CollSampList.Free;

  FCollectionSample.Free;

  for I := FElements.Count - 1 downto 0 do
    FElements.Objects[I].Free;
  FElements.Free;

  inherited;
end;

procedure TfAnatPathSpecimen.UpdateResponses;
var
  I: Integer;
begin
  if FCollectionSample <> nil then
  begin
    I := PageNumber;

    frmODAnatPath.Responses.Update('SPECIMEN', I, IntToStr(FCollectionSample.SpecimenID),
                                   FCollectionSample.SpecimenName);
    frmODAnatPath.Responses.Update('SPECDESC', I, FCollectionSample.SpecimenDescription,
                                   FCollectionSample.SpecimenDescription);

    if FCollectionSample.CollSampID > 0 then
      frmODAnatPath.Responses.Update('SAMPLE', I, IntToStr(FCollectionSample.CollSampID),
                                     FCollectionSample.CollSampName)
    else
      frmODAnatPath.Responses.Update('SAMPLE', I, '', '');
  end;
end;

end.
