unit fODAnatPathBuilder;

// Developer: Theodore Fontana

interface

uses
  Winapi.Windows, Winapi.Messages, Winapi.RichEdit, System.SysUtils,
  System.Variants, System.Classes, Vcl.Forms, Vcl.Controls, Vcl.StdCtrls,
  Vcl.ComCtrls, Vcl.ExtCtrls, Vcl.Graphics, Vcl.Menus, Vcl.CheckLst, ORCtrls,
  fODBase;

type
  TBuildReturn = class(TObject)
    IEN: string;
    PWB: string;
    PWV: string;
  end;

  TfAnatPathBuilder = class(TFrame)
    pnl1: TPanel;
    pnl2: TPanel;
    pnl4: TPanel;
    pnl3: TPanel;
    CheckList: TCheckListBox;
    spacer2: TPanel;
    spacer3: TPanel;
    lbCheckList: TLabel;
    spacer4: TPanel;
    spacer5: TPanel;
    lbWPField: TLabel;
    mNote: TCaptionMemo;
    popNoteMemo: TPopupMenu;
    popNoteMemoCut: TMenuItem;
    popNoteMemoCopy: TMenuItem;
    popNoteMemoPaste: TMenuItem;
    scrollbox1: TScrollBox;
    scrollBox2: TScrollBox;
    procedure pnl4Resize(Sender: TObject);
    procedure popNoteMemoCutClick(Sender: TObject);
    procedure popNoteMemoCopyClick(Sender: TObject);
    procedure popNoteMemoPasteClick(Sender: TObject);
  private
    FElements: TStringList;
    FResponseID: string;
    procedure UpdateFormForInput;
    procedure SetRequired(const Value: Boolean);
    function GetRequired: Boolean;
    function PageReference: string;
    function PageCaption: string;
  protected
    procedure ShowControl(AControl: TControl); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure UpdateResponses;
    function GetCaption: string;
    function GetText: TStringList;
    function GetTextEx: TStringList;
    function Valid: Boolean;
    property Required: Boolean read GetRequired write SetRequired Default False;
    property Elements: TStringList read FElements write FElements;
    property ResponseID: string read FResponseID write FResponseID;
  end;

implementation

{$R *.dfm}

uses
  ORNet, ORfn, VAUtils, uConst, rODAnatPath, fODAnatPath, fODAnatpathSpecimen;

procedure TfAnatPathBuilder.pnl4Resize(Sender: TObject);
const
  LEFT_MARGIN = 4;
  MAX_NOTE_WIDTH = 78;
begin
  inherited;

  LimitEditWidth(mNote, MAX_NOTE_WIDTH - 1);
  mNote.Width := pnl4.Width - 16;

  UpdateFormForInput;
end;

procedure TfAnatPathBuilder.popNoteMemoCutClick(Sender: TObject);
begin
  inherited;

  mNote.CutToClipboard;
end;

procedure TfAnatPathBuilder.popNoteMemoCopyClick(Sender: TObject);
begin
  inherited;

  mNote.CopyToClipboard;
end;

procedure TfAnatPathBuilder.popNoteMemoPasteClick(Sender: TObject);
begin
  inherited;

  Sendmessage(mNote.Handle, EM_PASTESPECIAL, CF_TEXT, 0);
  pnl4Resize(Self);
end;

// Private ---------------------------------------------------------------------

procedure TfAnatPathBuilder.UpdateFormForInput;
var
  idx,offset: Integer;
begin
  idx := GetSystemMetrics(SM_CXSCREEN);

  offset := 5;
  if(MainFontSize <> 8) then
    offset := ResizeWidth(BaseFont, Font, offset);
  dec(idx, offset + 10);
end;

procedure TfAnatPathBuilder.SetRequired(const Value: Boolean);
var
  tTab: TTabSheet;
  sCaption: string;
begin
  if ((Owner = nil) or not (Owner is TTabSheet)) then
    Exit;

  tTab := TTabSheet(Owner);
  sCaption := tTab.Caption;

  if sCaption <> '' then
  begin
    if Value then
    begin
      if sCaption[1] <> '*' then
        tTab.Caption := '*' + sCaption;
    end else
    begin
      if sCaption[1] = '*' then
      begin
        Delete(sCaption, 1, 1);
        tTab.Caption := sCaption;
      end;
    end;
  end;
end;

function TfAnatPathBuilder.GetRequired: Boolean;
var
  sCaption: string;
begin
  Result := False;

  sCaption := PageCaption;

  if sCaption <> '' then
    if sCaption[1] = '*' then
      Result := True;
end;

function TfAnatPathBuilder.PageReference: string;
begin
  Result := 'PG;0';

  if ((Owner = nil) or not (Owner is TTabSheet)) then
    Exit;

  Result := 'PG;' + IntToStr(TTabSheet(Owner).PageIndex + 1);
end;

function TfAnatPathBuilder.PageCaption: string;
begin
  Result := '';

  if ((Owner = nil) or not (Owner is TTabSheet)) then
    Exit;

  Result := TTabSheet(Owner).Caption;
end;

// Protected -------------------------------------------------------------------

procedure TfAnatPathBuilder.ShowControl(AControl: TControl);
begin
  inherited;

  mNote.SetFocus;
end;

// Public ----------------------------------------------------------------------

constructor TfAnatPathBuilder.Create(AOwner: TComponent);
var
  sl,BuildList: TStringList;
  I,J,cCount: Integer;
  Build: TBuildReturn;

  procedure AddControlItem(var cCount: Integer; vID,vTitle,vList,vDefault,vVals: string);
  var
    BuilderElement: TBuilderElement;

    procedure AddValuesCheckList(vVals,vDefault: string);
    var
      sl: TStringList;
      I: Integer;
    begin
      sl := TStringList.Create;
      try
        sl.Delimiter := '|';
        sl.StrictDelimiter := True;
        sl.DelimitedText := vVals;

        for I := 0 to sl.Count - 1 do
        begin
          CheckList.Items.Add(sl[I]);
          if sl[I] = vDefault then
            CheckList.Checked[CheckList.Count - 1] := True;
        end;
      finally
        sl.Free;
      end;
    end;

  begin
    // CheckList
    if vList = '1' then
    begin
      // Only one CheckList is allowed, if Pnl3 is visible then it was already
      // populated so any additional list entries will fail.
      if pnl3.Visible then
        Exit;

      pnl3.Visible := True;
      lbCheckList.Caption := vTitle;
      AddValuesCheckList(vVals,vDefault);
    end else
    begin
      Inc(cCount);
      if cCount < 4 then
      begin
        BuilderElement := TBuilderElement.Create(vID, Self);
        BuilderElement.Parent := scrollbox1;
      end else if cCount < 7 then
      begin
        BuilderElement := TBuilderElement.Create(vID, Self);
        BuilderElement.Parent := scrollbox2;
      end else Exit;

      BuilderElement.Caption := vTitle;
      BuilderElement.Add(vVals, vDefault, '');
      BuilderElement.Align := alTop;

      FElements.AddObject(vID, BuilderElement);
    end;
  end;

begin
  inherited Create(AOwner);

  FElements := TStringList.Create(True);

  sl := TStringList.Create;
  try
    try
      tCallV(sl, 'ORWLRAP1 CONFIG', [PageReference, ALabTest.TestID]);

      if (sl.Count < 1) or (sl[0] = '0') then
        Exit;

      // PWB^PAGE^ID^TITLE^LIST(1,0)^DEFAULT_VALUE
      // PWV^PAGE^ID^VAL;D;#|VAL;E;#| (D(ate),E(dit))
      // PWW^PAGE^TITLE
      BuildList := TStringList.Create(True);
      try
        for I := 0 to sl.Count - 1 do
        begin
          // PWW applies to the whole form and only has to be processed once
          if Piece(sl[I],U,1) = 'PWW' then
            lbWPField.Caption := Piece(sl[I],U,3)
          else
          begin
            // PWB applies to a Builder Element
            // PWV applies to the values of a Builder Element
            J := BuildList.IndexOf(Piece(sl[I],U,3));
            if J <> -1 then
              Build := TBuildReturn(BuildList.Objects[J])
            else
            begin
              Build := TBuildReturn.Create;
              BuildList.AddObject(Piece(sl[I],U,3), Build);
            end;

            if Piece(sl[I],U,1) = 'PWB' then
              Build.PWB := sl[I];

            if Piece(sl[I],U,1) = 'PWV' then
              Build.PWV := sl[I];
          end;
        end;

        cCount := 0;
        for I := 0 to BuildList.Count - 1 do
        begin
          Build := TBuildReturn(BuildList.Objects[I]);
          AddControlItem(cCount, Piece(Build.PWB,U,3), Piece(Build.PWB,U,4), Piece(Build.PWB,U,5),
                         Piece(Build.PWB,U,6), Piece(Build.PWV,U,4));
        end;
      finally
        BuildList.Free;
      end;
    except
      on E: Exception do
      ShowMsg(E.Message, smiError, smbOK);
    end;
  finally
    sl.Free;

    if ((not pnl1.Visible) and (not pnl2.Visible) and (not pnl3.Visible)) then
    begin
      mNote.Left := 5;
      if lbWPField.Caption = '' then
        mNote.Top := 9;
      mNote.Height := pnl4.Height - 18;
    end;
  end;
end;

destructor TfAnatPathBuilder.Destroy;
begin
  frmODAnatPath.Responses.Remove(ResponseID, 1);
  frmODAnatPath.UpdateOrderText;

  FElements.Free;

  inherited;
end;

procedure TfAnatPathBuilder.UpdateResponses;
var
  sl: TStringList;
begin
  sl := TStringList.Create;
  try
    if ResponseID <> '' then
      frmODAnatPath.Responses.Update(ResponseID, 1, TX_WPTYPE, sl.Text);
  finally
    sl.Free;
  end;
end;

function TfAnatPathBuilder.GetCaption;
var
  sCaption: string;
begin
  sCaption := PageCaption;
  if sCaption <> '' then
  begin
    if sCaption[1] = '*' then
      Delete(sCaption, 1, 1);
  end;
  Result := sCaption;
end;

function TfAnatPathBuilder.GetText: TStringList;
var
  I: Integer;
  sl: TStringList;
  Element: TBuilderElement;
begin
  Result := TStringList.Create;

  for I := 0 to FElements.Count - 1 do
  begin
    Element := TBuilderElement(FElements.Objects[I]);
    if Element <> nil then
      Result.Add(Element.Value + Element.ValueEx);
  end;

  if CheckList.Visible then
  begin
    sl := TStringList.Create;
    try
      for I := 0 to CheckList.Count - 1 do
        if CheckList.Checked[I] then
          sl.Add('  - ' + CheckList.Items[I]);

      if sl.Count > 0 then
      begin
        if lbCheckList.Caption <> '' then
          Result.Add(lbCheckList.Caption);

        Result.AddStrings(sl);
        Result.Add('');
      end;
    finally
      sl.Free;
    end;
  end;

  if mNote.Visible then
  begin
    if mNote.Lines.Count > 0 then
      Result.AddStrings(mNote.Lines);
  end;
end;

function TfAnAtPathBuilder.GetTextEx: TStringList;
var
  sCaption: string;
begin
  Result := GetText;

  if Result.Count > 0 then
  begin
    sCaption := GetCaption;
    if sCaption <> '' then
    begin
      Result.Insert(0, '--------------------------------------------------------------------------');
      Result.Insert(0, sCaption);
      Result.Add('');
    end;
  end;
end;

function TfAnatPathBuilder.Valid: Boolean;
var
  I: Integer;
  sl: TStringList;
begin
  Result := True;

  for I := 0 to FElements.Count - 1 do
    if not TBuilderElement(FElements.Objects[I]).Valid then
    begin
      Result := False;
      Break;
    end;

  if Result then
  begin
    sl := TStringList.Create;
    try
      if Required then
      begin
        sl := GetText;
        if sl.Count < 1 then
          Result := False;
      end;
    finally
      sl.Free;
    end;
  end;
end;

end.
