package gov.va.vamf.service.shifttransition.tasks.domain.rules;

import gov.va.vamf.service.shifttransition.tasks.domain.Schedule;
import gov.va.vamf.service.shifttransition.tasks.domain.rules.parsers.FrequencyParser;
import gov.va.vamf.service.shifttransition.tasks.domain.time.DueDate;
import org.junit.*;

import javax.ws.rs.WebApplicationException;
import java.util.*;

import static org.junit.Assert.*;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

public class FrequencyDateRuleTests {
    private Schedule schedule;

    @Before
    public void init() {
        schedule = mock(Schedule.class);
    }

    private void updateSchedule(Date startDate, Date nextDueDate, String scheduleFrequency) {
        when(schedule.startDate()).thenReturn(new DueDate(startDate));
        when(schedule.scheduleFrequency()).thenReturn(scheduleFrequency);

        if (nextDueDate == null)
            when(schedule.nextDueDate()).thenReturn(null);
        else
            when(schedule.nextDueDate()).thenReturn(new DueDate(nextDueDate));
    }

    @Test(expected = WebApplicationException.class)
    public void intervalMissingInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "D4;H6;M1");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void intervalNegativeInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I-4;D4;H6;M1");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void intervalNotNumberInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "Ifour;D4;H6;M1");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void intervalWrongInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "i6;D4;H6;M1");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void tooFewNumberPeriodsInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I1;D4;H6");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void tooManyNumberPeriodsInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I1;D4;H6;M0;S13");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void wrongPeriodFirstInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I10;H4;H6;M0");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void dayPeriodWrongInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I4;d4;H6;M0");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void wrongPeriodSecondInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I6;D4;M6;M0");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void hourPeriodWrongInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I5;D4;h6;M0");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void wrongPeriodThirdInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I4;D4;H6;H0");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void minutePeriodWringInScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I8;D4;H6;m0");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void nullScheduleFrequency() {
        updateSchedule(new Date(), new Date(), null);
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void nullPeriodValueScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I2;D;H2;M2");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void notNumberPeriodValueScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I2;Dseven;H2;M2");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void dayLessThanZeroScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I2;D-1;H2;M2");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void dayGreaterThanMaxScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I50;D" + FrequencyParser.MAX_DAYS +1 + ";H2;M2");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void hoursLessThanZeroScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I5;D0;H-2;M2");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void hoursGreaterThanMaxScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I9;D0;H" + FrequencyParser.MAX_HOURS + 1 + ";M2");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void minutesLessThanZeroScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I8;D0;H0;M-1");
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void minutesGreaterThanMaxScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "I10;D0;H0;M" + FrequencyParser.MAX_MINUTES + 1);
        new FrequencyDateRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void intervalZeroInScheduleFrequencyEndDateNull() {
        updateSchedule(new Date(), new Date(), "I0;D0;H0;M1");
        new FrequencyDateRule(schedule);
    }

    @Test()
    public void calculateNextDueDateWithNullLastDueDate() {
        Date startDate = getLaterDate(new Date());

        updateSchedule(startDate, null, "I1;D1;H1;M15");
        when(schedule.numTimesScheduled()).thenReturn(0);

        NextDueDateRule rule = new FrequencyDateRule(schedule);

        Date nextDueDate = rule.calculateNextDueDate();

        assertEquals(startDate, nextDueDate);
    }

    @Test()
    public void calculateNextDueDateWithEndDate() {
        Date startDate = getLaterDate(new Date());

        updateSchedule(startDate, null, "I0;D1;H1;M15");
        when(schedule.numTimesScheduled()).thenReturn(0);
        when(schedule.endDate()).thenReturn(new DueDate(startDate));

        NextDueDateRule rule = new FrequencyDateRule(schedule);

        Date nextDueDate = rule.calculateNextDueDate();

        assertEquals(startDate, nextDueDate);
    }

    @Test()
    public void calculateNextDueDateNumTimeScheduledGreaterThanInterval() {
        Date startDate = getLaterDate(new Date());

        updateSchedule(startDate, null, "I4;D1;H1;M15");
        when(schedule.numTimesScheduled()).thenReturn(5);

        NextDueDateRule rule = new FrequencyDateRule(schedule);

        Date nextDueDate = rule.calculateNextDueDate();

        assertNull(nextDueDate);
    }

    @Test()
    public void calculateNextDueDateWithLaterStartDate() {
        Date lastDueDate = new Date();

        //Make start date later than last due date
        Date startDate = getLaterDate(lastDueDate);

        updateSchedule(startDate, null, "I6;D1;H1;M15");
        when(schedule.numTimesScheduled()).thenReturn(0);

        NextDueDateRule rule = new FrequencyDateRule(schedule);

        Date nextDueDate = rule.calculateNextDueDate();

        assertEquals(startDate, nextDueDate);
    }

    @Test()
    public void calculateNextDueDate() {
        Date startDate = new Date();

        //Make last due date later than start date
        Date lastDueDate = getLaterDate(startDate);

        updateSchedule(startDate, lastDueDate, "I100;D1;H1;M15");
        when(schedule.numTimesScheduled()).thenReturn(0);

        NextDueDateRule rule = new FrequencyDateRule(schedule);

        Date nextDueDate = rule.calculateNextDueDate();

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(lastDueDate);
        calendar.add(Calendar.DATE, 1);
        calendar.add(Calendar.HOUR_OF_DAY, 1);
        calendar.add(Calendar.MINUTE, 15);

        assertEquals(calendar.getTime(), nextDueDate);
    }

    private Date getLaterDate(Date lastDueDate) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(lastDueDate);
        calendar.add(Calendar.MINUTE, 1);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);

        return calendar.getTime();
    }
}
