package gov.va.vamf.service.shifttransition.tasks.domain.rules;

import gov.va.vamf.service.shifttransition.tasks.domain.Schedule;
import gov.va.vamf.service.shifttransition.tasks.domain.time.DueDate;

import org.junit.*;

import javax.ws.rs.WebApplicationException;

import java.text.*;
import java.util.*;

import static org.junit.Assert.*;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

/**
 * Add a Description.  Should include class purpose (why not what) and assumptions.
 */
public class CustomRuleTests {

    private static final String AMERICA_CHICAGO_TIMEZONE = "America/Chicago";
    private static final TimeZone timezone = TimeZone.getTimeZone("America/Chicago");
    private static SimpleDateFormat formatter;

    private Schedule schedule;

    @BeforeClass
    public static void setup() {
        formatter = new SimpleDateFormat("HHmm");
        formatter.setTimeZone(TimeZone.getTimeZone(AMERICA_CHICAGO_TIMEZONE));
    }

    @Before
    public void init() {
        schedule = mock(Schedule.class);
    }

    private void updateSchedule(Date startDate, Date nextDueDate, String scheduleFrequency) {
    when(schedule.startDate()).thenReturn(new DueDate(startDate, AMERICA_CHICAGO_TIMEZONE));
        when(schedule.scheduleFrequency()).thenReturn(scheduleFrequency);

        if (nextDueDate == null)
            when(schedule.nextDueDate()).thenReturn(null);
        else
            when(schedule.nextDueDate()).thenReturn(new DueDate(nextDueDate, AMERICA_CHICAGO_TIMEZONE));
    }

    @Test(expected = WebApplicationException.class)
    public void nullScheduleFrequency() {
        updateSchedule(new Date(), new Date(), null);
        new CustomRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void blankScheduleFrequency() {
        updateSchedule(new Date(), new Date(), "");
        new CustomRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void invalidTimezone() {
        updateSchedule(new Date(), null, "EDT;0015;0630;1245;1800");
        new CustomRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void noTimes() {
        updateSchedule(new Date(), new Date(), AMERICA_CHICAGO_TIMEZONE);
        new CustomRule(schedule);
    }

    @Test(expected = WebApplicationException.class)
    public void invalidStringTimeValue() {
        updateSchedule(new Date(), new Date(), AMERICA_CHICAGO_TIMEZONE + ";seven");
        new CustomRule(schedule);
    }

    @Test
    public void beforeSingleTime() {
        updateSchedule(createDate(10, 45), null, AMERICA_CHICAGO_TIMEZONE + ";1100|All|Weekly");
        Date nextDueDate = calculateDateDueDate();

        assertEquals("1100", formatter.format(nextDueDate));
    }

    @Test
    public void betweenTwoTime() {
        updateSchedule(createDate(12,30), null, AMERICA_CHICAGO_TIMEZONE + ";1130;1245|All|Weekly");
        Date nextDueDate = calculateDateDueDate();

        assertEquals("1245", formatter.format(nextDueDate));
    }

    @Test
    public void startDateAfterLastDueDate() {

        Date startDate = createDate(12, 30);

        Calendar calendar = Calendar.getInstance(TimeZone.getTimeZone(AMERICA_CHICAGO_TIMEZONE));
        calendar.setTime(startDate);
        calendar.add(Calendar.HOUR_OF_DAY, -6);

        Date nextDueDate = calendar.getTime();

        updateSchedule(startDate, nextDueDate, AMERICA_CHICAGO_TIMEZONE + ";1130;1245|All|Weekly");
        nextDueDate = calculateDateDueDate();

        SimpleDateFormat formatter = new SimpleDateFormat("HHmm");
        formatter.setTimeZone(TimeZone.getTimeZone(AMERICA_CHICAGO_TIMEZONE));

        assertEquals("1245", formatter.format(nextDueDate));
    }

    private Date createDate(int hours, int minutes) {
        Calendar calendar = Calendar.getInstance(timezone);
        calendar.setTime(new Date());
        calendar.set(Calendar.HOUR_OF_DAY, hours);
        calendar.set(Calendar.MINUTE, minutes);

        return calendar.getTime();
    }

    @Test
    public void afterSingleTime() {
        Calendar calendar = Calendar.getInstance(timezone);
        calendar.setTime(new Date());
        calendar.set(Calendar.DATE, 12);
        calendar.set(Calendar.HOUR_OF_DAY, 12);
        calendar.set(Calendar.MINUTE, 30);

        updateSchedule(calendar.getTime(), null, AMERICA_CHICAGO_TIMEZONE + ";1115|All|Weekly");
        Date nextDueDate = calculateDateDueDate();

        SimpleDateFormat formatter = new SimpleDateFormat("ddHHmm");
        formatter.setTimeZone(timezone);

        assertEquals("131115", formatter.format(nextDueDate));
    }

    private Date calculateDateDueDate() {
        NextDueDateRule rule = new CustomRule(schedule);
        return rule.calculateNextDueDate();
    }
}
