package gov.va.vamf.service.shifttransition.tasks.domain.rules;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import gov.va.vamf.service.shifttransition.tasks.domain.Schedule;
import gov.va.vamf.service.shifttransition.tasks.domain.time.DueDate;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import org.junit.Before;
import org.junit.Test;

public class CustomRuleDayWeekTests {
	private static final String AMERICA_CHICAGO_TIMEZONE = "America/Chicago";
	private static final TimeZone timezone = TimeZone.getTimeZone("America/Chicago");
	
	private Schedule schedule;

//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Setup & Mock Services--------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	@Before
	public void init() {
		schedule = mock(Schedule.class);
	}
	
	private void updateSchedule(Date startDate, Date nextDueDate, String scheduleFrequency) {
	when(schedule.startDate()).thenReturn(new DueDate(startDate, AMERICA_CHICAGO_TIMEZONE));
		when(schedule.scheduleFrequency()).thenReturn(scheduleFrequency);

		if (nextDueDate == null)
			when(schedule.nextDueDate()).thenReturn(null);
		else
			when(schedule.nextDueDate()).thenReturn(new DueDate(nextDueDate, AMERICA_CHICAGO_TIMEZONE));
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Single Day One Time----------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	@Test
	public void tuesdaysOneTime() {
		validateNextDueDate("12/30/2014 11:15", null, "01/06/2015 11:15", "1115|Tue|Weekly");      //Every Tuesday
		validateNextDueDate("12/30/2014 11:15", null, "01/13/2015 11:15", "1115|Tue|EveryOther");  //Every Tuesday of every other week
		validateNextDueDate("12/30/2014 11:15", null, "01/20/2015 11:15", "1115|Tue|EveryThird");  //Every Tuesday of every third week
		validateNextDueDate("12/30/2014 11:15", null, "01/27/2015 11:15", "1115|Tue|EveryFourth"); //Every Tuesday of every fourth week
	}

	@Test
	public void tuesdaysOneTimeWeekly() {
		validateNextDueDate("12/30/2014 11:15", "01/06/2015 11:15", "01/13/2015 11:15", "1115|Tue|Weekly");
		validateNextDueDate("12/30/2014 11:15", "01/13/2015 11:15", "01/20/2015 11:15", "1115|Tue|Weekly");
		validateNextDueDate("12/30/2014 11:15", "01/20/2015 11:15", "01/27/2015 11:15", "1115|Tue|Weekly");
	}

	@Test
	public void tuesdaysOneTimeEveryOther() {
		validateNextDueDate("12/30/2014 11:15", "01/13/2015 11:15", "01/27/2015 11:15", "1115|Tue|EveryOther");
		validateNextDueDate("12/30/2014 11:15", "01/27/2015 11:15", "02/10/2015 11:15", "1115|Tue|EveryOther");
		validateNextDueDate("12/30/2014 11:15", "02/10/2015 11:15", "02/24/2015 11:15", "1115|Tue|EveryOther");
	}

	@Test
	public void tuesdaysOneTimeEveryThird() {
		validateNextDueDate("12/30/2014 11:15", "01/20/2015 11:15", "02/10/2015 11:15", "1115|Tue|EveryThird");
		validateNextDueDate("12/30/2014 11:15", "02/10/2015 11:15", "03/03/2015 11:15", "1115|Tue|EveryThird");
		validateNextDueDate("12/30/2014 11:15", "03/03/2015 11:15", "03/24/2015 11:15", "1115|Tue|EveryThird");
	}

	@Test
	public void tuesdaysOneTimeEveryFourth() {
		validateNextDueDate("12/30/2014 11:15", "01/27/2015 11:15", "02/24/2015 11:15", "1115|Tue|EveryFourth");
		validateNextDueDate("12/30/2014 11:15", "02/24/2015 11:15", "03/24/2015 11:15", "1115|Tue|EveryFourth");
		validateNextDueDate("12/30/2014 11:15", "03/24/2015 11:15", "04/21/2015 11:15", "1115|Tue|EveryFourth");
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Multiple Days One Time-------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	@Test
	public void monWedThurOneTime() {
		validateNextDueDate("12/29/2014 11:15", null, "12/31/2014 11:15", "1115|Mon;Wed;Thu|Weekly");
		validateNextDueDate("12/29/2014 11:15", null, "12/31/2014 11:15", "1115|Mon;Wed;Thu|EveryOther");
		validateNextDueDate("12/29/2014 11:15", null, "12/31/2014 11:15", "1115|Mon;Wed;Thu|EveryThird");
		validateNextDueDate("12/29/2014 11:15", null, "12/31/2014 11:15", "1115|Mon;Wed;Thu|EveryFourth");
	}

	@Test
	public void monWedThurOneTimeWeekly() {
		validateNextDueDate("12/29/2014 11:15", "01/05/2015 11:15", "01/07/2015 11:15", "1115|Mon;Wed;Thu|Weekly"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/07/2015 11:15", "01/08/2015 11:15", "1115|Mon;Wed;Thu|Weekly"); //Thur
		validateNextDueDate("12/29/2014 11:15", "01/08/2015 11:15", "01/12/2015 11:15", "1115|Mon;Wed;Thu|Weekly"); //Next Valid Monday
	}

	@Test
	public void monWedThurOneTimeEveryOther() {
		validateNextDueDate("12/29/2014 11:15", "01/12/2015 11:15", "01/14/2015 11:15", "1115|Mon;Wed;Thu|EveryOther"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/14/2015 11:15", "01/15/2015 11:15", "1115|Mon;Wed;Thu|EveryOther"); //Thur
		validateNextDueDate("12/29/2014 11:15", "01/15/2015 11:15", "01/26/2015 11:15", "1115|Mon;Wed;Thu|EveryOther"); //Next Valid Monday
	}

	@Test
	public void monWedThurOneTimeEveryThird() {
		validateNextDueDate("12/29/2014 11:15", "01/19/2015 11:15", "01/21/2015 11:15", "1115|Mon;Wed;Thu|EveryThird"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/21/2015 11:15", "01/22/2015 11:15", "1115|Mon;Wed;Thu|EveryThird"); //Thur
		validateNextDueDate("12/29/2014 11:15", "01/22/2015 11:15", "02/09/2015 11:15", "1115|Mon;Wed;Thu|EveryThird"); //Next Valid Monday
	}

	@Test
	public void monWedThurOneTimeEveryFourth() {
		validateNextDueDate("12/29/2014 11:15", "01/26/2015 11:15", "01/28/2015 11:15", "1115|Mon;Wed;Thu|EveryFourth"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/28/2015 11:15", "01/29/2015 11:15", "1115|Mon;Wed;Thu|EveryFourth"); //Thur
		validateNextDueDate("12/29/2014 11:15", "01/29/2015 11:15", "02/23/2015 11:15", "1115|Mon;Wed;Thu|EveryFourth"); //Next Valid Monday
	}
	
	@Test
	public void monTueWedThuFriOneTime() {
		validateNextDueDate("12/29/2014 11:15", null, "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly");
		validateNextDueDate("12/29/2014 11:15", null, "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther");
		validateNextDueDate("12/29/2014 11:15", null, "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird");
		validateNextDueDate("12/29/2014 11:15", null, "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth");
	}

	@Test
	public void monTueWedThuFriOneTimeWeekly() {
		validateNextDueDate("12/29/2014 11:15", "12/29/2014 11:15", "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Tue
		validateNextDueDate("12/29/2014 11:15", "12/30/2014 11:15", "12/31/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Wed
		validateNextDueDate("12/29/2014 11:15", "12/31/2014 11:15", "01/01/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/01/2015 11:15", "01/02/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/02/2015 11:15", "01/05/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Next Valid Monday
		
		//Check another period
		validateNextDueDate("12/29/2014 11:15", "01/05/2015 11:15", "01/06/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Tue
		validateNextDueDate("12/29/2014 11:15", "01/06/2015 11:15", "01/07/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/07/2015 11:15", "01/08/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/08/2015 11:15", "01/09/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/09/2015 11:15", "01/12/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|Weekly"); //Next Valid Monday
	}

	@Test
	public void monTueWedThuFriOneTimeEveryOther() {
		validateNextDueDate("12/29/2014 11:15", "12/29/2014 11:15", "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Tue
		validateNextDueDate("12/29/2014 11:15", "12/30/2014 11:15", "12/31/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Wed
		validateNextDueDate("12/29/2014 11:15", "12/31/2014 11:15", "01/01/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/01/2015 11:15", "01/02/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/02/2015 11:15", "01/12/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Next Valid Monday
		
		//Check another period
		validateNextDueDate("12/29/2014 11:15", "01/12/2015 11:15", "01/13/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Tue
		validateNextDueDate("12/29/2014 11:15", "01/13/2015 11:15", "01/14/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/14/2015 11:15", "01/15/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/15/2015 11:15", "01/16/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/16/2015 11:15", "01/26/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryOther"); //Next Valid Monday
	}

	@Test
	public void monTueWedThuFriOneTimeEveryThird() {
		validateNextDueDate("12/29/2014 11:15", "12/29/2014 11:15", "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Tue
		validateNextDueDate("12/29/2014 11:15", "12/30/2014 11:15", "12/31/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Wed
		validateNextDueDate("12/29/2014 11:15", "12/31/2014 11:15", "01/01/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/01/2015 11:15", "01/02/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/02/2015 11:15", "01/19/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Next Valid Monday
		
		//Check another period
		validateNextDueDate("12/29/2014 11:15", "01/19/2015 11:15", "01/20/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Tue
		validateNextDueDate("12/29/2014 11:15", "01/20/2015 11:15", "01/21/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/21/2015 11:15", "01/22/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/22/2015 11:15", "01/23/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/23/2015 11:15", "02/09/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryThird"); //Next Valid Monday
	}

	@Test
	public void monTueWedThuFriOneTimeEveryFourth() {
		validateNextDueDate("12/29/2014 11:15", "12/29/2014 11:15", "12/30/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Tue
		validateNextDueDate("12/29/2014 11:15", "12/30/2014 11:15", "12/31/2014 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Wed
		validateNextDueDate("12/29/2014 11:15", "12/31/2014 11:15", "01/01/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/01/2015 11:15", "01/02/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/09/2015 11:15", "01/26/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Next Valid Monday
		
		//Check another period
		validateNextDueDate("12/29/2014 11:15", "01/26/2015 11:15", "01/27/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Tue
		validateNextDueDate("12/29/2014 11:15", "01/27/2015 11:15", "01/28/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Wed
		validateNextDueDate("12/29/2014 11:15", "01/28/2015 11:15", "01/29/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Thu
		validateNextDueDate("12/29/2014 11:15", "01/29/2015 11:15", "01/30/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Fri
		
		validateNextDueDate("12/29/2014 11:15", "01/30/2015 11:15", "02/23/2015 11:15", "1115|Mon;Tue;Wed;Thu;Fri|EveryFourth"); //Next Valid Monday
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Single Day Multiple Times----------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	@Test
	public void tuesdaysMultipleTimes() {
		validateNextDueDate("12/30/2014 11:15", null, "12/30/2014 11:30", "1115;1130;1245|Tue|Weekly");      //Next Time Same Day
		validateNextDueDate("12/30/2014 11:15", null, "12/30/2014 11:30", "1115;1130;1245|Tue|EveryOther");  //Next Time Same Day
		validateNextDueDate("12/30/2014 11:15", null, "12/30/2014 11:30", "1115;1130;1245|Tue|EveryThird");  //Next Time Same Day
		validateNextDueDate("12/30/2014 11:15", null, "12/30/2014 11:30", "1115;1130;1245|Tue|EveryFourth"); //Next Time Same Day
	}

	@Test
	public void tuesdaysMultipleTimesWeekly() {
		validateNextDueDate("12/30/2014 11:15", "01/06/2015 11:15", "01/06/2015 11:30", "1115;1130;1245|Tue|Weekly"); //2nd time
		validateNextDueDate("12/30/2014 11:15", "01/06/2015 11:30", "01/06/2015 12:45", "1115;1130;1245|Tue|Weekly"); //3rd time
		validateNextDueDate("12/30/2014 11:15", "01/06/2015 12:45", "01/13/2015 11:15", "1115;1130;1245|Tue|Weekly"); //next date
	}

	@Test
	public void tuesdaysMultipleTimesEveryOther() {
		validateNextDueDate("12/30/2014 11:15", "01/13/2015 11:15", "01/13/2015 11:30", "1115;1130;1245|Tue|EveryOther"); //2nd time
		validateNextDueDate("12/30/2014 11:15", "01/13/2015 11:30", "01/13/2015 12:45", "1115;1130;1245|Tue|EveryOther"); //3rd time
		validateNextDueDate("12/30/2014 11:15", "01/13/2015 12:45", "01/27/2015 11:15", "1115;1130;1245|Tue|EveryOther"); //next date
	}

	@Test
	public void tuesdaysMultipleTimesEveryThird() {
		validateNextDueDate("12/30/2014 11:15", "01/20/2015 11:15", "01/20/2015 11:30", "1115;1130;1245|Tue|EveryThird"); //2nd time
		validateNextDueDate("12/30/2014 11:15", "01/20/2015 11:30", "01/20/2015 12:45", "1115;1130;1245|Tue|EveryThird"); //3rd time
		validateNextDueDate("12/30/2014 11:15", "01/20/2015 12:45", "02/10/2015 11:15", "1115;1130;1245|Tue|EveryThird"); //next date
	}

	@Test
	public void tuesdaysMultipleTimesEveryFourth() {
		validateNextDueDate("12/30/2014 11:15", "01/27/2015 11:15", "01/27/2015 11:30", "1115;1130;1245|Tue|EveryFourth"); //2nd time
		validateNextDueDate("12/30/2014 11:15", "01/27/2015 11:30", "01/27/2015 12:45", "1115;1130;1245|Tue|EveryFourth"); //3rd time
		validateNextDueDate("12/30/2014 11:15", "01/27/2015 12:45", "02/24/2015 11:15", "1115;1130;1245|Tue|EveryFourth"); //next date
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Multiple Days Multiple Times-------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	@Test
	public void monWedThurMultipleTimes() {
		validateNextDueDate("12/29/2014 11:15", null, "12/29/2014 11:30", "1115;1130;1245|Mon;Wed;Thu|Weekly");      //Next Time Same Day
		validateNextDueDate("12/29/2014 11:15", null, "12/29/2014 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryOther");  //Next Time Same Day
		validateNextDueDate("12/29/2014 11:15", null, "12/29/2014 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryThird");  //Next Time Same Day
		validateNextDueDate("12/29/2014 11:15", null, "12/29/2014 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //Next Time Same Day
	}

	@Test
	public void monWedThurMultipleTimesWeekly() {
		//Wed
		validateNextDueDate("12/29/2014 11:15", "01/05/2015 11:15", "01/05/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/05/2015 11:30", "01/05/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/05/2015 12:45", "01/07/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //next date
		
		//Thur
		validateNextDueDate("12/29/2014 11:15", "01/07/2015 11:15", "01/07/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/07/2015 11:30", "01/07/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/07/2015 12:45", "01/08/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //next date
		
		//Next Valid Monday
		validateNextDueDate("12/29/2014 11:15", "01/08/2015 11:15", "01/08/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/08/2015 11:30", "01/08/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/08/2015 12:45", "01/12/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|Weekly"); //next date
	}

	@Test
	public void monWedThurMultipleTimesEveryOther() {
		//Wed
		validateNextDueDate("12/29/2014 11:15", "01/12/2015 11:15", "01/12/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/12/2015 11:30", "01/12/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/12/2015 12:45", "01/14/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //next date

		//Thur
		validateNextDueDate("12/29/2014 11:15", "01/14/2015 11:15", "01/14/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/14/2015 11:30", "01/14/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/14/2015 12:45", "01/15/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //next date

		//Next Valid Monday
		validateNextDueDate("12/29/2014 11:15", "01/15/2015 11:15", "01/15/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/15/2015 11:30", "01/15/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/15/2015 12:45", "01/26/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryOther"); //next date
	}

	@Test
	public void monWedThurMultipleTimesEveryThird() {
		//Wed
		validateNextDueDate("12/29/2014 11:15", "01/19/2015 11:15", "01/19/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/19/2015 11:30", "01/19/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/19/2015 12:45", "01/21/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //next date
		
		//Thur
		validateNextDueDate("12/29/2014 11:15", "01/21/2015 11:15", "01/21/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/21/2015 11:30", "01/21/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/21/2015 12:45", "01/22/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //next date
		
		//Next Valid Monday
		validateNextDueDate("12/29/2014 11:15", "01/22/2015 11:15", "01/22/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/22/2015 11:30", "01/22/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/22/2015 12:45", "02/09/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryThird"); //next date
	}

	@Test
	public void monWedThurMultipleTimesEveryFourth() {
		//Wed
		validateNextDueDate("12/29/2014 11:15", "01/26/2015 11:15", "01/26/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/26/2015 11:30", "01/26/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/26/2015 12:45", "01/28/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //next date
		
		//Thur
		validateNextDueDate("12/29/2014 11:15", "01/28/2015 11:15", "01/28/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/28/2015 11:30", "01/28/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/28/2015 12:45", "01/29/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //next date
		
		//Next Valid Monday
		validateNextDueDate("12/29/2014 11:15", "01/29/2015 11:15", "01/29/2015 11:30", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //2nd time
		validateNextDueDate("12/29/2014 11:15", "01/29/2015 11:30", "01/29/2015 12:45", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //3rd time
		validateNextDueDate("12/29/2014 11:15", "01/29/2015 12:45", "02/23/2015 11:15", "1115;1130;1245|Mon;Wed;Thu|EveryFourth"); //next date
	}

//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------All With Only One Time-------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
	
	@Test
	public void allWithOneTime() {
		validateNextDueDate("01/01/2015 11:15", null, "01/02/2015 11:15", "1115|All|Weekly");      //Every Day
		validateNextDueDate("01/01/2015 11:15", null, "01/02/2015 11:15", "1115|All|EveryOther");  //Every Day of every other week
		validateNextDueDate("01/01/2015 11:15", null, "01/02/2015 11:15", "1115|All|EveryThird");  //Every Day of every third week
		validateNextDueDate("01/01/2015 11:15", null, "01/02/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
	}

	@Test
	public void allWithOneTimeWeekly() {
		validateNextDueDate("01/01/2015 11:15", "01/02/2015 11:15", "01/03/2015 11:15", "1115|All|Weekly"); //Every Day
		validateNextDueDate("01/01/2015 11:15", "01/03/2015 11:15", "01/04/2015 11:15", "1115|All|Weekly"); //Every Day
		validateNextDueDate("01/01/2015 11:15", "01/04/2015 11:15", "01/05/2015 11:15", "1115|All|Weekly"); //Every Day
		validateNextDueDate("01/01/2015 11:15", "01/05/2015 11:15", "01/06/2015 11:15", "1115|All|Weekly"); //Every Day
		validateNextDueDate("01/01/2015 11:15", "01/06/2015 11:15", "01/07/2015 11:15", "1115|All|Weekly"); //Every Day
		validateNextDueDate("01/01/2015 11:15", "01/07/2015 11:15", "01/08/2015 11:15", "1115|All|Weekly"); //Every Day
		validateNextDueDate("01/01/2015 11:15", "01/08/2015 11:15", "01/09/2015 11:15", "1115|All|Weekly"); //Every Day
		
		validateNextDueDate("01/01/2015 11:15", "01/09/2015 11:15", "01/10/2015 11:15", "1115|All|Weekly"); //Every Day
	}

	@Test
	public void allWithOneTimeEveryOther() {
		validateNextDueDate("01/01/2015 11:15", "01/15/2015 11:15", "01/16/2015 11:15", "1115|All|EveryOther"); //Every Day of every other week
		validateNextDueDate("01/01/2015 11:15", "01/16/2015 11:15", "01/17/2015 11:15", "1115|All|EveryOther"); //Every Day of every other week
		validateNextDueDate("01/01/2015 11:15", "01/17/2015 11:15", "01/18/2015 11:15", "1115|All|EveryOther"); //Every Day of every other week
		validateNextDueDate("01/01/2015 11:15", "01/18/2015 11:15", "01/19/2015 11:15", "1115|All|EveryOther"); //Every Day of every other week
		validateNextDueDate("01/01/2015 11:15", "01/19/2015 11:15", "01/20/2015 11:15", "1115|All|EveryOther"); //Every Day of every other week
		validateNextDueDate("01/01/2015 11:15", "01/20/2015 11:15", "01/21/2015 11:15", "1115|All|EveryOther"); //Every Day of every other week
		
		validateNextDueDate("01/01/2015 11:15", "01/21/2015 11:15", "01/29/2015 11:15", "1115|All|EveryOther"); //Every Day of every other week
	}

	@Test
	public void allWithOneTimeEveryThird() {
		validateNextDueDate("01/01/2015 11:15", "01/22/2015 11:15", "01/23/2015 11:15", "1115|All|EveryThird"); //Every Day of every third week
		validateNextDueDate("01/01/2015 11:15", "01/23/2015 11:15", "01/24/2015 11:15", "1115|All|EveryThird"); //Every Day of every third week
		validateNextDueDate("01/01/2015 11:15", "01/24/2015 11:15", "01/25/2015 11:15", "1115|All|EveryThird"); //Every Day of every third week
		validateNextDueDate("01/01/2015 11:15", "01/25/2015 11:15", "01/26/2015 11:15", "1115|All|EveryThird"); //Every Day of every third week
		validateNextDueDate("01/01/2015 11:15", "01/26/2015 11:15", "01/27/2015 11:15", "1115|All|EveryThird"); //Every Day of every third week
		validateNextDueDate("01/01/2015 11:15", "01/27/2015 11:15", "01/28/2015 11:15", "1115|All|EveryThird"); //Every Day of every third week
		
		validateNextDueDate("01/01/2015 11:15", "01/28/2015 11:15", "02/12/2015 11:15", "1115|All|EveryThird"); //Every Day of every third week
	}

	@Test
	public void allWithOneTimeEveryFourth() {
		validateNextDueDate("01/01/2015 11:15", "01/29/2015 11:15", "01/30/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
		validateNextDueDate("01/01/2015 11:15", "01/30/2015 11:15", "01/31/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
		validateNextDueDate("01/01/2015 11:15", "01/31/2015 11:15", "02/01/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
		validateNextDueDate("01/01/2015 11:15", "02/01/2015 11:15", "02/02/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
		validateNextDueDate("01/01/2015 11:15", "02/02/2015 11:15", "02/03/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
		validateNextDueDate("01/01/2015 11:15", "02/03/2015 11:15", "02/04/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
		
		validateNextDueDate("01/01/2015 11:15", "02/04/2015 11:15", "02/26/2015 11:15", "1115|All|EveryFourth"); //Every Day of every fourth week
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------All With Multiple Times------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
	
	@Test
	public void allWithMultipleTimes() {
		validateNextDueDate("01/01/2015 11:15", null, "01/01/2015 11:30", "1115;1130;1245|All|Weekly");      //Next Time Same Day
		validateNextDueDate("01/01/2015 11:15", null, "01/01/2015 11:30", "1115;1130;1245|All|EveryOther");  //Next Time Same Day
		validateNextDueDate("01/01/2015 11:15", null, "01/01/2015 11:30", "1115;1130;1245|All|EveryThird");  //Next Time Same Day
		validateNextDueDate("01/01/2015 11:15", null, "01/01/2015 11:30", "1115;1130;1245|All|EveryFourth"); //Next Time Same Day
	}

	@Test
	public void allWithMultipleTimesWeekly() {
		validateNextDueDate("01/01/2015 11:15", "01/02/2015 11:15", "01/02/2015 11:30", "1115;1130;1245|All|Weekly"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "01/02/2015 11:30", "01/02/2015 12:45", "1115;1130;1245|All|Weekly"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "01/02/2015 12:45", "01/03/2015 11:15", "1115;1130;1245|All|Weekly"); //next date
		
		validateNextDueDate("01/01/2015 11:15", "01/02/2015 11:15", "01/02/2015 11:30", "1115;1130;1245|All|Weekly"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "01/02/2015 11:30", "01/02/2015 12:45", "1115;1130;1245|All|Weekly"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "01/02/2015 12:45", "01/03/2015 11:15", "1115;1130;1245|All|Weekly"); //next date
	}

	@Test
	public void allWithMultipleTimesEveryOther() {
		validateNextDueDate("01/01/2015 11:15", "01/15/2015 11:15", "01/15/2015 11:30", "1115;1130;1245|All|EveryOther"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "01/15/2015 11:30", "01/15/2015 12:45", "1115;1130;1245|All|EveryOther"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "01/15/2015 12:45", "01/16/2015 11:15", "1115;1130;1245|All|EveryOther"); //next date
		
		validateNextDueDate("01/01/2015 11:15", "01/21/2015 11:15", "01/21/2015 11:30", "1115;1130;1245|All|EveryOther"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "01/21/2015 11:30", "01/21/2015 12:45", "1115;1130;1245|All|EveryOther"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "01/21/2015 12:45", "01/29/2015 11:15", "1115;1130;1245|All|EveryOther"); //next date
	}

	@Test
	public void allWithMultipleTimesEveryThird() {
		validateNextDueDate("01/01/2015 11:15", "01/22/2015 11:15", "01/22/2015 11:30", "1115;1130;1245|All|EveryThird"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "01/22/2015 11:30", "01/22/2015 12:45", "1115;1130;1245|All|EveryThird"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "01/22/2015 12:45", "01/23/2015 11:15", "1115;1130;1245|All|EveryThird"); //next date
		
		validateNextDueDate("01/01/2015 11:15", "01/28/2015 11:15", "01/28/2015 11:30", "1115;1130;1245|All|EveryThird"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "01/28/2015 11:30", "01/28/2015 12:45", "1115;1130;1245|All|EveryThird"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "01/28/2015 12:45", "02/12/2015 11:15", "1115;1130;1245|All|EveryThird"); //next date
	}

	@Test
	public void allWithMultipleTimesEveryFourth() {
		validateNextDueDate("01/01/2015 11:15", "01/29/2015 11:15", "01/29/2015 11:30", "1115;1130;1245|All|EveryFourth"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "01/29/2015 11:30", "01/29/2015 12:45", "1115;1130;1245|All|EveryFourth"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "01/29/2015 12:45", "01/30/2015 11:15", "1115;1130;1245|All|EveryFourth"); //next date
		
		validateNextDueDate("01/01/2015 11:15", "02/04/2015 11:15", "02/04/2015 11:30", "1115;1130;1245|All|EveryFourth"); //2nd time
		validateNextDueDate("01/01/2015 11:15", "02/04/2015 11:30", "02/04/2015 12:45", "1115;1130;1245|All|EveryFourth"); //3rd time
		validateNextDueDate("01/01/2015 11:15", "02/04/2015 12:45", "02/26/2015 11:15", "1115;1130;1245|All|EveryFourth"); //next date
	}
	
//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Validation Helper Method-----------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
	
	private void validateNextDueDate(String strStartDate, String strLastDueDate, String expectedDueDate, String timeDayWeek) {
		Date startDate = getDateFromString(strStartDate);
		assertNotNull("Unable to parse date: " + strStartDate, startDate);
		Date lastDueDate = getDateFromString(strLastDueDate);
		
		updateSchedule(startDate, lastDueDate, AMERICA_CHICAGO_TIMEZONE + ";" + timeDayWeek);
		Date calculatedDueDate = calculateDateDueDate();

		String actualDueDate = getStringFromDate(calculatedDueDate);
		assertEquals(expectedDueDate, actualDueDate);
	}

//----------------------------------------------------------------------------------------------------------------------
//-----------------------------------Helper Methods---------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	private Date getDateFromString(String str) {
		SimpleDateFormat sdf = new SimpleDateFormat("MM/dd/yyyy HH:mm");
		sdf.setTimeZone(timezone);
		Calendar calendar = Calendar.getInstance(timezone);
		Date retvalue = null;
		try {
			Date dt = sdf.parse(str);
			calendar.setTime(dt);
			retvalue = calendar.getTime();
		}
		catch (Exception e) {
			//Ignore
		}
		return retvalue;
	}
	
	private String getStringFromDate(Date dt) {
		SimpleDateFormat sdf = new SimpleDateFormat("MM/dd/yyyy HH:mm");
		sdf.setTimeZone(timezone);
		String retvalue = sdf.format(dt);
		return retvalue;
	}

	private Date calculateDateDueDate() {
		NextDueDateRule rule = new CustomRule(schedule);
		return rule.calculateNextDueDate();
	}
}
