package gov.va.vamf.service.shifttransition.tasks.domain;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.tasks.representations.NewScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTaskSummary;
import gov.va.vamf.service.shifttransition.tasks.representations.ScheduledTask;
import gov.va.vamf.service.shifttransition.tasks.representations.UpdatedScheduledTask;

import java.lang.reflect.Field;
import java.util.Date;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import com.google.common.collect.Lists;

public class TaskTests {

    private Task task;
    private Date startDate;

    @Before
    public void init() {
        startDate = new Date();

        NewScheduledTask newScheduleTask = new NewScheduledTask();
        newScheduleTask.name = "name";
        newScheduleTask.type = "type";
        newScheduleTask.externalTaskUrl = "url";
        newScheduleTask.startDate = startDate;
        newScheduleTask.comment = "test";
        task = new Task(newScheduleTask, "superman");
    }

    @Test
    public void newTask() {
        ScheduledTask scheduledTask = task.asScheduledTask();

        assertEquals("name", scheduledTask.name);
        assertEquals("type", scheduledTask.type);
        assertEquals("url", scheduledTask.externalTaskUrl);
        assertEquals("superman", scheduledTask.createdBy);
        assertEquals("test", scheduledTask.comment);

        assertNotNull(scheduledTask.lastModifiedDate);
    }

    @Test
    public void taskMatch() throws NoSuchFieldException, IllegalAccessException {
        assertFalse(task.match("1"));

        setTaskId();

        assertTrue(task.match("1"));
        assertFalse(task.match("2"));
        assertFalse(task.match(null));
    }

    private void setTaskId() throws NoSuchFieldException, IllegalAccessException {
        Field f = task.getClass().getDeclaredField("id");
        f.setAccessible(true);
        f.set(task, "1");
    }

    @Test
    public void taskUpdatedWithoutLastModifiedBy() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.comment = "test";
        updatedTask.startDate = startDate;

        task.updateTask(updatedTask, null);

        ScheduledTask scheduledTask = task.asScheduledTask();

        assertNull(scheduledTask.lastModifiedBy);
        assertEquals("test", scheduledTask.comment);
    }

    @Test
    public void taskUpdatedWithLastModifiedBy() {
        Date updatedStartDate = new Date();

        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.comment = "new comment";
        updatedTask.startDate = updatedStartDate;

        task.updateTask(updatedTask, "batman");

        ScheduledTask scheduledTask = task.asScheduledTask();

        assertEquals("batman", scheduledTask.lastModifiedBy);
        assertEquals("new comment", scheduledTask.comment);
        assertEquals(updatedStartDate, scheduledTask.startDate);
        assertNotNull(scheduledTask.lastModifiedDate);
    }

    @Test
    public void moreUrgentThanNull() {
        Task mostUrgent = task.moreUrgent(createRangeWithOffsets(0, 0), null);

        assertEquals(task, mostUrgent);
    }

    @Test
    public void moreUrgentThanOtherTask() {
        Task mostUrgent = task.moreUrgent(createRangeWithOffsets(-60000, 60000), Task.createNullTask());

        assertEquals(task, mostUrgent);
    }

    @Test
    public void otherTaskMoreUrgent() {
        NewScheduledTask newScheduleTask = new NewScheduledTask();
        newScheduleTask.startDate = startDate;

        Task otherTask = new Task(newScheduleTask, "robin");

        Task mostUrgent = task.moreUrgent(createRangeWithOffsets(60000, 120000), otherTask);

        assertEquals(otherTask, mostUrgent);
    }

    @Test
    public void addPatientTaskSummary() throws NoSuchFieldException, IllegalAccessException {
        setTaskId();

        List<PatientTaskSummary> patientTaskSumaries = Lists.newArrayList();

        task.addCurrentTaskSummary(createRangeWithOffsets(-60000, 60000), patientTaskSumaries);

        assertEquals("1", patientTaskSumaries.get(0).taskId);
    }

    private Range createRangeWithOffsets(int startDateOffset, int endDateOffset) {
        return new Range(new Date(startDate.getTime() + startDateOffset), new Date(startDate.getTime() + endDateOffset));
    }
}
