package gov.va.vamf.service.shifttransition.tasks.domain;

import gov.va.vamf.service.shifttransition.tasks.domain.time.*;
import gov.va.vamf.service.shifttransition.tasks.representations.*;

import org.junit.*;

import java.util.*;

import static org.junit.Assert.*;

public class ScheduleTests {
    private Date startDate;
    private Schedule schedule;

    @Before
    public void init() {
        startDate = new Date();

        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = startDate;

        schedule = new Schedule(updatedTask);
    }

    @Test
    public void canOnlyScheduleTaskOnceWhenScheduleTypeNone() {
        schedule = schedule.getNextSchedule();

        assertNull(schedule.nextDueDate());
        assertNull(schedule.endDate());
        assertEquals(1, schedule.numTimesScheduled());
    }

    @Test
    public void schedulingResetsNumOfTimesScheduled() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = startDate;
        updatedTask.scheduleType = "Frequency";
        updatedTask.scheduleFrequency = "I5;D0;H1;M0";

        schedule = new Schedule(updatedTask);
        schedule = schedule.getNextSchedule();
        schedule = schedule.getNextSchedule();

        assertEquals(3, schedule.numTimesScheduled());
    }

    @Test
    public void createPatientTaskSummaryOutOfRange() {
        PatientTaskSummary summary = schedule.createPatientTaskSummary(createRangeWithOffsets(600000, 800000));

        assertEquals(TaskStatus.NA.toString(), summary.status);
        assertNull(summary.nextDueDate);
    }

    @Test
    public void createPatientTaskSummaryStatusNA() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(91).get();
        schedule = new Schedule(updatedTask);

        PatientTaskSummary summary = schedule.createPatientTaskSummary(createRangeWithOffsets(-60000000, 800000000));

        assertEquals(TaskStatus.NA.toString(), summary.status);
        assertNull(summary.nextDueDate);
    }

    @Test
    public void createPatientTaskSummaryStatusPending() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(90).get();
        schedule = new Schedule(updatedTask);

        PatientTaskSummary summary = schedule.createPatientTaskSummary(createRangeWithOffsets(-60000000, 80000000));

        assertEquals("Pending", summary.status);
        assertNotNull(summary.nextDueDate);
    }

    private Range createRangeWithOffsets(int startDateOffset, int endDateOffset) {
        return new Range(new Date(startDate.getTime() + startDateOffset), new Date(startDate.getTime() + endDateOffset));
    }

    @Test
    public void endDateReachedForTask() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = startDate;
        updatedTask.scheduleType = "Frequency";
        updatedTask.scheduleFrequency = "I0;D0;H0;M1";
        updatedTask.endDate = new DueDate(startDate).addMinutes(1).get();

        schedule = new Schedule(updatedTask);
        schedule = schedule.getNextSchedule();
        schedule = schedule.getNextSchedule();

        PatientTasksStatus tasksStatus = schedule.generatePatientTasksStatus();

        assertEquals(TaskStatus.NA.toString(), tasksStatus.status);
        assertNull(tasksStatus.nextDueDate);
    }

    @Test
    public void statusIsDueForTaskMaxBoundary() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(30).get();
        schedule = new Schedule(updatedTask);

        PatientTaskSummary summary = schedule.createPatientTaskSummary(createRangeWithOffsets(-60000000, 800000000));

        assertEquals("Due", summary.status);
        assertNotNull(summary.nextDueDate);
    }

    @Test
    public void statusIsDueForTaskMinBoundary() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(1).get();
        schedule = new Schedule(updatedTask);

        PatientTaskSummary summary = schedule.createPatientTaskSummary(createRangeWithOffsets(-60000000, 800000000));

        assertEquals("Due", summary.status);
        assertNotNull(summary.nextDueDate);
    }

    @Test
    public void statusIsPastDueForTask() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(-1).get();
        schedule = new Schedule(updatedTask);

        PatientTaskSummary summary = schedule.createPatientTaskSummary(createRangeWithOffsets(-60000000, 800000000));

        assertEquals("Past Due", summary.status);
        assertNotNull(summary.nextDueDate);
    }

    @Test
    public void thisScheduleNotMoreUrgentWithNextDueDateOutOfRange() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(+11).get();
        Schedule other = new Schedule(updatedTask);

        assertFalse(schedule.moreUrgent(createRangeWithOffsets(600000, 800000), other));
    }

    @Test
    public void thisScheduleMoreUrgentWithMoreUrgentStatus() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(91).get();
        Schedule other = new Schedule(updatedTask);

        assertTrue(schedule.moreUrgent(createRangeWithOffsets(-60000000, 800000000), other));
    }

    @Test
    public void thisScheduleMoreUrgentWithSameStatusSoonerDueDate() {
        UpdatedScheduledTask updatedTask = new UpdatedScheduledTask();
        updatedTask.startDate = new DueDate(startDate).addMinutes(10).get();
        Schedule other = new Schedule(updatedTask);

        assertTrue(schedule.moreUrgent(createRangeWithOffsets(-60000000, 800000000), other));
    }
}
