package gov.va.vamf.service.shifttransition.watchlist;

import gov.va.vamf.service.shifttransition.dischargeprocess.*;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp400BadRequestException;
import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.application.resources.Routes;
import gov.va.vamf.service.shifttransition.tasks.representations.*;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import javax.ws.rs.*;
import javax.ws.rs.core.*;

import java.util.*;

/**
 * Resource that revolves around a nurse's list of patients that they are watching.
 */
@Service
@Scope("request")
@Path("/nsc-service/facility/{DFN-site-id}/my-patient-list/{user-id}")
public class PatientListResource {
    /**
     * Nurse's vista id for a specific facility.
     */
    @PathParam("user-id")
    private String userId;

    /**
     * Facility that matches the nurse's vista id i.e. facility nurse is working at.
     */
    @PathParam("DFN-site-id")
    private String siteId;

    private final RepositoryFactory repositoryFactory = new RepositoryFactory();

    /**
     * Returns a list of patient ids for patients on the nurse's watch list.  This should be the first
     * call by clients.
     *
     * @return  PatientList with zero or more patient ids and other resource urls.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public PatientList list(@Context UriInfo info, @Context HttpHeaders headers) {
        queueSiteForDischargeProcessing(headers);

        MyPatientsService service = new MyPatientsService(repositoryFactory);

        PatientList patients = service.getList(userId, siteId);
        patients.generateLinks(new Routes(info, siteId, userId));

        return patients;
    }

    /**
     * Kick off discharge process for a site each time a user gets their list of patients.
     *
     * See DischargeProcessBus for more information
     */
    private void queueSiteForDischargeProcessing(HttpHeaders headers) {
        String token = headers.getRequestHeaders().getFirst(HttpHeaders.AUTHORIZATION);

        if (token == null)
            return;

        DischargeProcessBus bus = new DischargeProcessBus();
        bus.post(new AddSiteEvent(siteId, token));
    }

    /**
     * Adds a single patient to a nurse's watch list.
     *
     * @param patient Valid patient from the same facility as the nurse is working.
     *
     * @return  Ok status if patient id was added successfully.  A 400 error will be returned if the patient is already
     *          on the nurse's watch list.
     */
    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    public Response add(@Context UriInfo info, @Context HttpHeaders headers, Person patient) {
        MyPatientsService service = new MyPatientsService(repositoryFactory);
        service.addPatient(userId, siteId, patient);

        return Response.ok().build();
    }

    /**
     * Removes a single patient id from the nurse's watch list.
     *
     * @param patientId Valid patient id from the same facility as the nurse is working.
     *
     * @return Ok status if the patient id was remove or the patient id was not part of the nurse's watch list.
     */
    @DELETE
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/patients/{patient-id}")
    public Response delete(@Context UriInfo info, @Context HttpHeaders headers, @PathParam("patient-id") String patientId) {
        MyPatientsService service = new MyPatientsService(repositoryFactory);
        service.deletePatientId(userId, siteId, patientId);

        return Response.ok().build();
    }

    /**
     * Get a list of all tasks including completed tasks for all patients on a nurse's watch list within a provided
     * time frame.  Any dismissed tasks are not included in this list.
     *
     * @param startDate Required query parameter.  The date must be an UTC timestamp.
     * @param endDate   Required query parameter.  The date must be an UTC timestamp.  This date must be after the start date.
     *
     * @return List of PatientTaskSummary, one per task.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/patients/tasks")
    public List<PatientTaskSummary> getPatientsTasks(@Context UriInfo info, @Context HttpHeaders headers,
                                                     @QueryParam("start-date") Long startDate, @QueryParam("end-date") Long endDate) {
    	if (null == startDate)
    		throw new WebApp400BadRequestException("start-date cannot be null");
    	if (null == endDate)
    		throw new WebApp400BadRequestException("end-date cannot be null");
    	
        MyPatientsService service = new MyPatientsService(repositoryFactory);
        List<PatientTaskSummary> summaries = service.getAllMyPatientsTasks(userId, siteId, new Date(startDate), new Date(endDate));

        for (PatientTaskSummary summary : summaries)
                summary.generateLinks(new Routes(info, siteId, summary.patientId));

        return summaries;
    }

    /**
     * Gets the most urgent patient task status and due date from all the tasks for each patient on the nurse's watch
     * list within a given time range.
     *
     * @param startDate Required query parameter.  The date must be an UTC timestamp.
     * @param endDate   Required query parameter.  The date must be an UTC timestamp.  This date must be after the start date.
     *
     * @return  List of PatientTasksStatus, one per patient on the nurse's watch list.  Data should be filtered on the client.
     */
    @GET
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    @Path("/patients/tasks/status")
    public List<PatientTasksStatus> getPatientsTaskStatus(@Context UriInfo info, @Context HttpHeaders headers,
                                                          @QueryParam("start-date") Long startDate, @QueryParam("end-date") Long endDate) {
    	if (null == startDate)
    		throw new WebApp400BadRequestException("start-date cannot be null");
    	if (null == endDate)
    		throw new WebApp400BadRequestException("end-date cannot be null");
    	
        MyPatientsService service = new MyPatientsService(repositoryFactory);
        List<PatientTasksStatus> patientTasksStatuses = service.getAllPatientsMostUrgentTaskStatus(userId, siteId, new Date(startDate), new Date(endDate));

        for (PatientTasksStatus tasksStatus :  patientTasksStatuses)
                tasksStatus.generateLinks(new Routes(info, siteId, userId));

        return patientTasksStatuses;
    }
}

