package gov.va.vamf.service.shifttransition.watchlist;

import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.completedtasks.CompletedTask;
import gov.va.vamf.service.shifttransition.completedtasks.CompletedTasksService;
import gov.va.vamf.service.shifttransition.tasks.PatientTaskService;
import gov.va.vamf.service.shifttransition.tasks.TaskTransformService;
import gov.va.vamf.service.shifttransition.tasks.domain.time.Range;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTaskSummary;
import gov.va.vamf.service.shifttransition.tasks.representations.PatientTasksStatus;

import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

/**
 * This is an application service used by end points (REST api) to interface with the shift transition domain
 * and repositories.  This application service is used to work with a nurse's patient list and group operations
 * for multiple patients.
 */
public class MyPatientsService {
    private static Logger logger = LoggerFactory.getLogger(MyPatientsService.class);

    private final RepositoryFactory repositoryFactory;

    public MyPatientsService(RepositoryFactory repositoryFactory) {
        this.repositoryFactory = repositoryFactory;
    }

    /**
     * Adds a patient to a nurse's watch list.
     *
     * @param userId        Nurse's vista access code that along with the siteId uniquely identifies a nurse.
     * @param siteId        Along with userId uniquely identifies a nurse.
     * @param patient       Patient to add to the nurse's watch list. Duplicate patient ids are not allowed on the
     *                      watch list.
     */
    public void addPatient(String userId, String siteId, Person patient) {
        patient.validateInput();

        logger.debug("Adding new patient {} for user {} at site {}.", patient.identity.uniqueId, userId, siteId);

        MyPatientsRepository repository = repositoryFactory.getMyPatientsRepository();
        MyPatients patients = repository.get(userId, siteId);

        patients.add(patient);
        repository.save(patients);
    }

    /**
     * Removes a patientId from the patient's watch list.
     *
     * @param userId        Nurse's vista access code that along with the siteId uniquely identifies a nurse.
     * @param siteId        Along with userId uniquely identifies a nurse.
     * @param patientId     Id of patient to remove from to the nurse's watch list.
     */
    public void deletePatientId(String userId, String siteId, String patientId) {
        logger.debug("Deleting patient {} for user {} at site {}.", patientId, userId, siteId);

        MyPatientsRepository repository = repositoryFactory.getMyPatientsRepository();
        MyPatients patients = repository.get(userId, siteId);

        if (patients.remove(patientId))
            repository.save(patients);
    }

    /**
     * Get a nurse's patient watch list.
     *
     * @param userId        Nurse's vista access code that along with the siteId uniquely identifies a nurse.
     * @param siteId        Along with userId uniquely identifies a nurse.
     * @return              List of patientIds.
     */
    public PatientList getList(String userId, String siteId) {
        MyPatients patients = repositoryFactory.getMyPatientsRepository().get(userId, siteId);

        return patients.asPatientList();
    }

    /**
     * Gets the most urgent task status for each patient.  Only one status is returned per patient on the nurse's
     * watch list.
     *
     * @param userId        Nurse's vista access code that along with the siteId uniquely identifies a nurse.
     * @param siteId        Along with userId uniquely identifies a nurse.
     * @param startDate     Part of date range that is used to find the most urgent status for each task.
     * @param endDate       Part of date range that is used to find the most urgent status for each task.
     * @return              List of PatientTasksStatus one per patient.
     */
    public List<PatientTasksStatus> getAllPatientsMostUrgentTaskStatus(String userId, String siteId, Date startDate, Date endDate) {
        logger.debug("Getting most urgent tasks status for user {} at site {}.", userId, siteId);

        PatientList patients = getList(userId, siteId);
        Range range = new Range(startDate, endDate);

        PatientTaskService service = new PatientTaskService(repositoryFactory);

        return service.getMostUrgentTaskStatusForPatients(range,siteId, patients.patientIds());
    }

    /**
     * Gets a task summary for all patient's on a nurse's watch list including completed tasks (not dismissed tasks)
     * within a given range.
     *
     * @param userId        Nurse's vista access code that along with the siteId uniquely identifies a nurse.
     * @param siteId        Along with userId uniquely identifies a nurse.
     * @param startDate     Part of date range that is used to return task summaries.
     * @param endDate       Part of date range that is used to return task summaries.
     * @return              List of scheduled tasks and completed tasks for all patients
     */
    public List<PatientTaskSummary> getAllMyPatientsTasks(String userId, String siteId, Date startDate, Date endDate) {
        logger.debug("Getting all patient tasks status for user {} at site {}.", userId, siteId);

        List<PatientTaskSummary> myPatientsTaskSummaries = Lists.newArrayList();

        PatientList patients = getList(userId, siteId);
        Range range = new Range(startDate, endDate);

        PatientTaskService patientTaskService = new PatientTaskService(repositoryFactory);
        myPatientsTaskSummaries.addAll(patientTaskService.getPatientSummariesForPatients(range,siteId, patients.patientIds()));

        //Completed tasks service call
        CompletedTasksService completedTasksService = new CompletedTasksService(repositoryFactory);
        List<CompletedTask> myPatientsCompletedTasks = completedTasksService.getCompletedTasksForPatients(siteId, startDate, endDate,
                                                                patients.patientIds());

        List<PatientTaskSummary> myPatientsCompletedTaskSummaries = new TaskTransformService()
                                                                            .createFromCompletedTasks(myPatientsCompletedTasks);

        myPatientsTaskSummaries.addAll(myPatientsCompletedTaskSummaries);

        sortPatientTaskSummaries(myPatientsTaskSummaries);

        return myPatientsTaskSummaries;
    }

    private void sortPatientTaskSummaries(List<PatientTaskSummary> myPatientsTaskSummaries) {
        logger.debug("Sorting patient task summaries.");

        Collections.sort(myPatientsTaskSummaries, new Comparator<PatientTaskSummary>() {
            public int compare(PatientTaskSummary o1, PatientTaskSummary o2) {
                if (o1.nextDueDate == null || o2.nextDueDate == null)
                    return 0;
                return o1.nextDueDate.compareTo(o2.nextDueDate);
            }
        });
    }
}
