package gov.va.vamf.service.shifttransition.userpreferences;

import gov.va.vamf.service.shifttransition.application.repositories.RepositoryFactory;
import gov.va.vamf.service.shifttransition.infrastructure.exception.WebApp500InternalServerErrorException;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriInfo;

import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import com.google.common.base.Strings;

/**
 * Resource that revolves around a nurse's preferences.
 */
@Service
@Scope("request")
@Path("/nsc-service/facility/{DFN-site-id}/user/{user-id}/shift-interval")
public class UserPreferencesResource {
	private final RepositoryFactory repositoryFactory = new RepositoryFactory();

	/**
	 * Nurse's vista id for a specific facility.
	 */
	@PathParam("user-id")
	private String staffId;
	
	/**
	 * Site id that combined with a user id uniquely identifies a nurse.
	 */
	@PathParam("DFN-site-id")
	private String siteId;
	
//----------------------------------------------------------------------------------------------------------------------
//---------------------------------GET Methods--------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
	
	/**
	 * Returns preferences for the logged in user (_id, staffId, siteId, startDate, endDate).  If there is no record
	 * saved in the database, JSON will be returned containing staffId and siteId only.
	 * <br/>
	 * <br/>
	 * 
	 * <h3>HTML STATUS CODES</h3>
	 * <table border="1">
	 * 	<tr><th>name</th><th>description</th></tr>
	 * 	<tr>
	 * 		<td>200</td>
	 * 		<td>The request was fulfilled</td>
	 * 	</tr>
	 *  <tr>
	 * 		<td>500</td>
	 * 		<td>Internal Server Error - consult the JSON to see what happened</td>
	 * 	</tr>
	 * </table>
	 * <br/>
	 * <br/>
	 *
	 * @return List<UserPreference> containing zero or more {@link MyPreferences}.
	 */
	@GET
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public MyPreferences list(@Context UriInfo info, @Context HttpHeaders headers) {
		validateUserIdSiteId(staffId, siteId);
		UserPreferencesRepository userPreferencesRepository = repositoryFactory.getUserPreferencesRepository();
		return userPreferencesRepository.get(staffId, siteId);
	}
	
//----------------------------------------------------------------------------------------------------------------------
//---------------------------------POST Methods-------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------
	
	/**
	 * Inserts a MyPreferences object into Mongo (_id, staffId, siteId, startDate, endDate) and returns the object that
	 * was inserted.  The UI can then use the newly populated _id variable for subsequent calls to PUT.<br/>
	 * POST should only be called once in the case where a GET call returns a record without _id populated.
	 * <br/>
	 * <br/>
	 * 
	 * <h3>HTML STATUS CODES</h3>
	 * <table border="1">
	 * 	<tr><th>name</th><th>description</th></tr>
	 * 	<tr>
	 * 		<td>200</td>
	 * 		<td>The request was fulfilled</td>
	 * 	</tr>
	 *  <tr>
	 * 		<td>400</td>
	 * 		<td>Bad Request - Something is wrong with the record or the record already existed - consult the JSON to see what happened</td>
	 * 	</tr>
	 * 	<tr>
	 * 		<td>500</td>
	 * 		<td>Internal Server Error - consult the JSON to see what happened</td>
	 * 	</tr>
	 * </table>
	 * <br/>
	 * <br/>
	 * 
	 * <h3>Validation</h3>
	 * <table border="1">
	 * 	<tr><th>json field</th><th>description</th></tr>
	 * 	<tr>
	 * 		<td>_id</td>
	 * 		<td>Must <b><font color="red">NOT</font></b> be populated.</td>
	 * 	</tr>
	 * 	<tr>
	 * 		<td>staffId</td>
	 * 		<td>Must be populated.</td>
	 * 	</tr>
	 * 	<tr>
	 * 		<td>siteId</td>
	 * 		<td>Must be populated.</td>
	 * 	</tr>
	 * </table>
	 * <br/>
	 * <br/>
	 * 
	 * @param userPreferences represents the @link {@link MyPreferences} to add.
	 * @return The record that was saved - The UI can then use the newly populated _id variable for subsequent calls to PUT.
	 */
	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public MyPreferences insert(@Context UriInfo info, @Context HttpHeaders headers, MyPreferences userPreferences) {
		validateUserIdSiteId(staffId, siteId);
		UserPreferencesRepository userPreferencesRepository = repositoryFactory.getUserPreferencesRepository();
		MyPreferences retvalue = userPreferencesRepository.insert(staffId, siteId, userPreferences);
		return retvalue;
	}

//----------------------------------------------------------------------------------------------------------------------
//---------------------------------PUT Methods--------------------------------------------------------------------------
//----------------------------------------------------------------------------------------------------------------------

	/**
	 * Updates a MyPreferences object in Mongo (_id, staffId, siteId, startDate, endDate) and returns the object that
	 * was updated.  PUT should be called anytime a record is returned from GET that has an _id populated.
	 * <br/>
	 * <br/>
	 * 
	 * <h3>HTML STATUS CODES</h3>
	 * <table border="1">
	 * 	<tr><th>name</th><th>description</th></tr>
	 * 	<tr>
	 * 		<td>200</td>
	 * 		<td>The request was fulfilled</td>
	 * 	</tr>
	 *  <tr>
	 * 		<td>400</td>
	 * 		<td>Bad Request - Something is wrong with the record</td>
	 * 	</tr>
	 * 	<tr>
	 * 		<td>500</td>
	 * 		<td>Internal Server Error - consult the JSON to see what happened</td>
	 * 	</tr>
	 * </table>
	 * <br/>
	 * <br/>
	 * 
	 * <h3>Validation</h3>
	 * <table border="1">
	 * 	<tr><th>json field</th><th>description</th></tr>
	 * 	<tr>
	 * 		<td>_id</td>
	 * 		<td>Must be populated.</td>
	 * 	</tr>
	 * 	<tr>
	 * 		<td>staffId</td>
	 * 		<td>Must be populated.</td>
	 * 	</tr>
	 * 	<tr>
	 * 		<td>siteId</td>
	 * 		<td>Must be populated.</td>
	 * 	</tr>
	 * </table>
	 * <br/>
	 * <br/>
	 * 
	 * @param userPreferences represents the @link {@link MyPreferences} to update.
	 * @return The record that was saved.
	 */
	@PUT
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public MyPreferences update(@Context UriInfo info, @Context HttpHeaders headers, MyPreferences userPreferences) {
		validateUserIdSiteId(staffId, siteId);
		UserPreferencesRepository userPreferencesRepository = repositoryFactory.getUserPreferencesRepository();
		MyPreferences retvalue = userPreferencesRepository.update(staffId, siteId, userPreferences);
		return retvalue;
	}
	
	private void validateUserIdSiteId(String staffId, String siteId) {
		//Because the staffId and siteId come to us from the server, they should in theory always be populated.
		if (Strings.isNullOrEmpty(staffId))
			throw new WebApp500InternalServerErrorException("staffId cannot be empty.");
		if (Strings.isNullOrEmpty(siteId))
			throw new WebApp500InternalServerErrorException("siteId cannot be empty.");
	}
}
